// Copyright (c) 2021 Giorgos Vougioukas
//
// The license can be found in the LICENSE file.

#include "alcyone/definitions.h"
#include "alcyone/app_info.h"
#include "alcyone/preferences.h"
#include "alcyone/client.h"

#include "WDL/heapbuf.h"
#include "WDL/wingui/wndsize.h"
#include "WDL/time_precise.h"

#define RSA_PROGRESS_CREATION 100
#define RSA_PROGRESS_CREATION_MS 50
#define RSA_MAIN_MENU_BAR 140
#define RSA_MAIN_MENU_BAR_MS 500

struct RSA_MainWndData
{
  HWND hwnd;
  UINT msg;
  WPARAM wparam;
  LPARAM lparam;
  int x, y, w, h;
  WDL_WndSizer resize;
  HMENU main_menu;
};

static RSA_MainWndData mwd;

static void WM_InitDialog()
{
  double wall0 = time_precise();

  g_mainwnd = mwd.hwnd;

  if (IsStableRelease())
  {
    SetWindowText(mwd.hwnd, ALCYONE_NAME_MARKETING);
  }
  else
  {
    SetWindowText(mwd.hwnd, ALCYONE_FULL_VERSION " (" ALCYONE_ARCH ")");
  }

  mwd.resize.init(mwd.hwnd);
  //mwd.resize.init_item(IDC_LIST1, 0.0f, 0.0f, 1.0f, 1.0f);

  mwd.main_menu = LoadMenu(g_inst, MAKEINTRESOURCE(IDR_MAIN_MENUBAR));
  SetMenu(mwd.hwnd, mwd.main_menu);

  int w = 0;
  WDL_FastString s;

  mwd.x = GetPrivateProfileInt(ALCYONE_NAME, "main_wnd_x", 50, g_inipath.Get());
  mwd.y = GetPrivateProfileInt(ALCYONE_NAME, "main_wnd_y", 50, g_inipath.Get());
  mwd.w = GetPrivateProfileInt(ALCYONE_NAME, "main_wnd_w", 450, g_inipath.Get());
  mwd.h = GetPrivateProfileInt(ALCYONE_NAME, "main_wnd_h", 50, g_inipath.Get());
  SetWindowPos(mwd.hwnd, NULL, mwd.x, mwd.y, mwd.w, mwd.h, SWP_NOZORDER | SWP_NOACTIVATE);

  int maximized = GetPrivateProfileInt(ALCYONE_NAME, "main_wnd_maximized", 0, g_inipath.Get());

  if (maximized)
  {
    ShowWindow(mwd.hwnd, SW_SHOWMAXIMIZED);
  }
  else
  {
    ShowWindow(mwd.hwnd, SW_SHOWNORMAL);
  }

  g_client = new RSA_Client;

  double wall1 = time_precise();

  if (!IsStableRelease())
  {
    const char warn_msg[] =
    {
      "Welcome to " ALCYONE_NAME_MARKETING " (the \"Software\").\n\n"
      "THIS IS AN UNSTABLE RELEASE STILL IN DEVELOPEMENT PHASE\n"
      "AND IS PROVIDED ON AN \"AS IS\" BASIS AND IS BELIEVED\n"
      "TO CONTAIN DEFECTS AND A PRIMARY PURPOSE OF THIS\n"
      "RELEASE IS TO OBTAIN FEEDBACK ON SOFTWARE PERFORMANCE\n"
      "AND THE IDENTIFICATION OF DEFECTS.\n\n"
      "IT IS ADVISED TO USE CAUTION AND NOT TO RELY IN ANY WAY\n"
      "ON THE CORRECT FUNCTIONING OF THIS RELEASE.\n\n"
      ALCYONE_NAME_MARKETING " version: " ALCYONE_NAKED_VERSION " (" ALCYONE_ARCH ")\n\n"
      "For the latest production release visit: " ALCYONE_WEBSITE_URL "\n\n"
      "Startup time: %.3f\n\n"
      ALCYONE_COPYRIGHT
    };

    WDL_FastString tmp;

    tmp.SetFormatted(1024, warn_msg, wall1 - wall0);

    MessageBox(mwd.hwnd, tmp.Get(), "Release for testing purposes", MB_OK);
  }

  int compressor = GetPrivateProfileInt(ALCYONE_NAME, "compressor", 1, g_inipath.Get());
  g_client->SetCompressor(compressor);

  SetTimer(mwd.hwnd, RSA_PROGRESS_CREATION, RSA_PROGRESS_CREATION_MS, NULL);
  SetTimer(mwd.hwnd, RSA_MAIN_MENU_BAR, RSA_MAIN_MENU_BAR_MS, NULL);
}

static void WM_SysCommand()
{
  if (mwd.wparam == SC_CLOSE)
  {
    SendMessage(mwd.hwnd, WM_COMMAND, ID_FILE_EXIT, 0);
  }
}

static int WM_Close()
{
#if defined(__linux__)
  has_requested_quit = true;
#endif
  return 0;
}

static void WM_Size()
{
  if (mwd.wparam != SIZE_MINIMIZED)
  {
    //m_title_capture = false;
    mwd.resize.onResize();
  }

  if (mwd.wparam != SIZE_MINIMIZED && mwd.wparam != SIZE_MAXIMIZED)
  {
    RECT r;
    GetWindowRect(mwd.hwnd, &r);

    mwd.x = r.left;
    mwd.y = r.top;
    mwd.w = r.right - r.left;
    mwd.h = r.bottom - r.top;
  }
}

static void WM_Move()
{
  int xpos = (int)(short) LOWORD(mwd.lparam); // horizontal position
  int ypos = (int)(short) HIWORD(mwd.lparam); // vertical position

  if (xpos >= 0 && ypos >= 0 && GetCapture() == mwd.hwnd)
  {
    RECT r;
    GetWindowRect(mwd.hwnd, &r);

    mwd.x = r.left;
    mwd.y = r.top;
    mwd.w = r.right - r.left;
    mwd.h = r.bottom - r.top;
  }
}

static int WM_Destroy()
{
  KillTimer(mwd.hwnd, RSA_PROGRESS_CREATION);
  KillTimer(mwd.hwnd, RSA_MAIN_MENU_BAR);
#if defined(_WIN32)
  WINDOWPLACEMENT wp = { sizeof(wp) };
  GetWindowPlacement(mwd.hwnd, &wp);

  //if (wp.showCmd != SW_MINIMIZE && wp.showCmd != SW_SHOWMINIMIZED &&
  //  wp.showCmd != SW_MAXIMIZE && wp.showCmd != SW_SHOWMAXIMIZED)
  //{
  //  RECT r;
  //  GetWindowRect(m_hwnd, &r);

  //  g_ini_file->write_int("main_wnd_x", r.left, ALCYONE_NAME);
  //  g_ini_file->write_int("main_wnd_y", r.top, ALCYONE_NAME);
  //  g_ini_file->write_int("main_wnd_w", r.right - r.left, ALCYONE_NAME);
  //  g_ini_file->write_int("main_wnd_h", r.bottom - r.top, ALCYONE_NAME);
  //}

  WDL_FastString xstr, ystr, wstr, hstr;
  xstr.SetFormatted(32, "%d", mwd.x);
  ystr.SetFormatted(32, "%d", mwd.y);
  wstr.SetFormatted(32, "%d", mwd.w);
  hstr.SetFormatted(32, "%d", mwd.h);

  WritePrivateProfileString(ALCYONE_NAME, "main_wnd_x", xstr.Get(), g_inipath.Get());
  WritePrivateProfileString(ALCYONE_NAME, "main_wnd_y", ystr.Get(), g_inipath.Get());
  WritePrivateProfileString(ALCYONE_NAME, "main_wnd_w", wstr.Get(), g_inipath.Get());
  WritePrivateProfileString(ALCYONE_NAME, "main_wnd_h", hstr.Get(), g_inipath.Get());

  if (wp.showCmd == SW_SHOWMAXIMIZED)
  {
    WritePrivateProfileString(ALCYONE_NAME, "main_wnd_maximized", "1", g_inipath.Get());
  }
  else
  {
    WritePrivateProfileString(ALCYONE_NAME, "main_wnd_maximized", "0", g_inipath.Get());
  }
#else
  RECT r;
  GetWindowRect(mwd.hwnd, &r);
  if (r.left >= 0 && r.top >= 0)
  {
    WDL_FastString xstr, ystr, wstr, hstr;
    xstr.SetFormatted(32, "%d", r.left);
    ystr.SetFormatted(32, "%d", r.top);
    wstr.SetFormatted(32, "%d", r.right - r.left);
    hstr.SetFormatted(32, "%d", r.bottom - r.top);

    WritePrivateProfileString(ALCYONE_NAME, "main_wnd_x", xstr.Get(), g_inipath.Get());
    WritePrivateProfileString(ALCYONE_NAME, "main_wnd_y", ystr.Get(), g_inipath.Get());
    WritePrivateProfileString(ALCYONE_NAME, "main_wnd_w", wstr.Get(), g_inipath.Get());
    WritePrivateProfileString(ALCYONE_NAME, "main_wnd_h", hstr.Get(), g_inipath.Get());

    //g_ini_file->write_int("main_wnd_x", r.left, ALCYONE_NAME);
    //g_ini_file->write_int("main_wnd_y", r.top, ALCYONE_NAME);
    //g_ini_file->write_int("main_wnd_w", r.right - r.left, ALCYONE_NAME);
    //g_ini_file->write_int("main_wnd_h", r.bottom - r.top, ALCYONE_NAME);
  }
  //g_ini_file->write_int("main_wnd_x", m_x, ALCYONE_NAME);
  //g_ini_file->write_int("main_wnd_y", m_y, ALCYONE_NAME);
  //g_ini_file->write_int("main_wnd_w", m_w, ALCYONE_NAME);
  //g_ini_file->write_int("main_wnd_h", m_h, ALCYONE_NAME);
#endif

  WDL_FastString compstr;
  int compressor = g_client->GetCompressor();
  compstr.SetFormatted(32, "%d", compressor);
  WritePrivateProfileString(ALCYONE_NAME, "compressor", compstr.Get(), g_inipath.Get());

  if (g_client) delete g_client;

#if defined(_WIN32)
  PostQuitMessage(0);
#elif defined(__APPLE__)
  SWELL_PostQuitMessage(m_hwnd);
#endif

  g_mainwnd = NULL;

#if defined(__linux__)
  has_requested_quit = true;
#endif

  return 0;
}

static void WM_Command()
{
  switch (LOWORD(mwd.wparam))
  {
  case ID_FILE_COMPRESSFOLDER:
    {
      g_client->CompressFolder();
    }
    break;
  case ID_FILE_COMPRESSFILE:
    {
      g_client->CompressFile();
    }
    break;
  case ID_FILE_DECOMPRESSFILE:
    {
      g_client->DecompressFile();
    }
    break;
  case ID_FILE_EXIT:
    {
      DestroyWindow(mwd.hwnd);
    }
    break;
  case ID_OPTIONS_ZLIB:
    {
      g_client->SetCompressor(0);
    }
    break;
  case ID_OPTIONS_ZSTD:
    {
      g_client->SetCompressor(1);
    }
    break;
  case ID_OPTIONS_PREFERENCES:
    {
      if (!g_preferences->WantCurses())
      {
        g_preferences->Open();
      }
      else
      {
        if (!g_prefwnd)
        {
          CreateDialog(g_inst, MAKEINTRESOURCE(IDD_PREFERENCES), mwd.hwnd, RSA_PreferencesWndProc);
          ShowWindow(g_prefwnd, SW_SHOW);
        }
        else
        {
          SetFocus(g_prefwnd);
        }
      }
    }
    break;
  case ID_HELP_DOCUMENTATION:
    {
      WDL_FastString doc;

      doc.Set(g_modpath.Get());
      doc.Append("documentation.txt");

      ShellExecute(mwd.hwnd, "", "notepad.exe", doc.Get(), "", SW_SHOWNORMAL);
    }
    break;
  case ID_HELP_ABOUTALCYONE:
    {
      if (!g_aboutwnd)
      {
        CreateDialog(g_inst, MAKEINTRESOURCE(IDD_ABOUT), g_aboutwnd, RSA_AboutWndProc);
        ShowWindow(g_aboutwnd, SW_SHOW);
      }
      else
      {
        SetFocus(g_aboutwnd);
      }
    }
    break;
  case ID_HELP_CHANGELOG:
    {
      if (!g_changelogwnd)
      {
        CreateDialog(g_inst, MAKEINTRESOURCE(IDD_CHANGELOG), mwd.hwnd, RSA_ChangelogWndProc);
        ShowWindow(g_changelogwnd, SW_SHOW);
      }
      else
      {
        SetFocus(g_changelogwnd);
      }
    }
    break;
  case ID_HELP_VERSION:
    {
      const char version_string[] =
      {
        ALCYONE_NAME_MARKETING " version: " ALCYONE_NAKED_VERSION " (" ALCYONE_ARCH ")\n\n"
        "Build timestamp: " ALCYONE_TIMESTAMP "\n"
        "Git commit: " ALCYONE_GIT_SHA
      };

      MessageBox(mwd.hwnd, version_string, "Version details", MB_OK);
    }
    break;
  case ID_HELP_OFFICIALWEBSITE:
    {
      ShellExecute(mwd.hwnd, "", ALCYONE_WEBSITE_URL, "", "", SW_SHOWNORMAL);
    }
    break;
  case ID_HELP_LICENSE:
    {
      if (!g_licwnd)
      {
        CreateDialog(g_inst, MAKEINTRESOURCE(IDD_LICENSE), mwd.hwnd, RSA_LicenseWndProc);
        ShowWindow(g_licwnd, SW_SHOW);
      }
      else
      {
        SetFocus(g_licwnd);
      }
    }
    break;
  }
}

static void WM_Timer()
{
  if (mwd.wparam == RSA_PROGRESS_CREATION)
  {
    if (!g_progwnd && g_client->IsInProgress())
    {
      DialogBox(g_inst, MAKEINTRESOURCE(IDD_PROGRESS), mwd.hwnd, RSA_ProgressWndProc);
    }
    else if (g_progwnd && !g_client->IsInProgress())
    {
      EndDialog(g_progwnd, 0);
    }
  }
  if (mwd.wparam == RSA_MAIN_MENU_BAR)
  {
    int compressor = g_client->GetCompressor();
    switch (compressor)
    {
    case 0:
      {
        CheckMenuItem(GetMenu(mwd.hwnd), ID_OPTIONS_ZLIB, MF_CHECKED);
        CheckMenuItem(GetMenu(mwd.hwnd), ID_OPTIONS_ZSTD, MF_UNCHECKED);
      }
      break;
    case 1:
      {
        CheckMenuItem(GetMenu(mwd.hwnd), ID_OPTIONS_ZLIB, MF_UNCHECKED);
        CheckMenuItem(GetMenu(mwd.hwnd), ID_OPTIONS_ZSTD, MF_CHECKED);
      }
      break;
    }
  }
}

WDL_DLGRET RSA_MainWndProc(HWND hwnd, UINT msg, WPARAM wparam, LPARAM lparam)
{
  mwd.hwnd = hwnd;
  mwd.msg = msg;
  mwd.wparam = wparam;
  mwd.lparam = lparam;

#if defined(__linux__)
  if (msg == WM_KEYDOWN && LOWORD(wparam) == VK_RETURN)
  {
    SendMessage(mwd.hwnd, WM_COMMAND, MAKEWPARAM(IDOK, 0), 0);
    return 0;
  }
#endif

  switch(msg)
  {
  case WM_INITDIALOG: WM_InitDialog(); break;
  case WM_SYSCOMMAND: WM_SysCommand(); break;
  case WM_CLOSE: return WM_Close(); break;
  case WM_SIZE: WM_Size(); break;
  case WM_MOVE: WM_Move(); break;
  case WM_DESTROY: return WM_Destroy(); break;
  case WM_COMMAND: WM_Command(); break;
  case WM_TIMER: WM_Timer(); break;
  }

  return 0;
}

#include "WDL/eel2/ns-eel.h"
#include "WDL/lice/lice.h"
#include "WDL/wingui/virtwnd-controls.h"
#include "WDL/wingui/scrollbar/coolscroll.h"

// an app should implement these
int WDL_STYLE_WantGlobalButtonBorders()
{
  return 0;
}

bool WDL_STYLE_WantGlobalButtonBackground(int *col)
{
  return false;
}

int WDL_STYLE_GetSysColor(int p)
{
  return GetSysColor(p);
}

void WDL_STYLE_ScaleImageCoords(int *x, int *y)
{}

bool WDL_Style_WantTextShadows(int *col)
{
  return false;
}

// this is the default, you can override per painter if you want.
// return values 0.0-1.0 for each, return false if no gradient desired
bool WDL_STYLE_GetBackgroundGradient(double *gradstart, double *gradslope)
{
  return false;
}

// for slider
LICE_IBitmap *WDL_STYLE_GetSliderBitmap2(bool vert)
{
  return NULL;
}

bool WDL_STYLE_AllowSliderMouseWheel()
{
  return false;
}

int WDL_STYLE_GetSliderDynamicCenterPos()
{
  return 0;
}

// TO BE IMPLEMENTED BY APP:
// implemented by calling app, can return a LICE_IBitmap **img for "scrollbar"
void *GetIconThemePointer(const char *name)
{
  if (!strcmp(name, "scrollbar"))
  {}

  return NULL;
}

// can be a passthrough to GetSysColor()
int CoolSB_GetSysColor(HWND m_hwnd, int val)
{
  return GetSysColor(val);
}

#include <WDL/eel2/ns-eel.h>

void NSEEL_HOSTSTUB_EnterMutex()
{}

void NSEEL_HOSTSTUB_LeaveMutex()
{}
