// Copyright (c) 2021 Giorgos Vougioukas
//
// The license can be found in the LICENSE file.

#include "alcyone/preferences.h"
#include "alcyone/definitions.h"
#include "alcyone/app_info.h"
#include "third_party/zstd/lib/zstd.h"

#include "WDL/fileread.h"
#include "WDL/filewrite.h"

static const char *default_preferences[] =
{
  "//",
  "// Welcome to " ALCYONE_NAME_MARKETING " " ALCYONE_ARCH " File Archiver, Version " ALCYONE_NAKED_VERSION,
  "//",
  "// Alcyone preferences",
  "//",
  "",
  "{",
  "  // Display preferences with Curses interface",
  "  // (supports only ASCII for text input).",
  "  // If set to false it uses an external editor.",
  "  \"curses_preferences\": true,",
  "  \"curses_right_focus\": true,",
  "",
  "  // Set zstd compression level (0-19).",
  "  // 0 level indicates no compression.",
  "  // 3 level is the zstd default.",
  "  \"zstd_compression_level\": 1,",
  "",
  "  // 32-bit checksum at end of frame.",
  "  \"zstd_enable_checksum\": false,",
  "",
  "  // Tarball.",
  "  \"zstd_keep_tarball\": false,",
  "",
  "  // Set zlib compression level (0-9).",
  "  // 0 level indicates no compression.",
  "  // 6 level is the zlib default.",
  "  \"zlib_compression_level\": 1,",
  "",
  "  // Action after compression.",
  "  // 0: Do nothing.",
  "  // 1: Ask where to save compressed file.",
  "  // 2: Open file explorer at related folder path.",
  "  \"action_after_compression\": 0,",
  "",
  "  // Action after decompression.",
  "  // 0: Do nothing.",
  "  // 1: Open file explorer at related folder path.",
  "  \"action_after_decompression\": 0,",
  "",
  "  // File dialog.",
  "  \"preserve_current_directory\": true",
  "}",
};

RSA_Preferences::RSA_Preferences()
  : m_pf(NULL)
  , m_relaunch(false)
{}

RSA_Preferences::~RSA_Preferences()
{
  cJSON_Delete(m_pf);
}

void RSA_Preferences::Create()
{
  WDL_FastString defpref, usrpref;

  for (int i = 0; i < (int)sizeof(default_preferences) / (int)sizeof(default_preferences[0]); i++)
  {
    defpref.AppendFormatted(2048, "%s\n", default_preferences[i]);
  }

  const char *user_preferences[] =
  {
    "{",
    "  // Add your personal preferences here.",
    "  // Your changes will take effect the next time you relaunch Alcyone.",
    "}"
  };

  for (int j = 0; j < (int)sizeof(user_preferences) / (int)sizeof(user_preferences[0]); j++)
  {
    usrpref.AppendFormatted(2048, "%s\n", user_preferences[j]);
  }

  WDL_FastString def(g_setpath), usr(g_setpath);
  def.Append("preferences-default.txt");
  usr.Append("preferences.txt");

  // Create default preferences file
  WDL_FileWrite *cdf = new WDL_FileWrite(def.Get());
  if (cdf && cdf->IsOpen())
  {
    cdf->Write(defpref.Get(), defpref.GetLength());
  }

  if (cdf) { delete cdf; cdf = NULL; }

  // Check if user preferences exist
  WDL_FileRead *uf = new WDL_FileRead(usr.Get());
  if (uf && !uf->IsOpen())
  {
    // Create user preferences file
    WDL_FileWrite *cuf = new WDL_FileWrite(usr.Get());
    if (cuf && cuf->IsOpen())
    {
      cuf->Write(usrpref.Get(), usrpref.GetLength());
    }

    if (cuf) { delete cuf; cuf = NULL; }
  }

  if (uf) { delete uf; uf = NULL; }
}

void RSA_Preferences::Open(bool default_file)
{
  WDL_FastString def(g_setpath), usr(g_setpath);
  def.Append("preferences-default.txt");
  usr.Append("preferences.txt");

  ShellExecute(g_mainwnd, "", "notepad.exe", default_file ? def.Get() : usr.Get(), "", SW_SHOWNORMAL);
}

bool RSA_Preferences::Parse(bool default_file)
{
  WDL_FastString def(g_setpath), usr(g_setpath);
  def.Append("preferences-default.txt");
  usr.Append("preferences.txt");

  WDL_FileRead fr(default_file ? def.Get() : usr.Get());
  WDL_TypedBuf<char> strbuf;
  strbuf.Resize((int)fr.GetSize());
  fr.Read(strbuf.Get(), strbuf.GetSize());
  strbuf.Add('\0');

  cJSON_Minify(strbuf.Get());
  m_pf = cJSON_Parse(strbuf.Get());

  if (!m_pf)
  {
    const char *error_ptr = cJSON_GetErrorPtr();
    if (error_ptr)
    {
      m_err.SetFormatted(2048, "Parse error before: %s\n", error_ptr);
      wdl_log(m_err.Get());
    }

    return false;
  }

  return true;
}

bool RSA_Preferences::WantCurses() const
{
  const cJSON *curses = NULL;
  curses = cJSON_GetObjectItemCaseSensitive(m_pf, "curses_preferences");
  if (cJSON_IsTrue(curses)) return true;
  else if (cJSON_IsFalse(curses)) return false;
  else return true;
}

bool RSA_Preferences::WantCursesRightFocus() const
{
  const cJSON *curses_right_focus = NULL;
  curses_right_focus = cJSON_GetObjectItemCaseSensitive(m_pf, "curses_right_focus");
  if (cJSON_IsTrue(curses_right_focus)) return true;
  else if (cJSON_IsFalse(curses_right_focus)) return false;
  else return true;
}

bool RSA_Preferences::WantPreserveCurrentDirectory() const
{
  const cJSON *preserve_current_directory = NULL;
  preserve_current_directory = cJSON_GetObjectItemCaseSensitive(m_pf, "preserve_current_directory");
  if (cJSON_IsTrue(preserve_current_directory)) return true;
  else if (cJSON_IsFalse(preserve_current_directory)) return false;
  else return true;
}

int RSA_Preferences::GetZstdCompressionLevel() const
{
  const cJSON *zstd_compression_level = NULL;
  zstd_compression_level = cJSON_GetObjectItemCaseSensitive(m_pf, "zstd_compression_level");
  if (cJSON_IsNumber(zstd_compression_level))
  {
    if (zstd_compression_level->valueint == 0)
    {
      return ZSTD_minCLevel();
    }
    if (zstd_compression_level->valueint >= 1 &&
      zstd_compression_level->valueint <= 19)
    {
      return zstd_compression_level->valueint;
    }
    else return 1;
  }
  else return 1;
}

bool RSA_Preferences::WantZstdKeepTarball() const
{
  const cJSON *zstd_keep_tarball = NULL;
  zstd_keep_tarball = cJSON_GetObjectItemCaseSensitive(m_pf, "zstd_keep_tarball");
  if (cJSON_IsTrue(zstd_keep_tarball)) return true;
  else if (cJSON_IsFalse(zstd_keep_tarball)) return false;
  else return false;
}

int RSA_Preferences::GetZstdChecksumFlag() const
{
  const cJSON *zstd_enable_checksum = NULL;
  zstd_enable_checksum = cJSON_GetObjectItemCaseSensitive(m_pf, "zstd_enable_checksum");
  if (cJSON_IsTrue(zstd_enable_checksum)) return 1;
  else if (cJSON_IsFalse(zstd_enable_checksum)) return 0;
  else return 0;
}

int RSA_Preferences::GetZlibCompressionLevel() const
{
  const cJSON *zlib_compression_level = NULL;
  zlib_compression_level = cJSON_GetObjectItemCaseSensitive(m_pf, "zlib_compression_level");
  if (cJSON_IsNumber(zlib_compression_level))
  {
    if (zlib_compression_level->valueint >= 0 &&
      zlib_compression_level->valueint <= 9)
    {
      return zlib_compression_level->valueint;
    }
    else return 1;
  }
  else return 1;
}

int RSA_Preferences::GetActionAfterCompression() const
{
  const cJSON *action_after_compression = NULL;
  action_after_compression = cJSON_GetObjectItemCaseSensitive(m_pf, "action_after_compression");
  if (cJSON_IsNumber(action_after_compression))
  {
    if (action_after_compression->valueint >= 0 &&
      action_after_compression->valueint <= 2)
    {
      return action_after_compression->valueint;
    }
    else return 0;
  }
  else return 0;
}

int RSA_Preferences::GetActionAfterDecompression() const
{
  const cJSON *action_after_decompression = NULL;
  action_after_decompression = cJSON_GetObjectItemCaseSensitive(m_pf, "action_after_decompression");
  if (cJSON_IsNumber(action_after_decompression))
  {
    if (action_after_decompression->valueint >= 0 &&
      action_after_decompression->valueint <= 1)
    {
      return action_after_decompression->valueint;
    }
    else return 0;
  }
  else return 0;
}
