// Copyright (c) 2021 Giorgos Vougioukas
//
// The license can be found in the LICENSE file.

#include "alcyone/client.h"
#include "alcyone/preferences.h"

#include <stdlib.h> // malloc, free, exit
#include <stdio.h> // fprintf, perror, fopen, etc.
#include <string.h> // strerror
#include <errno.h> // errno
#include <sys/stat.h> // stat

#include "WDL/filebrowse.h"
#include "WDL/dirscan.h"
#include "WDL/wdlcstring.h"
#include "WDL/wdlutf8.h"

//#define RSA_ENABLE_ZSTD_CHECKS 1

#define RSA_COMPRESS_FILES \
  "All files (*.*)\0*.*\0\0"
#define RSA_COMPRESS_FILES_DEFEXT "txt"

#define RSA_DECOMPRESS_FILE \
  "All supported files\0*.zip;*.tzst\0" \
  "ZIP files (*.zip)\0*.zip\0" \
  "Zstandard files (*.tzst)\0*.tzst\0" \
  "All files (*.*)\0*.*\0\0"
#define RSA_DECOMPRESS_FILE_DEFEXT "zip"

#define RSA_SAVE_COMPRESSED_FILE \
  "Zstandard file (*.tzst)\0*.tzst\0" \
  "ZIP file (*.zip)\0*.zip\0" \
  "All files (*.*)\0*.*\0\0"
//#define RSA_SAVE_COMPRESSED_FILE_DEFEXT "tzst"

// Define the returned error code from utility functions.
typedef enum
{
  ERROR_fsize = 1,
  ERROR_fopen = 2,
  ERROR_fclose = 3,
  ERROR_fread = 4,
  ERROR_fwrite = 5,
  ERROR_loadFile = 6,
  ERROR_saveFile = 7,
  ERROR_malloc = 8,
  ERROR_largeFile = 9,
} COMMON_ErrorCode;

// Check that the condition holds. If it doesn't print m_a message and die.
#define CHECK(cond, ...) \
  do \
  { \
    if (!(cond)) \
    { \
      fprintf(stderr, \
        "%s:%d CHECK(%s) failed: ", \
        __FILE__, \
        __LINE__, \
        #cond); \
      fprintf(stderr, "" __VA_ARGS__); \
      fprintf(stderr, "\n"); \
      exit(1); \
    } \
  } while (0)

// Check the zstd error code and die if an error occurred after printing m_a
// message.
#define CHECK_ZSTD(fn, ...) \
  do \
  { \
    size_t const err = (fn); \
    CHECK(!ZSTD_isError(err), "%s", ZSTD_getErrorName(err)); \
  } while (0)

RSA_Client::RSA_Client()
  : m_selfile(NULL)
  , m_compressfolder(false)
  , m_compressfile(false)
  , m_decompressfile(false)
  , m_thread(NULL)
  , m_killthread(false)
  , m_fileread(NULL)
  , m_filewrite(NULL)
  , m_zstdcctx(NULL)
  , m_a(NULL)
  , m_zstdclevel(3)
  , m_zstdchecksum(0)
  , m_zstddctx(NULL)
  , m_perc1(0)
  , m_perc2(0)
  , m_indexfile(0)
  , m_totalfiles(0)
  , m_inprogress(false)
  , m_abort(false)
  , m_compressor(0)
  , m_zf(NULL)
  , m_zlibclevel(6)
{
  m_zstdclevel = g_preferences->GetZstdCompressionLevel();
  m_zstdchecksum = g_preferences->GetZstdChecksumFlag();
  m_zlibclevel = g_preferences->GetZlibCompressionLevel();
  StartThread();
}

RSA_Client::~RSA_Client()
{
  StopThread();
  if (m_selfile) free(m_selfile);
  if (m_zstdcctx) ZSTD_freeCCtx(m_zstdcctx);
  if (m_zstddctx) ZSTD_freeDCtx(m_zstddctx);
}

void RSA_Client::CompressFolder()
{
  m_seldir.Resize(2048);

  bool preservecwd = g_preferences->WantPreserveCurrentDirectory();
  if (WDL_ChooseDirectory(g_mainwnd, "Compress folder", NULL,
    m_seldir.Get(), m_seldir.GetSize(), preservecwd))
  {
    m_compressfolder = true;
  }
}

void RSA_Client::CompressFile()
{
  if (m_selfile) { free(m_selfile); m_selfile = NULL; }

  bool preservecwd = g_preferences->WantPreserveCurrentDirectory();
  m_selfile = WDL_ChooseFileForOpen(g_mainwnd, "Compress file", NULL, NULL,
    RSA_COMPRESS_FILES, RSA_COMPRESS_FILES_DEFEXT, preservecwd, true);

  if (m_selfile)
  {
    m_compressfile = true;
  }
}

void RSA_Client::DecompressFile()
{
  if (m_selfile) { free(m_selfile); m_selfile = NULL; }

  bool preservecwd = g_preferences->WantPreserveCurrentDirectory();
  m_selfile = WDL_ChooseFileForOpen(g_mainwnd, "Decompress file", NULL, NULL,
    RSA_DECOMPRESS_FILE, RSA_DECOMPRESS_FILE_DEFEXT, preservecwd, false);

  if (m_selfile)
  {
    m_decompressfile = true;
  }
}

void RSA_Client::SetCompressor(int val)
{
  m_compressor = val;
}

int RSA_Client::GetCompressor() const
{
  return m_compressor;
}

bool RSA_Client::IsInProgress() const
{
  return m_inprogress;
}

void RSA_Client::AbortProgress()
{
  m_abort = true;
  m_inprogress = false;
}

void RSA_Client::Progress(int *perc1, int *perc2, WDL_FastString *stage) const
{
  *perc1 = m_perc1;
  *perc2 = m_perc2;
  *stage = &m_stage;
}

void RSA_Client::OnZstdCompressFolder()
{
  m_tarfn.Set(m_seldir.Get());
  m_tarfn.Append(".tar");

  m_a = archive_write_new();

  archive_write_add_filter_none(m_a);
  archive_write_set_format_gnutar(m_a);

  WCHAR widetarfn[2048];
  WDL_MBtoWideStr(widetarfn, m_tarfn.Get(), sizeof(widetarfn));
  int ret = archive_write_open_filename_w(m_a, widetarfn);
  if (ret != ARCHIVE_OK)
  {
    archive_write_close(m_a);
    archive_write_free(m_a);
    return;
  }

  m_perc1 = 0;
  m_perc2 = 0;
  m_abort = false;
  m_inprogress = true;
  m_indexfile = 0;
  m_totalfiles = 0;
  PrepareFiles(m_seldir.Get());
  ZstdScanFiles(m_seldir.Get());
  archive_write_close(m_a);
  archive_write_free(m_a);
  ZstdCompressArchive();
  m_inprogress = false;

  WDL_FastString cfn(m_tarfn.Get());
  cfn.remove_fileext();
  cfn.Append(".tzst");
  ActionAfterCompression(cfn.Get(), "tzst");
}

void RSA_Client::OnZstdCompressFile()
{
  m_abort = false;
  WDL_FastString path;

  const char *only_one = m_selfile;

  while (*only_one++);

  m_stage.Set("Writing tarball (1/2)");

  if (*only_one == '\0' && *(only_one + 1) == '\0')
  {
    path.Set(m_selfile);
    path.remove_fileext();
    m_tarfn.Set(path.Get());
    m_tarfn.Append(".tar");

    m_perc1 = 0;
    m_perc2 = 0;
    m_inprogress = true;
    WDL_FastString fn(m_selfile);
    WDL_FastString nakedfn(fn.get_filepart());

    m_a = archive_write_new();

    archive_write_add_filter_none(m_a);
    archive_write_set_format_gnutar(m_a);

    WCHAR widetarfn[2048];
    WDL_MBtoWideStr(widetarfn, m_tarfn.Get(), sizeof(widetarfn));
    int ret = archive_write_open_filename_w(m_a, widetarfn);
    if (ret != ARCHIVE_OK)
    {
      m_inprogress = false;
      archive_write_fail(m_a);
      archive_write_free(m_a);
      return;
    }

    WDL_HeapBuf data;
    data.Resize(4096);
    WDL_FileRead fr(fn.Get());

    if (fr.IsOpen())
    {
      struct archive *disk = archive_read_disk_new();
      //archive_read_disk_set_standard_lookup(disk);

      WCHAR widefn[2048];
      WDL_MBtoWideStr(widefn, fn.Get(), sizeof(widefn));
      ret = archive_read_disk_open_w(disk, widefn);
      if (ret != ARCHIVE_OK)
      {
        m_inprogress = false;
        archive_read_close(disk);
        archive_read_free(disk);
        archive_write_fail(m_a);
        archive_write_free(m_a);
        return;
      }

      struct archive_entry *ae = archive_entry_new2(m_a);
      ret = archive_read_next_header2(disk, ae);
      if (ret != ARCHIVE_OK)
      {
        m_inprogress = false;
        archive_entry_free(ae);
        archive_read_close(disk);
        archive_read_free(disk);
        archive_write_fail(m_a);
        archive_write_free(m_a);
        return;
      }

      archive_read_disk_descend(disk);

      //WCHAR widenakedfn[2048];
      //WDL_MBtoWideStr(widenakedfn, nakedfn.Get(), sizeof(widenakedfn));
      archive_entry_set_pathname(ae, nakedfn.Get());
      //archive_entry_set_pathname_utf8(ae, fn.Get());
      //archive_entry_set_symlink_type(ae, AE_SYMLINK_TYPE_FILE);
      //archive_entry_set_symlink_utf8(ae, nakedfn.Get());
      //archive_entry_set_size(ae, fr.GetSize());
      ret = archive_write_header(m_a, ae);
      if (ret != ARCHIVE_OK)
      {
        m_inprogress = false;
        archive_entry_free(ae);
        archive_read_close(disk);
        archive_read_free(disk);
        archive_write_fail(m_a);
        archive_write_free(m_a);
        return;
      }

      do
      {
        int read = fr.Read(data.Get(), data.GetSize());
        archive_write_data(m_a, data.Get(), read);
        m_perc1 = 100;
        m_perc2 = (int)(100 * fr.GetPosition() /
          wdl_clamp(fr.GetSize(), 1, fr.GetSize()));
      } while (fr.GetPosition() < fr.GetSize() && !m_abort);

      if (!m_abort)
      {
        archive_entry_free(ae);
        archive_read_close(disk);
        archive_read_free(disk);
        archive_write_close(m_a);
        archive_write_free(m_a);
        ZstdCompressArchive();
        m_inprogress = false;

        WDL_FastString cfn(m_tarfn.Get());
        cfn.remove_fileext();
        cfn.Append(".tzst");
        ActionAfterCompression(cfn.Get(), "tzst");
      }
      else
      {
        m_inprogress = false;
        archive_entry_free(ae);
        archive_read_close(disk);
        archive_read_free(disk);
        archive_write_fail(m_a);
        archive_write_free(m_a);
      }
    }
  }
  else
  {
    path.Set(m_selfile);
    path.Append(WDL_DIRCHAR_STR);

    char *s = m_selfile;
    while (*s++);

    m_tarfn.Set(path.Get());
    m_tarfn.Append(s);
    m_tarfn.remove_fileext();
    m_tarfn.Append(".tar");

    m_perc1 = 0;
    m_perc2 = 0;
    m_inprogress = true;

    WDL_PtrList_DeleteOnDestroy<WDL_FastString> filelist;

    do
    {
      WDL_FastString *fn = new WDL_FastString(path.Get());
      fn->Append(s);
      filelist.Add(fn);
      while (*s++);
    } while (*s != '\0' && *(s + 1) != '\0' && !m_abort);

    m_a = archive_write_new();

    archive_write_add_filter_none(m_a);
    archive_write_set_format_gnutar(m_a);

    WCHAR widetarfn[2048];
    WDL_MBtoWideStr(widetarfn, m_tarfn.Get(), sizeof(widetarfn));
    int ret = archive_write_open_filename_w(m_a, widetarfn);
    if (ret != ARCHIVE_OK)
    {
      m_inprogress = false;
      archive_write_fail(m_a);
      archive_write_free(m_a);
      return;
    }

    for (int i = 0; i < filelist.GetSize(); i++)
    {
      WDL_FastString *fn = filelist.Get(i);
      WDL_FastString nakedfn(fn->get_filepart());

      WDL_HeapBuf data;
      data.Resize(4096);
      WDL_FileRead fr(fn->Get());

      if (fr.IsOpen())
      {
        struct archive *disk = archive_read_disk_new();
        //archive_read_disk_set_standard_lookup(disk);

        WCHAR widefn[2048];
        WDL_MBtoWideStr(widefn, fn->Get(), sizeof(widefn));
        ret = archive_read_disk_open_w(disk, widefn);
        if (ret != ARCHIVE_OK)
        {
          m_inprogress = false;
          archive_read_close(disk);
          archive_read_free(disk);
          archive_write_fail(m_a);
          archive_write_free(m_a);
          return;
        }

        struct archive_entry *ae = archive_entry_new2(m_a);

        ret = archive_read_next_header2(disk, ae);
        if (ret != ARCHIVE_OK)
        {
          m_inprogress = false;
          archive_entry_free(ae);
          archive_read_close(disk);
          archive_read_free(disk);
          archive_write_fail(m_a);
          archive_write_free(m_a);
          return;
        }

        archive_read_disk_descend(disk);

        //WCHAR widenakedfn[2048];
        //WDL_MBtoWideStr(widenakedfn, nakedfn.Get(), sizeof(widenakedfn));
        archive_entry_set_pathname(ae, nakedfn.Get());
        //archive_entry_set_pathname_utf8(ae, fn.Get());
        //archive_entry_set_symlink_type(ae, AE_SYMLINK_TYPE_FILE);
        //archive_entry_set_symlink_utf8(ae, nakedfn.Get());
        //archive_entry_set_size(ae, fr.GetSize());
        ret = archive_write_header(m_a, ae);
        if (ret != ARCHIVE_OK)
        {
          m_inprogress = false;
          archive_entry_free(ae);
          archive_read_close(disk);
          archive_read_free(disk);
          archive_write_fail(m_a);
          archive_write_free(m_a);
          return;
        }

        do
        {
          int read = fr.Read(data.Get(), data.GetSize());
          archive_write_data(m_a, data.Get(), read);
          m_perc1 = (int)(100 * (i + 1) / 
            wdl_clamp(filelist.GetSize(), 1, filelist.GetSize()));
          m_perc2 = (int)(100 * fr.GetPosition() /
            wdl_clamp(fr.GetSize(), 1, fr.GetSize()));
        } while (fr.GetPosition() < fr.GetSize() && !m_abort);
        archive_entry_free(ae);
        archive_read_close(disk);
        archive_read_free(disk);
      }

      if (m_abort) break;
    }

    if (!m_abort)
    {
      archive_write_close(m_a);
      archive_write_free(m_a);
      ZstdCompressArchive();
      m_inprogress = false;

      WDL_FastString cfn(m_tarfn.Get());
      cfn.remove_fileext();
      cfn.Append(".tzst");
      ActionAfterCompression(cfn.Get(), "tzst");
    }
    else
    {
      m_inprogress = false;
      archive_write_fail(m_a);
      archive_write_free(m_a);
    }
  }
}

void RSA_Client::OnZstdDecompressFile()
{
  m_tzstfn.Set(m_selfile);
  m_perc1 = 0;
  m_perc2 = 0;
  m_abort = false;
  m_inprogress = true;
  ZstdDecompressArchive();
  m_inprogress = false;

  WDL_FastString path(m_selfile);
  path.remove_filepart();
  ActionAfterDecompression(path.Get());
}

void RSA_Client::ZstdCompressArchive()
{
  WDL_ASSERT(!m_zstdcctx);

  m_stage.Set("Compressing (2/2)");

  int insz = (int)ZSTD_CStreamInSize();
  int outsz = (int)ZSTD_CStreamOutSize();

  m_buffin.Resize(insz);
  m_buffout.Resize(outsz);

  m_zstdcctx = ZSTD_createCCtx();

#if RSA_ENABLE_ZSTD_CHECKS
  CHECK(m_zstdcctx != NULL, "ZSTD_createCCtx() failed");
  CHECK_ZSTD(ZSTD_CCtx_setParameter(m_zstdcctx, ZSTD_c_compressionLevel, m_zstdclevel));
  CHECK_ZSTD(ZSTD_CCtx_setParameter(m_zstdcctx, ZSTD_c_checksumFlag, m_zstdchecksum));
#else
  if (!m_zstdcctx)
  {
    wdl_log("ZSTD_createCCtx() failed\n");
    return;
  }
  ZSTD_CCtx_setParameter(m_zstdcctx, ZSTD_c_compressionLevel, m_zstdclevel);
  ZSTD_CCtx_setParameter(m_zstdcctx, ZSTD_c_checksumFlag, m_zstdchecksum);
#endif

  if (m_fileread) { delete m_fileread; m_fileread = NULL; }

  m_fileread = new WDL_FileRead(m_tarfn.Get());

  if (!m_fileread->IsOpen()) return;

  WDL_FastString outfilename(m_tarfn.Get());
  outfilename.remove_fileext();
  outfilename.Append(".tzst");
  m_filewrite = new WDL_FileWrite(outfilename.Get());
  if (!m_filewrite->IsOpen()) return;

  int nread = 0;

  do
  {
    nread = m_fileread->Read(m_buffin.Get(), m_buffin.GetSize());
    const int lastchunk = (nread < m_buffin.GetSize());
    const ZSTD_EndDirective mode = lastchunk ? ZSTD_e_end : ZSTD_e_continue;

    ZSTD_inBuffer input = { m_buffin.Get(), nread, 0 };

    m_perc1 = 100;
    m_perc2 = (int)(100 * m_fileread->GetPosition() /
      wdl_clamp(m_fileread->GetSize(), 1, m_fileread->GetSize()));

    int fin;
    do
    {
      ZSTD_outBuffer output = { m_buffout.Get(), m_buffout.GetSize(), 0 };
      const int remaining = (int)ZSTD_compressStream2(m_zstdcctx, &output, &input, mode);
#if RSA_ENABLE_ZSTD_CHECKS
      CHECK_ZSTD(remaining);
#else
      if (ZSTD_isError(remaining)) break;
#endif
      m_filewrite->Write(m_buffout.Get(), (int)output.pos);
      fin = lastchunk ? (remaining == 0) : (input.pos == input.size);
    } while (!fin && !m_abort);
#if RSA_ENABLE_ZSTD_CHECKS
    CHECK(input.pos == input.size, "zstd only returns 0 when the input is completely consumed");
#endif
  } while (nread && !m_abort);

  if (m_fileread) { delete m_fileread; m_fileread = NULL; }
  if (m_filewrite) { delete m_filewrite; m_filewrite = NULL; }
  if (m_zstdcctx) { ZSTD_freeCCtx(m_zstdcctx); m_zstdcctx = NULL; }

  bool keeptarball = g_preferences->WantZstdKeepTarball();
  if (!keeptarball)
  {
    DeleteFile(m_tarfn.Get());
  }
}

void RSA_Client::ZstdDecompressArchive()
{
  WDL_ASSERT(!m_zstddctx);

  m_stage.Set("Decompressing (1/2)");

  int insz = (int)ZSTD_DStreamInSize();
  int outsz = (int)ZSTD_DStreamOutSize();

  m_buffin.Resize(insz);
  m_buffout.Resize(outsz);

  m_zstddctx = ZSTD_createDCtx();

#if RSA_ENABLE_ZSTD_CHECKS
  CHECK(m_zstddctx != NULL, "ZSTD_createDCtx() failed");
#else
  if (!m_zstddctx)
  {
    wdl_log("ZSTD_createDCtx() failed\n");
    return;
  }
#endif

  m_fileread = new WDL_FileRead(m_tzstfn.Get());
  if (!m_fileread->IsOpen()) return;
  WDL_FastString outfilename(m_tzstfn.Get());
  outfilename.remove_fileext();
  outfilename.Append(".tar");
  m_tarfn.Set(outfilename.Get());
  m_filewrite = new WDL_FileWrite(outfilename.Get());
  if (!m_filewrite->IsOpen()) return;

  int nread = 0;
  int isempty = 1;
  int lastret = 0;
  m_abort = false;

  do
  {
    nread = m_fileread->Read(m_buffin.Get(), m_buffin.GetSize());

    isempty = 0;

    ZSTD_inBuffer input = { m_buffin.Get(), nread, 0 };

    m_perc1 = 100;
    m_perc2 = (int)(100 * m_fileread->GetPosition() /
      wdl_clamp(m_fileread->GetSize(), 1, m_fileread->GetSize()));

    while (input.pos < input.size && !m_abort)
    {
      ZSTD_outBuffer output = { m_buffout.Get(), m_buffout.GetSize(), 0 };
      const int ret = (int)ZSTD_decompressStream(m_zstddctx, &output, &input);
#if RSA_ENABLE_ZSTD_CHECKS
      CHECK_ZSTD(ret);
#else
      if (ZSTD_isError(ret)) break;
#endif
      m_filewrite->Write(m_buffout.Get(), (int)output.pos);
      lastret = ret;
    };
  } while (nread && !m_abort);

  if (isempty) { wdl_log("Input is empty\n"); }
  if (lastret != 0)
  {
    wdl_log("EOF before end of stream: %d", lastret);
  }

  if (m_fileread) { delete m_fileread; m_fileread = NULL; }
  if (m_filewrite) { delete m_filewrite; m_filewrite = NULL; }
  if (m_zstddctx) { ZSTD_freeDCtx(m_zstddctx); m_zstddctx = NULL; }

  WDL_FastString path(m_tarfn.Get());
  path.remove_fileext();
  path.Append(WDL_DIRCHAR_STR);

  m_stage.Set("Reading tarball (2/2)");

  m_perc1 = 0;
  m_perc2 = 0;

  if (1)
  {
    m_indexfile = 0;
    m_totalfiles = 0;

    struct archive *a = archive_read_new();

    archive_read_support_format_tar(a);
    archive_read_support_format_gnutar(a);

    WCHAR widetarfn[2048];
    WDL_MBtoWideStr(widetarfn, m_tarfn.Get(), sizeof(widetarfn));
    int ret = archive_read_open_filename_w(a, widetarfn, 10240);
    if (ret)
    {
      archive_read_close(a);
      archive_read_free(a);
      return;
    }

    struct archive_entry *ae;

    CreateDirectory(path.Get(), NULL);

    while (archive_read_next_header(a, &ae) == ARCHIVE_OK)
    {
      if (archive_entry_filetype(ae) == AE_IFREG) m_totalfiles++;
      archive_read_data_skip(a);
    }

    archive_read_close(a);
    archive_read_free(a);
  }

  struct archive *a = archive_read_new();

  archive_read_support_format_tar(a);
  archive_read_support_format_gnutar(a);

  WCHAR widetarfn[2048];
  WDL_MBtoWideStr(widetarfn, m_tarfn.Get(), sizeof(widetarfn));
  int ret = archive_read_open_filename_w(a, widetarfn, 10240);
  if (ret)
  {
    archive_read_close(a);
    archive_read_free(a);
    return;
  }

  struct archive_entry *ae;

  CreateDirectory(path.Get(), NULL);

  while (archive_read_next_header(a, &ae) == ARCHIVE_OK)
  {
    if (archive_entry_filetype(ae) == AE_IFDIR)
    {
      WDL_String dir(path.Get());
      dir.Append(archive_entry_pathname(ae));

      for (int i = 0; i < dir.GetLength(); i++)
      {
        if (dir.Get()[i] == '/')
        {
          dir.Get()[i] = WDL_DIRCHAR;
        }
      }

      CreateDirectory(dir.Get(), NULL);
    }
    else if (archive_entry_filetype(ae) == AE_IFREG)
    {
      WDL_String fn(path.Get());
      fn.Append(archive_entry_pathname(ae));

      for (int i = 0; i < fn.GetLength(); i++)
      {
        if (fn.Get()[i] == '/')
        {
          fn.Get()[i] = WDL_DIRCHAR;
        }
      }

      WDL_FileWrite fw(fn.Get());
      if (fw.IsOpen())
      {
        int read = 0;
        WDL_HeapBuf data;
        data.Resize(4096);
        la_int64_t filesize = 0;
        do
        {
          read = (int)archive_read_data(a, data.Get(), data.GetSize());
          fw.Write(data.Get(), read);
          filesize += read;
          m_perc1 = (int)(100 * (m_indexfile + 1) /
            wdl_clamp(m_totalfiles, 1, m_totalfiles));
          m_perc2 = (int)(100 * filesize /
            wdl_clamp(archive_entry_size(ae), 1, archive_entry_size(ae)));
        } while (read > 0);
      }
    }
  }

  archive_read_close(a);
  archive_read_free(a);

  bool keeptarball = g_preferences->WantZstdKeepTarball();
  if (!keeptarball)
  {
    DeleteFile(m_tarfn.Get());
  }
}

void RSA_Client::ZstdScanFiles(const char *path)
{
  WDL_DirScan dir;
  WDL_HeapBuf data;
  data.Resize(4096);

  m_stage.Set("Writing tarball (1/2)");

  if (!dir.First(path))
  {
    do
    {
      if (dir.GetCurrentIsDirectory())
      {
        WDL_FastString fn;
        dir.GetCurrentFullFN(&fn);

        if (strcmp(fn.get_filepart(), ".") &&
          strcmp(fn.get_filepart(), ".."))
        {
          m_relpath.Set(fn.Get());
          m_relpath.DeleteSub(0, (int)strlen(m_seldir.Get()) + 1);

          for (int i = 0; i < m_relpath.GetLength(); i++)
          {
            if (m_relpath.Get()[i] == WDL_DIRCHAR)
            {
              m_relpath.Get()[i] = '/';
            }
          }

          struct archive *disk = archive_read_disk_new();
          //archive_read_disk_set_standard_lookup(disk);

          WCHAR widefn[2048];
          WDL_MBtoWideStr(widefn, fn.Get(), sizeof(widefn));
          int ret = archive_read_disk_open_w(disk, widefn);
          if (ret != ARCHIVE_OK)
          {
            m_inprogress = false;
            archive_read_close(disk);
            archive_read_free(disk);
            return;
          }

          struct archive_entry *ae = archive_entry_new2(m_a);

          ret = archive_read_next_header2(disk, ae);
          if (ret != ARCHIVE_OK)
          {
            m_inprogress = false;
            archive_entry_free(ae);
            archive_read_close(disk);
            archive_read_free(disk);
            return;
          }

          archive_read_disk_descend(disk);

          //WCHAR widenakedfn[2048];
          //WDL_MBtoWideStr(widenakedfn, nakedfn.Get(), sizeof(widenakedfn));
          archive_entry_set_pathname(ae, m_relpath.Get());
          //archive_entry_set_pathname_utf8(ae, fn.Get());
          //archive_entry_set_symlink_type(ae, AE_SYMLINK_TYPE_FILE);
          //archive_entry_set_symlink_utf8(ae, nakedfn.Get());
          //archive_entry_set_size(ae, fr.GetSize());
          ret = archive_write_header(m_a, ae);
          if (ret != ARCHIVE_OK)
          {
            m_inprogress = false;
            archive_entry_free(ae);
            archive_read_close(disk);
            archive_read_free(disk);
            return;
          }

          archive_entry_free(ae);
          archive_read_close(disk);
          archive_read_free(disk);

          ZstdScanFiles(fn.Get());
        }
      }
      else
      {
        WDL_FastString fn;
        dir.GetCurrentFullFN(&fn);

        m_relpath.Set(fn.Get());
        m_relpath.DeleteSub(0, (int)strlen(m_seldir.Get()) + 1);

        for (int i = 0; i < m_relpath.GetLength(); i++)
        {
          if (m_relpath.Get()[i] == WDL_DIRCHAR)
          {
            m_relpath.Get()[i] = '/';
          }
        }

        struct archive *disk = archive_read_disk_new();
        //archive_read_disk_set_standard_lookup(disk);

        WCHAR widefn[2048];
        WDL_MBtoWideStr(widefn, fn.Get(), sizeof(widefn));
        int ret = archive_read_disk_open_w(disk, widefn);
        if (ret != ARCHIVE_OK)
        {
          m_inprogress = false;
          archive_read_close(disk);
          archive_read_free(disk);
          return;
        }

        struct archive_entry *ae = archive_entry_new2(m_a);

        ret = archive_read_next_header2(disk, ae);
        if (ret != ARCHIVE_OK)
        {
          m_inprogress = false;
          archive_entry_free(ae);
          archive_read_close(disk);
          archive_read_free(disk);
          return;
        }

        archive_read_disk_descend(disk);

        //WCHAR widenakedfn[2048];
        //WDL_MBtoWideStr(widenakedfn, nakedfn.Get(), sizeof(widenakedfn));
        archive_entry_set_pathname(ae, m_relpath.Get());
        //archive_entry_set_pathname_utf8(ae, fn.Get());
        //archive_entry_set_symlink_type(ae, AE_SYMLINK_TYPE_FILE);
        //archive_entry_set_symlink_utf8(ae, nakedfn.Get());
        //archive_entry_set_size(ae, fr.GetSize());
        ret = archive_write_header(m_a, ae);
        if (ret != ARCHIVE_OK)
        {
          m_inprogress = false;
          archive_entry_free(ae);
          archive_read_close(disk);
          archive_read_free(disk);
          return;
        }

        WDL_FileRead fr(fn.Get());
        if (fr.IsOpen())
        {
          do
          {
            int read = fr.Read(data.Get(), data.GetSize());
            archive_write_data(m_a, data.Get(), read);
            m_perc1 = (100 * (m_indexfile + 1) /
              wdl_clamp(m_totalfiles, 1, m_totalfiles));
            m_perc2 = (int)(100 * fr.GetPosition() /
              wdl_clamp(fr.GetSize(), 1, fr.GetSize()));
          } while (fr.GetPosition() < fr.GetSize() && !m_abort);

          m_indexfile++;
        }

        archive_entry_free(ae);
        archive_read_close(disk);
        archive_read_free(disk);
      }
    }
    while (!dir.Next() && !m_abort);
  }
}

void RSA_Client::PrepareFiles(const char *path)
{
  WDL_DirScan dir;

  m_stage.Set("Preparing...");

  if (!dir.First(path))
  {
    do
    {
      if (dir.GetCurrentIsDirectory())
      {
        WDL_FastString fn;
        dir.GetCurrentFullFN(&fn);

        if (strcmp(fn.get_filepart(), ".") &&
          strcmp(fn.get_filepart(), ".."))
        {
          PrepareFiles(fn.Get());
        }
      }
      else
      {
        m_totalfiles++;
      }
    }
    while (!dir.Next() && !m_abort);
  }
}

void RSA_Client::OnZlibCompressFolder()
{
  m_abort = false;
  m_zipfn.Set(m_seldir.Get());
  m_zipfn.Append(".zip");

  m_indexfile = 0;
  m_totalfiles = 0;

  m_zf = zipOpen64(m_zipfn.Get(), APPEND_STATUS_CREATE);
  if (m_zf)
  {
    m_perc1 = 0;
    m_perc2 = 0;
    m_inprogress = true;
    PrepareFiles(m_seldir.Get());
    ZlibScanFiles(m_seldir.Get());
    zipClose(m_zf, NULL);
    m_inprogress = false;
    ActionAfterCompression(m_zipfn.Get(), "zip");
  }
}

void RSA_Client::OnZlibCompressFile()
{
  m_abort = false;
  WDL_FastString path;

  m_stage.Set("Compressing");

  const char *only_one = m_selfile;

  while (*only_one++);

  if (*only_one == '\0' && *(only_one + 1) == '\0')
  {
    path.Set(m_selfile);
    path.remove_fileext();
    m_zipfn.Set(path.Get());
    m_zipfn.Append(".zip");
    m_zf = zipOpen64(m_zipfn.Get(), APPEND_STATUS_CREATE);
    if (m_zf)
    {
      m_perc1 = 0;
      m_perc2 = 0;
      m_inprogress = true;
      WDL_FastString fn(m_selfile);
      WDL_FastString nakedfn(fn.get_filepart());

      WDL_HeapBuf data;
      data.Resize(4096);
      WDL_FileRead fr(fn.Get());

      if (fr.IsOpen())
      {
        memset(&m_zfi, 0, sizeof(zip_fileinfo));
        int ret = zipOpenNewFileInZip64(m_zf, nakedfn.Get(), &m_zfi, NULL, 0,
          NULL, 0, NULL, Z_DEFLATED, m_zlibclevel, 1);
        if (ret == ZIP_OK)
        {
          do
          {
            int read = fr.Read(data.Get(), data.GetSize());
            zipWriteInFileInZip(m_zf, data.Get(), read);
            m_perc1 = 100;
            m_perc2 = (int)(100 * fr.GetPosition() /
              wdl_clamp(fr.GetSize(), 1, fr.GetSize()));
          } while (fr.GetPosition() < fr.GetSize() && !m_abort);

          zipCloseFileInZip(m_zf);
        }

        zipClose(m_zf, NULL);
      }

      m_inprogress = false;

      ActionAfterCompression(m_zipfn.Get(), "zip");
    }
  }
  else
  {
    path.Set(m_selfile);
    path.Append(WDL_DIRCHAR_STR);

    char *s = m_selfile;
    while (*s++);

    m_zipfn.Set(path.Get());
    m_zipfn.Append(s);
    m_zipfn.remove_fileext();
    m_zipfn.Append(".zip");
    m_zf = zipOpen64(m_zipfn.Get(), APPEND_STATUS_CREATE);
    if (m_zf)
    {
      m_perc1 = 0;
      m_perc2 = 0;
      m_inprogress = true;

      WDL_PtrList_DeleteOnDestroy<WDL_FastString> filelist;

      do
      {
        WDL_FastString *fn = new WDL_FastString(path.Get());
        fn->Append(s);
        filelist.Add(fn);
        while (*s++);
      } while (*s != '\0' && *(s + 1) != '\0' && !m_abort);

      for (int i = 0; i < filelist.GetSize(); i++)
      {
        WDL_FastString *fn = filelist.Get(i);
        WDL_FastString nakedfn(fn->get_filepart());

        WDL_HeapBuf data;
        data.Resize(4096);
        WDL_FileRead fr(fn->Get());

        if (fr.IsOpen())
        {
          memset(&m_zfi, 0, sizeof(zip_fileinfo));
          int ret = zipOpenNewFileInZip64(m_zf, nakedfn.Get(), &m_zfi, NULL, 0,
            NULL, 0, NULL, Z_DEFLATED, m_zlibclevel, 1);
          if (ret == ZIP_OK)
          {
            do
            {
              int read = fr.Read(data.Get(), data.GetSize());
              zipWriteInFileInZip(m_zf, data.Get(), read);
              m_perc1 = (int)(100 * (i + 1) /
                wdl_clamp(filelist.GetSize(), 1, filelist.GetSize()));
              m_perc2 = (int)(100 * fr.GetPosition() /
                wdl_clamp(fr.GetSize(), 1, fr.GetSize()));
            } while (fr.GetPosition() < fr.GetSize() && !m_abort);

            zipCloseFileInZip(m_zf);
          }
        }

        if (m_abort) break;
      }

      zipClose(m_zf, NULL);

      m_inprogress = false;

      ActionAfterCompression(m_zipfn.Get(), "zip");
    }
  }
}

void RSA_Client::OnZlibDecompressFile()
{
  m_stage.Set("Decompressing");

  m_zipfn.Set(m_selfile);
  m_perc1 = 0;
  m_perc2 = 0;
  m_abort = false;
  m_inprogress = true;

  WDL_FastString path(m_zipfn.Get());
  path.remove_fileext();
  path.Append(WDL_DIRCHAR_STR);

  unz_global_info64 globalinfo;
  unz_file_info64 fileinfo;
  char fn[2048];

  unzFile uzf = unzOpen64(m_zipfn.Get());
  if (uzf)
  {
    CreateDirectory(path.Get(), NULL);
    int ret = unzGetGlobalInfo64(uzf, &globalinfo);
    if (ret != UNZ_OK)
    {
      m_inprogress = false;
      unzClose(uzf);
      return;
    }

    for (int i = 0; i < globalinfo.number_entry; i++)
    {
      ret = unzGetCurrentFileInfo64(uzf, &fileinfo,
        fn, sizeof(fn), NULL, 0, NULL, 0);
      if (ret != UNZ_OK)
      {
        m_inprogress = false;
        unzClose(uzf);
        return;
      }

      if (fn[(int)strlen(fn) - 1] == '/') // WDL_DIRCHAR)
      {
        // Entry is m_a directory
        WDL_String dir(path.Get());
        dir.Append(fn);

        for (int j = 0; j < dir.GetLength(); j++)
        {
          if (dir.Get()[j] == '/')
          {
            dir.Get()[j] = WDL_DIRCHAR;
          }
        }

        CreateDirectory(dir.Get(), NULL);

        if ((i + 1) < globalinfo.number_entry)
        {
          ret = unzGoToNextFile(uzf);
          if (ret != UNZ_OK)
          {
            m_inprogress = false;
            unzClose(uzf);
            return;
          }
        }
      }
      else
      {
        ret = unzOpenCurrentFile(uzf);
        if (ret != UNZ_OK)
        {
          m_inprogress = false;
          unzCloseCurrentFile(uzf);
          unzClose(uzf);
          return;
        }

        WDL_String ffn(path.Get());
        ffn.Append(fn);
        wdl_log(ffn.Get());

        for (int j = 0; j < ffn.GetLength(); j++)
        {
          if (ffn.Get()[j] == '/')
          {
            ffn.Get()[j] = WDL_DIRCHAR;
          }
        }

        WDL_FileWrite fw(ffn.Get());
        if (fw.IsOpen())
        {
          WDL_HeapBuf hb;
          hb.Resize(4096);
          ZPOS64_T uncompsize = 0;

          do
          {
            ret = unzReadCurrentFile(uzf, hb.Get(), hb.GetSize());
            fw.Write(hb.Get(), ret);
            uncompsize += ret;
            m_perc1 = (int)(100 * (i + 1) /
              wdl_clamp(globalinfo.number_entry, 1, globalinfo.number_entry));
            m_perc2 = (int)(100 * uncompsize /
              wdl_clamp(fileinfo.uncompressed_size, 1, fileinfo.uncompressed_size));
          } while (ret > 0 && !m_abort);

          if (ret < 0)
          {
            m_inprogress = false;
            unzCloseCurrentFile(uzf);
            unzClose(uzf);
            return;
          }
        }
        else
        {
          m_inprogress = false;
          unzCloseCurrentFile(uzf);
          unzClose(uzf);
          return;
        }

        unzCloseCurrentFile(uzf);

        if ((i + 1) < globalinfo.number_entry)
        {
          ret = unzGoToNextFile(uzf);
          if (ret != UNZ_OK)
          {
            m_inprogress = false;
            unzClose(uzf);
            return;
          }
        }
      }
      if (m_abort) break;
    }
    unzClose(uzf);
  }

  m_inprogress = false;

  WDL_FastString actpath(m_selfile);
  actpath.remove_filepart();
  ActionAfterDecompression(actpath.Get());
}

void RSA_Client::ZlibScanFiles(const char *path)
{
  WDL_DirScan dir;

  m_stage.Set("Compressing");

  if (!dir.First(path))
  {
    do
    {
      if (dir.GetCurrentIsDirectory())
      {
        WDL_FastString fn;
        dir.GetCurrentFullFN(&fn);

        if (strcmp(fn.get_filepart(), ".") &&
          strcmp(fn.get_filepart(), ".."))
        {
          m_relpath.Set(fn.Get());
          m_relpath.DeleteSub(0, (int)strlen(m_seldir.Get()) + 1);
          m_relpath.Append("/");

          for (int i = 0; i < m_relpath.GetLength(); i++)
          {
            if (m_relpath.Get()[i] == WDL_DIRCHAR)
            {
              m_relpath.Get()[i] = '/';
            }
          }
          memset(&m_zfi, 0, sizeof(zip_fileinfo));
          int ret = zipOpenNewFileInZip64(m_zf, m_relpath.Get(), &m_zfi,
            NULL, 0, NULL, 0, NULL, 0, 0, 1);
          if (ret == ZIP_OK) { zipCloseFileInZip(m_zf); }

          ZlibScanFiles(fn.Get());
        }
      }
      else
      {
        WDL_FastString fn;
        dir.GetCurrentFullFN(&fn);

        m_relpath.Set(fn.Get());
        m_relpath.DeleteSub(0, (int)strlen(m_seldir.Get()) + 1);

        for (int i = 0; i < m_relpath.GetLength(); i++)
        {
          if (m_relpath.Get()[i] == WDL_DIRCHAR)
          {
            m_relpath.Get()[i] = '/';
          }
        }

        WDL_HeapBuf data;
        data.Resize(4096);

        WDL_FileRead fr(fn.Get());
        if (fr.IsOpen())
        {
          memset(&m_zfi, 0, sizeof(zip_fileinfo));
          int ret = zipOpenNewFileInZip64(m_zf, m_relpath.Get(), &m_zfi,
            NULL, 0, NULL, 0, NULL, Z_DEFLATED, m_zlibclevel, 1);
          if (ret == ZIP_OK)
          {
            do
            {
              int read = fr.Read(data.Get(), data.GetSize());
              zipWriteInFileInZip(m_zf, data.Get(), read);
              m_perc1 = (int)(100 * (m_indexfile + 1) /
                wdl_clamp(m_totalfiles, 1, m_totalfiles));
              m_perc2 = (int)(100 * fr.GetPosition() /
                wdl_clamp(fr.GetSize(), 1, fr.GetSize()));
            } while (fr.GetPosition() < fr.GetSize() && !m_abort);

            m_indexfile++;
            zipCloseFileInZip(m_zf);
          }
        }
        else { m_abort = true; }
      }
    }
    while (!dir.Next() && !m_abort);
  }
}

void RSA_Client::ActionAfterCompression(const char *filepath, const char *defext)
{
  int action = g_preferences->GetActionAfterCompression();
  switch (action)
  {
  case 1:
    {
      char fn[2048];
      if (WDL_ChooseFileForSave(g_mainwnd, "Save compressed file", NULL, NULL,
        RSA_SAVE_COMPRESSED_FILE, defext, true, fn, sizeof(fn)))
      {
        int read = 0;
        WDL_HeapBuf buf;
        buf.Resize(4096);
        WDL_FileRead *fr = new WDL_FileRead(filepath);
        if (!fr->IsOpen()) { delete fr; return; }
        WDL_FileWrite *fw = new WDL_FileWrite(fn);
        if (!fw->IsOpen()) { delete fr; delete fw; return; }

        do
        {
          read = fr->Read(buf.Get(), buf.GetSize());
          fw->Write(buf.Get(), read);
        } while (read);

        delete fr;
        delete fw;
        DeleteFile(filepath);
      }
    }
    break;
  case 2:
    {
      WDL_FastString explr(filepath);
      explr.remove_filepart();
      ShellExecute(NULL, "", "explorer.exe", explr.Get(), "", SW_SHOWNORMAL);
    }
    break;
  }
}

void RSA_Client::ActionAfterDecompression(const char *path)
{
  int action = g_preferences->GetActionAfterDecompression();
  switch (action)
  {
  case 1:
    {
      ShellExecute(NULL, "", "explorer.exe", path, "", SW_SHOWNORMAL);
    }
    break;
  }
}

int RSA_Client::Run()
{
  if (m_compressfolder)
  {
    switch (m_compressor)
    {
    case 0: OnZlibCompressFolder(); break;
    case 1: OnZstdCompressFolder(); break;
    }
    m_inprogress = false;
    m_compressfolder = false;
  }
  else if (m_compressfile)
  {
    switch (m_compressor)
    {
    case 0: OnZlibCompressFile(); break;
    case 1: OnZstdCompressFile(); break;
    }
    m_inprogress = false;
    m_compressfile = false;
  }
  else if (m_decompressfile)
  {
    if (!stricmp(WDL_get_fileext(m_selfile), ".zip")) OnZlibDecompressFile();
    if (!stricmp(WDL_get_fileext(m_selfile), ".tzst")) OnZstdDecompressFile();
    m_inprogress = false;
    m_decompressfile = false;
  }

  return 1;
}

void RSA_Client::StartThread()
{
  if (!m_thread)
  {
    unsigned int thread_id;
    m_thread = (HANDLE)_beginthreadex(NULL, 0, ThreadFunction, (void *)this, 0, &thread_id);
  }
}

void RSA_Client::StopThread()
{
  m_abort = true;
  m_killthread = true;

  if (m_thread)
  {
    WaitForSingleObject(m_thread, INFINITE);
    CloseHandle(m_thread);
    m_thread = NULL;
  }
}

unsigned int WINAPI RSA_Client::ThreadFunction(void *arg)
{
#if defined(_WIN32)
  CoInitialize(NULL);
#endif

  RSA_Client *self = (RSA_Client *)arg;

  if (WDL_NORMALLY(self))
  {
    self->m_killthread = false;

    while (!self->m_killthread)
    {
      self->m_mutex.Enter();
      while (!self->Run());
      self->m_mutex.Leave();
      Sleep(50);
    }
  }

#if defined(_WIN32)
  CoUninitialize();
#endif

  return 0;
}

#if 0
// Get the size of m_a given file path.
// return The size of m_a given file path.
size_t RSA_Client::zstd_fsize(const char *filename)
{
  struct stat st;
  if (stat(filename, &st) != 0)
  {
    // error
    perror(filename);
    exit(ERROR_fsize);
  }

  off_t const filesize = st.st_size;
  size_t const size = (size_t)filesize;
  // 1. filesize should be non-negative,
  // 2. if off_t -> size_t type conversion results in discrepancy,
  //    the file size is too large for type size_t.
  if ((filesize < 0) || (filesize != (off_t)size))
  {
    fprintf(stderr, "%s : filesize too large \n", filename);
    exit(ERROR_largeFile);
  }
  return size;
}

// Open m_a file using given file path and open option.
// return If successful this function will return m_a FILE pointer to an
// opened file otherwise it sends an error to stderr and exits.
FILE *RSA_Client::zstd_fopen(const char *filename, const char *instruction)
{
  FILE *const infile = fopen(filename, instruction);
  if (infile) return infile;
  // error
  perror(filename);
  exit(ERROR_fopen);
}

// Close an opened file using given FILE pointer.
void RSA_Client::zstd_fclose(FILE *file)
{
  if (!fclose(file)) { return; };
  // error
  perror("fclose");
  exit(ERROR_fclose);
}

// Read size_to_read bytes from m_a given file, storing them at the
// location given by buffer.
// return The number of bytes read.
size_t RSA_Client::zstd_fread(void *buffer, size_t size_to_read, FILE *file)
{
  size_t const readsize = fread(buffer, 1, size_to_read, file);
  if (readsize == size_to_read) return readsize; // good
  if (feof(file)) return readsize; // good, reached end of file
  // error
  perror("fread");
  exit(ERROR_fread);
}

// Write size_to_write bytes to m_a file pointed to by file, obtaining
// them from m_a location given by buffer.
// Note: This function will send an error to stderr and exit if it
// cannot write data to the given file pointer.
// return The number of bytes written.
size_t RSA_Client::zstd_fwrite(const void *buffer, size_t size_to_write, FILE *file)
{
  size_t const writtensize = fwrite(buffer, 1, size_to_write, file);
  if (writtensize == size_to_write) return size_to_write; // good
  // error
  perror("fwrite");
  exit(ERROR_fwrite);
}

// Allocate memory.
// return If successful this function returns m_a pointer to allo-
// cated memory.  If there is an error, this function will send that
// error to stderr and exit.
void *RSA_Client::zstd_malloc(size_t size)
{
  void *const buff = malloc(size);
  if (buff) return buff;
  // error
  perror("malloc");
  exit(ERROR_malloc);
}

// load file into buffer (memory).
// Note: This function will send an error to stderr and exit if it
// cannot read data from the given file path.
// return If successful this function will load file into buffer and
// return file size, otherwise it will printout an error to stderr and exit.
size_t RSA_Client::zstd_load_file(const char *filename, void *buffer, size_t buffer_size)
{
  size_t const filesize = zstd_fsize(filename);

#if RSA_ENABLE_ZSTD_CHECKS
  CHECK(filesize <= buffer_size, "File too large!");
#endif

  FILE *const infile = zstd_fopen(filename, "rb");
  size_t const readsize = fread(buffer, 1, filesize, infile);
  if (readsize != (size_t)filesize)
  {
    fprintf(stderr, "fread: %s : %s \n", filename, strerror(errno));
    exit(ERROR_fread);
  }
  fclose(infile); // can't fail, read only
  return filesize;
}

// allocate memory buffer and then load file into it.
// Note: This function will send an error to stderr and exit if memory allocation
// fails or it cannot read data from the given file path.
// return If successful this function will return buffer and buffer_size(=filesize),
// otherwise it will printout an error to stderr and exit.
void *RSA_Client::zstd_malloc_and_load_file(const char *filename, size_t *buffer_size)
{
  size_t const filesize = zstd_fsize(filename);
  *buffer_size = filesize;
  void* const buffer = zstd_malloc(*buffer_size);
  zstd_load_file(filename, buffer, *buffer_size);
  return buffer;
}

// Save buffSize bytes to m_a given file path, obtaining them from m_a location pointed
// to by buff.
// Note: This function will send an error to stderr and exit if it
// cannot write to m_a given file.
void RSA_Client::zstd_save_file(const char *filename, const void *buffer, size_t buffer_size)
{
  FILE *const ofile = zstd_fopen(filename, "wb");
  size_t const wsize = fwrite(buffer, 1, buffer_size, ofile);
  if (wsize != (size_t)buffer_size)
  {
    fprintf(stderr, "fwrite: %s : %s \n", filename, strerror(errno));
    exit(ERROR_fwrite);
  }
  if (fclose(ofile))
  {
    perror(filename);
    exit(ERROR_fclose);
  }
}

void RSA_Client::zstd_compress(const char *fname, const char *oname)
{
  size_t fsize;
  void *const fbuff = zstd_malloc_and_load_file(fname, &fsize);
  size_t const cbuffsize = ZSTD_compressBound(fsize);
  void *const cbuff = zstd_malloc(cbuffsize);

  // Compress.
  // If you are doing many compressions, you may want to reuse the context.
  // See the multiple_simple_compression.c example.
  size_t const csize = ZSTD_compress(cbuff, cbuffsize, fbuff, fsize, 1);

#if RSA_ENABLE_ZSTD_CHECKS
  CHECK_ZSTD(csize);
#endif

  zstd_save_file(oname, cbuff, csize);

  // success
  printf("%25s : %6u -> %7u - %s \n", fname, (unsigned)fsize, (unsigned)csize, oname);

  free(fbuff);
  free(cbuff);
}

char *RSA_Client::zstd_create_outfilename(const char *filename)
{
  size_t const inl = strlen(filename);
  size_t const outl = inl + 5;
  void* const outspace = zstd_malloc(outl);
  memset(outspace, 0, outl);
  strcat((char *)outspace, filename);
  strcat((char *)outspace, ".zst");
  return (char*)outspace;
}
#endif
