// Copyright (c) 2021 Giorgos Vougioukas
//
// The license can be found in the LICENSE file.

#ifndef _ALCYONE_CLIENT_H_
#define _ALCYONE_CLIENT_H_

#include "alcyone/definitions.h"
#include "third_party/zstd/lib/zstd.h"
#include "third_party/zstd/lib/zdict.h"
#include "third_party/libarchive/libarchive/archive.h"
#include "third_party/libarchive/libarchive/archive_entry.h"

#include <stdlib.h>
#include <string.h>

#include "WDL/zlib/zip.h"
#include "WDL/zlib/unzip.h"
#include "WDL/wdltypes.h"
#include "WDL/heapbuf.h"
#include "WDL/wdlstring.h"
#include "WDL/ptrlist.h"
#include "WDL/mutex.h"
#include "WDL/fileread.h"
#include "WDL/filewrite.h"

class RSA_Client
{
public:
  RSA_Client();
  ~RSA_Client();

  void CompressFolder();
  void CompressFile();
  void DecompressFile();

  void SetCompressor(int val);
  int GetCompressor() const;

  bool IsInProgress() const;
  void AbortProgress();
  void Progress(int *perc1, int *perc2, WDL_FastString *stage) const;

private:
#if 0
  // Get the size of a given file path.
  // return The size of a given file path.
  size_t zstd_fsize(const char *filename);

  // Open a file using given file path and open option.
  // return If successful this function will return a FILE pointer to an
  // opened file otherwise it sends an error to stderr and exits.
  FILE *zstd_fopen(const char *filename, const char *instruction);

  // Close an opened file using given FILE pointer.
  void zstd_fclose(FILE *file);

  // Read size_to_read bytes from a given file, storing them at the
  // location given by buffer.
  // return The number of bytes read.
  size_t zstd_fread(void *buffer, size_t size_to_read, FILE *file);

  // Write size_to_write bytes to a file pointed to by file, obtaining
  // them from a location given by buffer.
  // Note: This function will send an error to stderr and exit if it
  // cannot write data to the given file pointer.
  // return The number of bytes written.
  size_t zstd_fwrite(const void *buffer, size_t size_to_write, FILE *file);

  // Allocate memory.
  // return If successful this function returns a pointer to allo-
  // cated memory.  If there is an error, this function will send that
  // error to stderr and exit.
  void *zstd_malloc(size_t size);

  // load file into buffer (memory).
  // Note: This function will send an error to stderr and exit if it
  // cannot read data from the given file path.
  // return If successful this function will load file into buffer and
  // return file size, otherwise it will printout an error to stderr and exit.
  size_t zstd_load_file(const char *filename, void *buffer, size_t buffer_size);

  // allocate memory buffer and then load file into it.
  // Note: This function will send an error to stderr and exit if memory allocation
  // fails or it cannot read data from the given file path.
  // return If successful this function will return buffer and bufferSize(=fileSize),
  // otherwise it will printout an error to stderr and exit.
  void* zstd_malloc_and_load_file(const char *filename, size_t* buffer_size);

  // Save buffer_size bytes to a given file path, obtaining them from a location pointed
  // to by buff.
  // Note: This function will send an error to stderr and exit if it
  // cannot write to a given file.
  void zstd_save_file(const char *filename, const void *buffer, size_t buffer_size);

  void zstd_compress(const char *fname, const char *oname);
  char* zstd_create_outfilename(const char *filename);
#endif

  void OnZstdCompressFolder();
  void OnZstdCompressFile();
  void OnZstdDecompressFile();
  void ZstdCompressArchive();
  void ZstdDecompressArchive();
  void ZstdScanFiles(const char *path);
  void OnZlibCompressFolder();
  void OnZlibCompressFile();
  void OnZlibDecompressFile();
  void ZlibScanFiles(const char *path);
  void PrepareFiles(const char *path);
  void ActionAfterCompression(const char *filepath, const char *defext);
  void ActionAfterDecompression(const char *path);
  static unsigned int WINAPI ThreadFunction(void *arg);
  void StartThread();
  void StopThread();
  int Run();

  char *m_selfile;
  WDL_TypedBuf<char> m_seldir;
  bool m_compressfolder;
  bool m_compressfile;
  bool m_decompressfile;

  HANDLE m_thread;
  bool m_killthread;
  WDL_Mutex m_mutex;

  WDL_FileRead *m_fileread;
  WDL_FileWrite *m_filewrite;
  WDL_HeapBuf m_buffin;
  WDL_HeapBuf m_buffout;
  ZSTD_CCtx *m_zstdcctx;
  int m_zstdclevel;
  int m_zstdchecksum;
  ZSTD_DCtx *m_zstddctx;

  struct archive *m_a;

  WDL_FastString m_tarfn;
  WDL_FastString m_tzstfn;
  WDL_String m_relpath;
  int m_perc1;
  int m_perc2;
  int m_indexfile;
  int m_totalfiles;
  WDL_FastString m_stage;
  bool m_inprogress;
  bool m_abort;
  int m_compressor;

  WDL_FastString m_zipfn;
  zip_fileinfo m_zfi;
  zipFile m_zf;
  int m_zlibclevel;
};

#endif // _ALCYONE_CLIENT_H_
