// Copyright (c) 2018 Giorgos Vougioukas
//
// The license can be found in the LICENSE file.

#include "about_wnd.h"
#include "app_timers.h"

// 3:1 (ultra wide)
#define MAR_ABOUT_WIDTH 630
#define MAR_ABOUT_HEIGHT 210

#include <WDL/aggarray.h>

static const char *internal_master_project =
  "\xce\x9c\xce\xb1\xcf\x81\xce\xba\xce\xad\xce\xbb\xce\xbb\xce\xb1";

static const char *internal_different =
  "\xce\x93\xce\xb9\xcf\x8e\xcf\x81\xce\xb3\xce\xbf"
  "\xcf\x82\x20\xce\x92\xce\xbf\xcf\x85\xce\xb3\xce"
  "\xb9\xce\xbf\xcf\x8d\xce\xba\xce\xb1\xcf\x82";

static const char *internal_credits[] =
{
  MAR_NAME_MARKETING,
  "Version: " MAR_NAKED_VERSION " (" MAR_ARCH ")",
  "Build timestamp: " MAR_TIMESTAMP,
  "Git commit: " MAR_GIT_SHA,
  "",
  "Developed by:",
  "Giorgos Vougioukas",
  "",
  "WDL:",
  "Cockos Incorporated",
  "(when you care enough)",
  "",
  "Cool Scrollbar:",
  "J Brown",
  "",
  "Scrollbar theme is loosely based on",
  "REAPER's scrollbar made by:",
  "White Tie",
  "",
  "cJSON:",
  "Dave Gamble",
  "cJSON contributors",
  "",
  "libpng:",
  "Glenn Randers-Pehrson",
  "Simon-Pierre Cadieux",
  "Eric S. Raymond",
  "Mans Rullgard",
  "Cosmin Truta",
  "Gilles Vollant",
  "James Yu",
  "Tom Lane",
  "Willem van Schaik",
  "John Bowler",
  "Kevin Bracey",
  "Sam Bushell",
  "Magnus Holmgren",
  "Greg Roelofs",
  "Tom Tanner",
  "Andreas Dilger",
  "Dave Martindale",
  "Guy Eric Schalnat",
  "Paul Schmidt",
  "Tim Wegner",
  "",
  "Mersenne Twister:",
  "Makoto Matsumoto (MT inventor)",
  "Takuji Nishimura (MT inventor)",
  "Shawn Cokus",
  "Richard J. Wagner",
  "",
  "zlib:",
  "Jean-loup Gailly",
  "Mark Adler",
  "",
  "Windows installer:",
  "Nullsoft Scriptable Install System (NSIS)",
  "Compressor | ZLIB",
  "",
  "Windows portable:",
  "Archive format | ZIP",
  //"",
  //"Linux tarball:",
  //"gzip",
  "",
  "Giorgos wishes to thank the following",
  "for their assistance:",
  "Asimina",
  "Manolis",
  "Justin Frankel",
  "Charles Crowley",
  "Ken Silverman",
  "and",
  "Snoopy",
  "",
  "Contact | email:",
  "support@grafmin.gr",
  "",
  MAR_NAME_MARKETING,
  MAR_COPYRIGHT
};

static LOGFONT internal_lf =
{
  16, 0, 0, 0, FW_NORMAL, FALSE, FALSE, FALSE, DEFAULT_CHARSET,
  OUT_DEFAULT_PRECIS, CLIP_DEFAULT_PRECIS, DEFAULT_QUALITY, DEFAULT_PITCH,
  "Arial"
};

static WDL_PtrArray<const char *, WDL_ARRAYCOUNT(internal_credits)> internal_cr(internal_credits);

int MAR_AboutWnd::Run()
{
  RECT r, fr;
  int half_screen, half_text;

  HWND sta = GetDlgItem(m_hwnd, IDC_STATIC_ABOUT);

  GetWindowRect(sta, &r);
  ScreenToClient(m_hwnd, (LPPOINT)&r);
  ScreenToClient(m_hwnd, ((LPPOINT)&r)+1);

  GetWindowRect(sta, &fr);
  ScreenToClient(m_hwnd, (LPPOINT)&fr);
  ScreenToClient(m_hwnd, ((LPPOINT)&fr)+1);

  LICE_Clear(m_bm, LICE_RGBA(0, 0, 0, 255));

  m_font->SetBkColor(LICE_RGBA(0, 0, 0, 255));
  m_font->SetTextColor(LICE_RGBA(192, 192, 192, m_fontalpha));
  m_font->SetFromHFont(CreateFontIndirect(&internal_lf), LICE_FONT_FLAG_OWNS_HFONT);

  half_screen = (r.right - r.left) / 2;

  fr.top = MAR_ABOUT_HEIGHT - 32;
  fr.bottom = fr.top + MAR_ABOUT_HEIGHT;

  fr.top -= m_scrolloffsety;
  fr.bottom -= m_scrolloffsety;

  for (int i = 0; i < internal_cr.GetSize(); i++)
  {
    if (m_emptystring.Find(i) < 0)
    {
      if (i > m_emptystring.Get()[m_emptystring.GetSize() - 1])
      {
        fr.top += 32;
        fr.bottom += 32;
      }
      else
      {
        fr.top += 16;
        fr.bottom += 16;
      }

      if (fr.top >= -16 && fr.top <= MAR_ABOUT_HEIGHT)
      {
        if (!strcmp(internal_cr.Get()[i],
          "\x47\x69\x6f\x72\x67\x6f\x73\x20\x56\x6f\x75\x67\x69\x6f\x75\x6b\x61\x73")
          && (GetAsyncKeyState(VK_CONTROL) & 0x8000))
        {
          m_font->DrawText(m_bm, internal_different,
            (int)strlen(internal_different),
            &fr, DT_CALCRECT| DT_NOPREFIX | DT_SINGLELINE);
          half_text = (fr.right - fr.left) / 2;
          fr.left = 0;
          fr.left += half_screen - half_text;
          fr.right += half_screen - half_text;
          m_font->DrawText(m_bm, internal_different,
            (int)strlen(internal_different), &fr,
            LICE_DT_NEEDALPHA | LICE_DT_USEFGALPHA |
            DT_NOPREFIX | DT_SINGLELINE);
        }
        else if (!strcmp(internal_cr.Get()[i], "\x4d\x61\x72\x6b\x65\x6c\x6c\x61")
          && (GetAsyncKeyState(VK_CONTROL) & 0x8000))
        {
          m_font->DrawText(m_bm, internal_master_project,
            (int)strlen(internal_master_project),
            &fr, DT_CALCRECT| DT_NOPREFIX | DT_SINGLELINE);
          half_text = (fr.right - fr.left) / 2;
          fr.left = 0;
          fr.left += half_screen - half_text;
          fr.right += half_screen - half_text;
          m_font->DrawText(m_bm, internal_master_project,
            (int)strlen(internal_master_project), &fr,
            LICE_DT_NEEDALPHA | LICE_DT_USEFGALPHA |
            DT_NOPREFIX | DT_SINGLELINE);
        }
        else
        {
          m_font->DrawText(m_bm, internal_cr.Get()[i],
            (int)strlen(internal_cr.Get()[i]),
            &fr, DT_CALCRECT| DT_NOPREFIX | DT_SINGLELINE);
          half_text = (fr.right - fr.left) / 2;
          fr.left = 0;
          fr.left += half_screen - half_text;
          fr.right += half_screen - half_text;
          m_font->DrawText(m_bm, internal_cr.Get()[i],
            (int)strlen(internal_cr.Get()[i]), &fr,
            LICE_DT_NEEDALPHA | LICE_DT_USEFGALPHA |
            DT_NOPREFIX | DT_SINGLELINE);
        }
      }
    }
    else
    {
      if (i == m_emptystring.Get()[m_emptystring.GetSize() - 1])
      {
        fr.top += MAR_ABOUT_HEIGHT - 60;
        fr.bottom += MAR_ABOUT_HEIGHT - 60;
      }
      else
      {
        fr.top += 32;
        fr.bottom += 32;
      }
    }
  }

  if (m_calc)
  {
    m_creditsheight = fr.bottom - (MAR_ABOUT_HEIGHT / 2);
    m_calc = false;
  }

  m_x = r.left;
  m_y = r.top;
  m_w = r.right - r.left;
  m_h = r.bottom - r.top;

  LICE_Blit(m_bm, m_bm, m_x, m_y, m_w, m_h, m_bm->getWidth(),
    m_bm->getHeight(), 1.0f, LICE_BLIT_MODE_COPY);

  if (m_scrolloffsety > m_creditsheight)
  {
    if (m_fontalpha > 220) m_fontalpha -= 1;
    else m_fontalpha -= 20;

    if (m_fontalpha <= 0)
    {
      m_fontalpha = 255;
      m_scrolloffsety = -32;
    }
  }
  else
  {
    m_scrolloffsety++;
  }

  return 1;
}

void MAR_AboutWnd::DoPaint(HDC dc)
{
  BitBlt(dc, m_x, m_y, m_w, m_h, m_bm->getDC(), 0, 0, SRCCOPY);
}

void MAR_AboutWnd::OnInitDialog(WPARAM wparam, LPARAM lparam)
{
  HWND s = GetDlgItem(m_hwnd, IDC_STATIC_ABOUT);

  m_resize.init(m_hwnd);
  m_resize.init_itemhwnd(s);

  int x = GetPrivateProfileInt(MAR_NAME, "about_wnd_x", 0, g_inipath.Get());
  int y = GetPrivateProfileInt(MAR_NAME, "about_wnd_y", 50, g_inipath.Get());
  int w = MAR_ABOUT_WIDTH;
  int h = MAR_ABOUT_HEIGHT;
  SetWindowPos(m_hwnd, NULL, x, y, w, h, SWP_NOZORDER | SWP_NOACTIVATE);

  for (int i = 0; i < internal_cr.GetSize(); i++)
  {
    if (!strcmp(internal_cr.Get()[i], "")) m_emptystring.Add(&i, 1);
  }

  m_bm = new LICE_SysBitmap(w, h);
  m_font = new LICE_CachedFont();

  m_x = x;
  m_y = y;
  m_w = w;
  m_h = h;
  LICE_Clear(m_bm, LICE_RGBA(0, 0, 0, 255));
  SetTimer(m_hwnd, MAR_ABOUT_UPDATE_ID, MAR_ABOUT_UPDATE_MS, NULL);
}

void MAR_AboutWnd::OnPaint(WPARAM wparam, LPARAM lparam)
{
#if defined(_WIN32)
  PAINTSTRUCT ps;
  HWND sta = GetDlgItem(m_hwnd, IDC_STATIC_ABOUT);
  HDC dc = BeginPaint(sta, &ps);
  DoPaint(dc);
  EndPaint(sta, &ps);
#endif
}

void MAR_AboutWnd::OnTimer(WPARAM wparam, LPARAM lparam)
{
  if (wparam == MAR_ABOUT_UPDATE_ID)
  {
    Run();
#if defined(_WIN32)
    InvalidateRect(m_hwnd, NULL, FALSE);
#else
    {
      HWND h = GetDlgItem(m_hwnd, IDC_STATIC_ABOUT);
      HDC dc = GetWindowDC(h);
      DoPaint(dc);
      ReleaseDC(h,dc);
      //SWELL_FlushWindow(h); // for macos
    }
#endif
  }
}

void MAR_AboutWnd::OnSize(WPARAM wparam, LPARAM lparam)
{
  if (wparam != SIZE_MINIMIZED)
  {
    m_resize.onResize();
  }
}

INT_PTR MAR_AboutWnd::OnClose(WPARAM wparam, LPARAM lparam)
{
  DestroyWindow(m_hwnd);
  return 0;
}

void MAR_AboutWnd::OnDestroy(WPARAM wparam, LPARAM lparam)
{
  KillTimer(m_hwnd, MAR_ABOUT_UPDATE_ID);

  RECT r;
  GetWindowRect(m_hwnd, &r);

  WDL_FastString rleft, rtop;
  rleft.SetFormatted(32, "%d", r.left);
  rtop.SetFormatted(32, "%d", r.top);

  WritePrivateProfileString(MAR_NAME, "about_wnd_x", rleft.Get(), g_inipath.Get());
  WritePrivateProfileString(MAR_NAME, "about_wnd_y", rtop.Get(), g_inipath.Get());

  if (g_aboutwnd) { delete g_aboutwnd; g_aboutwnd = NULL; }
}

void MAR_AboutWnd::OnSysCommand(WPARAM wparam, LPARAM lparam)
{
  if (wparam == SC_CLOSE)
  {
    SendMessage(m_hwnd, WM_CLOSE, 0, 0);
  }
}

void MAR_AboutWnd::OnCommand(WPARAM wparam, LPARAM lparam)
{
  switch (LOWORD(wparam))
  {
  case IDCANCEL:
    {
      SendMessage(m_hwnd, WM_CLOSE, 0, 0);
    }
    break;
  }
}

WDL_DLGRET MAR_AboutWnd::AboutWndProc(HWND hwnd, UINT msg, WPARAM wparam, LPARAM lparam)
{
  MAR_AboutWnd *self = (MAR_AboutWnd *)GetWindowLongPtr(hwnd, GWLP_USERDATA);

  if (!self && msg == WM_INITDIALOG)
  {
    SetWindowLongPtr(hwnd, GWLP_USERDATA, lparam);
    self = (MAR_AboutWnd *)lparam;
    self->m_hwnd = hwnd;
  }

  if (WDL_likely(self))
  {
    return self->AboutWndLoop(msg, wparam, lparam);
  }
  else
  {
    return 0;
  }
}

WDL_DLGRET MAR_AboutWnd::AboutWndLoop(UINT msg, WPARAM wparam, LPARAM lparam)
{
  switch (msg)
  {
    case WM_INITDIALOG: OnInitDialog(wparam, lparam); break;
    case WM_PAINT: OnPaint(wparam, lparam); break;
    case WM_TIMER: OnTimer(wparam, lparam); break;
    case WM_SIZE: OnSize(wparam, lparam); break;
    case WM_ERASEBKGND: return 1; break;
    case WM_CLOSE: return OnClose(wparam, lparam); break;
    case WM_DESTROY: OnDestroy(wparam, lparam); break;
    case WM_SYSCOMMAND: OnSysCommand(wparam, lparam); break;
    case WM_COMMAND: OnCommand(wparam, lparam); break;
  }

  return 0;
}
