// Copyright (c) 2024 Giorgos Vougioukas
//
// The license can be found in the LICENSE file.

#include "gap.h"

#define MAR_GAPSIZE 24

MAR_Gap::MAR_Gap()
  : m_size(MAR_GAPSIZE)
  , m_startpos(0)
  , m_endpos(0)
  , m_flushed(true)
  , m_strbuf(NULL)
{}

MAR_Gap::~MAR_Gap()
{}

int MAR_Gap::GetSize() const
{
  return m_size;
}

int MAR_Gap::GetLength() const
{
  if (!m_flushed)
  {
    WDL_ASSERT(m_endpos >= m_startpos);
    int used = m_endpos - m_startpos;
    WDL_ASSERT(used <= m_size);
    return used;
  }
  return -1;
}

int MAR_Gap::GetPosition() const
{
  if (!m_flushed)
  {
    WDL_ASSERT(m_startpos <= m_strbuf->GetLength());
    return m_startpos;
  }
  return -1;
}

void MAR_Gap::Attach(MAR_GapString *strbuf)
{
  if (!m_strbuf)
  {
    m_strbuf = strbuf;
    New(0);
  }
}

void MAR_Gap::Insert(const char *str, int position, int maxlen)
{
  if (position == m_endpos)
  {
    if (maxlen > m_startpos + m_size - m_endpos)
    {
      int newpos = Flush(position);
      WDL_ASSERT(newpos <= m_strbuf->GetLength());
      m_strbuf->Insert(str, newpos, maxlen);
      New(newpos + maxlen);
    }
    else if (maxlen <= m_startpos + m_size - m_endpos)
    {
      memcpy(m_strbuf->Get() + m_endpos, str, maxlen);
      m_endpos += maxlen;
      if (m_endpos == m_startpos + m_size)
      {
        New(m_endpos);
      }
    }
  }
  else
  {
    int newpos = Flush(position);
    if (maxlen > m_startpos + m_size - m_endpos)
    {
      WDL_ASSERT(newpos <= m_strbuf->GetLength());
      m_strbuf->Insert(str, newpos, maxlen);
      New(newpos + maxlen);
    }
    else if (maxlen <= m_startpos + m_size - m_endpos)
    {
      New(newpos);
      memcpy(m_strbuf->Get() + m_endpos, str, maxlen);
      m_endpos += maxlen;
    }
  }
}

void MAR_Gap::Delete(int len)
{
  m_endpos -= len;
  const int unused = m_startpos + m_size - m_endpos;
  if (WDL_NORMALLY(unused > 0))
  {
    memset(m_strbuf->Get() + m_endpos, '$', unused);
  }
}

int MAR_Gap::Flush(int position)
{
  WDL_ASSERT(position >= 0);
  WDL_ASSERT(position <= m_strbuf->GetLength());
  int newpos = position;
  if (!m_flushed)
  {
    const int unused = m_startpos + m_size - m_endpos;
    if (WDL_NORMALLY(unused > 0))
    {
#if defined(_DEBUG)
      MAR_GapString gapcheck;
      gapcheck.SetLen(unused, false, '$');
      WDL_ASSERT(!strncmp(gapcheck.Get(), m_strbuf->Get() + m_endpos, unused));
      WDL_ASSERT(m_endpos <= m_strbuf->GetLength() - unused);
#endif
      m_strbuf->DeleteSub(m_endpos, unused);
      if (newpos > m_startpos + m_size) newpos -= unused;
      m_startpos = m_endpos = -1;
      m_flushed = true;
    }
  }
  return newpos;
}

bool MAR_Gap::IsFlushed() const
{
  return m_flushed;
}

void MAR_Gap::New(int position)
{
  WDL_ASSERT(m_strbuf);
  m_startpos = position;
  m_endpos = position;
  m_flushed = false;
  MAR_GapString gapinit;
  gapinit.SetLen(m_size, false, '$');
  m_strbuf->Insert(&gapinit, m_startpos, gapinit.GetLength());
}
