// Copyright (c) 2023 Giorgos Vougioukas
//
// The license can be found in the LICENSE file.

#include "line_wnd.h"
#include "app_timers.h"
#include "volissos.h"

#include <WDL/wingui/scrollbar/coolscroll.h>
#include <WDL/wdlutf8.h>

static const char *s_courier = "Courier";
static const char *s_couriernew = "Courier New";
static const char *s_consolas = "Consolas";

static LOGFONT lf =
{
#if defined(_WIN32)
  14, 0, 0, 0, FW_NORMAL, FALSE, FALSE, FALSE, DEFAULT_CHARSET,
  OUT_DEFAULT_PRECIS, CLIP_DEFAULT_PRECIS, DEFAULT_QUALITY, DEFAULT_PITCH,
  "Consolas"
#else
  11, 0, 0, 0, FW_NORMAL, FALSE, FALSE, FALSE, DEFAULT_CHARSET,
  OUT_DEFAULT_PRECIS, CLIP_DEFAULT_PRECIS, DEFAULT_QUALITY, DEFAULT_PITCH,
  "Arial"
#endif
};

static WNDPROC PrevLineWndProc = NULL;
static HWND s_linewnd = NULL;

MAR_LineWnd::MAR_LineWnd(HWND hwnd)
  : m_hwnd(hwnd)
  , m_bm(NULL)
  , m_font(NULL)
  , m_letterwidth(0)
  , m_letterheight(0)
  , m_top(0)
  , m_bottom(0)
  , m_curline(0)
  , m_maxwidth(0)
  , m_startspace(5)
{}

MAR_LineWnd::~MAR_LineWnd()
{
  if (m_bm) delete m_bm;
  if (m_font) delete m_font;
  m_lines.Empty(true);
  m_linesempty.Empty(true);
}

void MAR_LineWnd::Start(int line_start, int lines, int current_line, int column)
{
  m_curline = current_line;

  for (int i = 0; i < m_lines.GetSize(); i++)
  {
    WDL_FastString *str = m_lines.Get(i);
    if (str)
    {
      m_linesempty.Add(str);
    }
  }

  m_lines.Empty(false);

  for (int i = 0; i < lines; i++)
  {
    WDL_FastString *str = NULL;
    if (m_linesempty.GetSize())
    {
      str = m_linesempty.Get(m_linesempty.GetSize() - 1);
      m_linesempty.Delete(m_linesempty.GetSize() - 1, false);
    }
    else
    {
      str = new WDL_NEW WDL_FastString;
    }

    if (str)
    {
      WDL_FastString col;
      col.SetFormatted(128, "%03d", column);

      if (i != m_curline)
      {
        WDL_FastString pad;
        pad.SetLen(col.GetLength(), false, ' ');
        str->SetFormatted(128, "%s %04d", pad.Get(), line_start + i);
      }
      else
      {
        str->SetFormatted(128, "%s:%04d", col.Get(), line_start + i);
      }

      m_lines.Add(str);
    }
  }

  WDL_FastString *longest = m_lines.Get(m_lines.GetSize() - 1);
  if (longest)
  {
    RECT lr;
    m_font->DrawText(m_bm, longest->Get(), longest->GetLength(), &lr, DT_CALCRECT);
    m_maxwidth = lr.right - lr.left + m_startspace;
  }
}

int MAR_LineWnd::GetMaximumWidth() const
{
  return m_maxwidth;
}

void MAR_LineWnd::OnCreate(WPARAM wparam, LPARAM lparam)
{
  int text_font = 0;
  int text_font_height = 14;

  switch (text_font)
  {
  case 1:
    {
      int mx = (int)strlen(s_courier);
      WDL_ASSERT(mx > 0 && mx < 32);
      for (int i = 0; i < mx; i++)
      {
        lf.lfFaceName[i] = s_courier[i];
      }
      lf.lfFaceName[mx] = '\0';
      lf.lfHeight = text_font_height;
    } break;
  case 2:
    {
      int mx = (int)strlen(s_couriernew);
      WDL_ASSERT(mx > 0 && mx < 32);
      for (int i = 0; i < mx; i++)
      {
        lf.lfFaceName[i] = s_couriernew[i];
      }
      lf.lfFaceName[mx] = '\0';
      lf.lfHeight = text_font_height;
    } break;
  case 3:
    {
      int mx = (int)strlen(s_consolas);
      WDL_ASSERT(mx > 0 && mx < 32);
      for (int i = 0; i < mx; i++)
      {
        lf.lfFaceName[i] = s_consolas[i];
      }
      lf.lfFaceName[mx] = '\0';
      lf.lfHeight = text_font_height;
    } break;
  }

  m_bm = new LICE_SysBitmap();
  m_font = new LICE_CachedFont();
  m_font->SetFromHFont(CreateFontIndirect(&lf), LICE_FONT_FLAG_OWNS_HFONT);

  RECT let;
  m_font->DrawText(m_bm, "V", 1, &let, DT_CALCRECT);
  const int heightspace = 1;
  m_letterwidth = let.right - let.left;
  m_letterheight = let.bottom - let.top + heightspace;

  InitializeCoolSB(m_hwnd);

  RECT r;
  GetWindowRect(m_hwnd, &r);
  ScreenToClient(m_hwnd, (LPPOINT)&r);
  ScreenToClient(m_hwnd, ((LPPOINT)&r)+1);

  m_bm->resize(r.right - r.left, r.bottom - r.top);

  PrevLineWndProc = (WNDPROC)SetWindowLongPtrW(m_hwnd,
    GWLP_WNDPROC, (LONG_PTR)NewLineWndProc);
  s_linewnd = m_hwnd;
  //SetTimer(m_hwnd, MAR_LINE_UPDATE_ID, MAR_LINE_UPDATE_MS, FALSE);
}

void MAR_LineWnd::OnDestroy(WPARAM wparam, LPARAM lparam)
{
  //KillTimer(m_hwnd, MAR_LINE_UPDATE_ID);
  UninitializeCoolSB(m_hwnd);
}

void MAR_LineWnd::OnSize(WPARAM wparam, LPARAM lparam)
{
  if (wparam != SIZE_MINIMIZED)
  {
    m_resize.onResize();

    RECT r;
    GetClientRect(m_hwnd, &r);
    int w = r.right - r.left;
    int h = r.bottom - r.top;
    m_bm->resize(w, h);

    InvalidateRect(m_hwnd, NULL, FALSE);
  }
}

void MAR_LineWnd::OnCommand(WPARAM wparam, LPARAM lparam)
{}

void MAR_LineWnd::OnPaint(WPARAM wparam, LPARAM lparam)
{
  PAINTSTRUCT ps;
  HDC dc = BeginPaint(m_hwnd, &ps);

  DoPaint(dc);

  EndPaint(m_hwnd, &ps);
}

void MAR_LineWnd::OnLButtonDown(WPARAM wparam, LPARAM lparam)
{
  SetCapture(m_hwnd);

  POINT pt;
  pt.x = GET_X_LPARAM(lparam);
  pt.y = GET_Y_LPARAM(lparam);
}

void MAR_LineWnd::OnLButtonUp(WPARAM wparam, LPARAM lparam)
{
  POINT pt;
  pt.x = GET_X_LPARAM(lparam);
  pt.y = GET_Y_LPARAM(lparam);

  if (GetCapture())
  {
    ReleaseCapture();
  }
}

void MAR_LineWnd::OnMouseMove(WPARAM wparam, LPARAM lparam)
{
  SetCursor(LoadCursor(NULL, IDC_IBEAM));

  if (GetCapture())
  {

  }
}

void MAR_LineWnd::OnTimer(WPARAM wparam, LPARAM lparam)
{
  if (wparam == MAR_LINE_UPDATE_ID)
  {
    InvalidateRect(m_hwnd, NULL, FALSE);
  }
}

void MAR_LineWnd::OnKeyDown(WPARAM wparam, LPARAM lparam)
{
  //switch (wparam)
  //{}
}

void MAR_LineWnd::DoPaint(HDC dc)
{
  if (m_bm)
  {
    RECT r;
    GetClientRect(m_hwnd, &r);
    int w = r.right - r.left;
    int h = r.bottom - r.top;

    RECT let;
    m_font->SetBkMode(TRANSPARENT);
    m_font->SetTextColor(LICE_RGBA(70, 70, 70, 255));
    m_font->DrawText(m_bm, "V", 1, &let, DT_CALCRECT);
    const int heightspace = 1;
    m_letterwidth = let.right - let.left;
    m_letterheight = let.bottom - let.top + heightspace;
    m_textheight = h / m_letterheight;
    const int topmargin = 5;
    const int bottommargin = 3;
    const int top = topmargin;
    const int bottom = m_textheight - bottommargin;

    if (m_lines.GetSize())
    {
      LICE_Clear(m_bm, LICE_RGBA(0, 0, 0, 255));

      RECT lnr;
      for (int i = 0; i < m_lines.GetSize(); i++)
      {
        WDL_FastString *str = m_lines.Get(i);
        if (str)
        {
          m_font->DrawText(m_bm, str->Get(), str->GetLength(), &lnr, DT_CALCRECT);
          const int linewidth = lnr.right - lnr.left;
          //lnr.left = r.left + m_startspace;
          //lnr.right = lnr.left + linewidth;
          lnr.left = r.right - linewidth;
          lnr.right = r.right;
          lnr.top = i * m_letterheight;
          lnr.bottom = lnr.top + m_letterheight;
          if (i == m_curline) m_font->SetTextColor(LICE_RGBA(128, 128, 255, 255));
          m_font->DrawText(m_bm, str->Get(), str->GetLength(), &lnr, 0);
          m_font->SetTextColor(LICE_RGBA(70, 70, 70, 255));
        }
      }
    }
    else
    {
      LICE_Clear(m_bm, LICE_RGBA(48, 25, 52, 255));
    }

    BitBlt(dc, r.left, r.top, w, h, m_bm->getDC(), 0, 0, SRCCOPY);
  }
}

WDL_DLGRET MAR_LineWnd::NewLineWndProc(HWND hwnd, UINT msg, WPARAM wparam, LPARAM lparam)
{
  switch (msg)
  {
  case WM_KEYDOWN:
    {
      switch(wparam)
      {
      case VK_DELETE:
        {
          if (!(GetAsyncKeyState(VK_CONTROL) & 0x8000) &&
            !(GetAsyncKeyState(VK_SHIFT) & 0x8000) &&
            !(GetAsyncKeyState(VK_MENU) & 0x8000))
          {
            if (!GetCapture())
            {
              wdl_log("delete\n");
            }
          }
        }
        break;
      case VK_INSERT:
        {
          if (!(GetAsyncKeyState(VK_CONTROL) & 0x8000) &&
            !(GetAsyncKeyState(VK_SHIFT) & 0x8000) &&
            !(GetAsyncKeyState(VK_MENU) & 0x8000))
          {
            if (!GetCapture())
            {
              wdl_log("insert\n");
            }
          }
        }
        break;
      }
    }
    break;
  case WM_CHAR:
    {
      switch(wparam)
      {
      case VK_BACK:
        {
          if (!(GetAsyncKeyState(VK_CONTROL) & 0x8000) &&
            !(GetAsyncKeyState(VK_SHIFT) & 0x8000) &&
            !(GetAsyncKeyState(VK_MENU) & 0x8000))
          {
            if (!GetCapture())
            {
              wdl_log("back\n");
            }
          }
        }
        break;
      default:
        {
          wchar_t wc[2];
          wc[0] = (wchar_t)wparam;
          wc[1] = L'\0';
          char mb[32];
          int len = WDL_WideToMBStr(mb, wc, sizeof(mb));
        }
        break;
      }
    }
    break;
  }

  return CallWindowProc(PrevLineWndProc, hwnd, msg, wparam, lparam);
}

WDL_DLGRET MAR_LineWnd::LineWndProc(HWND hwnd, UINT msg, WPARAM wparam, LPARAM lparam)
{
  g_linewnd = (MAR_LineWnd *)GetWindowLongPtr(hwnd, GWLP_USERDATA);

#ifdef _WIN32
  if (!g_linewnd && msg == WM_NCCREATE)
#else
  if (!g_linewnd && msg == WM_CREATE)
#endif
  {
    g_linewnd = new MAR_LineWnd(hwnd);
    SetWindowLongPtr(hwnd, GWLP_USERDATA, (INT_PTR)g_linewnd);
#ifndef _WIN32
    //SWELL_SetClassName(hwnd, "waveform_superhost");
#endif
    if (g_linewnd) g_linewnd->OnCreate(wparam, lparam);
    return (g_linewnd != NULL);
  }

#ifdef _WIN32
  if (g_linewnd && msg == WM_NCDESTROY)
#else
  if (g_linewnd && msg == WM_DESTROY)
#endif
  {
    g_linewnd->OnDestroy(wparam, lparam);
    delete g_linewnd; g_linewnd = NULL;
  }

  if (WDL_likely(g_linewnd))
  {
    return g_linewnd->LineWndLoop(msg, wparam, lparam);
  }
  else
  {
    return 0;
  }
}

WDL_DLGRET MAR_LineWnd::LineWndLoop(UINT msg, WPARAM wparam, LPARAM lparam)
{
  switch (msg)
  {
#ifdef _WIN32
    case WM_NCCREATE: OnCreate(wparam, lparam); break;
    case WM_NCDESTROY: OnDestroy(wparam, lparam); break;
#else
    case WM_CREATE: OnCreate(wparam, lparam); break;
    case WM_DESTROY: OnDestroy(wparam, lparam); break;
#endif
    case WM_SIZE: OnSize(wparam, lparam); break;
    case WM_COMMAND: OnCommand(wparam, lparam); break;
    case WM_PAINT: OnPaint(wparam, lparam); break;
    case WM_LBUTTONDOWN: OnLButtonDown(wparam, lparam); break;
    case WM_LBUTTONUP: OnLButtonUp(wparam, lparam); break;
    case WM_MOUSEMOVE: OnMouseMove(wparam, lparam); break;
    case WM_TIMER: OnTimer(wparam, lparam); break;
    case WM_KEYDOWN: OnKeyDown(wparam, lparam); break;
  }

  return DefWindowProc(m_hwnd, msg, wparam, lparam);
}
