// Copyright (c) 2021 Giorgos Vougioukas
//
// The license can be found in the LICENSE file.

#include "main_wnd.h"
#include "app_timers.h"
#include "app_info.h"
#include "about_wnd.h"
#include "changelog_wnd.h"
#include "license_wnd.h"
#include "preferences_wnd.h"
#include "preferences.h"
#include "keymap.h"
#include "volissos.h"
#include "line_wnd.h"
#include "folders_wnd.h"
#include "switch_wnd.h"
#include "find_wnd.h"
#include "change_wnd.h"
#include "goto_wnd.h"

#include <WDL/wdlstring.h>
#include <WDL/time_precise.h>
#include <WDL/filebrowse.h>
#include <WDL/lice/lice.h>
#include <WDL/wingui/virtwnd.h>
#include <WDL/wingui/scrollbar/coolscroll.h>
#include <WDL/wdlutf8.h>
#include <WDL/time_precise.h>
#include <WDL/fileread.h>

#define MAR_TEXT_FILES "All supported files\0*.cpp;*.cc;*.cxx;*.hxx;*.hpp;*.hh;*.c;*.h;*.asm;*.s;*.S;*.bat;*.cmd;*.txt\0"\
  "C++ files (*.cpp;*.cc;*.h;*.cxx;*.hxx;*.hpp;*.hh)\0*.cpp;*.cc;*.h;*.cxx;*.hxx;*.hpp;*.hh\0"\
  "C files (*.c;*.h)\0*.c;*.h\0"\
  "Assembly files (*.asm;*.s;*.S)\0*.asm;*.s;*.S\0"\
  "Batch files (*.bat;*.cmd)\0*.bat;*.cmd\0"\
  "Text files (*.txt)\0*.txt\0"\
  "All files (*.*)\0*.*\0\0"
#define MAR_DEFAULT_EXT "cpp"

static WNDPROC Internal_PrevTreeProc = NULL;
static HWND internal_tree = NULL;
static WNDPROC Internal_PrevListProc = NULL;
static HWND internal_list = NULL;
static WNDPROC Internal_PrevEditProc = NULL;
static HWND internal_edit = NULL;
static bool internal_istreeroot = false;

static LOGFONT internal_lf =
{
#if defined(_WIN32)
  14, 0, 0, 0, FW_NORMAL, FALSE, FALSE, FALSE, DEFAULT_CHARSET,
  OUT_DEFAULT_PRECIS, CLIP_DEFAULT_PRECIS, DEFAULT_QUALITY, DEFAULT_PITCH,
  "Arial"
#else
  11, 0, 0, 0, FW_NORMAL, FALSE, FALSE, FALSE, DEFAULT_CHARSET,
  OUT_DEFAULT_PRECIS, CLIP_DEFAULT_PRECIS, DEFAULT_QUALITY, DEFAULT_PITCH,
  "Arial"
#endif
};

MAR_MainWnd::MAR_MainWnd()
  : m_hwnd(NULL)
  , m_line(NULL)
  , m_surface(NULL)
  , m_splitter(false)
  , m_x(0)
  , m_y(0)
  , m_w(0)
  , m_h(0)
  , m_mainmenu(NULL)
  , m_font(NULL)
  , m_fullscreen(false)
  , m_maximized(false)
  , m_showproject(false)
  , m_linenumber(false)
  , m_bg(0x000000)
  , m_filename(false)
{}

MAR_MainWnd::~MAR_MainWnd()
{
  if (m_font) delete m_font;
}

void MAR_MainWnd::ShowProjects()
{
  if (!g_folderswnd)
  {
    g_folderswnd = new MAR_FoldersWnd;
    CreateDialogParam(g_inst, MAKEINTRESOURCE(IDD_FOLDERS), m_hwnd,
      MAR_FoldersWnd::FoldersWndProc, (LPARAM)g_folderswnd);
    ShowWindow(g_folderswnd->Handle(), SW_SHOW);
  }
  else
  {
    SetFocus(g_folderswnd->Handle());
  }
}

void MAR_MainWnd::ShowSwitch()
{
  if (!g_switchwnd)
  {
    g_switchwnd = new MAR_SwitchWnd;
    CreateDialogParam(g_inst, MAKEINTRESOURCE(IDD_SWITCH), m_hwnd,
      MAR_SwitchWnd::SwitchWndProc, (LPARAM)g_switchwnd);
    ShowWindow(g_switchwnd->Handle(), SW_SHOW);
  }
  else
  {
    SetFocus(g_switchwnd->Handle());
  }
}

const char *MAR_MainWnd::GetOrigTitle() const
{
  return m_titleorig.Get();
}

int MAR_MainWnd::GetProjectsSize() const
{
  return m_folders.GetSize();
}

MAR_RootFolder *MAR_MainWnd::GetProject(int index) const
{
  if (WDL_NORMALLY(index < m_folders.GetSize()))
  {
    MAR_RootFolder *rf = m_folders.Get(index);
    if (rf) return rf;
  }

  return NULL;
}

void MAR_MainWnd::SetTitle(const char *title)
{
  SetWindowText(m_hwnd, title);
  SetTimer(m_hwnd, MAR_ORIGINAL_TITLE, MAR_ORIGINAL_TITLE_MS, NULL);
}

void MAR_MainWnd::DisplayFileName()
{
  if (g_volissos)
  {
    if (m_filename)
    {
      MAR_GapBuffer *gb = g_volissos->GetGapBuffer();
      if (gb)
      {
        WDL_FastString str;
        str.SetFormatted(2048, "%s - %s", gb->GetFileName(), m_titleorig.Get());
        SetWindowText(m_hwnd, str.Get());
      }
      else
      {
        SetWindowText(m_hwnd, m_titleorig.Get());
      }
    }
    else
    {
      SetWindowText(m_hwnd, m_titleorig.Get());
    }
  }
}

void MAR_MainWnd::ToggleFullscreen()
{
  m_fullscreen = !m_fullscreen;
  if (m_fullscreen)
  {
    RECT r, fsr;
    m_maximized = !!IsZoomed(m_hwnd);

    if (m_maximized)
    {
      SendMessage(m_hwnd, WM_SYSCOMMAND, SC_RESTORE, 0);
    }

    m_style = GetWindowLongPtr(m_hwnd, GWL_STYLE);
    m_exstyle = GetWindowLongPtr(m_hwnd, GWL_EXSTYLE);
    GetWindowRect(m_hwnd, &r);

    SetWindowLongPtr(m_hwnd, GWL_STYLE, m_style & ~(WS_CAPTION | WS_THICKFRAME));
    SetWindowLongPtr(m_hwnd, GWL_EXSTYLE, m_exstyle & ~(WS_EX_DLGMODALFRAME |
      WS_EX_WINDOWEDGE | WS_EX_CLIENTEDGE | WS_EX_STATICEDGE));

    MONITORINFO mi;
    mi.cbSize = sizeof(mi);
    GetMonitorInfo(MonitorFromWindow(m_hwnd, MONITOR_DEFAULTTONEAREST), &mi);

    CopyRect(&fsr, &mi.rcMonitor);
    SetWindowPos(m_hwnd, NULL, fsr.left, fsr.top, fsr.right - fsr.left, fsr.bottom - fsr.top,
      SWP_NOZORDER | SWP_NOACTIVATE | SWP_FRAMECHANGED);

    m_x = r.left;
    m_y = r.top;
    m_w = r.right - r.left;
    m_h = r.bottom - r.top;
  }
  else
  {
    SetWindowLongPtr(m_hwnd, GWL_STYLE, m_style);
    SetWindowLongPtr(m_hwnd, GWL_EXSTYLE, m_exstyle);
    SetWindowPos(m_hwnd, NULL, m_x, m_y, m_w, m_h,
      SWP_NOZORDER | SWP_NOACTIVATE | SWP_FRAMECHANGED);
    if (m_maximized) SendMessage(m_hwnd, WM_SYSCOMMAND, SC_MAXIMIZE, 0);
  }
}

void MAR_MainWnd::ToggleLineNumber()
{
  BOOL grayed = GetMenuState(m_mainmenu, ID_VIEW_TOGGLELINENUMBER, FALSE) & MFS_GRAYED;
  if (!grayed)
  {
    m_linenumber = !m_linenumber;
  }
}

void MAR_MainWnd::ToggleFileName()
{
  BOOL grayed = GetMenuState(m_mainmenu, ID_VIEW_TOGGLEFILENAME, FALSE) & MFS_GRAYED;
  if (!grayed)
  {
    m_filename = !m_filename;
    DisplayFileName();
  }
}

void MAR_MainWnd::ToggleRuler()
{
  BOOL grayed = GetMenuState(m_mainmenu, ID_VIEW_TOGGLERULER, FALSE) & MFS_GRAYED;
  if (!grayed)
  {
    if (g_surfacewnd) g_surfacewnd->ToggleRuler();
  }
}

void MAR_MainWnd::ToggleWhitespace()
{
  BOOL grayed = GetMenuState(m_mainmenu, ID_VIEW_TOGGLEWHITESPACE, FALSE) & MFS_GRAYED;
  if (!grayed)
  {
    if (g_surfacewnd) g_surfacewnd->ToggleWhitespace();
  }
}

void MAR_MainWnd::FindKeyWord()
{
  if (g_volissos && g_volissos->GetGapBufferSize())
  {
    if (!g_findwnd)
    {
      g_findwnd = new MAR_FindWnd;
      CreateDialogParam(g_inst, MAKEINTRESOURCE(IDD_FIND), m_hwnd,
        MAR_FindWnd::FindWndProc, (LPARAM)g_findwnd);
      ShowWindow(g_findwnd->Handle(), SW_SHOW);
    }
    else
    {
      SetFocus(g_findwnd->Handle());
    }
  }
}

void MAR_MainWnd::ChangeKeyWord()
{
  if (g_volissos && g_volissos->GetGapBufferSize())
  {
    if (!g_changewnd)
    {
      g_changewnd = new MAR_ChangeWnd;
      CreateDialogParam(g_inst, MAKEINTRESOURCE(IDD_CHANGE), m_hwnd,
        MAR_ChangeWnd::ChangeWndProc, (LPARAM)g_changewnd);
      ShowWindow(g_changewnd->Handle(), SW_SHOW);
    }
    else
    {
      SetFocus(g_changewnd->Handle());
    }
  }
}

void MAR_MainWnd::GoToLine()
{
  if (g_volissos && g_volissos->GetGapBufferSize())
  {
    if (!g_gotownd)
    {
      g_gotownd = new MAR_GoToWnd;
      CreateDialogParam(g_inst, MAKEINTRESOURCE(IDD_GOTO), m_hwnd,
        MAR_GoToWnd::GoToWndProc, (LPARAM)g_gotownd);
      ShowWindow(g_gotownd->Handle(), SW_SHOW);
    }
    else
    {
      SetFocus(g_gotownd->Handle());
    }
  }
}

void MAR_MainWnd::SwitchToPreviousFile()
{
  if (g_volissos) g_volissos->SwitchToPrevious();
}

void MAR_MainWnd::ReopenFile()
{
  if (g_volissos) g_volissos->Reopen();
}

void MAR_MainWnd::OnInitDialog(WPARAM wparam, LPARAM lparam)
{
  double wall0 = time_precise();

  if (MAR_IsStableRelease())
  {
    m_titleorig.Set(MAR_NAME_MARKETING);
  }
  else
  {
    m_titleorig.Set(MAR_FULL_VERSION " (" MAR_ARCH ")");
  }
  SetWindowText(m_hwnd, m_titleorig.Get());

  m_font = new LICE_CachedFont();
  m_font->SetFromHFont(CreateFontIndirect(&internal_lf), LICE_FONT_FLAG_OWNS_HFONT);

  m_line = GetDlgItem(m_hwnd, IDC_CUSTOM2);
  m_surface = GetDlgItem(m_hwnd, IDC_CUSTOM1);

  m_resize.init(m_hwnd);
  m_resize.init_itemhwnd(m_line, 0.0f, 0.0f, 0.0f, 1.0f);
  m_resize.init_itemhwnd(m_surface, 0.0f, 0.0f, 1.0f, 1.0f);

  m_mainmenu = LoadMenu(g_inst, MAKEINTRESOURCE(IDR_MAIN_MENUBAR));
  SetMenu(m_hwnd, m_mainmenu);

  g_keymap = new WDL_NEW MAR_KeyMap;
  if (WDL_NOT_NORMALLY(!g_keymap)) DestroyWindow(m_hwnd);

  g_volissos = new WDL_NEW MAR_Volissos;
  if (WDL_NOT_NORMALLY(!g_volissos)) DestroyWindow(m_hwnd);

  if (g_args.GetLength())
  {
    g_volissos->Open(g_args.Get());
  }

  m_x = GetPrivateProfileInt(MAR_NAME, "main_wnd_x", 50, g_inipath.Get());
  m_y = GetPrivateProfileInt(MAR_NAME, "main_wnd_y", 50, g_inipath.Get());
  m_w = GetPrivateProfileInt(MAR_NAME, "main_wnd_w", 900, g_inipath.Get());
  m_h = GetPrivateProfileInt(MAR_NAME, "main_wnd_h", 900, g_inipath.Get());
  SetWindowPos(m_hwnd, NULL, m_x, m_y, m_w, m_h, SWP_NOACTIVATE);

  int line_x, line_y, line_w, line_h;
  WDL_WndSizer__rec *recline = m_resize.get_itembywnd(m_line);
  line_x = 0;
  line_y = recline->orig.top;
  line_w = 0;
  line_h = recline->orig.bottom - recline->orig.top;
  line_x = GetPrivateProfileInt(MAR_NAME, "line_x", line_x, g_inipath.Get());
  line_y = GetPrivateProfileInt(MAR_NAME, "line_y", line_y, g_inipath.Get());
  line_w = GetPrivateProfileInt(MAR_NAME, "line_w", line_w, g_inipath.Get());
  line_h = GetPrivateProfileInt(MAR_NAME, "line_h", line_h, g_inipath.Get());
  recline->orig.left = line_x;
  recline->orig.right = line_x + line_w;

  int surface_x, surface_y, surface_w, surface_h;
  WDL_WndSizer__rec *recsurface = m_resize.get_itembywnd(m_surface);
  surface_x = recline->orig.right;
  surface_y = recsurface->orig.top;
  surface_w = recsurface->orig.right - recsurface->orig.left;
  surface_h = recsurface->orig.bottom - recsurface->orig.top;
  surface_x = GetPrivateProfileInt(MAR_NAME, "surface_x", surface_x, g_inipath.Get());
  surface_y = GetPrivateProfileInt(MAR_NAME, "surface_y", surface_y, g_inipath.Get());
  surface_w = GetPrivateProfileInt(MAR_NAME, "surface_w", surface_w, g_inipath.Get());
  surface_h = GetPrivateProfileInt(MAR_NAME, "surface_h", surface_h, g_inipath.Get());
  recsurface->orig.left = surface_x;
  m_resize.onResize();

  int maximized = GetPrivateProfileInt(MAR_NAME, "main_wnd_maximized", 0, g_inipath.Get());

  if (maximized)
  {
    ShowWindow(m_hwnd, SW_SHOWMAXIMIZED);
  }
  else
  {
    ShowWindow(m_hwnd, SW_SHOWNORMAL);
  }

  int linenumber = GetPrivateProfileInt(MAR_NAME, "linenumber", 1, g_inipath.Get());
  if (linenumber) ToggleLineNumber();
  int filename = GetPrivateProfileInt(MAR_NAME, "filename", 1, g_inipath.Get());
  if (filename) ToggleFileName();

  //Internal_PrevTreeProc = (WNDPROC)SetWindowLongPtrW(GetDlgItem(m_hwnd, IDC_TREE1),
  //GWLP_WNDPROC, (LONG_PTR)NewTreeProc);

  //Internal_PrevListProc = (WNDPROC)SetWindowLongPtrW(GetDlgItem(m_hwnd, IDC_LIST1),
  //GWLP_WNDPROC, (LONG_PTR)NewListProc);

  //Internal_PrevEditProc = (WNDPROC)SetWindowLongPtrW(GetDlgItem(m_hwnd, IDC_EDIT1),
  //  GWLP_WNDPROC, (LONG_PTR)NewEditProc);

  int bk = 0x000000; //g_preferences->GetBackgroundColor();
  int fg = 0xffffff; //g_preferences->GetTextColor();
  //ListView_SetBkColor(m_surface, RGB((bk >> 16) & 0xFF,(bk >> 8) & 0xFF, bk & 0xFF));
  //ListView_SetTextBkColor(m_surface, RGB((bk >> 16) & 0xFF,(bk >> 8) & 0xFF, bk & 0xFF));
  //ListView_SetTextColor(m_surface, RGB((fg >> 16) & 0xFF,(fg >> 8) & 0xFF, fg & 0xFF));

  double wall1 = time_precise();

  if (WDL_unlikely(!MAR_IsStableRelease()))
  {
    const char warn_msg[] =
    {
      "Welcome to " MAR_NAME_MARKETING " (the \"Software\").\n\n"
      "THIS IS AN UNSTABLE RELEASE STILL IN DEVELOPEMENT PHASE\n"
      "AND IS PROVIDED ON AN \"AS IS\" BASIS AND IS BELIEVED\n"
      "TO CONTAIN DEFECTS AND A PRIMARY PURPOSE OF THIS\n"
      "RELEASE IS TO OBTAIN FEEDBACK ON SOFTWARE PERFORMANCE\n"
      "AND THE IDENTIFICATION OF DEFECTS.\n\n"
      "IT IS ADVISED TO USE CAUTION AND NOT TO RELY IN ANY WAY\n"
      "ON THE CORRECT FUNCTIONING OF THIS RELEASE.\n\n"
      MAR_NAME_MARKETING " version: " MAR_NAKED_VERSION " (" MAR_ARCH ")\n\n"
      "For the latest production release visit: " MAR_WEBSITE_URL "\n\n"
      "Startup time: %.3f sec\n\n"
      MAR_COPYRIGHT
    };

    WDL_FastString tmp;

    tmp.SetFormatted(1024, warn_msg, wall1 - wall0);

    MessageBox(m_hwnd, tmp.Get(), "Release for testing purposes", MB_OK);
  }

  SetTimer(m_hwnd, MAR_MAIN_SCREEN_UPDATE_ID, MAR_MAIN_SCREEN_UPDATE_MS, NULL);
}

void MAR_MainWnd::OnSysCommand(WPARAM wparam, LPARAM lparam)
{
  if (wparam == SC_CLOSE)
  {
    SendMessage(m_hwnd, WM_CLOSE, 0, 0);
  }
}

INT_PTR MAR_MainWnd::OnClose(WPARAM wparam, LPARAM lparam)
{
  DestroyWindow(m_hwnd);
  return 0;
}

void MAR_MainWnd::OnSize(WPARAM wparam, LPARAM lparam)
{
  RECT r;
  GetClientRect(m_hwnd, &r);
  int w = r.right - r.left;
  int h = r.bottom - r.top;

  if (wparam != SIZE_MINIMIZED)
  {
    m_resize.onResize();
  }

  if (wparam != SIZE_MINIMIZED && wparam != SIZE_MAXIMIZED)
  {
    RECT r;
    GetWindowRect(m_hwnd, &r);

    m_x = r.left;
    m_y = r.top;
    m_w = r.right - r.left;
    m_h = r.bottom - r.top;
  }
}

void MAR_MainWnd::OnMove(WPARAM wparam, LPARAM lparam)
{
  int xpos = (int)(short)LOWORD(lparam); // horizontal position
  int ypos = (int)(short)HIWORD(lparam); // vertical position

  if (xpos >= 0 && ypos >= 0 && GetCapture() == m_hwnd)
  {
    RECT r;
    GetWindowRect(m_hwnd, &r);

    m_x = r.left;
    m_y = r.top;
    m_w = r.right - r.left;
    m_h = r.bottom - r.top;
  }
}

void MAR_MainWnd::OnDestroy(WPARAM wparam, LPARAM lparam)
{
  KillTimer(m_hwnd, MAR_MAIN_SCREEN_UPDATE_ID);

#ifdef _WIN32
  WINDOWPLACEMENT wp = { sizeof(wp) };
  GetWindowPlacement(m_hwnd, &wp);

  WDL_FastString xstr, ystr, wstr, hstr;
  xstr.SetFormatted(32, "%d", m_x);
  ystr.SetFormatted(32, "%d", m_y);
  wstr.SetFormatted(32, "%d", m_w);
  hstr.SetFormatted(32, "%d", m_h);

  if (wp.showCmd == SW_SHOWMAXIMIZED)
  {
    WritePrivateProfileString(MAR_NAME, "main_wnd_maximized", "1", g_inipath.Get());
  }
  else
  {
    WritePrivateProfileString(MAR_NAME, "main_wnd_maximized", "0", g_inipath.Get());
    WritePrivateProfileString(MAR_NAME, "main_wnd_x", xstr.Get(), g_inipath.Get());
    WritePrivateProfileString(MAR_NAME, "main_wnd_y", ystr.Get(), g_inipath.Get());
    WritePrivateProfileString(MAR_NAME, "main_wnd_w", wstr.Get(), g_inipath.Get());
    WritePrivateProfileString(MAR_NAME, "main_wnd_h", hstr.Get(), g_inipath.Get());
  }

  RECT r;
  GetWindowRect(m_line, &r);
  ScreenToClient(m_hwnd, (LPPOINT)&r);
  ScreenToClient(m_hwnd, ((LPPOINT)&r)+1);
  xstr.SetFormatted(32, "%d", r.left);
  ystr.SetFormatted(32, "%d", r.top);
  wstr.SetFormatted(32, "%d", r.right - r.left);
  hstr.SetFormatted(32, "%d", r.bottom - r.top);
  WritePrivateProfileString(MAR_NAME, "line_x", xstr.Get(), g_inipath.Get());
  WritePrivateProfileString(MAR_NAME, "line_y", ystr.Get(), g_inipath.Get());
  WritePrivateProfileString(MAR_NAME, "line_w", wstr.Get(), g_inipath.Get());
  WritePrivateProfileString(MAR_NAME, "line_h", hstr.Get(), g_inipath.Get());

  GetWindowRect(m_surface, &r);
  ScreenToClient(m_hwnd, (LPPOINT)&r);
  ScreenToClient(m_hwnd, ((LPPOINT)&r)+1);
  xstr.SetFormatted(32, "%d", r.left);
  ystr.SetFormatted(32, "%d", r.top);
  wstr.SetFormatted(32, "%d", r.right - r.left);
  hstr.SetFormatted(32, "%d", r.bottom - r.top);
  WritePrivateProfileString(MAR_NAME, "surface_x", xstr.Get(), g_inipath.Get());
  WritePrivateProfileString(MAR_NAME, "surface_y", ystr.Get(), g_inipath.Get());
  WritePrivateProfileString(MAR_NAME, "surface_w", wstr.Get(), g_inipath.Get());
  WritePrivateProfileString(MAR_NAME, "surface_h", hstr.Get(), g_inipath.Get());

  if (m_linenumber) WritePrivateProfileString(MAR_NAME, "linenumber", "1", g_inipath.Get());
  else WritePrivateProfileString(MAR_NAME, "linenumber", "0", g_inipath.Get());
  if (m_filename) WritePrivateProfileString(MAR_NAME, "filename", "1", g_inipath.Get());
  else WritePrivateProfileString(MAR_NAME, "filename", "0", g_inipath.Get());
#else
  RECT r;
  GetWindowRect(m_hwnd, &r);

  WDL_FastString xstr, ystr, wstr, hstr;
  xstr.SetFormatted(32, "%d", r.left);
  ystr.SetFormatted(32, "%d", r.top);
  wstr.SetFormatted(32, "%d", r.right - r.left);
  hstr.SetFormatted(32, "%d", r.bottom - r.top);

  if (r.left >= 0 && r.top >= 0)
  {
    WritePrivateProfileString(MAR_NAME, "main_wnd_x", xstr.Get(), g_inipath.Get());
    WritePrivateProfileString(MAR_NAME, "main_wnd_y", ystr.Get(), g_inipath.Get());
    WritePrivateProfileString(MAR_NAME, "main_wnd_w", wstr.Get(), g_inipath.Get());
    WritePrivateProfileString(MAR_NAME, "main_wnd_h", hstr.Get(), g_inipath.Get());
  }

  #ifdef __APPLE__
  SWELL_PostQuitMessage(g_mainwnd);
  #endif
  g_hasrequestedquit = true;
#endif

  if (g_volissos) delete g_volissos;
  if (g_keymap) delete g_keymap;
  if (g_scrollbar) delete g_scrollbar;

  PostQuitMessage(0);
}

void MAR_MainWnd::OnCommand(WPARAM wparam, LPARAM lparam)
{
  switch (LOWORD(wparam))
  {
  //case IDOK:
  //  {
  //    SetDlgItemText(m_hwnd, IDC_EDIT1, "");
  //  }
  //  break;
  case ID_FILE_NEW_PROJECT:
    {
      g_volissos->New();
    }
    break;
  case ID_FILE_OPEN_PROJECT:
    {
      char *fn = NULL;
      bool preservecwd = g_preferences->WantPreserveDirectory();
      fn = WDL_ChooseFileForOpen(m_hwnd, "Open file", NULL, NULL,
        MAR_TEXT_FILES, MAR_DEFAULT_EXT, preservecwd, false);
      if (fn)
      {
        if (!g_volissos->Open(fn))
        {
          MessageBox(m_hwnd, MAR_NAME_MARKETING " cannot open the file.", "Open file", MB_OK);
        }

        free(fn);
      }
    }
    break;
  case ID_FILE_REOPEN:
    {
      ReopenFile();
    }
    break;
  case ID_FILE_OPENFOLDER_PROJECT:
    {
      char fn[2048];
      bool preservecwd = g_preferences->WantPreserveDirectory();
      if (WDL_ChooseDirectory(m_hwnd, "Open folder", NULL, fn, sizeof(fn), preservecwd))
      {
        m_folders.Open(fn);
      }
    }
    break;
  case ID_FILE_SAVE_PROJECT:
    {
      char fn[2048];
      MAR_GapBuffer *gb = g_volissos->GetGapBuffer();
      if (gb)
      {
        if (!gb->Save())
        {
          WDL_FastString oldfn(gb->GetFilePath());
          if (WDL_ChooseFileForSave(m_hwnd, "File save as", NULL, NULL,
            MAR_TEXT_FILES, MAR_DEFAULT_EXT, true, fn, sizeof(fn)))
          {
            if (gb->SaveAs(fn))
            {
              g_volissos->RenameFile(oldfn.Get(), fn);
              g_volissos->ActiveFile(fn);
            }
          }
        }
      }
    }
    break;
  case ID_FILE_SAVEAS_PROJECT:
    {
      char fn[2048];
      MAR_GapBuffer *gb = g_volissos->GetGapBuffer();
      if (gb)
      {
        WDL_FastString oldfn(gb->GetFilePath());
        if (WDL_ChooseFileForSave(m_hwnd, "File save as", NULL, NULL,
          MAR_TEXT_FILES, MAR_DEFAULT_EXT, true, fn, sizeof(fn)))
        {
          if (gb->SaveAs(fn))
          {
            g_volissos->RenameFile(oldfn.Get(), fn);
            g_volissos->ActiveFile(fn);
          }
        }
      }
    }
    break;
  case ID_FILE_SAVEALL_PROJECT:
    {
      char fn[2048];
      for (int i = 0; i < g_volissos->GetGapBufferSize(); i++)
      {
        MAR_GapBuffer *gb = g_volissos->GetGapBuffer(i);
        if (gb)
        {
          if (!gb->Save())
          {
            WDL_FastString oldfn(gb->GetFilePath());
            if (WDL_ChooseFileForSave(m_hwnd, "File save as", NULL, NULL,
              MAR_TEXT_FILES, MAR_DEFAULT_EXT, true, fn, sizeof(fn)))
            {
              if (gb->SaveAs(fn))
              {
                g_volissos->RenameFile(oldfn.Get(), fn);
                g_volissos->ActiveFile(fn);
              }
            }
          }
        }
      }
    }
    break;
  case ID_FILE_CLOSEFILE:
    {
      g_volissos->Close();
    }
    break;
  case ID_FILE_CLOSETHEREST:
    {
      g_volissos->CloseTheRest();
    }
    break;
  case ID_FILE_EXIT:
    {
      if (m_fullscreen) ToggleFullscreen();
      SendMessage(m_hwnd, WM_CLOSE, 0, 0);
    }
    break;
  case ID_EDIT_CUTTEXT:
    {
      MAR_GapBuffer *gb = g_volissos->GetGapBuffer();
      if (gb)
      {
        gb->Cut();
      }
    }
    break;
  case ID_EDIT_COPYTEXT:
    {
      MAR_GapBuffer *gb = g_volissos->GetGapBuffer();
      if (gb)
      {
        gb->Copy();
      }
    }
    break;
  case ID_EDIT_PASTETEXT:
    {
      MAR_GapBuffer *gb = g_volissos->GetGapBuffer();
      if (gb)
      {
        gb->Paste();
      }
    }
    break;
  case ID_EDIT_SELECTALL:
    {
      MAR_GapBuffer *gb = g_volissos->GetGapBuffer();
      if (gb)
      {
        gb->SelectAll();
      }
    }
    break;
  case ID_EDIT_FINDTEXT:
    {
      FindKeyWord();
    }
    break;
  case ID_EDIT_CHANGETEXT:
    {
      ChangeKeyWord();
    }
    break;
  case ID_EDIT_GOTOLINE:
    {
      GoToLine();
    }
    break;
  case ID_EDIT_COMMENT:
    {
      MAR_GapBuffer *gb = g_volissos->GetGapBuffer();
      if (gb)
      {
        gb->CommentLine();
      }
    }
    break;
  case ID_EDIT_UNCOMMENT:
    {
      MAR_GapBuffer *gb = g_volissos->GetGapBuffer();
      if (gb)
      {
        gb->UncommentLine();
      }
    }
    break;
  case ID_VIEW_PREVIOUS:
    {
      SwitchToPreviousFile();
    }
    break;
  case ID_VIEW_SWITCH:
    {
      ShowSwitch();
    }
    break;
  case ID_VIEW_FOLDERS:
    {
      ShowProjects();
    }
    break;
  case ID_VIEW_TOGGLEFULLSCREEN:
    {
      ToggleFullscreen();
    }
    break;
  case ID_VIEW_TOGGLELINENUMBER:
    {
      ToggleLineNumber();
    }
    break;
  case ID_VIEW_TOGGLEFILENAME:
    {
      ToggleFileName();
    }
    break;
  case ID_VIEW_TOGGLERULER:
    {
      ToggleRuler();
    }
    break;
  case ID_VIEW_TOGGLEWHITESPACE:
    {
      ToggleWhitespace();
    }
    break;
  case ID_OPTIONS_PREFERENCES:
    {
      if (!g_prefwnd)
      {
        g_prefwnd = new MAR_PreferencesWnd;
        CreateDialogParam(g_inst, MAKEINTRESOURCE(IDD_PREFERENCES), m_hwnd,
          MAR_PreferencesWnd::PreferencesWndProc, (LPARAM)g_prefwnd);
        ShowWindow(g_prefwnd->Handle(), SW_SHOW);
      }
      else
      {
        SetFocus(g_prefwnd->Handle());
      }
    }
    break;
  case ID_HELP_DOCUMENTATION:
    {
      WDL_FastString doc;
      doc.Set(g_modpath.Get());
      doc.Append("documentation.txt");

      ShellExecute(m_hwnd, "", "notepad.exe", doc.Get(), "", SW_SHOWNORMAL);
    }
    break;
  case ID_HELP_ABOUT:
    {
      if (!g_aboutwnd)
      {
        g_aboutwnd = new MAR_AboutWnd;
        CreateDialogParam(g_inst, MAKEINTRESOURCE(IDD_ABOUT), m_hwnd,
          MAR_AboutWnd::AboutWndProc, (LPARAM)g_aboutwnd);
        ShowWindow(g_aboutwnd->Handle(), SW_SHOW);
      }
      else
      {
        SetFocus(g_aboutwnd->Handle());
      }
    }
    break;
  case ID_HELP_CHANGELOG:
    {
      if (!g_changelogwnd)
      {
        g_changelogwnd = new MAR_ChangelogWnd;
        CreateDialogParam(g_inst, MAKEINTRESOURCE(IDD_CHANGELOG), m_hwnd,
          MAR_ChangelogWnd::ChangelogWndProc, (LPARAM)g_changelogwnd);
        ShowWindow(g_changelogwnd->Handle(), SW_SHOW);
      }
      else
      {
        SetFocus(g_changelogwnd->Handle());
      }
    }
    break;
  case ID_HELP_VERSION:
    {
      const char version_str[] =
      {
        MAR_NAME_MARKETING " version: " MAR_NAKED_VERSION " (" MAR_ARCH ")\n\n"
        "Build timestamp: " MAR_TIMESTAMP "\n"
        "Git commit: " MAR_GIT_SHA
      };

      MessageBox(m_hwnd, version_str, "Version details", MB_OK);
    }
    break;
  case ID_HELP_OFFICIALWEBSITE:
    {
      ShellExecute(m_hwnd, "", MAR_WEBSITE_URL, "", "", SW_SHOWNORMAL);
    }
    break;
  case ID_HELP_LICENSE:
    {
      if (!g_licwnd)
      {
        g_licwnd = new MAR_LicenseWnd;
        CreateDialogParam(g_inst, MAKEINTRESOURCE(IDD_LICENSE), m_hwnd,
          MAR_LicenseWnd::LicenseWndProc, (LPARAM)g_licwnd);
        ShowWindow(g_licwnd->Handle(), SW_SHOW);
      }
      else
      {
        SetFocus(g_licwnd->Handle());
      }
    }
    break;
  }
}

void MAR_MainWnd::OnTimer(WPARAM wparam, LPARAM lparam)
{
  if (wparam == MAR_MAIN_SCREEN_UPDATE_ID)
  {
    if (g_volissos)
    {
      MAR_GapBuffer *gb = g_volissos->GetGapBuffer();
      if (gb)
      {
        EnableMenuItem(m_mainmenu, ID_FILE_SAVE_PROJECT, MF_ENABLED);
        EnableMenuItem(m_mainmenu, ID_FILE_SAVEAS_PROJECT, MF_ENABLED);
        EnableMenuItem(m_mainmenu, ID_FILE_SAVEALL_PROJECT, MF_ENABLED);
        EnableMenuItem(m_mainmenu, ID_FILE_CLOSEFILE, MF_ENABLED);
        EnableMenuItem(m_mainmenu, ID_FILE_CLOSETHEREST, MF_ENABLED);
        EnableMenuItem(m_mainmenu, ID_EDIT_SELECTALL, MF_ENABLED);
        EnableMenuItem(m_mainmenu, ID_EDIT_FINDTEXT, MF_ENABLED);
        EnableMenuItem(m_mainmenu, ID_EDIT_CHANGETEXT, MF_ENABLED);
        EnableMenuItem(m_mainmenu, ID_EDIT_GOTOLINE, MF_ENABLED);
        EnableMenuItem(m_mainmenu, ID_EDIT_COMMENT, MF_ENABLED);
        EnableMenuItem(m_mainmenu, ID_EDIT_UNCOMMENT, MF_ENABLED);
        EnableMenuItem(m_mainmenu, ID_VIEW_TOGGLELINENUMBER, MF_ENABLED);
        EnableMenuItem(m_mainmenu, ID_VIEW_TOGGLEFILENAME, MF_ENABLED);
        EnableMenuItem(m_mainmenu, ID_VIEW_TOGGLERULER, MF_ENABLED);
        EnableMenuItem(m_mainmenu, ID_VIEW_TOGGLEWHITESPACE, MF_ENABLED);

        if (g_volissos->IsOpen())
        {
          EnableMenuItem(m_mainmenu, ID_FILE_REOPEN, MF_ENABLED);
        }
        else
        {
          EnableMenuItem(m_mainmenu, ID_FILE_REOPEN, MF_GRAYED);
        }

        if (g_volissos->HasPrevious())
        {
          EnableMenuItem(m_mainmenu, ID_VIEW_PREVIOUS, MF_ENABLED);
        }
        else
        {
          EnableMenuItem(m_mainmenu, ID_VIEW_PREVIOUS, MF_GRAYED);
        }

        if (gb->HasSelection())
        {
          EnableMenuItem(m_mainmenu, ID_EDIT_CUTTEXT, MF_ENABLED);
          EnableMenuItem(m_mainmenu, ID_EDIT_COPYTEXT, MF_ENABLED);
        }
        else
        {
          EnableMenuItem(m_mainmenu, ID_EDIT_CUTTEXT, MF_GRAYED);
          EnableMenuItem(m_mainmenu, ID_EDIT_COPYTEXT, MF_GRAYED);
        }

        if (
#if defined(_WIN32)
          (IsClipboardFormatAvailable(CF_TEXT)) &&
#endif
          (g_mainwnd && OpenClipboard(g_mainwnd->Handle())))
        {
          HANDLE buf = GetClipboardData(CF_TEXT);
          if (buf)
          {
            EnableMenuItem(m_mainmenu, ID_EDIT_PASTETEXT, MF_ENABLED);
          }
          CloseClipboard();
        }
        else
        {
          EnableMenuItem(m_mainmenu, ID_EDIT_PASTETEXT, MF_GRAYED);
        }
      }
      else
      {
        EnableMenuItem(m_mainmenu, ID_FILE_REOPEN, MF_GRAYED);
        EnableMenuItem(m_mainmenu, ID_FILE_SAVE_PROJECT, MF_GRAYED);
        EnableMenuItem(m_mainmenu, ID_FILE_SAVEAS_PROJECT, MF_GRAYED);
        EnableMenuItem(m_mainmenu, ID_FILE_SAVEALL_PROJECT, MF_GRAYED);
        EnableMenuItem(m_mainmenu, ID_FILE_CLOSEFILE, MF_GRAYED);
        EnableMenuItem(m_mainmenu, ID_FILE_CLOSETHEREST, MF_GRAYED);
        EnableMenuItem(m_mainmenu, ID_EDIT_CUTTEXT, MF_GRAYED);
        EnableMenuItem(m_mainmenu, ID_EDIT_COPYTEXT, MF_GRAYED);
        EnableMenuItem(m_mainmenu, ID_EDIT_PASTETEXT, MF_GRAYED);
        EnableMenuItem(m_mainmenu, ID_EDIT_SELECTALL, MF_GRAYED);
        EnableMenuItem(m_mainmenu, ID_EDIT_FINDTEXT, MF_GRAYED);
        EnableMenuItem(m_mainmenu, ID_EDIT_CHANGETEXT, MF_GRAYED);
        EnableMenuItem(m_mainmenu, ID_EDIT_GOTOLINE, MF_GRAYED);
        EnableMenuItem(m_mainmenu, ID_EDIT_COMMENT, MF_GRAYED);
        EnableMenuItem(m_mainmenu, ID_EDIT_UNCOMMENT, MF_GRAYED);
        EnableMenuItem(m_mainmenu, ID_VIEW_PREVIOUS, MF_GRAYED);
        EnableMenuItem(m_mainmenu, ID_VIEW_TOGGLELINENUMBER, MF_GRAYED);
        EnableMenuItem(m_mainmenu, ID_VIEW_TOGGLEFILENAME, MF_GRAYED);
        EnableMenuItem(m_mainmenu, ID_VIEW_TOGGLERULER, MF_GRAYED);
        EnableMenuItem(m_mainmenu, ID_VIEW_TOGGLEWHITESPACE, MF_GRAYED);
      }

      if (m_fullscreen) CheckMenuItem(m_mainmenu, ID_VIEW_TOGGLEFULLSCREEN, MF_CHECKED);
      else CheckMenuItem(m_mainmenu, ID_VIEW_TOGGLEFULLSCREEN, MF_UNCHECKED);
      if (m_linenumber) CheckMenuItem(m_mainmenu, ID_VIEW_TOGGLELINENUMBER, MF_CHECKED);
      else CheckMenuItem(m_mainmenu, ID_VIEW_TOGGLELINENUMBER, MF_UNCHECKED);
      if (m_filename) CheckMenuItem(m_mainmenu, ID_VIEW_TOGGLEFILENAME, MF_CHECKED);
      else CheckMenuItem(m_mainmenu, ID_VIEW_TOGGLEFILENAME, MF_UNCHECKED);

      if (g_surfacewnd)
      {
        bool ruler = g_surfacewnd->HasRuler();
        bool whitespace = g_surfacewnd->HasWhitespace();
        if (ruler) CheckMenuItem(m_mainmenu, ID_VIEW_TOGGLERULER, MF_CHECKED);
        else CheckMenuItem(m_mainmenu, ID_VIEW_TOGGLERULER, MF_UNCHECKED);
        if (whitespace) CheckMenuItem(m_mainmenu, ID_VIEW_TOGGLEWHITESPACE, MF_CHECKED);
        else CheckMenuItem(m_mainmenu, ID_VIEW_TOGGLEWHITESPACE, MF_UNCHECKED);
      }
    }

    if (g_linewnd)
    {
      RECT r;
      GetClientRect(m_hwnd, &r);
      WDL_WndSizer__rec *recline = m_resize.get_itembywnd(m_line);
      WDL_WndSizer__rec *recsurface = m_resize.get_itembywnd(m_surface);
      if (recline && recsurface)
      {
        if (m_linenumber)
        {
          int maxw = g_linewnd->GetMaximumWidth();
          if (maxw != recline->orig.right - recline->orig.left)
          {
            recline->orig.left = r.left;
            recline->orig.right = recline->orig.left + maxw;
            recsurface->orig.left = recline->orig.right;
            m_resize.onResize();
          }
        }
        else
        {
          if (recline->orig.right - recline->orig.left != 0)
          {
            recline->orig.left = 0;
            recline->orig.right = 0;
            recsurface->orig.left = recline->orig.right;
            m_resize.onResize();
          }
        }
      }
    }
  }

  if (wparam == MAR_ORIGINAL_TITLE)
  {
    SetWindowText(m_hwnd, m_titleorig.Get());
    KillTimer(m_hwnd, MAR_ORIGINAL_TITLE);
  }
}

void MAR_MainWnd::OnPaint(WPARAM wparam, LPARAM lparam)
{
  RECT r;
  GetClientRect(m_hwnd, &r);
  int w = r.right - r.left;
  int h = r.bottom - r.top;

  //PAINTSTRUCT ps;
  //HDC dc = BeginPaint(m_hwnd, &ps);
  //LICE_Clear(m_tbm, LICE_RGBA((m_bg >> 16) & 0xff, (m_bg >> 8) & 0xff, m_bg & 0xff, 255));
  //LICE_Clear(m_lbm, LICE_RGBA((m_bg >> 16) & 0xff, (m_bg >> 8) & 0xff, m_bg & 0xff, 255));
  //BitBlt(dc, r.left, r.top, w, 7, m_tbm->getDC(), 0, 0, SRCCOPY);
  //BitBlt(dc, r.left, r.top, 7, h, m_lbm->getDC(), 0, 0, SRCCOPY);
  //EndPaint(m_hwnd, &ps);

  m_painter.PaintBegin(m_hwnd, RGB(25, 25, 25));
  int xo, yo;
  LICE_IBitmap *bm = m_painter.GetBuffer(&xo, &yo);

  m_painter.PaintEnd();
}

INT_PTR MAR_MainWnd::OnNotify(WPARAM wparam, LPARAM lparam)
{
  //switch (LOWORD(wparam))
  {

  }

  return 0;
}

void MAR_MainWnd::OnContextMenu(WPARAM wparam, LPARAM lparam)
{
  POINT pt;
  pt.x = GET_X_LPARAM(lparam);
  pt.y = GET_Y_LPARAM(lparam);
  HANDLE hwnd = (HANDLE)wparam;

  if (hwnd == m_hwnd)
  {
    ScreenToClient(m_hwnd, (LPPOINT)&pt);

  }
}

void MAR_MainWnd::OnLButtonDown(WPARAM wparam, LPARAM lparam)
{
  POINT pt;
  pt.x = GET_X_LPARAM(lparam);
  pt.y = GET_Y_LPARAM(lparam);
}

void MAR_MainWnd::OnLButtonUp(WPARAM wparam, LPARAM lparam)
{

}

void MAR_MainWnd::OnMouseMove(WPARAM wparam, LPARAM lparam)
{
  POINT pt;
  pt.x = GET_X_LPARAM(lparam);
  pt.y = GET_Y_LPARAM(lparam);

  RECT r, lr, sr;
  GetWindowRect(m_hwnd, &r);
  ScreenToClient(m_hwnd, (LPPOINT)&r);
  ScreenToClient(m_hwnd, ((LPPOINT)&r)+1);
  GetWindowRect(m_line, &lr);
  ScreenToClient(m_hwnd, (LPPOINT)&lr);
  ScreenToClient(m_hwnd, ((LPPOINT)&lr)+1);
  GetWindowRect(m_surface, &sr);
  ScreenToClient(m_hwnd, (LPPOINT)&sr);
  ScreenToClient(m_hwnd, ((LPPOINT)&sr)+1);

  if (pt.x >= r.right && pt.x <= r.right &&
    pt.y >= r.top && pt.y <= r.bottom)
  {
    SetCursor(LoadCursor(NULL, IDC_SIZEWE));
  }
}

INT_PTR MAR_MainWnd::OnCtrlColorEdit(UINT msg, WPARAM wparam, LPARAM lparam)
{
  if (0)//if((HWND)lparam == GetDlgItem(m_hwnd, IDC_EDIT1))
  {
    int bk = 0x000000; //g_preferences->GetBackgroundColor();
    int fg = 0xffffff; //g_preferences->GetTextColor();
    SetBkMode((HDC)wparam, RGB((bk >> 16) & 0xFF,(bk >> 8) & 0xFF, bk & 0xFF));
    SetTextColor((HDC)wparam, RGB((fg >> 16) & 0xFF,(fg >> 8) & 0xFF, fg & 0xFF));
    if (!m_brush) m_brush = CreateSolidBrush(RGB((bk >> 16) & 0xFF,(bk >> 8) & 0xFF, bk & 0xFF));
    return (INT_PTR)m_brush;
  }
  else
  {
    return DefWindowProc(m_hwnd, msg, wparam, lparam);
  }
}

void MAR_MainWnd::OnActivateApp(WPARAM wparam, LPARAM lparam)
{
  if (wparam == TRUE)
  {

  }
  else
  {

  }
}

void MAR_MainWnd::OnNCLButtonDown(WPARAM wparam, LPARAM lparam)
{

}

WDL_DLGRET MAR_MainWnd::NewTreeProc(HWND hwnd, UINT msg, WPARAM wparam, LPARAM lparam)
{
  switch (msg)
  {
  case WM_MOUSEWHEEL:
  case WM_MOUSEHWHEEL:
    if (GetAsyncKeyState(VK_SHIFT) & 0x8000)
    {
      return 0;
    }
    break;
  case WM_SYSKEYUP:
  case WM_KEYUP:
    {
      switch(wparam)
      {
      case VK_UP:
        {
          HTREEITEM sel = TreeView_GetSelection(internal_tree);
          if (!sel) return 0;

          if (sel == TreeView_GetRoot(internal_tree) && internal_istreeroot)
          {
            internal_istreeroot = false;
            SetFocus(internal_edit);
          }
          else if (sel == TreeView_GetRoot(internal_tree))
          {
            internal_istreeroot = true;
          }
          else
          {
            internal_istreeroot = false;
          }
        }
        break;
      }
    }
    break;
  }

  return CallWindowProc(Internal_PrevTreeProc, hwnd, msg, wparam, lparam);
}

WDL_DLGRET MAR_MainWnd::NewListProc(HWND hwnd, UINT msg, WPARAM wparam, LPARAM lparam)
{
  switch (msg)
  {
  case WM_MOUSEWHEEL:
  case WM_MOUSEHWHEEL:
    if (GetAsyncKeyState(VK_SHIFT) & 0x8000)
    {
      return 0;
    }
    break;
  case WM_SETFOCUS:
    {
      int one_ping_only = ListView_GetNextItem(internal_list, -1, LVNI_SELECTED);
      if (one_ping_only == -1)
      {
        ListView_SetItemState(internal_list, 0,
          LVNI_SELECTED|LVNI_FOCUSED,
          LVNI_SELECTED|LVNI_FOCUSED)
      }
    }
    break;
  }

  return CallWindowProc(Internal_PrevListProc, hwnd, msg, wparam, lparam);
}

WDL_DLGRET MAR_MainWnd::NewEditProc(HWND hwnd, UINT msg, WPARAM wparam, LPARAM lparam)
{
  switch (msg)
  {
  case WM_MOUSEWHEEL:
  case WM_MOUSEHWHEEL:
    if (GetAsyncKeyState(VK_SHIFT) & 0x8000)
    {
      return 0;
    }
    break;
  case WM_SYSKEYDOWN:
  case WM_KEYDOWN:
    {
      switch(wparam)
      {
      case VK_HOME:
        {

        } break;
      case VK_END:
        {

        } break;
      case VK_PRIOR:
        {

        } break;
      case VK_NEXT:
        {

        } break;
      case VK_UP:
        {

        } break;
      case VK_DOWN:
        {

        }
        break;
      case VK_BACK:
        {
          if ((GetAsyncKeyState(VK_CONTROL) & 0x8000))
          {

          }
        }
        break;
      }
    }
    break;
  case WM_SETFOCUS:
    {

    } break;
  case WM_KILLFOCUS:
    {

    } break;
  }

  return CallWindowProc(Internal_PrevEditProc, hwnd, msg, wparam, lparam);
}

WDL_DLGRET MAR_GotoLineWndProc(HWND hwnd, UINT msg, WPARAM wparam, LPARAM lparam)
{
  switch(msg)
  {
    case WM_INITDIALOG:
    {
      //g_gotolinewnd = hwnd;
      int x = GetPrivateProfileInt(MAR_NAME, "goto_wnd_x", 0, g_inipath.Get());
      int y = GetPrivateProfileInt(MAR_NAME, "goto_wnd_y", 50, g_inipath.Get());
      SetWindowPos(hwnd, NULL, x, y, 0, 0, SWP_NOSIZE | SWP_NOZORDER | SWP_NOACTIVATE);

      ShowWindow(hwnd, SW_SHOWDEFAULT);
    } break;
    case WM_DESTROY:
    {
      RECT r;
      GetWindowRect(hwnd, &r);

      int x = r.left;
      int y = r.top;

      WDL_FastString xstr, ystr;
      xstr.SetFormatted(32, "%d", x);
      ystr.SetFormatted(32, "%d", y);

      WritePrivateProfileString(MAR_NAME, "goto_wnd_x", xstr.Get(), g_inipath.Get());
      WritePrivateProfileString(MAR_NAME, "goto_wnd_y", ystr.Get(), g_inipath.Get());
      //g_gotolinewnd = hwnd = NULL;
    } break;
    case WM_COMMAND:
    {
      switch(LOWORD(wparam))
      {
      case IDOK:
        {
          DestroyWindow(hwnd);
        } break;
      case IDCANCEL:
        {
          DestroyWindow(hwnd);
        } break;
      }
    } break;
  }

  return 0;
}

WDL_DLGRET MAR_GotoLineOffsetWndProc(HWND hwnd, UINT msg, WPARAM wparam, LPARAM lparam)
{
  switch(msg)
  {
    case WM_INITDIALOG:
    {
      //g_gotolineoffsetwnd = hwnd;
      int x = GetPrivateProfileInt(MAR_NAME, "goto_wnd_x", 0, g_inipath.Get());
      int y = GetPrivateProfileInt(MAR_NAME, "goto_wnd_y", 50, g_inipath.Get());
      SetWindowPos(hwnd, NULL, x, y, 0, 0, SWP_NOSIZE | SWP_NOZORDER | SWP_NOACTIVATE);

      ShowWindow(hwnd, SW_SHOWDEFAULT);
    } break;
    case WM_DESTROY:
    {
      RECT r;
      GetWindowRect(hwnd, &r);

      int x = r.left;
      int y = r.top;

      WDL_FastString xstr, ystr;
      xstr.SetFormatted(32, "%d", x);
      ystr.SetFormatted(32, "%d", y);

      WritePrivateProfileString(MAR_NAME, "goto_wnd_x", xstr.Get(), g_inipath.Get());
      WritePrivateProfileString(MAR_NAME, "goto_wnd_y", ystr.Get(), g_inipath.Get());
      //g_gotolineoffsetwnd = hwnd = NULL;
    } break;
    case WM_COMMAND:
    {
      switch(LOWORD(wparam))
      {
      case IDOK:
        {
          DestroyWindow(hwnd);
        } break;
      case IDCANCEL:
        {
          DestroyWindow(hwnd);
        } break;
      }
    } break;
  }

  return 0;
}

WDL_DLGRET MAR_MainWnd::MainWndProc(HWND hwnd, UINT msg, WPARAM wparam, LPARAM lparam)
{
  MAR_MainWnd *self = (MAR_MainWnd *)GetWindowLongPtr(hwnd, GWLP_USERDATA);

  if (!self && msg == WM_INITDIALOG)
  {
    SetWindowLongPtr(hwnd, GWLP_USERDATA, lparam);
    self = (MAR_MainWnd *)lparam;
    self->m_hwnd = hwnd;
  }

  if (WDL_likely(self))
  {
    return self->MainWndLoop(msg, wparam, lparam);
  }
  else
  {
    return 0;
  }
}

WDL_DLGRET MAR_MainWnd::MainWndLoop(UINT msg, WPARAM wparam, LPARAM lparam)
{
#if defined(__linux__)
  if (msg == WM_KEYDOWN && LOWORD(wparam) == VK_RETURN)
  {
    SendMessage(m_hwnd, WM_COMMAND, MAKEWPARAM(IDOK, 0), 0);
    return 0;
  }
#endif

  switch(msg)
  {
    case WM_INITDIALOG: OnInitDialog(wparam, lparam); break;
    case WM_SYSCOMMAND: OnSysCommand(wparam, lparam); break;
    case WM_CLOSE: return OnClose(wparam, lparam); break;
    case WM_SIZE: OnSize(wparam, lparam); break;
    case WM_MOVE: OnMove(wparam, lparam); break;
    case WM_DESTROY: OnDestroy(wparam, lparam); break;
    case WM_COMMAND: OnCommand(wparam, lparam); break;
    case WM_TIMER: OnTimer(wparam, lparam); break;
    case WM_PAINT: OnPaint(wparam, lparam); break;
    case WM_ERASEBKGND: return 1; break;
    case WM_NOTIFY: return OnNotify(wparam, lparam); break;
    case WM_CONTEXTMENU: OnContextMenu(wparam, lparam); break;
    case WM_LBUTTONDOWN: OnLButtonDown(wparam, lparam); break;
    case WM_LBUTTONUP: OnLButtonUp(wparam, lparam); break;
    case WM_MOUSEMOVE: OnMouseMove(wparam, lparam); break;
      //case WM_CTLCOLOREDIT: return OnCtrlColorEdit(msg, wparam, lparam); break;
    case WM_ACTIVATEAPP: OnActivateApp(wparam, lparam); break;
    case WM_NCLBUTTONDOWN: OnNCLButtonDown(wparam, lparam); break;
  }

  return 0;
}

#include <WDL/eel2/ns-eel.h>
#include <WDL/lice/lice.h>
#include <WDL/wingui/virtwnd-controls.h>
#include <WDL/wingui/scrollbar/coolscroll.h>


int WDL_STYLE_WantGlobalButtonBorders() { return 0; }
bool WDL_STYLE_WantGlobalButtonBackground(int *col) { return false; }
bool WDL_STYLE_GetBackgroundGradient(double *gradstart, double *gradslope) { return false; }
LICE_IBitmap *WDL_STYLE_GetSliderBitmap2(bool vert) { return NULL; }
bool WDL_STYLE_AllowSliderMouseWheel() { return true; }
int WDL_STYLE_GetSliderDynamicCenterPos() { return 500; }


// TO BE IMPLEMENTED BY APP:
// implemented by calling app, can return a LICE_IBitmap **img for "scrollbar"
void *GetIconThemePointer(const char *name)
{
  if (!strcmp(name, "scrollbar"))
  {
    if (!g_scrollbar)
    {
      WDL_FastString sb(g_modpath);
      sb.Append("skin\\faithful\\scrollbar.png");
      g_scrollbar = LICE_LoadPNG(sb.Get());
      return &g_scrollbar;
    }
  }

  return NULL;
}

// can be a passthrough to GetSysColor()
int CoolSB_GetSysColor(HWND m_hwnd, int val)
{
  if (!g_scrollbar)
  {
    return GetSysColor(val);
  }
  else
  {
    return RGB(51, 51, 51);
  }
}

#include <WDL/eel2/ns-eel.h>

void NSEEL_HOSTSTUB_EnterMutex()
{}

void NSEEL_HOSTSTUB_LeaveMutex()
{}
