// Copyright (c) 2021 Giorgos Vougioukas
//
// The license can be found in the LICENSE file.

#include "preferences.h"
#include "definitions.h"
#include "app_info.h"
#include "main_wnd.h"

#include <WDL/fileread.h>
#include <WDL/filewrite.h>
#include <WDL/aggarray.h>

static const char *internal_default_preferences[] =
{
  "//",
  "// Welcome to " MAR_NAME_MARKETING " " MAR_ARCH " Text Editor, Version " MAR_NAKED_VERSION,
  "//",
  "// " MAR_NAME_MARKETING " preferences",
  "// Your changes will take effect the next time you relaunch " MAR_NAME_MARKETING "",
  "//",
  "",
  "{",
  "  // Focus side on startup.",
  "  \"curses_right_focus\": true,",
  "",
  "  // Always spaces.",
  "  \"tab_size\": 2,",
  //"  \"spaces\": true,",
  "",
  "  // Preferred end-of-line for new file.",
  "  // 0: LF, 1: CRLF, 2: CR.",
#if defined(_WIN32)
  "  \"end_of_line\": 1,",
#else
  "  \"end_of_line\": 0,",
#endif
  "",
  "  // Ruler for maximum character column.",
  "  \"ruler\": 120,",
  "",
  "  // Ancient wisdom.",
  "  \"ancient_wisdom\": false,",
  "",
  "  // File dialog.",
  "  \"preserve_directory\": true",
  "}",
};

static WDL_PtrArray<const char *, WDL_ARRAYCOUNT(internal_default_preferences)> internal_dpref(internal_default_preferences);

MAR_Preferences::MAR_Preferences()
  : m_pf(NULL)
  , m_relaunch(false)
{}

MAR_Preferences::~MAR_Preferences()
{
  cJSON_Delete(m_pf);
}

void MAR_Preferences::Create()
{
  WDL_FastString defpref, usrpref;

  for (int i = 0; i < internal_dpref.GetSize(); i++)
  {
    defpref.AppendFormatted(2048, "%s\n", internal_dpref.Get()[i]);
  }

  const char *user_preferences[] =
  {
    "{",
    "  // Add your personal preferences here.",
    "}"
  };

  WDL_PtrArray<const char *, WDL_ARRAYCOUNT(user_preferences)> upref(user_preferences);

  for (int j = 0; j < upref.GetSize(); j++)
  {
    usrpref.AppendFormatted(2048, "%s\n", upref.Get()[j]);
  }

  WDL_FastString def(g_setpath), usr(g_setpath);
  def.Append("preferences-default.txt");
  usr.Append("preferences.txt");

  // Create default preferences file
  WDL_FileWrite *cdf = new WDL_FileWrite(def.Get());
  if (cdf && cdf->IsOpen())
  {
    cdf->Write(defpref.Get(), defpref.GetLength());
  }

  if (cdf) { delete cdf; cdf = NULL; }

  // Check if user preferences exist
  WDL_FileRead *uf = new WDL_FileRead(usr.Get());
  if (uf && !uf->IsOpen())
  {
    // Create user preferences file
    WDL_FileWrite *cuf = new WDL_FileWrite(usr.Get());
    if (cuf && cuf->IsOpen())
    {
      cuf->Write(usrpref.Get(), usrpref.GetLength());
    }

    if (cuf) { delete cuf; cuf = NULL; }
  }

  if (uf) { delete uf; uf = NULL; }
}

void MAR_Preferences::Open(bool default_file)
{
  WDL_FastString def(g_setpath), usr(g_setpath);
  def.Append("preferences-default.txt");
  usr.Append("preferences.txt");

  ShellExecute(g_mainwnd->Handle(), "", "notepad.exe", default_file ? def.Get() : usr.Get(), "", SW_SHOWNORMAL);
}

bool MAR_Preferences::Parse(bool default_file)
{
  WDL_FastString def(g_setpath), usr(g_setpath);
  def.Append("preferences-default.txt");
  usr.Append("preferences.txt");

  WDL_FileRead fr(default_file ? def.Get() : usr.Get());
  WDL_TypedBuf<char> strbuf;
  strbuf.Resize((int)fr.GetSize());
  fr.Read(strbuf.Get(), strbuf.GetSize());
  strbuf.Add('\0');

  cJSON_Minify(strbuf.Get());
  m_pf = cJSON_Parse(strbuf.Get());

  if (!m_pf)
  {
    const char *error_ptr = cJSON_GetErrorPtr();
    if (error_ptr)
    {
      m_err.SetFormatted(2048, "Parse error before: %s\n", error_ptr);
      wdl_log(m_err.Get());
    }

    return false;
  }

  return true;
}

bool MAR_Preferences::WantCursesRightFocus() const
{
  const cJSON *cursesrightfocus = NULL;
  cursesrightfocus = cJSON_GetObjectItemCaseSensitive(m_pf, "curses_right_focus");
  if (cJSON_IsTrue(cursesrightfocus)) return true;
  else if (cJSON_IsFalse(cursesrightfocus)) return false;
  else return true;
}

int MAR_Preferences::GetTabSize() const
{
  const cJSON *tabsize = NULL;
  tabsize = cJSON_GetObjectItemCaseSensitive(m_pf, "tab_size");
  if (cJSON_IsNumber(tabsize))
  {
    if (tabsize->valueint >= 2) return tabsize->valueint; else return 2;
  }
  else return 2;
}

bool MAR_Preferences::WantSpaces() const
{
  const cJSON *spaces = NULL;
  spaces = cJSON_GetObjectItemCaseSensitive(m_pf, "spaces");
  if (cJSON_IsTrue(spaces)) return true;
  else if (cJSON_IsFalse(spaces)) return false;
  else return true;
}

int MAR_Preferences::EndOfLine() const
{
  const cJSON *endofline = NULL;
  endofline = cJSON_GetObjectItemCaseSensitive(m_pf, "end_of_line");
  if (cJSON_IsNumber(endofline))
  {
    if (endofline >= 0 && endofline->valueint <= 2)
    {
      return endofline->valueint;
    }
    else
    {
#if defined(_WIN32)
      return 1;
#else
      return 0;
#endif
    }
  }
  else
  {
#if defined(_WIN32)
    return 1;
#else
    return 0;
#endif
  }
}

bool MAR_Preferences::WantPreserveDirectory() const
{
  const cJSON *preservedirectory = NULL;
  preservedirectory = cJSON_GetObjectItemCaseSensitive(m_pf, "preserve_directory");
  if (cJSON_IsTrue(preservedirectory)) return true;
  else if (cJSON_IsFalse(preservedirectory)) return false;
  else return true;
}

int MAR_Preferences::GetRuler() const
{
  const cJSON *ruler = NULL;
  ruler = cJSON_GetObjectItemCaseSensitive(m_pf, "ruler");
  if (cJSON_IsNumber(ruler))
  {
    if (ruler->valueint >= 70) return ruler->valueint; else return 120;
  }
  else return 120;
}

bool MAR_Preferences::WantAncientWisdom() const
{
  const cJSON *ancientwisdom = NULL;
  ancientwisdom = cJSON_GetObjectItemCaseSensitive(m_pf, "ancient_wisdom");
  if (cJSON_IsTrue(ancientwisdom)) return true;
  else if (cJSON_IsFalse(ancientwisdom)) return false;
  else return false;
}
