// Copyright (c) 2024 Giorgos Vougioukas
//
// The license can be found in the LICENSE file.

#include "scan_files.h"

#include <WDL/dirscan.h>

MAR_ScanFiles::MAR_ScanFiles()
  : m_threadid(0u)
  , m_threadhandle(NULL)
  , m_killthread(false)
  , m_running(false)
  , m_cancel(false)
  , m_scan(false)
{
  StartThread();
}

MAR_ScanFiles::~MAR_ScanFiles()
{
  m_cancel = true;
  if (IsRunning()) StopThread();
  files.Empty(true);
}

void MAR_ScanFiles::Open(const char *path)
{
  m_path.Set(path); m_scan = true;
}

bool MAR_ScanFiles::IsScanning() const
{
  return m_scan;
}

const char *MAR_ScanFiles::GetPath() const
{
  return m_path.Get();
}

void MAR_ScanFiles::Scan(const char *path, bool empty_buffer)
{
  WDL_DirScan dir;
  WDL_FastString fn;

  m_lp.parse(g_textext.Get());
  if (empty_buffer) files.Empty(true);

  if (!dir.First(path))
  {
    do
    {
      if (dir.GetCurrentIsDirectory())
      {
        dir.GetCurrentFullFN(&fn);
        if (wdl_filename_cmp(fn.get_filepart(), ".") &&
          wdl_filename_cmp(fn.get_filepart(), ".."))
        {
          Scan(fn.Get(), false);
        }
      }
      else
      {
        WDL_FastString *str = new WDL_NEW WDL_FastString();
        if (str)
        {
          dir.GetCurrentFullFN(str);
          bool valid_ext = false;

          for (int i = 0; i < m_lp.getnumtokens(); i++)
          {
            if (wdl_filename_cmp(str->get_fileext(), m_lp.gettoken_str(i)) == 0)
            {
              valid_ext = true; break;
            }
          }

          if (valid_ext) files.Add(str);
          else delete str;
        }
      }
    } while (!dir.Next() && !m_cancel);
  }

  qsort(files.GetList(), files.GetSize(),
    sizeof(WDL_FastString *), sort_by_file_path);
}

void MAR_ScanFiles::StartThread()
{
  m_killthread = false;
  m_threadhandle = (HANDLE)_beginthreadex(NULL, 0,
    ThreadFunction, (void *)this, 0, &m_threadid);
  SetThreadPriority(m_threadhandle, THREAD_PRIORITY_NORMAL);
  m_running = true;
}

void MAR_ScanFiles::StopThread()
{
  m_killthread = true;
  WaitForSingleObject(m_threadhandle, INFINITE);
  CloseHandle(m_threadhandle);
  m_running = false;
}

bool MAR_ScanFiles::IsRunning() const
{
  return m_running;
}

int MAR_ScanFiles::Run()
{
  if (m_scan && m_path.GetLength())
  {
    Scan(m_path.Get());
    m_scan = false;
  }

  return 1;
}

unsigned int WINAPI MAR_ScanFiles::ThreadFunction(void *arg)
{
  MAR_ScanFiles *self = (MAR_ScanFiles *)arg;

  if (WDL_NORMALLY(self))
  {
    int sleepstep = 50;

    self->m_killthread = false;

    while (!self->m_killthread)
    {
      self->m_mutex.Enter();
      while (!self->Run());
      self->m_mutex.Leave();
      Sleep(sleepstep);
    }
  }

  return 0;
}
