// Copyright (c) 2023 Giorgos Vougioukas
//
// The license can be found in the LICENSE file.

#include "surface_wnd.h"
#include "delphi.h"
#include "app_info.h"
#include "app_timers.h"
#include "volissos.h"
#include "line_wnd.h"
#include "preferences.h"
#include "main_wnd.h"

#include <WDL/wingui/scrollbar/coolscroll.h>
#include <WDL/MersenneTwister.h>
#include <WDL/wdlutf8.h>
#include <WDL/aggarray.h>

static const char *s_courier = "Courier";
static const char *s_couriernew = "Courier New";
static const char *s_consolas = "Consolas";

static LOGFONT lf =
{
#if defined(_WIN32)
  14, 0, 0, 0, FW_NORMAL, FALSE, FALSE, FALSE, DEFAULT_CHARSET,
  OUT_DEFAULT_PRECIS, CLIP_DEFAULT_PRECIS, DEFAULT_QUALITY, DEFAULT_PITCH,
  "Consolas"
#else
  11, 0, 0, 0, FW_NORMAL, FALSE, FALSE, FALSE, DEFAULT_CHARSET,
  OUT_DEFAULT_PRECIS, CLIP_DEFAULT_PRECIS, DEFAULT_QUALITY, DEFAULT_PITCH,
  "Arial"
#endif
};

static WNDPROC PrevSurfaceWndProc = NULL;
static HWND s_surfacewnd = NULL;
static WDL_PtrArray<const char *, WDL_ARRAYCOUNT(delphic_maxims)> delphi(delphic_maxims);

MAR_SurfaceWnd::MAR_SurfaceWnd(HWND hwnd)
  : m_hwnd(hwnd)
  , m_bm(NULL)
  , m_font(NULL)
  , m_ctx(NULL)
  , m_ruler(0)
  , m_lazy(false)
  , m_whitespace(false)
  , m_onepingonly(false)
  , m_maxim(0)
{
  int ruler = GetPrivateProfileInt(MAR_NAME, "ruler", 0, g_inipath.Get());
  if (ruler) ToggleRuler();
  int whitespace = GetPrivateProfileInt(MAR_NAME, "whitespace", 1, g_inipath.Get());
  if (whitespace) ToggleWhitespace();
  m_onepingonly = g_preferences->WantAncientWisdom();
  if (m_onepingonly)
  {
    MTRand mt;
    m_maxim = mt.randInt(delphi.GetSize() - 1);
    WDL_ASSERT(m_maxim < delphi.GetSize());
  }
}

MAR_SurfaceWnd::~MAR_SurfaceWnd()
{
  if (m_bm) delete m_bm;
  if (m_font) delete m_font;

  if (m_ruler) WritePrivateProfileString(MAR_NAME, "ruler", "1", g_inipath.Get());
  else WritePrivateProfileString(MAR_NAME, "ruler", "0", g_inipath.Get());
  if (m_whitespace) WritePrivateProfileString(MAR_NAME, "whitespace", "1", g_inipath.Get());
  else WritePrivateProfileString(MAR_NAME, "whitespace", "0", g_inipath.Get());
}

HWND MAR_SurfaceWnd::Handle() const
{
  return m_hwnd;
}

int MAR_SurfaceWnd::GetLines() const
{
  return m_ctx ? m_ctx->minsurfacelines : 0;
}

void MAR_SurfaceWnd::RequestRedraw()
{
  if (m_ctx) m_ctx->fulldraw = true;
}

void MAR_SurfaceWnd::SetPageOneLineUp()
{
  if (m_ctx->yoffset > 0)
  {
    if (m_ctx->cursorline < m_ctx->bottom)
    {
      m_ctx->yoffset -= 1;
      m_ctx->cursorline += 1;
      m_ctx->fulldraw = true;
    }
  }
}

void MAR_SurfaceWnd::SetPageOneLineDown()
{
  if (m_ctx->yoffset < m_ctx->textheight - m_ctx->lines)
  {
    if (m_ctx->cursorline > m_ctx->top)
    {
      m_ctx->yoffset += 1;
      m_ctx->cursorline -= 1;
      m_ctx->fulldraw = true;
    }
  }
}

void MAR_SurfaceWnd::FindKeyWord(const char *find, bool case_sensitive, bool match_whole_word)
{
  if (g_volissos)
  {
    MAR_GapBuffer *gb = g_volissos->GetGapBuffer();
    if (gb)
    {
      const int ln = gb->GetCursorLine();
      const int col = gb->GetCursorColumn();
      if (gb->FindKeyWord(find, case_sensitive, match_whole_word, true, false))
      {
        if (ln != m_ctx->lastline || col != m_ctx->lastcolumn) m_ctx->cursorline = -1;
      }
    }
  }
}

void MAR_SurfaceWnd::ChangeKeyWord(const char *find, const char *change, bool case_sensitive, bool match_whole_word)
{
  if (g_volissos)
  {
    MAR_GapBuffer *gb = g_volissos->GetGapBuffer();
    if (gb)
    {
      int changes = gb->ChangeKeyWord(find, change, case_sensitive, match_whole_word);
      if (changes)
      {
        if (g_mainwnd)
        {
          WDL_FastString inf;
          if (changes == 1) inf.SetFormatted(512, "%s | %d change", g_mainwnd->GetOrigTitle(), changes);
          else inf.SetFormatted(512, "%s | %d changes", g_mainwnd->GetOrigTitle(), changes);
          g_mainwnd->SetTitle(inf.Get());
          m_ctx->cursorline = -1;
        }
      }
    }
  }
}

void MAR_SurfaceWnd::ToggleRuler()
{
  if (!m_ruler)
  {
    m_ruler = g_preferences->GetRuler();
  }
  else
  {
    m_ruler = 0;
  }
  if (m_ctx) m_ctx->fulldraw = true;
}

bool MAR_SurfaceWnd::HasRuler() const
{
  return m_ruler > 0;
}

void MAR_SurfaceWnd::ToggleWhitespace()
{
  m_whitespace = !m_whitespace;
  if (m_ctx) m_ctx->fulldraw = true;
}

bool MAR_SurfaceWnd::HasWhitespace() const
{
  return m_whitespace;
}

void MAR_SurfaceWnd::OnCreate(WPARAM wparam, LPARAM lparam)
{
  int text_font = 0;
  int text_font_height = 14;

  switch (text_font)
  {
  case 1:
    {
      int mx = (int)strlen(s_courier);
      WDL_ASSERT(mx > 0 && mx < 32);
      for (int i = 0; i < mx; i++)
      {
        lf.lfFaceName[i] = s_courier[i];
      }
      lf.lfFaceName[mx] = '\0';
      lf.lfHeight = text_font_height;
    } break;
  case 2:
    {
      int mx = (int)strlen(s_couriernew);
      WDL_ASSERT(mx > 0 && mx < 32);
      for (int i = 0; i < mx; i++)
      {
        lf.lfFaceName[i] = s_couriernew[i];
      }
      lf.lfFaceName[mx] = '\0';
      lf.lfHeight = text_font_height;
    } break;
  case 3:
    {
      int mx = (int)strlen(s_consolas);
      WDL_ASSERT(mx > 0 && mx < 32);
      for (int i = 0; i < mx; i++)
      {
        lf.lfFaceName[i] = s_consolas[i];
      }
      lf.lfFaceName[mx] = '\0';
      lf.lfHeight = text_font_height;
    } break;
  }

  m_bm = new LICE_SysBitmap();
  m_font = new LICE_CachedFont();
  m_font->SetFromHFont(CreateFontIndirect(&lf), LICE_FONT_FLAG_OWNS_HFONT);

  InitializeCoolSB(m_hwnd);

  RECT r;
  GetWindowRect(m_hwnd, &r);
  ScreenToClient(m_hwnd, (LPPOINT)&r);
  ScreenToClient(m_hwnd, ((LPPOINT)&r)+1);

  m_bm->resize(r.right - r.left, r.bottom - r.top);

  SCROLLINFO vsi;
  vsi.cbSize = sizeof(vsi);
  vsi.fMask = SIF_PAGE|SIF_RANGE; //SIF_PAGE|SIF_POS|SIF_RANGE,
  vsi.nMin = 0;
  vsi.nMax = 0;
  vsi.nPage = 0;

  CoolSB_SetScrollInfo(m_hwnd, SB_VERT, &vsi, TRUE);
  CoolSB_ShowScrollBar(m_hwnd, SB_VERT, FALSE);
  CoolSB_SetMinThumbSize(m_hwnd, SB_VERT, 25);

  PrevSurfaceWndProc = (WNDPROC)SetWindowLongPtrW(m_hwnd,
    GWLP_WNDPROC, (LONG_PTR)NewSurfaceWndProc);
  s_surfacewnd = m_hwnd;
  SetTimer(m_hwnd, MAR_SURFACE_UPDATE_ID, MAR_SURFACE_UPDATE_MS, FALSE);
}

void MAR_SurfaceWnd::OnDestroy(WPARAM wparam, LPARAM lparam)
{
  KillTimer(m_hwnd, MAR_SURFACE_UPDATE_ID);
  UninitializeCoolSB(m_hwnd);
}

void MAR_SurfaceWnd::OnSize(WPARAM wparam, LPARAM lparam)
{
  if (wparam != SIZE_MINIMIZED)
  {
    m_resize.onResize();

    RECT r;
    GetClientRect(m_hwnd, &r);
    int w = r.right - r.left;
    int h = r.bottom - r.top;
    m_bm->resize(w, h);

    if (!CoolSB_IsCoolScrollEnabled(m_hwnd))
    {
      CoolSB_ShowScrollBar(m_hwnd, SB_VERT, TRUE);
    }

    int cx = LOWORD(lparam);
    int cy = HIWORD(lparam);

    if (m_ctx)
    {
      SCROLLINFO vsi;
      vsi.cbSize = sizeof(vsi);
      vsi.fMask = SIF_PAGE|SIF_RANGE; //SIF_PAGE|SIF_POS|SIF_RANGE,
      vsi.nMin = 0;
      vsi.nMax = m_ctx->textheight;
      vsi.nPage = cy / m_ctx->letterheight;

      CoolSB_SetScrollInfo(m_hwnd, SB_VERT, &vsi, TRUE);

      m_ctx->fulldraw = true;
      InvalidateRect(m_hwnd, NULL, FALSE);
    }
    else
    {
      m_lazy = false;
      InvalidateRect(m_hwnd, NULL, FALSE);
    }
  }
}

void MAR_SurfaceWnd::OnCommand(WPARAM wparam, LPARAM lparam)
{}

void MAR_SurfaceWnd::OnPaint(WPARAM wparam, LPARAM lparam)
{
  PAINTSTRUCT ps;
  HDC dc = BeginPaint(m_hwnd, &ps);

  DoPaint(dc);

  EndPaint(m_hwnd, &ps);
}

void MAR_SurfaceWnd::OnLButtonDown(WPARAM wparam, LPARAM lparam)
{
  POINT pt;
  pt.x = GET_X_LPARAM(lparam);
  pt.y = GET_Y_LPARAM(lparam);

  MAR_GapBuffer *gb = g_volissos->GetGapBuffer();
  if (gb)
  {
    m_ctx->mouseln = pt.y ? wdl_max(pt.y / m_ctx->letterheight, 0) : 0;
    m_ctx->mousecol = pt.x ? wdl_max(pt.x / m_ctx->letterwidth, 0) : 0;
    const int ln = m_ctx->yoffset + m_ctx->mouseln;
    const int col = m_ctx->xoffset + m_ctx->mousecol;
    gb->SetSelect(false, false, false);
    gb->GoToLine(ln, col);
    gb->CheckSelect(true);
  }

  SetCapture(m_hwnd);
}

void MAR_SurfaceWnd::OnLButtonUp(WPARAM wparam, LPARAM lparam)
{
  POINT pt;
  pt.x = GET_X_LPARAM(lparam);
  pt.y = GET_Y_LPARAM(lparam);

  if (GetCapture())
  {
    MAR_GapBuffer *gb = g_volissos->GetGapBuffer();
    if (gb)
    {
      int oldmouseln = m_ctx->mouseln;
      int oldmousecol = m_ctx->mousecol;
      m_ctx->mouseln = pt.y ? wdl_max(pt.y / m_ctx->letterheight, 0) : 0;
      m_ctx->mousecol = pt.x ? wdl_max(pt.x / m_ctx->letterwidth, 0) : 0;

      WDL_TypedBuf<MAR_StringView> *sv;
      gb->GetStringView(&sv);
      const int ln = wdl_clamp(m_ctx->yoffset + m_ctx->mouseln, 0, m_ctx->textheight - 1);
      const int col = wdl_clamp(m_ctx->xoffset + m_ctx->mousecol, 0, sv->Get()[ln].len);
      gb->GoToLine(ln, col);

      if (oldmouseln < m_ctx->mouseln)
      {
        gb->SetSelect(true, true, false);
      }
      else if (oldmouseln > m_ctx->mouseln)
      {
        gb->SetSelect(true, true, true);
      }
      else
      {
        if (oldmousecol < m_ctx->mousecol)
        {
          gb->SetSelect(true, true, false);
        }
        else if (oldmousecol > m_ctx->mousecol)
        {
          gb->SetSelect(true, true, true);
        }
        else
        {
          gb->SetSelect(false, false, false);
        }
      }
    }
    ReleaseCapture();
  }
}

void MAR_SurfaceWnd::OnMouseMove(WPARAM wparam, LPARAM lparam)
{
  POINT pt;
  pt.x = GET_X_LPARAM(lparam);
  pt.y = GET_Y_LPARAM(lparam);
  SetCursor(LoadCursor(NULL, IDC_IBEAM));

  if (GetCapture())
  {
    MAR_GapBuffer *gb = g_volissos->GetGapBuffer();
    if (gb)
    {
      int mouseln = pt.y ? wdl_max(pt.y / m_ctx->letterheight, 0) : 0;
      int mousecol = pt.x ? wdl_max(pt.x / m_ctx->letterwidth, 0) : 0;

      WDL_TypedBuf<MAR_StringView> *sv;
      gb->GetStringView(&sv);
      const int ln = wdl_clamp(m_ctx->yoffset + mouseln, 0, m_ctx->textheight - 1);
      const int col = wdl_clamp(m_ctx->xoffset + mousecol, 0, sv->Get()[ln].len);
      gb->MouseMoveSelect(col, ln);
    }
  }
}

void MAR_SurfaceWnd::OnTimer(WPARAM wparam, LPARAM lparam)
{
  if (wparam == MAR_SURFACE_UPDATE_ID)
  {
    InvalidateRect(m_hwnd, NULL, FALSE);
  }
}

void MAR_SurfaceWnd::OnVScroll(WPARAM wparam, LPARAM lparam)
{
  SCROLLINFO vsi = { sizeof(SCROLLINFO), SIF_POS|SIF_PAGE|SIF_RANGE|SIF_TRACKPOS, 0 };
  CoolSB_GetScrollInfo(m_hwnd, SB_VERT, &vsi);
  switch (LOWORD(wparam))
  {
  case SB_THUMBTRACK:
    vsi.nPos = vsi.nTrackPos;
    break;
  case SB_LINEUP:
    vsi.nPos -= 1;
    break;
  case SB_LINEDOWN:
    vsi.nPos += 1;
    break;
  case SB_PAGEUP:
    vsi.nPos -= (int)vsi.nPage;
    break;
  case SB_PAGEDOWN:
    vsi.nPos += (int)vsi.nPage;
    break;
  case SB_TOP:
    vsi.nPos = 0;
    break;
  case SB_BOTTOM:
    vsi.nPos = vsi.nMax-(int)vsi.nPage;
    break;
  }
  if (vsi.nPos > vsi.nMax-(int)vsi.nPage) vsi.nPos = vsi.nMax-(int)vsi.nPage;
  if (vsi.nPos < 0) vsi.nPos = 0;

  if (m_ctx)
  {
    int ln = vsi.nPos;
    m_ctx->cursorline = -1;
    m_ctx->yoffset = ln;
    m_ctx->fulldraw = true;
    CoolSB_SetScrollPos(m_hwnd, SB_VERT, ln, TRUE);
  }

  InvalidateRect(m_hwnd, NULL, FALSE);
}

void MAR_SurfaceWnd::OnKeyDown(WPARAM wparam, LPARAM lparam)
{
  switch (wparam)
  {
  case VK_HOME:
    {
      CoolSB_SetScrollPos(m_hwnd, SB_VERT, 0, TRUE);
      InvalidateRect(m_hwnd, NULL, FALSE);
    }
    break;
  case VK_END:
    {
      SCROLLINFO vsi = { sizeof(SCROLLINFO), SIF_POS|SIF_PAGE|SIF_RANGE|SIF_TRACKPOS, 0 };
      CoolSB_GetScrollInfo(m_hwnd, SB_VERT, &vsi);

      CoolSB_SetScrollPos(m_hwnd, SB_VERT, vsi.nMax, TRUE);
      InvalidateRect(m_hwnd, NULL, FALSE);
    }
    break;
  case VK_PRIOR:
    {
      SCROLLINFO vsi = { sizeof(SCROLLINFO), SIF_POS|SIF_PAGE|SIF_RANGE|SIF_TRACKPOS, 0 };
      CoolSB_GetScrollInfo(m_hwnd, SB_VERT, &vsi);

      int pos = vsi.nPos - vsi.nPage;
      if (pos < 0) pos = 0;

      CoolSB_SetScrollPos(m_hwnd, SB_VERT, pos, TRUE);
      InvalidateRect(m_hwnd, NULL, FALSE);
    }
    break;
  case VK_NEXT:
    {
      SCROLLINFO vsi = { sizeof(SCROLLINFO), SIF_POS|SIF_PAGE|SIF_RANGE|SIF_TRACKPOS, 0 };
      CoolSB_GetScrollInfo(m_hwnd, SB_VERT, &vsi);

      int pos = vsi.nPos + vsi.nPage;
      if (pos > vsi.nMax) pos = vsi.nMax;


      CoolSB_SetScrollPos(m_hwnd, SB_VERT, pos, TRUE);
      InvalidateRect(m_hwnd, NULL, FALSE);
    }
    break;
  case VK_UP:
    {
      SCROLLINFO vsi = { sizeof(SCROLLINFO), SIF_POS|SIF_PAGE|SIF_RANGE|SIF_TRACKPOS, 0 };
      CoolSB_GetScrollInfo(m_hwnd, SB_VERT, &vsi);

      int pos = vsi.nPos - (m_ctx ? m_ctx->letterheight : 0);
      if (pos < 0) pos = 0;

      CoolSB_SetScrollPos(m_hwnd, SB_VERT, pos, TRUE);
      InvalidateRect(m_hwnd, NULL, FALSE);
    }
    break;
  case VK_DOWN:
    {
      SCROLLINFO vsi = { sizeof(SCROLLINFO), SIF_POS|SIF_PAGE|SIF_RANGE|SIF_TRACKPOS, 0 };
      CoolSB_GetScrollInfo(m_hwnd, SB_VERT, &vsi);

      int pos = vsi.nPos + (m_ctx ? m_ctx->letterheight : 0);
      if (pos > vsi.nMax) pos = vsi.nMax;

      CoolSB_SetScrollPos(m_hwnd, SB_VERT, pos, TRUE);
      InvalidateRect(m_hwnd, NULL, FALSE);
    }
    break;
  }
}

void MAR_SurfaceWnd::OnMouseWheel(WPARAM wparam, LPARAM lparam)
{
  POINT p = { GET_X_LPARAM(lparam), GET_Y_LPARAM(lparam) };
  ScreenToClient(m_hwnd, &p);

  int l = (short)HIWORD(wparam);
  float pos = -l / 120.0f;

  SCROLLINFO vsi = { sizeof(SCROLLINFO), SIF_POS|SIF_PAGE|SIF_RANGE|SIF_TRACKPOS, 0 };
  CoolSB_GetScrollInfo(m_hwnd, SB_VERT, &vsi);

  pos *= 3; //(int)vsi.nPage;
  pos += vsi.nPos;

  const float mv = (float)(vsi.nMax - (int)vsi.nPage);
  if (pos > mv) pos = mv;
  if (pos < 0) pos = 0;

  if (m_ctx)
  {
    int ln = (int)(pos + 0.5);
    if (!GetCapture()) m_ctx->cursorline = -1;
    m_ctx->yoffset = ln;
    m_ctx->fulldraw = true;
    CoolSB_SetScrollPos(m_hwnd, SB_VERT, ln, TRUE);
  }

  InvalidateRect(m_hwnd, NULL, FALSE);
}

void MAR_SurfaceWnd::DoPaint(HDC dc)
{
  if (m_bm && g_volissos)
  {
    RECT r;
    GetClientRect(m_hwnd, &r);
    const int w = r.right - r.left;
    const int h = r.bottom - r.top;

    RECT let;
    m_ctx = NULL;
    m_font->SetBkMode(TRANSPARENT);
    m_font->SetTextColor(LICE_RGBA(192, 192, 192, 255));
    m_font->DrawText(m_bm, "V", 1, &let, DT_CALCRECT);
    const int heightspace = 1;

    MAR_GapBuffer *gb = g_volissos->GetGapBuffer();
    if (gb && gb->GetSurfaceContext())
    {
      m_lazy = false;

      if (WDL_unlikely(m_onepingonly))
      {
        m_onepingonly = m_lazy;
      }

      m_ctx = gb->GetSurfaceContext();

      if (wdl_filename_cmp(m_prevfn.Get(), gb->GetFilePath()))
      {
        if (m_ctx->vsi.cbSize)
        {
          CoolSB_SetScrollInfo(m_hwnd, SB_VERT, &m_ctx->vsi, TRUE);
          InvalidateRect(m_hwnd, NULL, FALSE);
        }
        m_prevfn.Set(gb->GetFilePath());
        m_ctx->maxline = -1;
        m_ctx->fulldraw = true;
      }

      m_ctx->letterwidth = let.right - let.left;
      m_ctx->letterheight = let.bottom - let.top + heightspace;
      m_ctx->lines = h / m_ctx->letterheight;
      m_ctx->columns = w / m_ctx->letterwidth - (1 * m_ctx->letterwidth);
      const int topmargin = 5;
      const int bottommargin = 3;
      m_ctx->top = topmargin;
      m_ctx->bottom = m_ctx->lines - bottommargin;
      m_ctx->minsurfacelines = m_ctx->bottom - m_ctx->top;
      const int column = gb->GetCursorColumn();
      const int line = gb->GetCursorLine();
      const int ins = gb->GetInsert();
      m_ctx->textheight = gb->GetTotalLines();
      const int tabsize = g_preferences->GetTabSize();
      const bool selection = gb->HasSelection();

      RECT lnr;

      WDL_TypedBuf<MAR_StringView> *sv;
      gb->GetStringView(&sv);

      if (m_ctx->fulldraw || selection || selection != m_ctx->selection || line != m_ctx->lastline ||
        column != m_ctx->lastcolumn || m_ctx->maxline != m_ctx->textheight)
      {
        LICE_Clear(m_bm, LICE_RGBA(0, 0, 0, 255));

        if (line != m_ctx->lastline || m_ctx->maxline != m_ctx->textheight)
        {
          SCROLLINFO vsi;
          CoolSB_GetScrollInfo(m_hwnd, SB_VERT, &vsi);
          vsi.cbSize = sizeof(vsi);
          vsi.fMask = SIF_POS|SIF_RANGE|SIF_PAGE; //SIF_PAGE|SIF_POS|SIF_RANGE,
          vsi.nMin = 0;
          vsi.nMax = m_ctx->textheight ? m_ctx->textheight : 0;
          vsi.nPage = (r.bottom - r.top) / m_ctx->letterheight;
          vsi.nPos = line > (int)vsi.nPage ? line - (int)vsi.nPage : 0;
          CoolSB_SetScrollInfo(m_hwnd, SB_VERT, &vsi, TRUE);
        }

        m_ctx->maxline = m_ctx->textheight;
        //const int quot = line / m_ctx->lines;
        //const int rem = line - (m_ctx->lines * quot);

        int ydiff = line - m_ctx->lastline;

        if (m_ctx->textheight <= m_ctx->lines)
        {
          ydiff = m_ctx->textheight - line;
          m_ctx->cursorline = line;
          m_ctx->yoffset = 0;
        }
        else if (m_ctx->cursorline >= 0)
        {
          if (line < m_ctx->top)
          {
            m_ctx->cursorline = line;
            m_ctx->yoffset = 0;
          }
          else if (ydiff > 0 && line >= m_ctx->top && line <= m_ctx->maxline - bottommargin)
          {
            m_ctx->cursorline = wdl_clamp(m_ctx->cursorline + ydiff, m_ctx->top, m_ctx->bottom);
            if (m_ctx->cursorline == m_ctx->bottom)
            {
              m_ctx->yoffset = line - m_ctx->bottom;
            }
          }
          else if (ydiff < 0 && line >= m_ctx->top && line <= m_ctx->maxline - bottommargin)
          {
            m_ctx->cursorline = wdl_clamp(m_ctx->cursorline + ydiff, m_ctx->top, m_ctx->bottom);
            if (m_ctx->cursorline == m_ctx->top)
            {
              m_ctx->yoffset = line - m_ctx->top;
            }
          }
          else if (line >= m_ctx->lines && line > m_ctx->maxline - bottommargin)
          {
            ydiff = m_ctx->maxline - line;
            m_ctx->cursorline = m_ctx->lines - ydiff;
            m_ctx->yoffset = m_ctx->maxline - m_ctx->lines;
          }
        }
        else if ((line != m_ctx->lastline || column != m_ctx->lastcolumn) && m_ctx->cursorline == -1)
        {
          if (m_ctx->mouseln >= m_ctx->top && m_ctx->mouseln <= m_ctx->bottom)
          {
            m_ctx->cursorline = m_ctx->mouseln;
            m_ctx->yoffset = line - m_ctx->cursorline;
            m_ctx->mouseln = m_ctx->mousecol = 0;
          }
          else if (line < m_ctx->top)
          {
            m_ctx->cursorline = line;
            m_ctx->yoffset = 0;
          }
          else if (line > m_ctx->maxline - m_ctx->lines)
          {
            m_ctx->yoffset = m_ctx->maxline - m_ctx->lines;
            m_ctx->cursorline = m_ctx->lines - (m_ctx->maxline - line);
          }
          else
          {
            m_ctx->cursorline = m_ctx->top;
            m_ctx->yoffset = line - m_ctx->top;
          }
        }

        const int negdiff = wdl_min((m_ctx->columns * m_ctx->letterwidth) - (column * m_ctx->letterwidth), 0);
        const int xdiff = column - m_ctx->lastcolumn;

        if (xdiff > 0 && m_ctx->cursorcolumn + xdiff >= m_ctx->columns)
        {
          m_ctx->xoffset = negdiff ? negdiff : 0;
          m_ctx->cursorcolumn = m_ctx->columns;
        }
        else if (xdiff > 0 && m_ctx->cursorcolumn + xdiff < m_ctx->columns)
        {
          m_ctx->cursorcolumn += xdiff;
        }
        else if (xdiff < 0 && m_ctx->cursorcolumn <= 3 && column > 3)
        {
          m_ctx->xoffset -= xdiff * m_ctx->letterwidth;
          m_ctx->cursorcolumn = wdl_min(column, 3);
        }
        else if (xdiff < 0 && m_ctx->cursorcolumn <= 3 && column <= 3)
        {
          m_ctx->xoffset = 0;
          m_ctx->cursorcolumn = column;
        }
        else if (xdiff < 0 && m_ctx->cursorcolumn > 0)
        {
          m_ctx->cursorcolumn = wdl_min(m_ctx->cursorcolumn + xdiff, m_ctx->columns);
          if (m_ctx->cursorcolumn < 0)
          {
            m_ctx->xoffset = negdiff ? negdiff : 0;
            m_ctx->cursorcolumn = wdl_max(column, 0);
          }
          else if (m_ctx->cursorcolumn < 3)
          {
            m_ctx->xoffset = negdiff ? negdiff : 0;
            m_ctx->cursorcolumn = wdl_max(column, 0);
          }
        }
        else if (!negdiff)
        {
          m_ctx->cursorcolumn += xdiff;
        }

        m_ctx->lastline = line;
        m_ctx->lastcolumn = column;
        m_ctx->selection = selection;

        if (g_linewnd)
        {
          g_linewnd->Start(m_ctx->yoffset + 1, wdl_min(m_ctx->lines, m_ctx->textheight),
            m_ctx->cursorline, column + 1);
          InvalidateRect(g_linewnd->Handle(), NULL, FALSE);
        }

        RECT cur;
        //cur.left = wdl_min(m_ctx->cursorcolumn * m_ctx->letterwidth + m_ctx->startspace,
        //  m_ctx->columns * m_ctx->letterwidth + m_ctx->startspace);
        cur.left = m_ctx->cursorcolumn * m_ctx->letterwidth + m_ctx->startspace;
        cur.top = m_ctx->cursorline * m_ctx->letterheight;
        cur.right = cur.left + m_ctx->letterwidth;
        cur.bottom = cur.top + m_ctx->letterheight;

        m_ctx->fulldraw = true;
        m_ctx->lastcursor.left = cur.left;
        m_ctx->lastcursor.top = cur.top;
        m_ctx->lastcursor.right = cur.right;
        m_ctx->lastcursor.bottom = cur.bottom;
        m_ctx->cursornoblink = true;

        for (int ln = m_ctx->yoffset, sl = 0; ln < sv->GetSize() &&
          ln < m_ctx->textheight && sl < m_ctx->lines; ln++, sl++)
        {
          MAR_StringView *v = &sv->Get()[ln];
          if (!v) continue;

          if (v->len) m_strbuf.Set(v->str, v->len);
          else m_strbuf.Set("");

          for (int i = 0; i < m_strbuf.GetLength(); i++)
          {
            if (m_strbuf.Get()[i] == '\t')
            {
              m_strbuf.DeleteSub(i, 1);
              //for (int j = 0; j < tabsize; j++)
              {
                m_strbuf.Insert(" ", i);
              }
            }
          }

          if (m_strbuf.GetLength())
          {
            m_font->DrawText(m_bm, m_strbuf.Get(), m_strbuf.GetLength(), &lnr, DT_CALCRECT);
            const int linewidth = lnr.right - lnr.left;
            lnr.left = (r.left + m_ctx->xoffset) + m_ctx->startspace;
            lnr.right = lnr.left + linewidth;
            lnr.top = sl * m_ctx->letterheight;
            lnr.bottom = lnr.top + m_ctx->letterheight;
            m_font->DrawText(m_bm, m_strbuf.Get(), m_strbuf.GetLength(), &lnr, 0);
          }

          if (m_whitespace)
          {
            for (int i = 0; i < m_strbuf.GetLength(); i++)
            {
              if (m_strbuf.Get()[i] == ' ')
              {
                lnr.left = r.left + m_ctx->xoffset +
                  m_ctx->startspace + (i * m_ctx->letterwidth);
                lnr.right = lnr.left + m_ctx->letterwidth;
                lnr.top = sl * m_ctx->letterheight;
                lnr.bottom = lnr.top + m_ctx->letterheight;
                int pixelx = lnr.left + m_ctx->letterwidth / 2;
                int pixely = lnr.top + m_ctx->letterheight / 2;
                LICE_PutPixel(m_bm, pixelx, pixely,
                  LICE_RGBA(20, 72, 82, 255),
                  1.0f, LICE_BLIT_MODE_COPY);
              }
            }
          }

          if (m_ctx->xoffset)
          {
            m_font->DrawText(m_bm, "<< ", 3, &lnr, DT_CALCRECT);
            const int linewidth = lnr.right - lnr.left;
            lnr.left = r.left + m_ctx->startspace;
            lnr.right = lnr.left + linewidth;
            lnr.top = sl * m_ctx->letterheight;
            lnr.bottom = lnr.top + m_ctx->letterheight;
            LICE_FillRect(m_bm, lnr.left - m_ctx->startspace,
              lnr.top, lnr.left + m_ctx->letterwidth * 3,
              lnr.bottom - lnr.top, LICE_RGBA(0, 0, 0, 255));
            m_font->SetTextColor(LICE_RGBA(128, 128, 255, 255));
            m_font->DrawText(m_bm, "<< ", 3, &lnr, 0);
            m_font->SetTextColor(LICE_RGBA(192, 192, 192, 255));
          }
        }

        m_ctx->lastcursorbm.resize(m_ctx->letterwidth, m_ctx->letterheight);
        LICE_Blit(&m_ctx->lastcursorbm, m_bm, 0, 0, cur.left, cur.top,
          cur.right - cur.left, cur.bottom - cur.top, 1.0f,
          LICE_BLIT_MODE_COPY);

        if (WDL_unlikely(selection))
        {
          int x1, y1, x2, y2;
          gb->GetSelectionRegion(&x1, &y1, &x2, &y2);
          x1 = x1 * m_ctx->letterwidth + m_ctx->startspace;
          y1 = (y1 - m_ctx->yoffset) * m_ctx->letterheight;
          x2 = x2 * m_ctx->letterwidth + m_ctx->startspace;
          y2 = (y2 - m_ctx->yoffset) * m_ctx->letterheight;

          if (m_ctx->xoffset)
          {
            if (x2 > x1)
            {
              const int xw = x2 - x1;
              x1 += m_ctx->xoffset;
              x2 = x1 + xw;
            }
            else if (x1 > x2)
            {
              const int xw = x1 - x2;
              x2 += m_ctx->xoffset;
              x1 = x2 + xw;
            }
          }

          if (y1 < y2)
          {
            LICE_FillRect(m_bm, x1, y1, r.right - x1, m_ctx->letterheight - heightspace,
              LICE_RGBA(128, 128, 255, 255), 0.6f, LICE_BLIT_MODE_COPY);
            for (int i = y1 + m_ctx->letterheight; i < y2; i += m_ctx->letterheight)
            {
              LICE_FillRect(m_bm, m_ctx->startspace, i, w, m_ctx->letterheight - heightspace,
                LICE_RGBA(128, 128, 255, 255), 0.6f, LICE_BLIT_MODE_COPY);
            }
            LICE_FillRect(m_bm, m_ctx->startspace, y2, x2 - m_ctx->startspace, m_ctx->letterheight - heightspace,
              LICE_RGBA(128, 128, 255, 255), 0.6f, LICE_BLIT_MODE_COPY);
          }
          else if (y1 > y2)
          {
            LICE_FillRect(m_bm, x2, y2, r.right - x2, m_ctx->letterheight - heightspace,
              LICE_RGBA(128, 128, 255, 255), 0.6f, LICE_BLIT_MODE_COPY);
            for (int i = y2 + m_ctx->letterheight; i < y1; i += m_ctx->letterheight)
            {
              LICE_FillRect(m_bm, m_ctx->startspace, i, w, m_ctx->letterheight - heightspace,
                LICE_RGBA(128, 128, 255, 255), 0.6f, LICE_BLIT_MODE_COPY);
            }
            LICE_FillRect(m_bm, m_ctx->startspace, y1, x1 - m_ctx->startspace, m_ctx->letterheight - heightspace,
              LICE_RGBA(128, 128, 255, 255), 0.6f, LICE_BLIT_MODE_COPY);
          }
          else
          {
            if (x1 < x2)
            {
              LICE_FillRect(m_bm, x1, y1, x2 - x1, m_ctx->letterheight - heightspace,
                LICE_RGBA(128, 128, 255, 255), 0.6f, LICE_BLIT_MODE_COPY);
            }
            else if (x1 > x2)
            {
              LICE_FillRect(m_bm, x2, y2, x1 - x2, m_ctx->letterheight - heightspace,
                LICE_RGBA(128, 128, 255, 255), 0.6f, LICE_BLIT_MODE_COPY);
            }
          }
        }
      }

      m_ctx->blinkrate += MAR_SURFACE_UPDATE_MS;
      if ((m_ctx->blinkrate >= 100 || m_ctx->cursornoblink) && m_ctx->cursorline >= 0)
      {
        if (m_ctx->cursornoblink)
        {
          m_ctx->viewcursor = true;
          m_ctx->cursornoblink = false;
        }
        else m_ctx->viewcursor = !m_ctx->viewcursor;
        m_ctx->blinkrate = 0;

        if (m_ctx->viewcursor)
        {
          if (WDL_unlikely(selection))
          {
            int x1, y1, x2, y2;
            gb->GetSelectionRegion(&x1, &y1, &x2, &y2);
            x1 = x1 * m_ctx->letterwidth + m_ctx->startspace;
            y1 = (y1 - m_ctx->yoffset) * m_ctx->letterheight;
            x2 = x2 * m_ctx->letterwidth + m_ctx->startspace;
            y2 = (y2 - m_ctx->yoffset) * m_ctx->letterheight;

            if (m_ctx->xoffset)
            {
              if (x2 > x1)
              {
                const int xw = x2 - x1;
                x1 += m_ctx->xoffset;
                x2 = x1 + xw;
              }
              else if (x1 > x2)
              {
                const int xw = x1 - x2;
                x2 += m_ctx->xoffset;
                x1 = x2 + xw;
              }
            }

            LICE_Line(m_bm, x2, y2, x2, y2 + m_ctx->letterheight - heightspace, LICE_RGBA(255, 0, 0, 255));
          }
          else
          {
            if (WDL_unlikely(ins))
            {
              LICE_Blit(m_bm, &m_ctx->lastcursorbm, m_ctx->lastcursor.left,
                m_ctx->lastcursor.top, 0, 0, m_ctx->lastcursorbm.getWidth(),
                m_ctx->lastcursorbm.getHeight(), 1.0f, LICE_BLIT_MODE_COPY);
              LICE_FillRect(m_bm, m_ctx->lastcursor.left, m_ctx->lastcursor.top,
                m_ctx->lastcursor.right - m_ctx->lastcursor.left,
                m_ctx->lastcursor.bottom - heightspace - m_ctx->lastcursor.top,
                LICE_RGBA(128, 128, 255, 255), 0.6f);
            }
            else
            {
              LICE_Line(m_bm, r.left, m_ctx->lastcursor.bottom, r.right,
                m_ctx->lastcursor.bottom, LICE_RGBA(0, 0, 0, 255));
              LICE_Line(m_bm, r.left, m_ctx->lastcursor.bottom - heightspace, r.right,
                m_ctx->lastcursor.bottom - heightspace, LICE_RGBA(0, 0, 0, 255));
              LICE_Line(m_bm, m_ctx->lastcursor.left, m_ctx->lastcursor.bottom,
                m_ctx->lastcursor.right, m_ctx->lastcursor.bottom, LICE_RGBA(128, 128, 255, 255));
              LICE_Line(m_bm, m_ctx->lastcursor.left, m_ctx->lastcursor.bottom - heightspace,
                m_ctx->lastcursor.right, m_ctx->lastcursor.bottom - heightspace,
                LICE_RGBA(128, 128, 255, 255));
            }
          }
        }
        else
        {
          if (WDL_unlikely(ins))
          {
            LICE_Blit(m_bm, &m_ctx->lastcursorbm, m_ctx->lastcursor.left,
              m_ctx->lastcursor.top, 0, 0, m_ctx->lastcursorbm.getWidth(),
              m_ctx->lastcursorbm.getHeight(), 1.0f, LICE_BLIT_MODE_COPY);
            LICE_Line(m_bm, r.left, m_ctx->lastcursor.bottom, r.right,
              m_ctx->lastcursor.bottom, LICE_RGBA(0, 0, 0, 255));
            LICE_Line(m_bm, r.left, m_ctx->lastcursor.bottom - heightspace,
              r.right, m_ctx->lastcursor.bottom - heightspace, LICE_RGBA(0, 0, 0, 255));
          }
          else
          {
            LICE_Blit(m_bm, &m_ctx->lastcursorbm, m_ctx->lastcursor.left,
              m_ctx->lastcursor.top, 0, 0, m_ctx->lastcursorbm.getWidth(),
              m_ctx->lastcursorbm.getHeight(), 1.0f, LICE_BLIT_MODE_COPY);
            LICE_Line(m_bm, r.left, m_ctx->lastcursor.bottom, r.right,
              m_ctx->lastcursor.bottom, LICE_RGBA(0, 0, 0, 255));
            LICE_Line(m_bm, r.left, m_ctx->lastcursor.bottom - heightspace,
              r.right, m_ctx->lastcursor.bottom - heightspace, LICE_RGBA(0, 0, 0, 255));
          }
        }
      }

      if (m_ruler)
      {
        const int x1 = m_ruler * m_ctx->letterwidth + m_ctx->startspace;
        const int y1 = 0;
        const int x2 = x1;
        const int y2 = y1 + h;
        LICE_Line(m_bm, x1, y1, x2, y2, LICE_RGBA(25, 25, 25, 255));
      }
    }
    else
    {
      if (!m_lazy)
      {
        m_lazy = true;
        CoolSB_ShowScrollBar(m_hwnd, SB_VERT, FALSE);
        LICE_Clear(m_bm, LICE_RGBA(48, 25, 52, 255));

        if (g_linewnd)
        {
          g_linewnd->Start(0, 0, 0, 0);
          InvalidateRect(g_linewnd->Handle(), NULL, FALSE);
        }

        if (m_onepingonly)
        {
          RECT lnr;
          m_font->DrawText(m_bm, delphi.Get()[m_maxim], -1, &lnr, DT_CALCRECT);
          const int lw = lnr.right - lnr.left;
          const int lh = lnr.bottom - lnr.top;
          const int hlw = (lnr.right - lnr.left) / 2;
          const int hlh = (lnr.bottom - lnr.top) / 2;
          lnr.left = (r.right - r.left) / 2 - hlw;
          lnr.right = lnr.left + lw;
          lnr.top = (r.bottom - r.top) / 2 - hlh;
          lnr.bottom = lnr.top + lh;
          m_font->SetTextColor(LICE_RGBA(255, 255, 255, 255));
          m_font->DrawText(m_bm, delphi.Get()[m_maxim], -1, &lnr, 0);
          m_font->SetTextColor(LICE_RGBA(192, 192, 192, 255));
        }

        BitBlt(dc, r.left, r.top, w, h, m_bm->getDC(), 0, 0, SRCCOPY);
      }
    }

    if (m_ctx && !m_ctx->fulldraw)
    {
      BitBlt(dc, r.left, m_ctx->lastcursor.bottom - heightspace, w,
        (m_ctx->lastcursor.bottom + heightspace) - (m_ctx->lastcursor.bottom - heightspace),
        m_bm->getDC(), r.left, m_ctx->lastcursor.bottom - heightspace, SRCCOPY);
      BitBlt(dc, m_ctx->lastcursor.left, m_ctx->lastcursor.top, m_ctx->letterwidth,
        m_ctx->letterheight, m_bm->getDC(), m_ctx->lastcursor.left, m_ctx->lastcursor.top, SRCCOPY);
    }
    else
    {
      if (!m_lazy)
      {
        GetClientRect(m_hwnd, &r);
        const int fw = r.right - r.left;
        const int fh = r.bottom - r.top;
        BitBlt(dc, r.left, r.top, fw, fh, m_bm->getDC(), 0, 0, SRCCOPY);
        if (m_ctx) m_ctx->fulldraw = false;
      }
    }

    if (m_ctx)
    {
      SCROLLINFO vsi;
      CoolSB_GetScrollInfo(m_hwnd, SB_VERT, &vsi);
      m_ctx->vsi.cbSize = vsi.cbSize;
      m_ctx->vsi.fMask = vsi.fMask;
      m_ctx->vsi.nMin = vsi.nMin;
      m_ctx->vsi.nMax = vsi.nMax;
      m_ctx->vsi.nPage = vsi.nPage;
      m_ctx->vsi.nPos = vsi.nPos;
    }
  }
}

WDL_DLGRET MAR_SurfaceWnd::NewSurfaceWndProc(HWND hwnd, UINT msg, WPARAM wparam, LPARAM lparam)
{
  switch (msg)
  {
  case WM_KEYDOWN:
    {
      switch(wparam)
      {
      case VK_DELETE:
        {
          if (!(GetAsyncKeyState(VK_CONTROL) & 0x8000) &&
            !(GetAsyncKeyState(VK_SHIFT) & 0x8000) &&
            !(GetAsyncKeyState(VK_MENU) & 0x8000))
          {
            if (!GetCapture())
            {
              wdl_log("delete\n");
            }
          }
        }
        break;
      case VK_INSERT:
        {
          if (!(GetAsyncKeyState(VK_CONTROL) & 0x8000) &&
            !(GetAsyncKeyState(VK_SHIFT) & 0x8000) &&
            !(GetAsyncKeyState(VK_MENU) & 0x8000))
          {
            if (!GetCapture())
            {
              wdl_log("insert\n");
            }
          }
        }
        break;
      }
    }
    break;
  case WM_CHAR:
    {
      switch(wparam)
      {
      case VK_BACK:
        {
          if (!(GetAsyncKeyState(VK_CONTROL) & 0x8000) &&
            !(GetAsyncKeyState(VK_SHIFT) & 0x8000) &&
            !(GetAsyncKeyState(VK_MENU) & 0x8000))
          {
            if (!GetCapture())
            {
              wdl_log("back\n");

            }
          }
        }
        break;
      default:
        {
          wchar_t wc[2];
          wc[0] = (wchar_t)wparam;
          wc[1] = L'\0';
          char mb[32];
          int len = WDL_WideToMBStr(mb, wc, sizeof(mb));
          if (len) g_volissos->InsertText(mb, len);
        }
        break;
      }
    }
    break;
  }

  return CallWindowProc(PrevSurfaceWndProc, hwnd, msg, wparam, lparam);
}

WDL_DLGRET MAR_SurfaceWnd::SurfaceWndProc(HWND hwnd, UINT msg, WPARAM wparam, LPARAM lparam)
{
  g_surfacewnd = (MAR_SurfaceWnd *)GetWindowLongPtr(hwnd, GWLP_USERDATA);

#ifdef _WIN32
  if (!g_surfacewnd && msg == WM_NCCREATE)
#else
  if (!g_surfacewnd && msg == WM_CREATE)
#endif
  {
    g_surfacewnd = new MAR_SurfaceWnd(hwnd);
    SetWindowLongPtr(hwnd, GWLP_USERDATA, (INT_PTR)g_surfacewnd);
#ifndef _WIN32
    //SWELL_SetClassName(hwnd, "waveform_superhost");
#endif
    if (g_surfacewnd) g_surfacewnd->OnCreate(wparam, lparam);
    return (g_surfacewnd != NULL);
  }

#ifdef _WIN32
  if (g_surfacewnd && msg == WM_NCDESTROY)
#else
  if (g_surfacewnd && msg == WM_DESTROY)
#endif
  {
    g_surfacewnd->OnDestroy(wparam, lparam);
    delete g_surfacewnd; g_surfacewnd = NULL;
  }

  if (WDL_likely(g_surfacewnd))
  {
    return g_surfacewnd->SurfaceWndLoop(msg, wparam, lparam);
  }
  else
  {
    return 0;
  }
}

WDL_DLGRET MAR_SurfaceWnd::SurfaceWndLoop(UINT msg, WPARAM wparam, LPARAM lparam)
{
  switch (msg)
  {
#ifdef _WIN32
    case WM_NCCREATE: OnCreate(wparam, lparam); break;
    case WM_NCDESTROY: OnDestroy(wparam, lparam); break;
#else
    case WM_CREATE: OnCreate(wparam, lparam); break;
    case WM_DESTROY: OnDestroy(wparam, lparam); break;
#endif
    case WM_SIZE: OnSize(wparam, lparam); break;
    case WM_COMMAND: OnCommand(wparam, lparam); break;
    case WM_PAINT: OnPaint(wparam, lparam); break;
    case WM_LBUTTONDOWN: OnLButtonDown(wparam, lparam); break;
    case WM_LBUTTONUP: OnLButtonUp(wparam, lparam); break;
    case WM_MOUSEMOVE: OnMouseMove(wparam, lparam); break;
    case WM_TIMER: OnTimer(wparam, lparam); break;
    case WM_VSCROLL: OnVScroll(wparam, lparam); break;
    case WM_KEYDOWN: OnKeyDown(wparam, lparam); break;
    case WM_MOUSEWHEEL: OnMouseWheel(wparam, lparam); break;
  }

  return DefWindowProc(m_hwnd, msg, wparam, lparam);
}
