// Copyright (c) 2024 Giorgos Vougioukas
//
// The license can be found in the LICENSE file.

#include "switch_wnd.h"
#include "app_info.h"
#include "app_timers.h"
#include "volissos.h"

MAR_SwitchWnd::MAR_SwitchWnd()
  : m_hwnd(NULL)
  , m_list(NULL)
{}

MAR_SwitchWnd::~MAR_SwitchWnd()
{}

HWND MAR_SwitchWnd::Handle() const
{
  return m_hwnd;
}

void MAR_SwitchWnd::SelectGapBuffer()
{
  int pos = ListView_GetNextItem(m_list, -1, LVIS_SELECTED|LVIS_FOCUSED);
  if (pos >= 0)
  {
    if (g_volissos)
    {
      MAR_GapBuffer *gb = g_volissos->GetGapBuffer(pos);
      if (gb)
      {
        g_volissos->ActiveFile(gb->GetFilePath());
      }
    }
  }
  SendMessage(m_hwnd, WM_CLOSE, 0, 0);
}

void MAR_SwitchWnd::SelectNextGapBuffer()
{
  int pos = ListView_GetNextItem(m_list, -1, LVIS_SELECTED|LVIS_FOCUSED);
  if (pos >= 0 && pos + 1 < ListView_GetItemCount(m_list))
  {
    ListView_SetItemState(m_list, pos, ~LVIS_SELECTED, LVIS_SELECTED);
    ListView_SetItemState(m_list, pos + 1, LVIS_SELECTED|LVIS_FOCUSED, LVIS_SELECTED|LVIS_FOCUSED);
    ListView_EnsureVisible(m_list, pos + 1, FALSE);
  }
  else
  {
    ListView_SetItemState(m_list, pos, ~LVIS_SELECTED, LVIS_SELECTED);
    ListView_SetItemState(m_list, 0, LVIS_SELECTED|LVIS_FOCUSED, LVIS_SELECTED|LVIS_FOCUSED);
    ListView_EnsureVisible(m_list, 0, FALSE);
  }
}

void MAR_SwitchWnd::SelectPreviousGapBuffer()
{
  int pos = ListView_GetNextItem(m_list, -1, LVIS_SELECTED|LVIS_FOCUSED);
  if (pos > 0)
  {
    ListView_SetItemState(m_list, pos, ~LVIS_SELECTED, LVIS_SELECTED);
    ListView_SetItemState(m_list, pos - 1, LVIS_SELECTED|LVIS_FOCUSED, LVIS_SELECTED|LVIS_FOCUSED);
    ListView_EnsureVisible(m_list, pos - 1, FALSE);
  }
  else
  {
    ListView_SetItemState(m_list, pos, ~LVIS_SELECTED, LVIS_SELECTED);
    ListView_SetItemState(m_list, ListView_GetItemCount(m_list) - 1,
      LVIS_SELECTED|LVIS_FOCUSED, LVIS_SELECTED|LVIS_FOCUSED);
    ListView_EnsureVisible(m_list, ListView_GetItemCount(m_list) - 1, FALSE);
  }
}

void MAR_SwitchWnd::OnInitDialog(WPARAM wparam, LPARAM lparam)
{
  m_list = GetDlgItem(m_hwnd, IDC_LIST1);
  WDL_UTF8_HookListView(m_list);

  m_resize.init(m_hwnd);
  m_resize.init_itemhwnd(m_list, 0.0f, 0.0f, 1.0f, 1.0f);

  int x = GetPrivateProfileInt(MAR_NAME, "switch_wnd_x", 0, g_inipath.Get());
  int y = GetPrivateProfileInt(MAR_NAME, "switch_wnd_y", 50, g_inipath.Get());
  int w = GetPrivateProfileInt(MAR_NAME, "switch_wnd_w", 200, g_inipath.Get());
  int h = GetPrivateProfileInt(MAR_NAME, "switch_wnd_h", 600, g_inipath.Get());
  SetWindowPos(m_hwnd, NULL, x, y, w, h, SWP_NOZORDER | SWP_NOACTIVATE);

#if defined(_WIN32)
  ListView_SetExtendedListViewStyleEx(m_list,
    LVS_EX_HEADERDRAGDROP | LVS_EX_FULLROWSELECT | LVS_EX_DOUBLEBUFFER,
    LVS_EX_HEADERDRAGDROP | LVS_EX_FULLROWSELECT | LVS_EX_DOUBLEBUFFER);
#else
  ListView_SetExtendedListViewStyleEx(m_list,
    LVS_EX_HEADERDRAGDROP | LVS_EX_FULLROWSELECT,
    LVS_EX_HEADERDRAGDROP | LVS_EX_FULLROWSELECT);
#endif

  ListView_SetBkColor(m_list, RGB(0, 0, 0));
  ListView_SetTextBkColor(m_list, RGB(0, 0, 0));
  ListView_SetTextColor(m_list, RGB(192, 192, 192));

  if (g_volissos)
  {
    LVCOLUMN lvc = { 0, };
    lvc.mask = LVCF_FMT | LVCF_TEXT | LVCF_WIDTH;

    RECT r;
    GetClientRect(m_list, &r);

    lvc.cx = r.right - r.left - 7;
    lvc.fmt = LVCFMT_LEFT;
    lvc.pszText = (char *)"";
    lvc.cchTextMax = (int)0;
    ListView_InsertColumn(m_list, 0, &lvc);

    MAR_GapBuffer *gb = g_volissos->GetGapBuffer();
    if (gb)
    {
      for (int i = 0; i < g_volissos->GetGapBufferSize(); i++)
      {
        gb = g_volissos->GetGapBuffer(i);

        if (gb)
        {
          LVITEM lvi = { 0, };
          lvi.mask = LVIF_TEXT | LVIF_PARAM;
          lvi.lParam = (LPARAM)i;
          lvi.iItem = i;
          lvi.iSubItem = 0;
          lvi.pszText = (char *)gb->GetFileName();
          lvi.cchTextMax = (int)strlen(gb->GetFileName());
          ListView_InsertItem(m_list, &lvi);
        }
      }
    }
  }

  ListView_SetItemState(m_list, 0, LVNI_SELECTED | LVNI_FOCUSED, LVNI_SELECTED | LVNI_FOCUSED);
}

INT_PTR MAR_SwitchWnd::OnClose(WPARAM wparam, LPARAM lparam)
{
  DestroyWindow(m_hwnd);
  return 0;
}

void MAR_SwitchWnd::OnDestroy(WPARAM wparam, LPARAM lparam)
{
  RECT r;
  GetWindowRect(m_hwnd, &r);

  WDL_FastString rleft, rtop, rwidth, rheight;
  rleft.SetFormatted(32, "%d", r.left);
  rtop.SetFormatted(32, "%d", r.top);
  rwidth.SetFormatted(32, "%d", r.right - r.left);
  rheight.SetFormatted(32, "%d", r.bottom - r.top);

  WritePrivateProfileString(MAR_NAME, "switch_wnd_x", rleft.Get(), g_inipath.Get());
  WritePrivateProfileString(MAR_NAME, "switch_wnd_y", rtop.Get(), g_inipath.Get());
  WritePrivateProfileString(MAR_NAME, "switch_wnd_w", rwidth.Get(), g_inipath.Get());
  WritePrivateProfileString(MAR_NAME, "switch_wnd_h", rheight.Get(), g_inipath.Get());

  if (g_switchwnd) { delete g_switchwnd; g_switchwnd = NULL; }
}

void MAR_SwitchWnd::OnSize(WPARAM wparam, LPARAM lparam)
{
  if (wparam != SIZE_MINIMIZED)
  {
    m_resize.onResize();

    RECT r;
    GetClientRect(m_list, &r);
    int cw = r.right - r.left - 7;
    ListView_SetColumnWidth(m_list, 0, cw);
  }
}

void MAR_SwitchWnd::OnSysCommand(WPARAM wparam, LPARAM lparam)
{
  if (wparam == SC_CLOSE)
  {
    SendMessage(m_hwnd, WM_CLOSE, 0, 0);
  }
}

void MAR_SwitchWnd::OnCommand(WPARAM wparam, LPARAM lparam)
{
  switch(LOWORD(wparam))
  {
  case IDOK:
    {
      SelectGapBuffer();
      SendMessage(m_hwnd, WM_CLOSE, 0, 0);
    } break;
  case IDCANCEL:
    {
      SendMessage(m_hwnd, WM_CLOSE, 0, 0);
    } break;
  }
}

void MAR_SwitchWnd::OnPaint(WPARAM wparam, LPARAM lparam)
{
  PAINTSTRUCT ps;
  HDC dc = BeginPaint(m_hwnd, &ps);
  EndPaint(m_hwnd, &ps);
}

void MAR_SwitchWnd::OnLButtonDown(WPARAM wparam, LPARAM lparam)
{
  SetCapture(m_hwnd);

  POINT pt;
  pt.x = GET_X_LPARAM(lparam);
  pt.y = GET_Y_LPARAM(lparam);
}

void MAR_SwitchWnd::OnLButtonUp(WPARAM wparam, LPARAM lparam)
{
  POINT pt;
  pt.x = GET_X_LPARAM(lparam);
  pt.y = GET_Y_LPARAM(lparam);

  if (GetCapture())
  {
    ReleaseCapture();
  }
}

void MAR_SwitchWnd::OnMouseMove(WPARAM wparam, LPARAM lparam)
{
  SetCursor(LoadCursor(NULL, IDC_IBEAM));

  if (GetCapture())
  {

  }
}

void MAR_SwitchWnd::OnTimer(WPARAM wparam, LPARAM lparam)
{
  if (wparam == MAR_SWITCH_UPDATE_ID)
  {

  }
}

void MAR_SwitchWnd::OnKeyDown(WPARAM wparam, LPARAM lparam)
{
  //switch (wparam)
  //{}
}

INT_PTR MAR_SwitchWnd::OnNotify(WPARAM wparam, LPARAM lparam)
{
  switch (LOWORD(wparam))
  {
  case IDC_LIST1:
    {
      switch (((LPNMHDR)lparam)->code)
      {
      case NM_DBLCLK:
        {
          SelectGapBuffer();
        } break;
      }
    } break;
  }

  return 0;
}

WDL_DLGRET MAR_SwitchWnd::SwitchWndProc(HWND hwnd, UINT msg, WPARAM wparam, LPARAM lparam)
{
  MAR_SwitchWnd *self = (MAR_SwitchWnd *)GetWindowLongPtr(hwnd, GWLP_USERDATA);

  if (!self && msg == WM_INITDIALOG)
  {
    SetWindowLongPtr(hwnd, GWLP_USERDATA, lparam);
    self = (MAR_SwitchWnd *)lparam;
    self->m_hwnd = hwnd;
  }

  if (WDL_likely(self))
  {
    return self->SwitchWndLoop(msg, wparam, lparam);
  }
  else
  {
    return 0;
  }
}

WDL_DLGRET MAR_SwitchWnd::SwitchWndLoop(UINT msg, WPARAM wparam, LPARAM lparam)
{
  switch (msg)
  {
    case WM_INITDIALOG: OnInitDialog(wparam, lparam); break;
    case WM_CLOSE: return OnClose(wparam, lparam); break;
    case WM_DESTROY: OnDestroy(wparam, lparam); break;
    case WM_SIZE: OnSize(wparam, lparam); break;
    case WM_SYSCOMMAND: OnSysCommand(wparam, lparam); break;
    case WM_COMMAND: OnCommand(wparam, lparam); break;
    case WM_PAINT: OnPaint(wparam, lparam); break;
    case WM_LBUTTONDOWN: OnLButtonDown(wparam, lparam); break;
    case WM_LBUTTONUP: OnLButtonUp(wparam, lparam); break;
    case WM_MOUSEMOVE: OnMouseMove(wparam, lparam); break;
    case WM_TIMER: OnTimer(wparam, lparam); break;
    case WM_KEYDOWN: OnKeyDown(wparam, lparam); break;
    case WM_NOTIFY: OnNotify(wparam, lparam); break;
  }

  return 0;
}
