// Copyright (c) 2023 Giorgos Vougioukas
//
// The license can be found in the LICENSE file.

#include "volissos.h"
#include "main_wnd.h"

static void del_gapbuffer(MAR_GapBuffer *gb) { if (gb) delete gb; }

MAR_Volissos::MAR_Volissos()
  : m_threadid(0u)
  , m_threadhandle(NULL)
  , m_killthread(false)
  , m_running(false)
  , m_files(true, del_gapbuffer)
  , m_original(0)
{
  StartThread();
}

MAR_Volissos::~MAR_Volissos()
{
  if (IsRunning()) StopThread();
  m_files.DeleteAll();
}

void MAR_Volissos::New()
{
  MAR_GapBuffer *gb = new WDL_NEW MAR_GapBuffer;
  if (WDL_NORMALLY(gb))
  {
    SetPrevious();
    WDL_FastString fn;
    fn.SetFormatted(256, "original-%d.mem", m_original++);
    gb->New(fn.Get());
    m_files.Insert(fn.Get(), gb);
    m_activefile.Set(fn.Get());
    if (g_mainwnd) g_mainwnd->DisplayFileName();
  }
}

bool MAR_Volissos::Open(const char *filename)
{
  if (m_files.Exists(filename))
  {
    SetPrevious();
    m_activefile.Set(filename);
    return true;
  }
  else
  {
    MAR_GapBuffer *gb = new WDL_NEW MAR_GapBuffer;
    if (WDL_NORMALLY(gb))
    {
      if (gb->Open(filename))
      {
        SetPrevious();
        m_files.Insert(filename, gb);
        m_activefile.Set(filename);
        if (g_mainwnd) g_mainwnd->DisplayFileName();

        return true;
      }
      else
      {
        if (gb) delete gb;
      }
    }
  }

  return false;
}

void MAR_Volissos::Reopen()
{
  if (wdl_filename_cmp(m_activefile.get_fileext(), ".mem"))
  {
    if (m_files.Exists(m_activefile.Get()))
    {
      m_files.Delete(m_activefile.Get());
      Open(m_activefile.Get());
    }
  }
}

bool MAR_Volissos::IsOpen() const
{
  if (wdl_filename_cmp(m_activefile.get_fileext(), ".mem"))
  {
    return m_files.Exists(m_activefile.Get());
  }

  return false;
}

void MAR_Volissos::SwitchToPrevious()
{
  if (m_files.Exists(m_previousfile.Get()) &&
    m_files.GetSize() > 1)
  {
    WDL_FastString tmp(m_activefile.Get());
    m_activefile.Set(m_previousfile.Get());
    m_previousfile.Set(tmp.Get());
    if (g_mainwnd) g_mainwnd->DisplayFileName();
  }
}

bool MAR_Volissos::HasPrevious() const
{
  return m_files.Exists(m_previousfile.Get()) &&
    m_files.GetSize() > 1;
}

void MAR_Volissos::Close()
{
  if (m_files.GetSize())
  {
    int idx = m_files.GetIdx(m_activefile.Get());
    if (idx >= 0)
    {
      m_activefile.Set("");
      m_files.DeleteByIndex(idx);
      if (m_files.GetSize())
      {
        if (idx >= m_files.GetSize()) idx = m_files.GetSize() - 1;

        const char *key;
        m_files.Enumerate(idx, &key);
        if (key) m_activefile.Set(key);
      }
      if (g_mainwnd) g_mainwnd->DisplayFileName();
    }
  }
}

void MAR_Volissos::CloseTheRest()
{
  if (m_files.GetSize())
  {
    for (int i = 0; i < m_files.GetSize();)
    {
      const char *key;
      m_files.Enumerate(i, &key);
      if (strcmp(m_activefile.Get(), key))
      {
        m_files.DeleteByIndex(i);
      }
      else i++;
    }
  }
}

void MAR_Volissos::ActiveFile(const char *filename)
{
  if (WDL_NORMALLY(m_files.Exists(filename)))
  {
    SetPrevious();
    m_activefile.Set(filename);
    if (g_mainwnd) g_mainwnd->DisplayFileName();
  }
}

void MAR_Volissos::RenameFile(const char *old_filename, const char *new_filename)
{
  if (WDL_NORMALLY(m_files.Exists(old_filename)))
  {
    m_files.ChangeKey(old_filename, new_filename);
  }
}

void MAR_Volissos::InsertText(const char *text, int length)
{
  MAR_GapBuffer *gb = m_files.Get(m_activefile.Get());
  if (gb)
  {
    gb->InsertText(text, length);
  }
}

void MAR_Volissos::DeleteText(int length, bool forward)
{
  MAR_GapBuffer *gb = m_files.Get(m_activefile.Get());
  if (gb)
  {
    gb->DeleteText(length, forward);
  }
}

MAR_GapBuffer *MAR_Volissos::GetGapBuffer() const
{
  MAR_GapBuffer *gb = m_files.Get(m_activefile.Get());
  if (gb)
  {
    return gb;
  }

  return NULL;
}

MAR_GapBuffer *MAR_Volissos::GetGapBuffer(int index) const
{
  if (WDL_NORMALLY(index < m_files.GetSize()))
  {
    return m_files.Enumerate(index);
  }

  return NULL;
}

int MAR_Volissos::GetGapBufferSize() const
{
  return m_files.GetSize();
}

void MAR_Volissos::SetPrevious()
{
  if (m_activefile.GetLength())
  {
    m_previousfile.Set(m_activefile.Get());
  }
}

void MAR_Volissos::StartThread()
{
  m_killthread = false;
  m_threadhandle = (HANDLE)_beginthreadex(NULL, 0,
    ThreadFunction, (void *)this, 0, &m_threadid);
  SetThreadPriority(m_threadhandle, THREAD_PRIORITY_NORMAL);
  m_running = true;
}

void MAR_Volissos::StopThread()
{
  m_killthread = true;
  WaitForSingleObject(m_threadhandle, INFINITE);
  CloseHandle(m_threadhandle);
  m_running = false;
}

bool MAR_Volissos::IsRunning() const
{
  return m_running;
}

int MAR_Volissos::Run()
{
  return 1;
}

unsigned int WINAPI MAR_Volissos::ThreadFunction(void *arg)
{
  MAR_Volissos *self = (MAR_Volissos *)arg;

  if (WDL_NORMALLY(self))
  {
    int sleepstep = 50;

    self->m_killthread = false;

    while (!self->m_killthread)
    {
      self->m_mutex.Enter();
      while (!self->Run());
      self->m_mutex.Leave();
      Sleep(sleepstep);
    }
  }

  return 0;
}
