// Copyright (c) 2024 Giorgos Vougioukas
//
// The license can be found in the LICENSE file.

#include "find_wnd.h"
#include "app_info.h"
#include "surface_wnd.h"

MAR_FindWnd::MAR_FindWnd()
  : m_hwnd(NULL)
  , m_find(NULL)
  , m_matchcase(NULL)
  , m_matchword(NULL)
{}

MAR_FindWnd::~MAR_FindWnd()
{}

HWND MAR_FindWnd::Handle() const
{
  return m_hwnd;
}

void MAR_FindWnd::OnInitDialog(WPARAM wparam, LPARAM lparam)
{
  m_find = GetDlgItem(m_hwnd, IDC_EDIT1);
  m_matchcase = GetDlgItem(m_hwnd, IDC_CHECK1);
  m_matchword = GetDlgItem(m_hwnd, IDC_CHECK2);
  SendMessage(m_matchcase, BM_SETCHECK, BST_CHECKED, 0);
  SendMessage(m_matchword, BM_SETCHECK, BST_CHECKED, 0);

  int x = GetPrivateProfileInt(MAR_NAME, "find_wnd_x", 0, g_inipath.Get());
  int y = GetPrivateProfileInt(MAR_NAME, "find_wnd_y", 50, g_inipath.Get());
  SetWindowPos(m_hwnd, NULL, x, y, 0, 0, SWP_NOSIZE | SWP_NOZORDER | SWP_NOACTIVATE);
}

INT_PTR MAR_FindWnd::OnClose(WPARAM wparam, LPARAM lparam)
{
  DestroyWindow(m_hwnd);
  return 0;
}

void MAR_FindWnd::OnDestroy(WPARAM wparam, LPARAM lparam)
{
  RECT r;
  GetWindowRect(m_hwnd, &r);

  WDL_FastString rleft, rtop, rwidth, rheight;
  rleft.SetFormatted(32, "%d", r.left);
  rtop.SetFormatted(32, "%d", r.top);

  WritePrivateProfileString(MAR_NAME, "find_wnd_x", rleft.Get(), g_inipath.Get());
  WritePrivateProfileString(MAR_NAME, "find_wnd_y", rtop.Get(), g_inipath.Get());

  if (g_findwnd) { delete g_findwnd; g_findwnd = NULL; }
}

void MAR_FindWnd::OnSysCommand(WPARAM wparam, LPARAM lparam)
{
  if (wparam == SC_CLOSE)
  {
    SendMessage(m_hwnd, WM_CLOSE, 0, 0);
  }
}

void MAR_FindWnd::OnCommand(WPARAM wparam, LPARAM lparam)
{
  switch(LOWORD(wparam))
  {
  case IDOK:
    {
      bool case_sensitive = false;
      bool match_whole_word = false;
      LRESULT chk = SendMessage(m_matchcase, BM_GETCHECK, 0, 0);
      if (chk == BST_CHECKED) case_sensitive = true;
      chk = SendMessage(m_matchword, BM_GETCHECK, 0, 0);
      if (chk == BST_CHECKED) match_whole_word = true;

      char text[2048];
      int len = GetWindowText(m_find, text, sizeof(text));
      if (len && g_surfacewnd)
      {
        g_surfacewnd->FindKeyWord(text, case_sensitive, match_whole_word);
      }

      SendMessage(m_hwnd, WM_CLOSE, 0, 0);
    } break;
  case IDCANCEL:
    {
      SendMessage(m_hwnd, WM_CLOSE, 0, 0);
    } break;
  }
}

WDL_DLGRET MAR_FindWnd::FindWndProc(HWND hwnd, UINT msg, WPARAM wparam, LPARAM lparam)
{
  MAR_FindWnd *self = (MAR_FindWnd *)GetWindowLongPtr(hwnd, GWLP_USERDATA);

  if (!self && msg == WM_INITDIALOG)
  {
    SetWindowLongPtr(hwnd, GWLP_USERDATA, lparam);
    self = (MAR_FindWnd *)lparam;
    self->m_hwnd = hwnd;
  }

  if (WDL_likely(self))
  {
    return self->FindWndLoop(msg, wparam, lparam);
  }
  else
  {
    return 0;
  }
}

WDL_DLGRET MAR_FindWnd::FindWndLoop(UINT msg, WPARAM wparam, LPARAM lparam)
{
  switch (msg)
  {
    case WM_INITDIALOG: OnInitDialog(wparam, lparam); break;
    case WM_CLOSE: return OnClose(wparam, lparam); break;
    case WM_DESTROY: OnDestroy(wparam, lparam); break;
    case WM_SYSCOMMAND: OnSysCommand(wparam, lparam); break;
    case WM_COMMAND: OnCommand(wparam, lparam); break;
  }

  return 0;
}
