// Copyright (c) 2023 Giorgos Vougioukas
//
// The license can be found in the LICENSE file.

#include "gap_buffer.h"
#include "macros.h"
#include "preferences.h"
#include "main_wnd.h"

#define MAR_HIDEGAP 1

#include <WDL/fileread.h>
#include <WDL/filewrite.h>
#include <WDL/wdlutf8.h>

MAR_GapBuffer::MAR_GapBuffer()
  : m_strbuf(NULL)
  , m_cursorpos(0)
  , m_cachedline(0)
  , m_cachedcolumn(0)
  , m_ins(false)
  , m_eol(0)
  , m_ctx(NULL)
  , m_tabsize(2)
  , m_wantspaces(true)
  , m_findpos(NULL)
  , m_changepos(NULL)
  , m_casesensitive(true)
  , m_matchwholeword(true)
  , m_select(false)
  , m_selectinit(0)
  , m_selectstart(0)
  , m_selectend(0)
  , m_selectx1(0)
  , m_selecty1(0)
  , m_selectx2(0)
  , m_selecty2(0)
{
  m_strbuf = new MAR_GapString;
  m_ctx = new MAR_SurfaceContext;
  m_tabsize = g_preferences->GetTabSize();
  m_wantspaces = g_preferences->WantSpaces();
  m_eol = g_preferences->EndOfLine();
}

MAR_GapBuffer::~MAR_GapBuffer()
{
  if (m_ctx) delete m_ctx;
  if (m_strbuf) delete m_strbuf;
  m_svgapcache.Empty(true);
  m_svgapempties.Empty(true);
}

void MAR_GapBuffer::New(const char *filename)
{
  m_eol = g_preferences->EndOfLine();
  m_fn.Set(filename);
  m_strbuf->Set("");
  SetPosition(0);
  m_gap.Attach(m_strbuf);
  BuildStringView();
}

bool MAR_GapBuffer::Save()
{
  if (wdl_filename_cmp(m_fn.get_fileext(), ".mem"))
  {
    WDL_FileWrite fw(m_fn.Get());
    if (fw.IsOpen())
    {
      FlushGap();
      for (int i = 0, j = 0; i < m_strbuf->GetLength();)
      {
        if (m_strbuf->Get()[j] == '\n')
        {
          fw.Write(m_strbuf->Get() + i, j - i);
          switch (m_eol)
          {
            // 0: LF, 1: CRLF, 2: CR
            case 0: fw.Write("\n", 1); break;
            case 1: fw.Write("\r\n", 2); break;
            case 2: fw.Write("\r", 1); break;
          };
          j++; i = j;
        }
        else if (j == m_strbuf->GetLength() - 1)
        {
          fw.Write(m_strbuf->Get() + i, m_strbuf->GetLength() - i);
          j++; i = j;
        }
        else j++;
      }
      return true;
    }
  }

  return false;
}

bool MAR_GapBuffer::SaveAs(const char *filename)
{
  WDL_FileWrite fw(filename);
  if (fw.IsOpen())
  {
    FlushGap();
    for (int i = 0, j = 0; i < m_strbuf->GetLength();)
    {
      if (m_strbuf->Get()[j] == '\n')
      {
        fw.Write(m_strbuf->Get() + i, j - i);
        switch (m_eol)
        {
          // 0: LF, 1: CRLF, 2: CR
          case 0: fw.Write("\n", 1); break;
          case 1: fw.Write("\r\n", 2); break;
          case 2: fw.Write("\r", 1); break;
        };
        j++; i = j;
      }
      else if (j == m_strbuf->GetLength() - 1)
      {
        fw.Write(m_strbuf->Get() + i, m_strbuf->GetLength() - i);
        j++; i = j;
      }
      else j++;
    }
    m_fn.Set(filename);
    return true;
  }

  return false;
}

bool MAR_GapBuffer::Open(const char *filename)
{
  WDL_MutexLock lock(&m_mutex);
  m_fn.Set(filename);

  int filesize = 0;
  {
    WDL_FileRead fr(m_fn.Get());

    if (fr.IsOpen())
     {
      if (fr.GetSize() < MAR_STORAGE_MB(16))
      {
        filesize = (int)fr.GetSize();
      }
    }
  }

  if (filesize)
  {
#if 1
    FILE *fp = fopen(m_fn.Get(), "r");
    if (fp)
    {
      char str[4096];
      WDL_FastString ln;
      while (fgets(str, sizeof(str), fp) != NULL)
      {
        ln.Set(str);
        const int cr = ln.GetLength() - 2;
        const int lf = ln.GetLength() - 1;

        if (ln.Get()[cr] == '\r' && ln.Get()[lf] == '\n')
        {
          ln.DeleteSub(cr, 2); m_eol = 1;
        }
        else if (ln.Get()[lf] == '\n')
        {
          ln.DeleteSub(lf, 1); m_eol = 0;
        }
        else if (ln.Get()[lf] == '\r')
        {
          ln.DeleteSub(lf, 1); m_eol = 2;
        }

        for (int i = 0; i < ln.GetLength(); i++)
        {
          if (ln.Get()[i] == '\t')
          {
            ln.DeleteSub(i, 1);
            for (int j = 0; j < m_tabsize; j++)
            {
              ln.Insert(" ", i);
            }
          }
        }

        m_strbuf->Append(ln.Get());
        m_strbuf->Append("\n");
      }

      fclose(fp);

      int utf8 = WDL_DetectUTF8(m_strbuf->Get());
      if (utf8 >= 0)
      {
        SetPosition(0);
        m_gap.Attach(m_strbuf);
        BuildStringView();
        return true;
      }
    }
#else
    WDL_FileRead fr(m_fn.Get());

    if (fr.IsOpen())
    {
      if (fr.GetSize() < MAR_STORAGE_MB(16))
      {
        int filesize = (int)fr.GetSize();
        m_strbuf->SetLen(filesize, false);
        unsigned char buf[4096];
        int read = 0;
        int pos = 0;

        do
        {
          read = fr.Read(buf, sizeof(buf));
          if (read)
          {
            int rem = 0;
            char *sta = (char *)buf;
            char *end = (char *)&buf[read - 1];

            do
            {
              if (*sta == '\r' && *(sta + 1) == '\n')
              {
                memmove(sta, sta + 1, end - sta + 1); rem += 1; m_eol = 1;
              }
            } while (sta++ < end);

            read -= rem;
            if (!rem)
            {
              for (int i = 0; i < read; i++)
              {
                if (buf[i] == '\r') { buf[i] = '\n'; m_eol = 2; }
              }
            }

            memcpy(m_strbuf->Get() + pos, (char *)buf, read);
            pos += read;
          }
        } while (read);

        m_strbuf->Get()[pos] = '\0';
        m_strbuf->SetLen(pos, false);
        int utf8 = WDL_DetectUTF8(m_strbuf->Get());
        if (utf8 >= 0)
        {
          SetPosition(0);
          m_gap.Attach(m_strbuf);
          BuildStringView();
          return true;
        }
      }
    }
#endif
  }

  return false;
}

void MAR_GapBuffer::SetPositionStartOfFile(bool select)
{
  CheckSelect(select);
  SetPosition(0);
  SetSelect(select, true, true);
}

void MAR_GapBuffer::SetPositionEndOfFile(bool select)
{
  CheckSelect(select);
  SetPosition(m_strbuf->GetLength());
  SetSelect(select, true, false);
}

void MAR_GapBuffer::SetPositionStartOfLine(bool select)
{
  CheckSelect(select);
  char *start = GetCursorPositionPtr();
  const char *old = start;
  while (start-- > m_strbuf->Get())
  {
    if (*start == '\n') break;
  }

  start++; const char *beg = start; int bytepos;
  while (*start == ' ' || *start == '\t') start++;
  if (start == old) bytepos = (int)(beg - m_strbuf->Get());
  else bytepos = (int)(start - m_strbuf->Get());
  SetPosition(bytepos);
  SetSelect(select, true, true);
}

void MAR_GapBuffer::SetPositionEndOfLine(bool select)
{
  CheckSelect(select);
  char *start = GetCursorPositionPtr();
  if (*start == '\n') { SetSelect(select, true, false); return; }
  const char *end = m_strbuf->Get() + m_strbuf->GetLength();
  while (start++ < end)
  {
    if (*start == '\n') break;
  }

  const int bytepos = (int)(start - m_strbuf->Get());
  SetPosition(bytepos);
  SetSelect(select, true, false);
}

void MAR_GapBuffer::SetPositionNextChar(bool select)
{
  CheckSelect(select);
  const char *rd = GetCursorPositionPtr();
  const int bytepos = m_cursorpos + wdl_utf8_parsechar(rd, NULL);
  SetPosition(bytepos);
  SetSelect(select, true, false);
}

void MAR_GapBuffer::SetPositionPreviousChar(bool select)
{
  CheckSelect(select);
  int c = 0; int bytesize = 0;
  const char *rd = GetCursorPositionPtr();
  while (rd-- >= m_strbuf->Get())
  {
    char text[16];
    c <<= 8; c = rd[0] & 0xff;
    bytesize = wdl_utf8_makechar(c, text, sizeof(text));
    if (bytesize > 0) break;
  }
  if (bytesize < 0) bytesize = 0;
  const int bytepos = m_cursorpos - bytesize;
  SetPosition(bytepos);
  SetSelect(select, true, true);
}

void MAR_GapBuffer::SetPositionNextWord(bool select)
{
  CheckSelect(select);
  char *start = GetCursorPositionPtr();
  const char *end = m_strbuf->Get() + m_strbuf->GetLength();
  while (start++ < end)
  {
    if ((*(start - 1) == ' ' || *(start - 1) == '\n') && *start != ' ')
    {
      if (*start == '\n' || start == end) continue;
      const int bytepos = (int)(start - m_strbuf->Get());
      SetPosition(bytepos); SetSelect(select, true, false); break;
    }
  }
}

void MAR_GapBuffer::SetPositionPreviousWord(bool select)
{
  CheckSelect(select);
  char *start = GetCursorPositionPtr();
  const char *beg = m_strbuf->Get();
  while (start-- > beg)
  {
    if ((start == beg || *(start - 1) == ' ' || *(start - 1) == '\n') && *start != ' ')
    {
      if (*start == '\n') continue;
      const int bytepos = (int)(start - m_strbuf->Get());
      SetPosition(bytepos); SetSelect(select, true, true); break;
    }
  }
}

void MAR_GapBuffer::SetPositionNextLine(bool select)
{
  CheckSelect(select);
  const int ln = GetCursorLine();
  const int maxln = GetTotalLines() - 1;
  GoToLine(wdl_min(ln + 1, maxln));
  SetSelect(select, true, false);
}

void MAR_GapBuffer::SetPositionPreviousLine(bool select)
{
  CheckSelect(select);
  const int ln = GetCursorLine();
  GoToLine(wdl_max(ln - 1, 0));
  SetSelect(select, true, true);
}

void MAR_GapBuffer::SetPositionPageUp(int lines, bool select)
{
  CheckSelect(select);
  const int ln = GetCursorLine();
  GoToLine(wdl_max(ln - lines, 0));
  SetSelect(select, true, true);
}

void MAR_GapBuffer::SetPositionPageDown(int lines, bool select)
{
  CheckSelect(select);
  const int ln = GetCursorLine();
  const int maxln = GetTotalLines() - 1;
  GoToLine(wdl_min(ln + lines, maxln));
  SetSelect(select, true, false);
}

int MAR_GapBuffer::GetCursorLine() const
{
  return m_cachedline;
}

int MAR_GapBuffer::GetCursorColumn() const
{
  return m_cachedcolumn;
}

void MAR_GapBuffer::GoToLine(int line, int column)
{
  WDL_ASSERT(line >= 0);
  const int col = column >= 0 ? column : GetCursorColumn();

  int ln = m_cachedline;
  int bytepos = m_cursorpos;
  char *start = m_strbuf->Get() + m_cursorpos;
  const char *end = m_strbuf->Get() + m_strbuf->GetLength();

  if (line > ln)
  {
    do
    {
      if (*start == '\n')
      {
        ln++;
        if (ln == line)
        {
          bytepos = (int)(start + 1 - m_strbuf->Get());
          break;
        }
      }
    } while (start++ <= end);
  }
  else if (line < ln)
  {
    char *skip = start;
    while (skip-- >= m_strbuf->Get())
    {
      if (*skip == '\n') { start = skip; break; }
    }

    while (start-- >= m_strbuf->Get())
    {
      if (*start == '\n')
      {
        ln--;
        if (ln == line)
        {
          bytepos = (int)(start + 1 - m_strbuf->Get());
          break;
        }
      }
      else if (start == m_strbuf->Get())
      {
        ln--;
        if (ln == line)
        {
          bytepos = (int)(start - m_strbuf->Get());
          break;
        }
      }
    }
  }
  else if (line == ln)
  {
    while (start-- >= m_strbuf->Get())
    {
      if (*start == '\n')
      {
        bytepos = (int)(start + 1 - m_strbuf->Get());
        break;
      }
      else if (start == m_strbuf->Get())
      {
        bytepos = (int)(start - m_strbuf->Get());
        break;
      }
    }
  }

  const int bytecol = WDL_utf8_charpos_to_bytepos(start, col);
  while (start++ <= m_strbuf->Get() + bytepos + bytecol)
  {
    if (*start == '\n') break;
  }

  const int bytecolsum = bytepos + bytecol;
  bytepos = (int)(start - m_strbuf->Get());
  bytepos = wdl_min(bytecolsum, bytepos);
  SetPosition(bytepos);
}

void MAR_GapBuffer::InsertText(const char *text, int length)
{
  if (m_select)
  {
    m_strbuf->DeleteSub(m_selectstart, m_selectend - m_selectstart);
    SetPosition(m_selectstart, false);
    SetSelect(false, false, false);
  }
  else if (m_ins)
  {
    DeleteText(length, true);
  }

  m_gap.Insert(text, m_cursorpos, length);
  SetPosition(m_cursorpos + length);
  BuildStringView();
}

void MAR_GapBuffer::DeleteText(int length, bool forward)
{
  if (m_select)
  {
    m_strbuf->DeleteSub(m_selectstart, m_selectend - m_selectstart);
    SetPosition(m_selectstart, false);
    SetSelect(false, false, false);
  }
  else
  {
    if (forward)
    {
      FlushGap();
      const int charpos = WDL_utf8_bytepos_to_charpos(m_strbuf->Get(), m_cursorpos);
      const int bytepos = WDL_utf8_charpos_to_bytepos(m_strbuf->Get(), charpos + length);
      m_strbuf->DeleteSub(m_cursorpos, bytepos - m_cursorpos);
      SetPosition(m_cursorpos, false);
    }
    else
    {
      if (m_cursorpos == m_gap.GetPosition() + m_gap.GetLength())
      {
        const int charpos = WDL_utf8_bytepos_to_charpos(m_strbuf->Get(), m_cursorpos);
        const int bytepos = WDL_utf8_charpos_to_bytepos(m_strbuf->Get(), charpos - length);
        int len = m_cursorpos - bytepos;
        if (len <= m_gap.GetLength())
        {
          m_gap.Delete(len);
        }
        else
        {
          FlushGap();
          m_strbuf->DeleteSub(bytepos, len);
        }
        SetPosition(bytepos, false);
      }
      else
      {
        FlushGap();
        const int charpos = WDL_utf8_bytepos_to_charpos(m_strbuf->Get(), m_cursorpos);
        const int bytepos = WDL_utf8_charpos_to_bytepos(m_strbuf->Get(), charpos - length);
        m_strbuf->DeleteSub(bytepos, m_cursorpos - bytepos);
        SetPosition(bytepos, false);
      }
    }
  }
  BuildStringView();
}

void MAR_GapBuffer::DeleteWord(bool forward)
{
  if (m_select)
  {
    m_strbuf->DeleteSub(m_selectstart, m_selectend - m_selectstart);
    SetPosition(m_selectstart, false);
    SetSelect(false, false, false);
  }
  else
  {
    FlushGap();
    if (forward)
    {
      char *start = GetCursorPositionPtr();
      char chk = *start;
      while (1)
      {
        if (chk == '\n')
        {
          m_strbuf->DeleteSub(m_cursorpos, 1); SetPosition(m_cursorpos, false); break;
        }
        else if (chk == ' ')
        {
          if (*start == '\n' || start == m_strbuf->Get() + m_strbuf->GetLength() || *start != ' ') break;
          if (*start == ' ') { m_strbuf->DeleteSub(m_cursorpos, 1); SetPosition(m_cursorpos, false); }
        }
        else
        {
          if (*start == '\n' || start == m_strbuf->Get() + m_strbuf->GetLength() || *start == ' ') break;
          if (*start != ' ') { m_strbuf->DeleteSub(m_cursorpos, 1); SetPosition(m_cursorpos, false); }
        }
      }
    }
    else
    {
      char *start = GetCursorPositionPtr();
      char chk = *(start - 1);
      while (start--)
      {
        if (chk == '\n')
        {
          if (m_cursorpos)
          {
            m_strbuf->DeleteSub(m_cursorpos - 1, 1); SetPosition(m_cursorpos - 1, false); break;
          }
        }
        else if (chk == ' ')
        {
          if (*start == '\n' || start < m_strbuf->Get() || *start != ' ') break;
          if (*start == ' ') { m_strbuf->DeleteSub(m_cursorpos - 1, 1); SetPosition(m_cursorpos - 1, false); }
        }
        else
        {
          if (*start == '\n' || start < m_strbuf->Get() || *start == ' ') break;
          if (*start != ' ') { m_strbuf->DeleteSub(m_cursorpos - 1, 1); SetPosition(m_cursorpos - 1, false); }
        }
      }
    }
  }
  BuildStringView();
}

bool MAR_GapBuffer::FindKeyWord(const char *find,
  bool case_sensitive, bool match_whole_word, bool start, bool reverse)
{
  FlushGap(); //BuildStringView();
  if (m_select) SetSelect(false, false, false);
  m_casesensitive = case_sensitive;
  m_matchwholeword = match_whole_word;
  const char *end = m_strbuf->Get() + m_strbuf->GetLength();

  if (reverse && !start)
  {
    do
    {
      if (m_casesensitive)
      {
        if (!strncmp(m_findpos, m_find.Get(), m_find.GetLength()))
        {
          if (m_matchwholeword)
          {
            if (m_findpos == m_strbuf->Get() &&
              ((*(m_findpos + m_find.GetLength()) == ' ' ||
              *(m_findpos + m_find.GetLength()) == '\n') ||
              m_findpos + m_find.GetLength() == end))
            {}
            else if ((*(m_findpos - 1) == ' ' || *(m_findpos - 1) == '\n') &&
              ((*(m_findpos + m_find.GetLength()) == ' ' ||
              *(m_findpos + m_find.GetLength()) == '\n') ||
              m_findpos + m_find.GetLength() == end))
            {}
            else continue;
          }
          const int bytepos = (int)(m_findpos - m_strbuf->Get());
          SetPosition(bytepos);
          return true;
        }
      }
      else
      {
        if (!strnicmp(m_findpos, m_find.Get(), m_find.GetLength()))
        {
          if (m_matchwholeword)
          {
            if (m_findpos == m_strbuf->Get() &&
              ((*(m_findpos + m_find.GetLength()) == ' ' ||
              *(m_findpos + m_find.GetLength()) == '\n') ||
              m_findpos + m_find.GetLength() == end))
            {}
            else if ((*(m_findpos - 1) == ' ' || *(m_findpos - 1) == '\n') &&
              ((*(m_findpos + m_find.GetLength()) == ' ' ||
              *(m_findpos + m_find.GetLength()) == '\n') ||
              m_findpos + m_find.GetLength() == end))
            {}
            else continue;
          }
          const int bytepos = (int)(m_findpos - m_strbuf->Get());
          SetPosition(bytepos);
          return true;
        }
      }
    } while (m_findpos-- >= m_strbuf->Get());
    m_findpos = m_strbuf->Get() + m_strbuf->GetLength();
  }
  else
  {
    if (start || !m_findpos || m_findpos >= end)
    {
      if (start) { m_find.Set(find); m_change.Set(""); }
      m_findpos = GetCursorPositionPtr();
    }

    do
    {
      if (m_casesensitive)
      {
        if (!strncmp(m_findpos, m_find.Get(), m_find.GetLength()))
        {
          if (m_matchwholeword)
          {
            if (m_findpos == m_strbuf->Get() &&
              ((*(m_findpos + m_find.GetLength()) == ' ' ||
              *(m_findpos + m_find.GetLength()) == '\n') ||
              m_findpos + m_find.GetLength() == end))
            {}
            else if ((*(m_findpos - 1) == ' ' || *(m_findpos - 1) == '\n') &&
              ((*(m_findpos + m_find.GetLength()) == ' ' ||
              *(m_findpos + m_find.GetLength()) == '\n') ||
              m_findpos + m_find.GetLength() == end))
            {}
            else continue;
          }
          const int bytepos = (int)(m_findpos - m_strbuf->Get());
          SetPosition(bytepos);
          return true;
        }
      }
      else
      {
        if (!strnicmp(m_findpos, m_find.Get(), m_find.GetLength()))
        {
          if (m_matchwholeword)
          {
            if (m_findpos == m_strbuf->Get() &&
              ((*(m_findpos + m_find.GetLength()) == ' ' ||
              *(m_findpos + m_find.GetLength()) == '\n') ||
              m_findpos + m_find.GetLength() == end))
            {}
            else if ((*(m_findpos - 1) == ' ' || *(m_findpos - 1) == '\n') &&
              ((*(m_findpos + m_find.GetLength()) == ' ' ||
              *(m_findpos + m_find.GetLength()) == '\n') ||
              m_findpos + m_find.GetLength() == end))
            {}
            else continue;
          }
          const int bytepos = (int)(m_findpos - m_strbuf->Get());
          SetPosition(bytepos);
          return true;
        }
      }
    } while (m_findpos++ < end - m_find.GetLength());
    m_findpos = m_strbuf->Get();
  }

  return false;
}

int MAR_GapBuffer::ChangeKeyWord(const char *find, const char *change,
  bool case_sensitive, bool match_whole_word)
{
  FlushGap(); //BuildStringView();
  if (m_select) SetSelect(false, false, false);
  m_findpos = m_strbuf->Get();
  const char *end = m_strbuf->Get() + m_strbuf->GetLength();
  m_find.Set(find); m_change.Set(change);
  int changes = 0;

  do
  {
    if (case_sensitive)
    {
      if (!strncmp(m_findpos, m_find.Get(), m_find.GetLength()))
      {
        if (match_whole_word)
        {
          if (m_findpos == m_strbuf->Get() &&
            ((*(m_findpos + m_find.GetLength()) == ' ' ||
            *(m_findpos + m_find.GetLength()) == '\n') ||
            m_findpos + m_find.GetLength() == end))
          {}
          else if ((*(m_findpos - 1) == ' ' || *(m_findpos - 1) == '\n') &&
            ((*(m_findpos + m_find.GetLength()) == ' ' ||
            *(m_findpos + m_find.GetLength()) == '\n') ||
            m_findpos + m_find.GetLength() == end))
          {}
          else continue;
        }
        const int bytepos = (int)(m_findpos - m_strbuf->Get());
        SetPosition(bytepos);
        DeleteText(m_find.GetLength(), true);
        InsertText(m_change.Get(), m_change.GetLength());
        SetPosition(bytepos);
        changes++;
      }
    }
    else
    {
      if (!strnicmp(m_findpos, m_find.Get(), m_find.GetLength()))
      {
        if (match_whole_word)
        {
          if (m_findpos == m_strbuf->Get() &&
            ((*(m_findpos + m_find.GetLength()) == ' ' ||
            *(m_findpos + m_find.GetLength()) == '\n') ||
            m_findpos + m_find.GetLength() == end))
          {}
          else if ((*(m_findpos - 1) == ' ' || *(m_findpos - 1) == '\n') &&
            ((*(m_findpos + m_find.GetLength()) == ' ' ||
            *(m_findpos + m_find.GetLength()) == '\n') ||
            m_findpos + m_find.GetLength() == end))
          {}
          else continue;
        }
        const int bytepos = (int)(m_findpos - m_strbuf->Get());
        SetPosition(bytepos);
        DeleteText(m_find.GetLength(), true);
        InsertText(m_change.Get(), m_change.GetLength());
        SetPosition(bytepos);
        changes++;
      }
    }
  } while (m_findpos++ < end - m_find.GetLength());

  return changes;
}

void MAR_GapBuffer::FindNextKeyWord()
{
  if (m_select) SetSelect(false, false, false);
  if (m_find.GetLength())
  {
    if (m_findpos > m_strbuf->Get()) m_findpos++;
    FindKeyWord(m_find.Get(), m_casesensitive, m_matchwholeword, false, false);
  }
}

void MAR_GapBuffer::FindPreviousKeyWord()
{
  if (m_select) SetSelect(false, false, false);
  if (m_find.GetLength())
  {
    if (m_findpos > m_strbuf->Get()) m_findpos--;
    FindKeyWord(m_find.Get(), m_casesensitive, m_matchwholeword, false, true);
  }
}

void MAR_GapBuffer::ToggleInsert()
{
  m_ins = !m_ins;
}

bool MAR_GapBuffer::GetInsert() const
{
  return m_ins;
}

void MAR_GapBuffer::InsertTab()
{
  if (m_wantspaces)
  {
    if (m_select)
    {
      int pad = 0;
      const int bytepos = m_cursorpos;
      for (int i = wdl_min(m_selecty1, m_selecty2); i <= wdl_max(m_selecty1, m_selecty2); i++)
      {
        GoToLine(i, 0);
        for (int j = 0; j < m_tabsize; j++)
        {
          m_strbuf->Insert(" ", m_cursorpos, 1);
          SetPosition(m_cursorpos + 1);
          pad++;
        }
      }
      const bool nw = IsNorthWest();
      SetPosition(m_cursorpos - m_tabsize);
      SetSelect(true, true, nw);
    }
    else
    {
      for (int i = 0; i < m_tabsize; i++)
      {
        InsertText(" ", 1);
      }
    }
  }
  else
  {
    if (m_select)
    {
      int pad = 0;
      const int bytepos = m_cursorpos;
      for (int i = wdl_min(m_selecty1, m_selecty2); i <= wdl_max(m_selecty1, m_selecty2); i++)
      {
        GoToLine(i, 0);
        m_strbuf->Insert("\t", m_cursorpos, 1);
        SetPosition(m_cursorpos + 1);
        pad++;
      }
      const bool nw = IsNorthWest();
      SetPosition(m_cursorpos - 1);
      SetSelect(true, true, nw);
    }
    else
    {
      InsertText("\t", 1);
    }
  }
  BuildStringView();
}

void MAR_GapBuffer::RemoveTab()
{
  FlushGap();
  if (m_wantspaces)
  {
    if (m_select)
    {
      const int bytepos = m_cursorpos;
      for (int i = wdl_min(m_selecty1, m_selecty2); i <= wdl_max(m_selecty1, m_selecty2); i++)
      {
        GoToLine(i, 0);
        for (int j = 0; j < m_tabsize; j++)
        {
          if (m_strbuf->Get()[m_cursorpos] == ' ')
          {
            m_strbuf->DeleteSub(m_cursorpos, 1);
            SetPosition(m_cursorpos);
          }
        }
      }
      const bool nw = IsNorthWest();
      SetSelect(true, true, nw);
    }
    else
    {
      int space = 0;
      char *start = GetCursorPositionPtr();
      for (int i = 0; i < m_tabsize; i++)
      {
        if (*--start == ' ') space++;
      }
      if (space)
      {
        m_strbuf->DeleteSub(m_cursorpos - space, space);
        SetPosition(m_cursorpos - space);
      }
    }
  }
  else
  {
    if (m_select)
    {
      const int bytepos = m_cursorpos;
      for (int i = wdl_min(m_selecty1, m_selecty2); i <= wdl_max(m_selecty1, m_selecty2); i++)
      {
        GoToLine(i, 0);
        if (m_strbuf->Get()[m_cursorpos] == '\t')
        {
          m_strbuf->DeleteSub(m_cursorpos, 1);
          SetPosition(m_cursorpos);
        }
      }
      const bool nw = IsNorthWest();
      SetSelect(true, true, nw);
    }
    else
    {
      int space = 0;
      char *tab = GetCursorPositionPtr();
      if (*--tab == '\t') space++;
      if (space)
      {
        m_strbuf->DeleteSub(m_cursorpos - space, space);
        SetPosition(m_cursorpos - space);
      }
    }
  }
  BuildStringView();
}

void MAR_GapBuffer::CommentLine()
{
  if (m_select)
  {
    int pad = 0;
    const int bytepos = m_cursorpos;
    for (int i = wdl_min(m_selecty1, m_selecty2); i <= wdl_max(m_selecty1, m_selecty2); i++)
    {
      GoToLine(i, 0);
      for (int j = 0; j < 2; j++)
      {
        m_strbuf->Insert("/", m_cursorpos, 1);
        SetPosition(m_cursorpos + 1);
        pad++;
      }
    }
    const bool nw = IsNorthWest();
    SetPosition(m_cursorpos - m_tabsize);
    SetSelect(true, true, nw);
  }
  else
  {
    for (int i = 0; i < 2; i++)
    {
      InsertText("/", 1);
    }
  }
  BuildStringView();
}

void MAR_GapBuffer::UncommentLine()
{
  FlushGap();
  if (m_select)
  {
    const int bytepos = m_cursorpos;
    for (int i = wdl_min(m_selecty1, m_selecty2); i <= wdl_max(m_selecty1, m_selecty2); i++)
    {
      GoToLine(i, 0);
      for (int j = 0; j < 2; j++)
      {
        if (m_strbuf->Get()[m_cursorpos] == '/')
        {
          m_strbuf->DeleteSub(m_cursorpos, 1);
          SetPosition(m_cursorpos);
        }
      }
    }
    const bool nw = IsNorthWest();
    SetSelect(true, true, nw);
  }
  else
  {
    int space = 0;
    char *start = GetCursorPositionPtr();
    for (int i = 0; i < 2; i++)
    {
      if (*--start == '/') space++;
    }
    if (space)
    {
      m_strbuf->DeleteSub(m_cursorpos - space, space);
      SetPosition(m_cursorpos - space);
    }
  }
  BuildStringView();
}

int MAR_GapBuffer::GetTotalLines() const
{
  return m_sv.GetSize();
}

void MAR_GapBuffer::GetStringView(WDL_TypedBuf<MAR_StringView> **sv)
{
  *sv = &m_sv;
}

MAR_SurfaceContext *MAR_GapBuffer::GetSurfaceContext() const
{
  return m_ctx ? m_ctx : NULL;
}

const char *MAR_GapBuffer::GetFilePath() const
{
  return m_fn.Get();
}

const char *MAR_GapBuffer::GetFileName() const
{
  return m_fn.get_filepart();
}

bool MAR_GapBuffer::HasSelection() const
{
  return m_select;
}

void MAR_GapBuffer::GetSelectionRegion(int *x1, int *y1, int *x2, int *y2)
{
  *x1 = m_selectx1;
  *y1 = m_selecty1;
  *x2 = m_selectx2;
  *y2 = m_selecty2;
}

void MAR_GapBuffer::SetPosition(int bytepos, bool diff)
{
  const int previouspos = diff ? m_cursorpos : 0;
  m_cursorpos = wdl_clamp(bytepos, 0, m_strbuf->GetLength());
  CacheLine(previouspos, diff); CacheColumn();
}

void MAR_GapBuffer::CheckSelect(bool select)
{
  FlushGap();
  if (!m_select && select)
  {
    m_selectinit = m_selectstart = m_selectend = m_cursorpos;
    m_selectx1 = m_selectx2 = m_cachedcolumn;
    m_selecty1 = m_selecty2 = m_cachedline;
    m_select = true;
  }
}

void MAR_GapBuffer::MouseMoveSelect(int x, int y)
{
  m_selectx2 = x;
  m_selecty2 = y;
}

void MAR_GapBuffer::SetSelect(bool select, bool direction, bool northwest)
{
  if (select)
  {
    if (m_cursorpos > m_selectinit)
    {
      m_selectend = m_cursorpos;
    }
    else if (m_cursorpos < m_selectinit)
    {
      m_selectstart = m_cursorpos;
    }

    m_selectx2 = m_cachedcolumn;
    m_selecty2 = m_cachedline;

    if (m_selectx1 == m_selectx2 && m_selecty1 == m_selecty2)
    {
      m_selectstart = m_selectend = m_cursorpos;
    }
  }
  else
  {
    if (m_select)
    {
      if (direction)
      {
        if (northwest)
        {
          SetPosition(m_selectstart);
        }
        else
        {
          SetPosition(m_selectend);
        }
      }

      m_select = false;
      m_selectstart = 0;
      m_selectend = 0;
      m_selectx1 = 0;
      m_selecty1 = 0;
      m_selectx2 = 0;
      m_selecty2 = 0;
    }
  }
}

bool MAR_GapBuffer::IsNorthWest() const
{
  if (m_selecty1 < m_selecty2)
  {
    return false;
  }
  else if (m_selecty1 > m_selecty2)
  {
    return true;
  }
  else
  {
    if (m_selectx1 < m_selectx2)
    {
      return false;
    }
    else if (m_selectx1 > m_selectx2)
    {
      return true;
    }
    else
    {
      return false;
    }
  }
}

void MAR_GapBuffer::SelectAll()
{
  SetPositionStartOfFile(false);
  SetPositionEndOfFile(true);
}

void MAR_GapBuffer::Cut()
{
  if (m_select)
  {
    const int sta = m_selectstart;
    const int end = m_selectend;
    const int len = m_selectend - m_selectstart;
    if (g_mainwnd && OpenClipboard(g_mainwnd->Handle()))
    {
      EmptyClipboard();
      HANDLE buf = GlobalAlloc(GMEM_MOVEABLE, len + 1);
      if (buf)
      {
        char *ptr = (char *)GlobalLock(buf);
        memcpy(ptr, &m_strbuf->Get()[sta], len);
        ptr[len] = 0;
        GlobalUnlock(buf);
        SetClipboardData(CF_TEXT, buf);
        DeleteText(len, true);
      }
      CloseClipboard();
    }
  }
}

void MAR_GapBuffer::Copy()
{
  if (m_select)
  {
    const int sta = m_selectstart;
    const int end = m_selectend;
    const int len = m_selectend - m_selectstart;
    if (g_mainwnd && OpenClipboard(g_mainwnd->Handle()))
    {
      EmptyClipboard();
      HANDLE buf = GlobalAlloc(GMEM_MOVEABLE, len + 1);
      if (buf)
      {
        char *ptr = (char *)GlobalLock(buf);
        memcpy(ptr, &m_strbuf->Get()[sta], len);
        ptr[len] = 0;
        GlobalUnlock(buf);
        SetClipboardData(CF_TEXT, buf);
      }
      CloseClipboard();
    }
  }
}

void MAR_GapBuffer::Paste()
{
  if (
#if defined(_WIN32)
    (IsClipboardFormatAvailable(CF_TEXT)) &&
#endif
    (g_mainwnd && OpenClipboard(g_mainwnd->Handle())))
  {
    HANDLE buf = GetClipboardData(CF_TEXT);
    if (buf)
    {
      char *ptr = (char *)GlobalLock(buf);
      if (ptr)
      {
        WDL_HeapBuf hb;
        int len = (int)strlen(ptr);
        hb.Resize(len);
        memcpy(hb.Get(), ptr, len);

        int rem = 0;

        char *sta = (char *)hb.Get();
        char *end = (char *)hb.Get() + hb.GetSize();

        do
        {
          if (*sta == '\r' && *(sta + 1) == '\n')
          {
            memmove(sta, sta + 1, end - sta + 1); rem += 1;
          }
        } while (sta++ < end);

        len -= rem;
        if (!rem)
        {
          for (int i = 0; i < len; i++)
          {
            char *ch = (char *)hb.Get();
            if (ch[i] == '\r') { ch[i] = '\n'; }
          }
        }

        InsertText((char *)hb.Get(), len);
        GlobalUnlock(buf);
      }
    }
    CloseClipboard();
  }
}

void MAR_GapBuffer::CacheLine(int previous_pos, bool diff)
{
  int ln = diff ? m_cachedline : 0;
  int bytepos = m_cursorpos;
  if (previous_pos < bytepos)
  {
    for (int i = previous_pos; i < bytepos; i++)
    {
      if (m_strbuf->Get()[i] == '\n') ln++;
    }
  }
  else if (previous_pos > bytepos)
  {
    for (int i = bytepos; i < previous_pos; i++)
    {
      if (m_strbuf->Get()[i] == '\n') ln--;
    }
  }

  m_cachedline = wdl_max(ln, 0);
}

void MAR_GapBuffer::CacheColumn()
{
  int col = 0;
  char *start, *end;
  start = end = GetCursorPositionPtr();
  while (start-- > m_strbuf->Get())
  {
    if (*start == '\n') break;
  }

  start++;
  WDL_ASSERT(end >= start);
  int bytepos = (int)(end - start);
  col = WDL_utf8_bytepos_to_charpos(start, bytepos);
  m_cachedcolumn = wdl_max(col, 0);
}

void MAR_GapBuffer::FlushGap()
{
  if (!m_gap.IsFlushed())
  {
    m_cursorpos = m_gap.Flush(m_cursorpos);
    //SetPosition(m_cursorpos);
    BuildStringView();
  }
}

void MAR_GapBuffer::BuildStringView()
{
  WDL_MutexLock lock(&m_mutex);

  m_sv.Resize(0, false);
  char *beg = m_strbuf->Get();
  char *act = m_strbuf->Get();
  const char *end = m_strbuf->Get() + m_strbuf->GetLength();
  const char *startgap = m_strbuf->Get() + m_gap.GetPosition();
  const char *starthide = m_strbuf->Get() + m_gap.GetPosition() + m_gap.GetLength();
  const char *endhide = m_strbuf->Get() + m_gap.GetPosition() + m_gap.GetSize();

  if (!m_gap.IsFlushed() && MAR_HIDEGAP)
  {
    for (int i = 0; i < m_svgapcache.GetSize(); i++)
    {
      WDL_FastString *str = m_svgapcache.Get(i);
      if (str) m_svgapempties.Add(str);
    }
    m_svgapcache.Empty(false);

    char *a = act = (char *)startgap;
    while (act-- >= beg)
    {
      if (*act == '\n') { a = act + 1; break; }
      else if (act == beg) { a = act; break; }
    }

    char *b = act = (char *)endhide;

    do
    {
      if (*act == '\n') { b = act; break; }
      else if (act == end) { b = act; break; }
    } while (act++ <= end);

    const int aaa = (int)(a - beg);
    int sss = (int)(starthide - beg);
    int eee = (int)(endhide - beg);
    sss = sss - aaa;
    eee = eee - aaa;

    const int blocksize = (int)(b - a);
    m_svblock.Resize(blocksize + 1, false);
    memset(m_svblock.Get(), 0, blocksize + 1);
    memcpy(m_svblock.Get(), a, blocksize);
    memmove(m_svblock.Get() + sss,
      m_svblock.Get() + eee,
      blocksize + 1 - eee);

    beg = act = m_svblock.Get();

    do
    {
      if (*act == '\n')
      {
        MAR_StringView sv;
        sv.str = beg;
        sv.len = (int)(act - beg);

        WDL_FastString *str = NULL;
        if (m_svgapempties.GetSize())
        {
          int la = m_svgapempties.GetSize() - 1;
          str = m_svgapempties.Get(la);
          m_svgapempties.Delete(la, false);
        }
        else
        {
          str = new WDL_NEW WDL_FastString;
        }

        if (WDL_NORMALLY(str))
        {
          if (sv.len) str->Set(sv.str, sv.len);
          else str->Set("");
          m_svgapcache.Add(str);
          beg = act + 1;
        }
      }
      else if (act == m_svblock.Get() + blocksize)
      {
        MAR_StringView sv;
        sv.str = beg;
        sv.len = (int)(act - beg);

        WDL_FastString *str = NULL;
        if (m_svgapempties.GetSize())
        {
          int la = m_svgapempties.GetSize() - 1;
          str = m_svgapempties.Get(la);
          m_svgapempties.Delete(la, false);
        }
        else
        {
          str = new WDL_NEW WDL_FastString;
        }

        if (WDL_NORMALLY(str))
        {
          if (sv.len) str->Set(sv.str, sv.len);
          else str->Set("");
          m_svgapcache.Add(str);
        }
      }
    } while (act++ < m_svblock.Get() + blocksize);

    beg = act = m_strbuf->Get();
    int cnt, ln; cnt = ln = 0;

    do
    {
      if (*act == '\n')
      {
        cnt++;
        if (act == a - 1) ln = cnt;
        MAR_StringView sv;
        sv.str = beg;
        sv.len = (int)(act - beg);
        m_sv.Add(sv);
        beg = act + 1;
      }
      else if (act == end)
      {
        MAR_StringView sv;
        sv.str = beg;
        sv.len = (int)(act - beg);
        m_sv.Add(sv);
      }
    } while (act++ < end);

    for (int i = 0; i < m_svgapcache.GetSize(); i++)
    {
      MAR_StringView *sv = m_sv.Get();
      WDL_FastString *str = m_svgapcache.Get(i);
      if (WDL_NORMALLY(sv && str && ln + i < m_sv.GetSize()))
      {
        sv[ln + i].str = (char *)str->Get();
        sv[ln + i].len = str->GetLength();
      }
    }
  }
  else
  {
    do
    {
      if (*act == '\n')
      {
        MAR_StringView sv;
        sv.str = beg;
        sv.len = (int)(act - beg);
        m_sv.Add(sv);
        beg = act + 1;
      }
      else if (act == end)
      {
        MAR_StringView sv;
        sv.str = beg;
        sv.len = (int)(act - beg);
        m_sv.Add(sv);
      }
    } while (act++ < end);
  }
  if (m_ctx) m_ctx->fulldraw = true;
}

char *MAR_GapBuffer::GetCursorPositionPtr() const
{
  WDL_ASSERT(m_cursorpos <= m_strbuf->GetLength());
  const int bytepos = wdl_clamp(m_cursorpos, 0, m_strbuf->GetLength());
  return (char *)&m_strbuf->Get()[bytepos];
}
