// Copyright (c) 2023 Giorgos Vougioukas
//
// The license can be found in the LICENSE file.

#ifndef _MARKELLA_GAP_BUFFER_H_
#define _MARKELLA_GAP_BUFFER_H_

#include "definitions.h"
#include "surface_wnd.h"
#include "gap_string.h"
#include "gap.h"

#include <WDL/heapbuf.h>
#include <WDL/wdlstring.h>
#include <WDL/ptrlist.h>
#include <WDL/mutex.h>
#include <WDL/lice/lice.h>

struct MAR_StringView
{
  char *str;
  int len;
};

class MAR_GapBuffer
{
public:
  MAR_GapBuffer();
  ~MAR_GapBuffer();

  void New(const char *filename);
  bool Save();
  bool SaveAs(const char *filename);
  bool Open(const char *filename);
  void SetPositionStartOfFile(bool select);
  void SetPositionEndOfFile(bool select);
  void SetPositionStartOfLine(bool select);
  void SetPositionEndOfLine(bool select);
  void SetPositionNextChar(bool select);
  void SetPositionPreviousChar(bool select);
  void SetPositionNextWord(bool select);
  void SetPositionPreviousWord(bool select);
  void SetPositionNextLine(bool select);
  void SetPositionPreviousLine(bool select);
  void SetPositionPageUp(int lines, bool select);
  void SetPositionPageDown(int lines, bool select);

  int GetCursorLine() const;
  int GetCursorColumn() const;
  void GoToLine(int index, int column = -1);

  void InsertText(const char *text, int length);
  void DeleteText(int length, bool forward);
  void DeleteWord(bool forward);
  bool FindKeyWord(const char *find, bool case_sensitive, bool match_whole_word, bool start, bool reverse);
  int ChangeKeyWord(const char *find, const char *change, bool case_sensitive, bool match_whole_word);
  void FindNextKeyWord();
  void FindPreviousKeyWord();

  void ToggleInsert();
  bool GetInsert() const;

  void InsertTab();
  void RemoveTab();
  void CommentLine();
  void UncommentLine();

  int GetTotalLines() const;
  void GetStringView(WDL_TypedBuf<MAR_StringView> **sv);
  MAR_SurfaceContext *GetSurfaceContext() const;

  const char *GetFilePath() const;
  const char *GetFileName() const;

  bool HasSelection() const;
  void GetSelectionRegion(int *x1, int *y1, int *x2, int *y2);
  void CheckSelect(bool select);
  void MouseMoveSelect(int x, int y);
  void SetSelect(bool select, bool direction, bool northwest);
  bool IsNorthWest() const;
  void SelectAll();

  void Cut();
  void Copy();
  void Paste();

private:
  void SetPosition(int bytepos, bool diff = true);
  void CacheLine(int previous_pos, bool diff);
  void CacheColumn();
  void FlushGap();
  void BuildStringView();
  char *GetCursorPositionPtr() const;
  WDL_FastString m_fn;
  MAR_GapString *m_strbuf;
  MAR_Gap m_gap;
  WDL_Mutex m_mutex;

  WDL_TypedBuf<MAR_StringView> m_sv;
  WDL_PtrList<WDL_FastString> m_svgapcache;
  WDL_PtrList<WDL_FastString> m_svgapempties;
  WDL_TypedBuf<char> m_svblock;

  int m_cursorpos;
  int m_cachedline;
  int m_cachedcolumn;
  bool m_ins;
  int m_eol; // 0: LF, 1: CRLF, 2: CR
  MAR_SurfaceContext *m_ctx;
  int m_tabsize;
  bool m_wantspaces;
  WDL_FastString m_find;
  WDL_FastString m_change;
  char *m_findpos;
  char *m_changepos;
  bool m_casesensitive;
  bool m_matchwholeword;
  bool m_select;
  int m_selectinit;
  int m_selectstart;
  int m_selectend;
  int m_selectx1;
  int m_selecty1;
  int m_selectx2;
  int m_selecty2;
};

#endif // _MARKELLA_GAP_BUFFER_H_
