/*
  gap_string.cpp derived from WDL - wdlstring.h
  Copyright (C) 2005 and later, Cockos Incorporated
  Modifications copyright (c) 2023, Giorgos Vougioukas

  This software is provided 'as-is', without any express or implied
  warranty.  In no event will the authors be held liable for any damages
  arising from the use of this software.

  Permission is granted to anyone to use this software for any purpose,
  including commercial applications, and to alter it and redistribute it
  freely, subject to the following restrictions:

  1. The origin of this software must not be misrepresented; you must not
      claim that you wrote the original software. If you use this software
      in a product, an acknowledgment in the product documentation would be
      appreciated but is not required.
  2. Altered source versions must be plainly marked as such, and must not be
      misrepresented as being the original software.
  3. This notice may not be removed or altered from any source distribution.
*/

#include "gap_string.h"

#include <stdio.h>
#include <stdarg.h>

MAR_GapString::MAR_GapString(int hbgran)
  : m_hb(hbgran)
{}

MAR_GapString::MAR_GapString(const char *initial, int initial_len)
  : m_hb(128)
{
  if (initial) Set(initial, initial_len);
}

MAR_GapString::MAR_GapString(const MAR_GapString &s)
  : m_hb(128)
{
  Set(&s);
}

MAR_GapString::MAR_GapString(const MAR_GapString *s)
  : m_hb(128)
{
  if (s && s != this) Set(s);
}

MAR_GapString::~MAR_GapString()
{}

char *MAR_GapString::Get()
{
  if (m_hb.GetSize()) return (char *)m_hb.Get();
  static char c; c = 0; return &c; // don't return "", in case it gets written to.
}

const char *MAR_GapString::Get() const
{
  return m_hb.GetSize() ? (char*)m_hb.Get() : "";
}

int MAR_GapString::GetLength() const
{
  int a = m_hb.GetSize(); return a > 0 ? a - 1 : 0;
}

void MAR_GapString::SetRaw(const char *str, int len) { __doSet(0, str, len, 0); }
void MAR_GapString::AppendRaw(const char *str, int len) { __doSet(GetLength(), str, len, 0); }
void MAR_GapString::InsertRaw(const char *str, int position, int ilen)
{
  const int srclen = GetLength();
  if (position < 0) position = 0;
  else if (position > srclen) position = srclen;
  if (ilen > 0) __doSet(position, str, ilen, srclen - position);
}

void MAR_GapString::Set(const char *str, int maxlen)
{
  int s = 0;
  if (str)
  {
    if (maxlen > 0) while (s < maxlen && str[s]) s++;
    else s = (int)strlen(str);
  }
  __doSet(0, str, s, 0);
}

void MAR_GapString::Set(const MAR_GapString *str, int maxlen)
{
  int s = str ? str->GetLength() : 0;
  if (maxlen > 0 && maxlen < s) s = maxlen;

  __doSet(0, str ? str->Get() : NULL, s, 0);
}

void MAR_GapString::Append(const char *str, int maxlen)
{
  int s = 0;
  if (str)
  {
    if (maxlen > 0) while (s < maxlen && str[s]) s++;
    else s = (int)strlen(str);
  }

  __doSet(GetLength(), str, s, 0);
}

void MAR_GapString::Append(const MAR_GapString *str, int maxlen)
{
  int s = str ? str->GetLength() : 0;
  if (maxlen > 0 && maxlen < s) s = maxlen;

  __doSet(GetLength(), str ? str->Get() : NULL, s, 0);
}

void MAR_GapString::DeleteSub(int position, int len)
{
  int l = m_hb.GetSize() - 1;
  char *p = (char *)m_hb.Get();
  if (l < 0 || !*p || position < 0 || position >= l) return;
  if (position + len > l) len = l - position;
  if (len > 0)
  {
    memmove(p + position, p + position + len, l - position - len + 1);
    m_hb.Resize(l + 1 - len, false);
  }
}

void MAR_GapString::Insert(const char *str, int position, int maxlen)
{
  int ilen = 0;
  if (str)
  {
    if (maxlen > 0) while (ilen < maxlen && str[ilen]) ilen++;
    else ilen = (int)strlen(str);
  }

  const int srclen = GetLength();
  if (position < 0) position = 0;
  else if (position > srclen) position = srclen;
  if (ilen > 0) __doSet(position, str, ilen, srclen - position);
}

void MAR_GapString::Insert(const MAR_GapString *str, int position, int maxlen)
{
  int ilen = str ? str->GetLength() : 0;
  if (maxlen > 0 && maxlen < ilen) ilen = maxlen;

  const int srclen = m_hb.GetSize() > 0 ? m_hb.GetSize() - 1 : 0;
  if (position < 0) position = 0;
  else if (position > srclen) position = srclen;
  if (ilen > 0) __doSet(position, str->Get(), ilen, srclen - position);
}

bool MAR_GapString::SetLen(int length, bool resizeDown, char fillchar)
{
  int osz = m_hb.GetSize() - 1;
  if (osz < 0) osz = 0;
  if (length < 0) length = 0;
  char *b = (char*)m_hb.ResizeOK(length + 1, resizeDown);
  if (b)
  {
    const int fill = length - osz;
    if (fill > 0) memset(b + osz, fillchar, fill);
    b[length] = 0;
    return true;
  }
  return false;
}

void MAR_GapString::SetAppendFormattedArgs(bool append, int maxlen, const char* fmt, va_list arglist)
{
  int offs = append ? GetLength() : 0;
  char *b = (char*)m_hb.ResizeOK(offs + maxlen + 1, false);

  if (!b) return;

  b += offs;

#ifdef _WIN32
    int written = _vsnprintf(b, maxlen+1, fmt, arglist);
    if (written < 0 || written >= maxlen) b[written = b[0] ? maxlen : 0] = 0;
#else
    int written = vsnprintf(b, maxlen + 1, fmt, arglist);
    if (written > maxlen) written = maxlen;
#endif

  m_hb.Resize(offs + written + 1, false);
}

void WDL_VARARG_WARN(printf, 3, 4) MAR_GapString::SetFormatted(int maxlen, const char *fmt, ...)
{
  va_list arglist;
  va_start(arglist, fmt);
  SetAppendFormattedArgs(false, maxlen, fmt, arglist);
  va_end(arglist);
}

void WDL_VARARG_WARN(printf, 3, 4) MAR_GapString::AppendFormatted(int maxlen, const char *fmt, ...)
{
  va_list arglist;
  va_start(arglist, fmt);
  SetAppendFormattedArgs(true, maxlen, fmt, arglist);
  va_end(arglist);
}

void MAR_GapString::Ellipsize(int minlen, int maxlen)
{
  if (maxlen >= 4 && m_hb.GetSize() && GetLength() > maxlen)
  {
    if (minlen < 0) minlen = 0;
    char *b = (char *)m_hb.Get();
    int i;
    for (i = maxlen - 4; i >= minlen; --i)
    {
      if (b[i] == ' ')
      {
        memcpy(b + i, "...", 4);
        m_hb.Resize(i + 4, false);
        break;
      }
    }
    if (i < minlen && maxlen >= 4)
    {
      memcpy(b + maxlen - 4, "...", 4);
      m_hb.Resize(maxlen, false);
    }
  }
}

void MAR_GapString::__doSet(int offs, const char *str, int len, int trailkeep)
{
  // if non-empty, or (empty and allocated and Set() rather than append/insert), then allow update, otherwise do nothing
  if (len == 0 && !trailkeep && !offs)
  {
#ifdef MAR_GAP_STRING_FREE_ON_CLEAR
      m_hb.Resize(0, true);
#else
      char *p = (char *)m_hb.ResizeOK(1, false);
      if (p) *p = 0;
#endif
  }
  else if (len > 0 && offs >= 0)
  {
    const int oldsz = m_hb.GetSize();
    const int newsz = offs + len + trailkeep + 1;
    const int growamt = newsz - oldsz;
    if (growamt > 0)
    {
      const char *oldb = (const char *)m_hb.Get();
      const char *newb = (const char *)m_hb.ResizeOK(newsz, false); // resize up if necessary

      // in case str overlaps with input, keep it valid
      if (str && newb != oldb && str >= oldb && str < oldb + oldsz) str = newb + (str - oldb);
    }

    if (m_hb.GetSize() >= newsz)
    {
      char *newbuf = (char *)m_hb.Get();
      if (trailkeep > 0) memmove(newbuf + offs + len, newbuf + offs, trailkeep);
      if (str) memmove(newbuf + offs, str, len);
      newbuf[newsz - 1] = 0;

      // resize down if necessary
      if (growamt < 0) m_hb.Resize(newsz, false);
    }
  }
}
