// Copyright (c) 2024 Giorgos Vougioukas
//
// The license can be found in the LICENSE file.

#include "goto_wnd.h"
#include "app_info.h"
#include "gap_buffer.h"
#include "volissos.h"

MAR_GoToWnd::MAR_GoToWnd()
  : m_hwnd(NULL)
  , m_goto(NULL)
{}

MAR_GoToWnd::~MAR_GoToWnd()
{}

HWND MAR_GoToWnd::Handle() const
{
  return m_hwnd;
}

void MAR_GoToWnd::OnInitDialog(WPARAM wparam, LPARAM lparam)
{
  m_goto = GetDlgItem(m_hwnd, IDC_EDIT1);

  int x = GetPrivateProfileInt(MAR_NAME, "goto_wnd_x", 0, g_inipath.Get());
  int y = GetPrivateProfileInt(MAR_NAME, "goto_wnd_y", 50, g_inipath.Get());
  SetWindowPos(m_hwnd, NULL, x, y, 0, 0, SWP_NOSIZE | SWP_NOZORDER | SWP_NOACTIVATE);
}

INT_PTR MAR_GoToWnd::OnClose(WPARAM wparam, LPARAM lparam)
{
  DestroyWindow(m_hwnd);
  return 0;
}

void MAR_GoToWnd::OnDestroy(WPARAM wparam, LPARAM lparam)
{
  RECT r;
  GetWindowRect(m_hwnd, &r);

  WDL_FastString rleft, rtop, rwidth, rheight;
  rleft.SetFormatted(32, "%d", r.left);
  rtop.SetFormatted(32, "%d", r.top);

  WritePrivateProfileString(MAR_NAME, "goto_wnd_x", rleft.Get(), g_inipath.Get());
  WritePrivateProfileString(MAR_NAME, "goto_wnd_y", rtop.Get(), g_inipath.Get());

  if (g_gotownd) { delete g_gotownd; g_gotownd = NULL; }
}

void MAR_GoToWnd::OnSysCommand(WPARAM wparam, LPARAM lparam)
{
  if (wparam == SC_CLOSE)
  {
    SendMessage(m_hwnd, WM_CLOSE, 0, 0);
  }
}

void MAR_GoToWnd::OnCommand(WPARAM wparam, LPARAM lparam)
{
  switch(LOWORD(wparam))
  {
  case IDOK:
    {
      char text[2048];
      int len = GetWindowText(m_goto, text, sizeof(text));
      if (len && g_volissos)
      {
        MAR_GapBuffer *gb = g_volissos->GetGapBuffer();
        if (gb)
        {
          const int maxln = gb->GetTotalLines() - 1;
          const int ln = wdl_clamp(atoi(text) - 1, 0, maxln);
          gb->GoToLine(ln, 0);
        }
      }

      SendMessage(m_hwnd, WM_CLOSE, 0, 0);
    } break;
  case IDCANCEL:
    {
      SendMessage(m_hwnd, WM_CLOSE, 0, 0);
    } break;
  }
}

WDL_DLGRET MAR_GoToWnd::GoToWndProc(HWND hwnd, UINT msg, WPARAM wparam, LPARAM lparam)
{
  MAR_GoToWnd *self = (MAR_GoToWnd *)GetWindowLongPtr(hwnd, GWLP_USERDATA);

  if (!self && msg == WM_INITDIALOG)
  {
    SetWindowLongPtr(hwnd, GWLP_USERDATA, lparam);
    self = (MAR_GoToWnd *)lparam;
    self->m_hwnd = hwnd;
  }

  if (WDL_likely(self))
  {
    return self->GoToWndLoop(msg, wparam, lparam);
  }
  else
  {
    return 0;
  }
}

WDL_DLGRET MAR_GoToWnd::GoToWndLoop(UINT msg, WPARAM wparam, LPARAM lparam)
{
  switch (msg)
  {
    case WM_INITDIALOG: OnInitDialog(wparam, lparam); break;
    case WM_CLOSE: return OnClose(wparam, lparam); break;
    case WM_DESTROY: OnDestroy(wparam, lparam); break;
    case WM_SYSCOMMAND: OnSysCommand(wparam, lparam); break;
    case WM_COMMAND: OnCommand(wparam, lparam); break;
  }

  return 0;
}
