// Copyright (c) 2021 Giorgos Vougioukas
//
// The license can be found in the LICENSE file.

#include "preferences_wnd.h"
#include "preferences.h"
#include "app_timers.h"

static win32CursesCtx internal_preview_ctx;
static win32CursesCtx internal_editor_ctx;

MAR_PreferencesWnd::MAR_PreferencesWnd()
  : m_hwnd(NULL)
  , m_preview(&internal_preview_ctx)
  , m_editor(&internal_editor_ctx)
{}

MAR_PreferencesWnd::~MAR_PreferencesWnd()
{}

void MAR_PreferencesWnd::OnInitDialog(WPARAM wparam, LPARAM lparam)
{
  HWND ccp = GetDlgItem(m_hwnd, IDC_CUSTOM3);
  HWND cce = GetDlgItem(m_hwnd, IDC_CUSTOM2);
  curses_setWindowContext(ccp, &internal_preview_ctx);
  curses_setWindowContext(cce, &internal_editor_ctx);

  m_resize.init(m_hwnd);
  m_resize.init_itemhwnd(ccp, 0.0f, 0.0f, 0.5f, 1.0f);
  m_resize.init_itemhwnd(cce, 0.5f, 0.0f, 1.0f, 1.0f);

  int x = GetPrivateProfileInt(MAR_NAME, "preferences_wnd_x", 0, g_inipath.Get());
  int y = GetPrivateProfileInt(MAR_NAME, "preferences_wnd_y", 50, g_inipath.Get());
  int w = GetPrivateProfileInt(MAR_NAME, "preferences_wnd_w", 1140, g_inipath.Get());
  int h = GetPrivateProfileInt(MAR_NAME, "preferences_wnd_h", 720, g_inipath.Get());
  SetWindowPos(m_hwnd, NULL, x, y, w, h, SWP_NOZORDER | SWP_NOACTIVATE);

  WDL_FastString defpref;
  defpref.Set(g_setpath.Get());
  defpref.Append("preferences-default.txt");

  WDL_FastString usrpref;
  usrpref.Set(g_setpath.Get());
  usrpref.Append("preferences.txt");

  m_preview.SetParent(m_hwnd);
  m_preview.init(defpref.Get());
  m_preview.draw();
  m_preview.setCursor();

  m_editor.SetParent(m_hwnd);
  m_editor.init(usrpref.Get());
  m_editor.draw();
  m_editor.setCursor();

  bool crf = g_preferences->WantCursesRightFocus();
  if (crf) SetFocus(cce); else SetFocus(ccp);
  SetTimer(m_hwnd, MAR_PREFERENCES_UPDATE_ID, MAR_PREFERENCES_UPDATE_MS, NULL);
}

void MAR_PreferencesWnd::OnTimer(WPARAM wparam, LPARAM lparam)
{
  if (wparam == MAR_PREFERENCES_UPDATE_ID)
  {
    m_preview.RunEditor();
    m_editor.RunEditor();
  }
}

INT_PTR MAR_PreferencesWnd::OnClose(WPARAM wparam, LPARAM lparam)
{
  DestroyWindow(m_hwnd);
  return 0;
}

void MAR_PreferencesWnd::OnDestroy(WPARAM wparam, LPARAM lparam)
{
  KillTimer(m_hwnd, MAR_PREFERENCES_UPDATE_ID);

  m_preview.ClearText();
  m_editor.ClearText();

  RECT r;
  GetWindowRect(m_hwnd, &r);

  WDL_FastString rleft, rtop, rwidth, rheight;
  rleft.SetFormatted(32, "%d", r.left);
  rtop.SetFormatted(32, "%d", r.top);
  rwidth.SetFormatted(32, "%d", r.right - r.left);
  rheight.SetFormatted(32, "%d", r.bottom - r.top);

  WritePrivateProfileString(MAR_NAME, "preferences_wnd_x", rleft.Get(), g_inipath.Get());
  WritePrivateProfileString(MAR_NAME, "preferences_wnd_y", rtop.Get(), g_inipath.Get());
  WritePrivateProfileString(MAR_NAME, "preferences_wnd_w", rwidth.Get(), g_inipath.Get());
  WritePrivateProfileString(MAR_NAME, "preferences_wnd_h", rheight.Get(), g_inipath.Get());

  if (g_prefwnd) { delete g_prefwnd; g_prefwnd = NULL; }
}

void MAR_PreferencesWnd::OnSize(WPARAM wparam, LPARAM lparam)
{
  if (wparam != SIZE_MINIMIZED)
  {
    m_resize.onResize();
  }
}

void MAR_PreferencesWnd::OnSysCommand(WPARAM wparam, LPARAM lparam)
{
  if (wparam == SC_CLOSE)
  {
    SendMessage(m_hwnd, WM_CLOSE, 0, 0);
  }
}

void MAR_PreferencesWnd::OnCommand(WPARAM wparam, LPARAM lparam)
{
  switch(LOWORD(wparam))
  {
    case IDCANCEL:
    {
      SendMessage(m_hwnd, WM_CLOSE, 0, 0);
    }
    break;
  }
}

void MAR_PreferencesWnd::OnToggleFocus(WPARAM wparam, LPARAM lparam)
{
  HWND ccp = GetDlgItem(m_hwnd, IDC_CUSTOM3);
  HWND cce = GetDlgItem(m_hwnd, IDC_CUSTOM2);
  if (GetFocus() == ccp) SetFocus(cce); else SetFocus(ccp);
}

void MAR_PreferencesWnd::OnPaint(WPARAM wparam, LPARAM lparam)
{
  RECT r;
  GetClientRect(m_hwnd, &r);

  m_painter.PaintBegin(m_hwnd, RGB(255, 255, 128));
  m_painter.PaintEnd();
}

WDL_DLGRET MAR_PreferencesWnd::PreferencesWndProc(HWND hwnd, UINT msg, WPARAM wparam, LPARAM lparam)
{
  MAR_PreferencesWnd *self = (MAR_PreferencesWnd *)GetWindowLongPtr(hwnd, GWLP_USERDATA);

  if (!self && msg == WM_INITDIALOG)
  {
    SetWindowLongPtr(hwnd, GWLP_USERDATA, lparam);
    self = (MAR_PreferencesWnd *)lparam;
    self->m_hwnd = hwnd;
  }

  if (WDL_likely(self))
  {
    return self->PreferencesWndLoop(msg, wparam, lparam);
  }
  else
  {
    return 0;
  }
}

WDL_DLGRET MAR_PreferencesWnd::PreferencesWndLoop(UINT msg, WPARAM wparam, LPARAM lparam)
{
  switch (msg)
  {
    case WM_INITDIALOG: OnInitDialog(wparam, lparam); break;
    case WM_TIMER: OnTimer(wparam, lparam); break;
    case WM_CLOSE: return OnClose(wparam, lparam); break;
    case WM_DESTROY: OnDestroy(wparam, lparam); break;
    case WM_SIZE: OnSize(wparam, lparam); break;
    case WM_SYSCOMMAND: OnSysCommand(wparam, lparam); break;
    case WM_COMMAND: OnCommand(wparam, lparam); break;
    case WM_TOGGLE_FOCUS: OnToggleFocus(wparam, lparam); break;
    case WM_PAINT: OnPaint(wparam, lparam); break;
  }

  return 0;
}
