// Copyright (c) 2021 Giorgos Vougioukas
//
// The license can be found in the LICENSE file.

#include "phoebe/main_wnd.h"

#define PHO_MAIN_SCREEN_UPDATE 300
#define PHO_MAIN_SCREEN_UPDATE_MS 50

#include "phoebe/app_info.h"
#include "phoebe/plugin.h"
#include "phoebe/about_wnd.h"
#include "phoebe/changelog_wnd.h"
#include "phoebe/license_wnd.h"
#include "phoebe/preferences.h"
#include "phoebe/preferences_wnd.h"
#include "phoebe/super_editor.h"
#include "phoebe/cursor.h"
#include "phoebe/switch_wnd.h"

#include "WDL/fpcmp.h"
#include "WDL/wdlstring.h"
#include "WDL/time_precise.h"
#include "WDL/filebrowse.h"
#include "WDL/lice/lice.h"
#include "WDL/wingui/virtwnd.h"
#include "WDL/wingui/scrollbar/coolscroll.h"
#include "WDL/wdlutf8.h"
#include "WDL/time_precise.h"
#include "WDL/dirscan.h"
#include "WDL/fileread.h"

#define PHO_TEXT_FILES "All supported files\0*.c;*.h;*.cpp;*.cc;*.txt;*.asm;*.s;*.S;*.m;*.mm;*.sql;*.cs;*.java;*.js;*.pl;*.php;*.mak;*.mk;*.sh;*.bat;*.cmd;*.nsi;*.tex;*.lua;*.eel;*.kc;*.html;*.css;*.xml\0"\
  "C files (*.c;*.h)\0*.c;*.h\0"\
  "C++ files (*.cpp;*.h;*.cc)\0*.cpp;*.h;*.cc\0"\
  "Assembly files (*.asm;*.s;*.S)\0*.asm;*.s;*.S\0"\
  "Text files (*.txt)\0*.txt\0"\
  "C# files (*.cs)\0*.cs\0"\
  "Java files (*.java)\0*.java\0"\
  "JavaScript files (*.js)\0*.js\0"\
  "Objective-C files (*.h;*.m;*.mm)\0*.h;*.m;*.mm\0"\
  "SQL files (*.sql)\0*.sql\0"\
  "Perl files (*.pl)\0*.pl\0"\
  "PHP files (*.php)\0*.php\0"\
  "Makefile files (*.mak;*.mk)\0*.mak;*.mk\0"\
  "Shell files (*.sh)\0*.sh\0"\
  "Batch files (*.bat;*.cmd)\0*.bat;*.cmd\0"\
  "NSIS files (*.nsi)\0*.nsi\0"\
  "TeX files (*.tex)\0*.tex\0"\
  "Lua files (*.lua)\0*.lua\0"\
  "EEL2 files (*.eel)\0*.eel\0"\
  "KC files (*.kc)\0*.kc\0"\
  "HTML files (*.html)\0*.html\0"\
  "CSS files (*.css)\0*.css\0"\
  "XML files (*.xml)\0*.xml\0"\
  "All files (*.*)\0*.*\0\0"
#define PHO_DEFAULT_EXT "c"

static WNDPROC PrevTreeProc = NULL;
static HWND s_tree = NULL;
static WNDPROC PrevListProc = NULL;
static HWND s_list = NULL;
static WNDPROC PrevEditProc = NULL;
static HWND s_edit = NULL;
static bool s_istreeroot = false;

static LOGFONT lf =
{
#if defined(_WIN32)
  14, 0, 0, 0, FW_NORMAL, FALSE, FALSE, FALSE, DEFAULT_CHARSET,
  OUT_DEFAULT_PRECIS, CLIP_DEFAULT_PRECIS, DEFAULT_QUALITY, DEFAULT_PITCH,
  "Arial"
#else
  11, 0, 0, 0, FW_NORMAL, FALSE, FALSE, FALSE, DEFAULT_CHARSET,
  OUT_DEFAULT_PRECIS, CLIP_DEFAULT_PRECIS, DEFAULT_QUALITY, DEFAULT_PITCH,
  "Arial"
#endif
};

PHO_MainWnd::PHO_MainWnd()
  : m_hwnd(NULL)
  , m_x(0)
  , m_y(0)
  , m_w(0)
  , m_h(0)
  , m_main_menu(NULL)
  , m_tbm(NULL)
  , m_lbm(NULL)
  , m_font(NULL)
  , m_fullscreen(false)
  , m_maximized(false)
  , m_bg(g_preferences->GetBackgroundColor())
{}

PHO_MainWnd::~PHO_MainWnd()
{
  DestroyWindow(m_hwnd);
  if (m_font) delete m_font;
  if (m_tbm) delete m_tbm;
  if (m_lbm) delete m_lbm;
}

void PHO_MainWnd::GotoLine()
{
  if (!g_gotolinewnd)
  {
    CreateDialog(g_inst, MAKEINTRESOURCE(IDD_GOTO_LINE), m_hwnd, PHO_GotoLineWndProc);
  }
  else
  {
    SetFocus(g_gotolinewnd);
  }
}

void PHO_MainWnd::GotoLineOffset()
{
  if (!g_gotolineoffsetwnd)
  {
    CreateDialog(g_inst, MAKEINTRESOURCE(IDD_GOTO_LINE_OFFSET), m_hwnd, PHO_GotoLineOffsetWndProc);
  }
  else
  {
    SetFocus(g_gotolineoffsetwnd);
  }
}

void PHO_MainWnd::ToggleFullscreen()
{
  m_fullscreen = !m_fullscreen;
  if (m_fullscreen)
  {
    RECT r, fsr;
    m_maximized = !!IsZoomed(m_hwnd);

    if (m_maximized)
    {
      SendMessage(m_hwnd, WM_SYSCOMMAND, SC_RESTORE, 0);
    }

    m_style = GetWindowLongPtr(m_hwnd, GWL_STYLE);
    m_exstyle = GetWindowLongPtr(m_hwnd, GWL_EXSTYLE);
    GetWindowRect(m_hwnd, &r);

    SetWindowLongPtr(m_hwnd, GWL_STYLE, m_style & ~(WS_CAPTION | WS_THICKFRAME));
    SetWindowLongPtr(m_hwnd, GWL_EXSTYLE, m_exstyle & ~(WS_EX_DLGMODALFRAME |
      WS_EX_WINDOWEDGE | WS_EX_CLIENTEDGE | WS_EX_STATICEDGE));

    MONITORINFO mi;
    mi.cbSize = sizeof(mi);
    GetMonitorInfo(MonitorFromWindow(m_hwnd, MONITOR_DEFAULTTONEAREST), &mi);

    CopyRect(&fsr, &mi.rcMonitor);
    SetWindowPos(m_hwnd, NULL, fsr.left, fsr.top, fsr.right - fsr.left, fsr.bottom - fsr.top,
      SWP_NOZORDER | SWP_NOACTIVATE | SWP_FRAMECHANGED);

    m_x = r.left;
    m_y = r.top;
    m_w = r.right - r.left;
    m_h = r.bottom - r.top;
  }
  else
  {
    SetWindowLongPtr(m_hwnd, GWL_STYLE, m_style);
    SetWindowLongPtr(m_hwnd, GWL_EXSTYLE, m_exstyle);
    SetWindowPos(m_hwnd, NULL, m_x, m_y, m_w, m_h,
      SWP_NOZORDER | SWP_NOACTIVATE | SWP_FRAMECHANGED);
    if (m_maximized) SendMessage(m_hwnd, WM_SYSCOMMAND, SC_MAXIMIZE, 0);
  }
}

void PHO_MainWnd::SwitchWindow()
{
  if (!g_switchwnd)
  {
    g_switchwnd = new PHO_SwitchWnd;
    CreateDialogParam(g_inst, MAKEINTRESOURCE(IDD_SWITCH), m_hwnd,
      PHO_SwitchWnd::SwitchWndProc, (LPARAM)g_switchwnd);
    ShowWindow(g_switchwnd->Handle(), SW_SHOW);
  }
  else
  {
    SetFocus(g_switchwnd->Handle());
  }
}

void PHO_MainWnd::Find()
{
  if (!g_findwnd)
  {
    CreateDialog(g_inst, MAKEINTRESOURCE(IDD_FIND), m_hwnd, PHO_FindWndProc);
  }
  else
  {
    SetFocus(g_findwnd);
  }
}

void PHO_MainWnd::Replace()
{
  if (!g_replacewnd)
  {
    CreateDialog(g_inst, MAKEINTRESOURCE(IDD_REPLACE), m_hwnd, PHO_ReplaceWndProc);
  }
  else
  {
    SetFocus(g_replacewnd);
  }
}

void PHO_MainWnd::FindReplaceNext()
{
  if (g_supereditor->IsReplaceEnabled()) g_supereditor->ReplaceNext();
  else if (g_supereditor->IsFindEnabled()) g_supereditor->FindNext();
}

void PHO_MainWnd::FindReplacePrevious()
{
  if (g_supereditor->IsReplaceEnabled()) g_supereditor->ReplacePrevious();
  else if (g_supereditor->IsFindEnabled()) g_supereditor->FindPrevious();
}

const char *PHO_MainWnd::GetOrigTitle() const
{
  return m_titleorig.Get();
}

void PHO_MainWnd::OnInitDialog(WPARAM wparam, LPARAM lparam)
{
  double wall0 = time_precise();

  if (IsStableRelease())
  {
    m_titleorig.Set(PHO_NAME_MARKETING);
  }
  else
  {
    m_titleorig.Set(PHO_FULL_VERSION " (" PHO_ARCH ")");
  }
  SetWindowText(m_hwnd, m_titleorig.Get());

  m_tbm = new LICE_SysBitmap;
  m_lbm = new LICE_SysBitmap;
  m_font = new LICE_CachedFont();
  m_font->SetFromHFont(CreateFontIndirect(&lf), LICE_FONT_FLAG_OWNS_HFONT);

  m_resize.init(m_hwnd);

  m_main_menu = LoadMenu(g_inst, MAKEINTRESOURCE(IDR_MAIN_MENUBAR));
  SetMenu(m_hwnd, m_main_menu);

  m_x = GetPrivateProfileInt(PHO_NAME, "main_wnd_x", 50, g_inipath.Get());
  m_y = GetPrivateProfileInt(PHO_NAME, "main_wnd_y", 50, g_inipath.Get());
  m_w = GetPrivateProfileInt(PHO_NAME, "main_wnd_w", 1120, g_inipath.Get());
  m_h = GetPrivateProfileInt(PHO_NAME, "main_wnd_h", 900, g_inipath.Get());
  SetWindowPos(m_hwnd, NULL, m_x, m_y, m_w, m_h, SWP_NOACTIVATE);

  int maximized = GetPrivateProfileInt(PHO_NAME, "main_wnd_maximized", 0, g_inipath.Get());

  if (maximized)
  {
    ShowWindow(m_hwnd, SW_SHOWMAXIMIZED);
  }
  else
  {
    ShowWindow(m_hwnd, SW_SHOWNORMAL);
  }

  //PrevTreeProc = (WNDPROC)SetWindowLongPtrW(GetDlgItem(m_hwnd, IDC_TREE1),
  //GWLP_WNDPROC, (LONG_PTR)NewTreeProc);

  //PrevListProc = (WNDPROC)SetWindowLongPtrW(GetDlgItem(m_hwnd, IDC_LIST1),
  //GWLP_WNDPROC, (LONG_PTR)NewListProc);

  //PrevEditProc = (WNDPROC)SetWindowLongPtrW(GetDlgItem(m_hwnd, IDC_EDIT1),
  //  GWLP_WNDPROC, (LONG_PTR)NewEditProc);

  g_cursor = new PHO_Cursor;
  g_supereditor = new PHO_SuperEditor;

  int bk = g_preferences->GetBackgroundColor();
  int fg = g_preferences->GetTextColor();
  //ListView_SetBkColor(m_list, RGB((bk >> 16) & 0xFF,(bk >> 8) & 0xFF, bk & 0xFF));
  //ListView_SetTextBkColor(m_list, RGB((bk >> 16) & 0xFF,(bk >> 8) & 0xFF, bk & 0xFF));
  //ListView_SetTextColor(m_list, RGB((fg >> 16) & 0xFF,(fg >> 8) & 0xFF, fg & 0xFF));

  double wall1 = time_precise();

  if (WDL_unlikely(!IsStableRelease()))
  {
    const char warn_msg[] =
    {
      "Welcome to " PHO_NAME_MARKETING " (the \"Software\").\n\n"
      "THIS IS AN UNSTABLE RELEASE STILL IN DEVELOPEMENT PHASE\n"
      "AND IS PROVIDED ON AN \"AS IS\" BASIS AND IS BELIEVED\n"
      "TO CONTAIN DEFECTS AND A PRIMARY PURPOSE OF THIS\n"
      "RELEASE IS TO OBTAIN FEEDBACK ON SOFTWARE PERFORMANCE\n"
      "AND THE IDENTIFICATION OF DEFECTS.\n\n"
      "IT IS ADVISED TO USE CAUTION AND NOT TO RELY IN ANY WAY\n"
      "ON THE CORRECT FUNCTIONING OF THIS RELEASE.\n\n"
      PHO_NAME_MARKETING " version: " PHO_NAKED_VERSION " (" PHO_ARCH ")\n\n"
      "For the latest production release visit: " PHO_WEBSITE_URL "\n\n"
      "Startup time: %.3f sec\n\n"
      PHO_COPYRIGHT
    };

    WDL_FastString tmp;

    tmp.SetFormatted(1024, warn_msg, wall1 - wall0);

    MessageBox(m_hwnd, tmp.Get(), "Release for testing purposes", MB_OK);
  }

  m_strbuf.SetFormatted(512, "%s - *", m_titleorig.Get());
  SetWindowText(m_hwnd, m_strbuf.Get());

  SetTimer(m_hwnd, PHO_MAIN_SCREEN_UPDATE, PHO_MAIN_SCREEN_UPDATE_MS, NULL);

  bool cmdlinefile = false;
  if (g_argfile.GetLength())
  {
    WDL_FileRead fr(g_argfile.Get());
    if (fr.IsOpen()) cmdlinefile = true;
  }
  if (cmdlinefile) g_supereditor->Open(g_argfile.Get());
  else g_supereditor->New();
  g_argfile.SetLen(0, true);
}

void PHO_MainWnd::OnSysCommand(WPARAM wparam, LPARAM lparam)
{
  if (wparam == SC_CLOSE)
  {
    bool active = false;

    //if (active && g_preferences->AskExit())
    if (0)
    {
      int res = MessageBox(m_hwnd, "Playback is active. Do you really want to Exit?", "Ask Exit", MB_YESNO);
      if (res == IDYES) DestroyWindow(m_hwnd);
    }
    else
    {
      DestroyWindow(m_hwnd);
    }
  }
}

INT_PTR PHO_MainWnd::OnClose(WPARAM wparam, LPARAM lparam)
{
  return 0;
}

void PHO_MainWnd::OnSize(WPARAM wparam, LPARAM lparam)
{
  RECT r;
  GetClientRect(m_hwnd, &r);
  HWND cus = GetDlgItem(m_hwnd, IDC_CUSTOM1);
  int w = r.right - r.left;
  int h = r.bottom - r.top;

  if (wparam != SIZE_MINIMIZED)
  {
    m_resize.onResize();

    m_tbm->resize(w, 7);
    m_lbm->resize(7, h);

    SetWindowPos(cus, NULL, r.left + 7, r.top + 7, w - 7, h - 7, SWP_NOACTIVATE | SWP_NOZORDER);
  }

  if (wparam != SIZE_MINIMIZED && wparam != SIZE_MAXIMIZED)
  {
    RECT r;
    GetWindowRect(m_hwnd, &r);

    m_x = r.left;
    m_y = r.top;
    m_w = r.right - r.left;
    m_h = r.bottom - r.top;
  }
}

void PHO_MainWnd::OnMove(WPARAM wparam, LPARAM lparam)
{
  int xpos = (int)(short) LOWORD(lparam); // horizontal position
  int ypos = (int)(short) HIWORD(lparam); // vertical position

  if (xpos >= 0 && ypos >= 0 && GetCapture() == m_hwnd)
  {
    RECT r;
    GetWindowRect(m_hwnd, &r);

    m_x = r.left;
    m_y = r.top;
    m_w = r.right - r.left;
    m_h = r.bottom - r.top;
  }
}

void PHO_MainWnd::OnDestroy(WPARAM wparam, LPARAM lparam)
{
  KillTimer(m_hwnd, PHO_MAIN_SCREEN_UPDATE);

#ifdef _WIN32
  WINDOWPLACEMENT wp = { sizeof(wp) };
  GetWindowPlacement(m_hwnd, &wp);

  WDL_FastString xstr, ystr, wstr, hstr;
  xstr.SetFormatted(32, "%d", m_x);
  ystr.SetFormatted(32, "%d", m_y);
  wstr.SetFormatted(32, "%d", m_w);
  hstr.SetFormatted(32, "%d", m_h);

  if (wp.showCmd == SW_SHOWMAXIMIZED)
  {
    WritePrivateProfileString(PHO_NAME, "main_wnd_maximized", "1", g_inipath.Get());
  }
  else
  {
    WritePrivateProfileString(PHO_NAME, "main_wnd_maximized", "0", g_inipath.Get());
    WritePrivateProfileString(PHO_NAME, "main_wnd_x", xstr.Get(), g_inipath.Get());
    WritePrivateProfileString(PHO_NAME, "main_wnd_y", ystr.Get(), g_inipath.Get());
    WritePrivateProfileString(PHO_NAME, "main_wnd_w", wstr.Get(), g_inipath.Get());
    WritePrivateProfileString(PHO_NAME, "main_wnd_h", hstr.Get(), g_inipath.Get());
  }

#else
  RECT r;
  GetWindowRect(m_hwnd, &r);

  WDL_FastString xstr, ystr, wstr, hstr;
  xstr.SetFormatted(32, "%d", r.left);
  ystr.SetFormatted(32, "%d", r.top);
  wstr.SetFormatted(32, "%d", r.right - r.left);
  hstr.SetFormatted(32, "%d", r.bottom - r.top);

  if (r.left >= 0 && r.top >= 0)
  {
    WritePrivateProfileString(PHO_NAME, "main_wnd_x", xstr.Get(), g_inipath.Get());
    WritePrivateProfileString(PHO_NAME, "main_wnd_y", ystr.Get(), g_inipath.Get());
    WritePrivateProfileString(PHO_NAME, "main_wnd_w", wstr.Get(), g_inipath.Get());
    WritePrivateProfileString(PHO_NAME, "main_wnd_h", hstr.Get(), g_inipath.Get());
  }

  #ifdef __APPLE__
  SWELL_PostQuitMessage(g_mainwnd);
  #endif
  g_hasrequestedquit = true;
#endif

  if (g_scrollbar) delete g_scrollbar;
  if (g_supereditor) delete g_supereditor;
  if (g_cursor) delete g_cursor;

  PostQuitMessage(0);
}

void PHO_MainWnd::OnCommand(WPARAM wparam, LPARAM lparam)
{
  switch (LOWORD(wparam))
  {
  //case IDOK:
  //  {
  //    SetDlgItemText(m_hwnd, IDC_EDIT1, "");
  //  }
  //  break;
  case ID_FILE_NEW_TEXT:
    {
      g_supereditor->New();
    }
    break;
  case ID_FILE_OPEN_TEXT:
    {
      char *fl = NULL;
      bool preservecwd = g_preferences->WantPreserveDirectory();
      fl = WDL_ChooseFileForOpen(m_hwnd, "Open file", NULL, NULL,
       PHO_TEXT_FILES, PHO_DEFAULT_EXT, preservecwd, false);

      if (fl)
      {
        g_supereditor->Open(fl); free(fl);
      }
    }
    break;
  case ID_FILE_SAVE_TEXT:
    {
      char fn[2048];

      if (g_supereditor->IsOpen())
      {
        g_supereditor->Save();
      }
      else if (WDL_ChooseFileForSave(m_hwnd, "Save file", NULL, NULL,
        PHO_TEXT_FILES, PHO_DEFAULT_EXT, true, fn, sizeof(fn)))
      {
        g_supereditor->SaveAs(fn);
      }
    }
    break;
  case ID_FILE_SAVEAS_TEXT:
    {
      char fn[2048];

      if (WDL_ChooseFileForSave(m_hwnd, "Save file", NULL, NULL,
        PHO_TEXT_FILES, PHO_DEFAULT_EXT, true, fn, sizeof(fn)))
      {
        g_supereditor->SaveAs(fn);
      }
    }
    break;
  case ID_FILE_EXIT:
    {
      if (m_fullscreen) ToggleFullscreen();
      DestroyWindow(g_mainwnd->Handle());
    }
    break;
  case ID_OPTIONS_PREFERENCES:
    {
      if (!g_preferences->WantCurses())
      {
        g_preferences->Open();
      }
      else
      {
        if (!g_prefwnd)
        {
          g_prefwnd = new PHO_PreferencesWnd;
          CreateDialogParam(g_inst, MAKEINTRESOURCE(IDD_PREFERENCES), m_hwnd,
            PHO_PreferencesWnd::PreferencesWndProc, (LPARAM)g_prefwnd);
          ShowWindow(g_prefwnd->Handle(), SW_SHOW);
        }
        else
        {
          SetFocus(g_prefwnd->Handle());
        }
      }
    }
    break;
  case ID_HELP_DOCUMENTATION:
    {
      WDL_FastString doc;
      doc.Set(g_modpath.Get());
      doc.Append("documentation.txt");

      ShellExecute(m_hwnd, "", "notepad.exe", doc.Get(), "", SW_SHOWNORMAL);
    }
    break;
  case ID_HELP_ABOUT:
    {
      if (!g_aboutwnd)
      {
        g_aboutwnd = new PHO_AboutWnd;
        CreateDialogParam(g_inst, MAKEINTRESOURCE(IDD_ABOUT), m_hwnd,
          PHO_AboutWnd::AboutWndProc, (LPARAM)g_aboutwnd);
        ShowWindow(g_aboutwnd->Handle(), SW_SHOW);
      }
      else
      {
        SetFocus(g_aboutwnd->Handle());
      }
    }
    break;
  case ID_HELP_CHANGELOG:
    {
      if (!g_changelogwnd)
      {
        g_changelogwnd = new PHO_ChangelogWnd;
        CreateDialogParam(g_inst, MAKEINTRESOURCE(IDD_CHANGELOG), m_hwnd,
          PHO_ChangelogWnd::ChangelogWndProc, (LPARAM)g_changelogwnd);
        ShowWindow(g_changelogwnd->Handle(), SW_SHOW);
      }
      else
      {
        SetFocus(g_changelogwnd->Handle());
      }
    }
    break;
  case ID_HELP_VERSION:
    {
      const char version_str[] =
      {
        PHO_NAME_MARKETING " version: " PHO_NAKED_VERSION " (" PHO_ARCH ")\n\n"
        "Build timestamp: " PHO_TIMESTAMP "\n"
        "Git commit: " PHO_GIT_SHA
      };

      MessageBox(m_hwnd, version_str, "Version details", MB_OK);
    }
    break;
  case ID_HELP_OFFICIALWEBSITE:
    {
      ShellExecute(m_hwnd, "", PHO_WEBSITE_URL, "", "", SW_SHOWNORMAL);
    }
    break;
  case ID_HELP_LICENSE:
    {
      if (!g_licwnd)
      {
        g_licwnd = new PHO_LicenseWnd;
        CreateDialogParam(g_inst, MAKEINTRESOURCE(IDD_LICENSE), m_hwnd,
          PHO_LicenseWnd::LicenseWndProc, (LPARAM)g_licwnd);
        ShowWindow(g_licwnd->Handle(), SW_SHOW);
      }
      else
      {
        SetFocus(g_licwnd->Handle());
      }
    }
    break;
  }
}

void PHO_MainWnd::OnTimer(WPARAM wparam, LPARAM lparam)
{
  if (wparam == PHO_MAIN_SCREEN_UPDATE)
  {

  }
}

void PHO_MainWnd::OnPaint(WPARAM wparam, LPARAM lparam)
{
  RECT r;
  GetClientRect(m_hwnd, &r);
  int w = r.right - r.left;
  int h = r.bottom - r.top;

  PAINTSTRUCT ps;
  HDC dc = BeginPaint(m_hwnd, &ps);
  LICE_Clear(m_tbm, LICE_RGBA((m_bg >> 16) & 0xff, (m_bg >> 8) & 0xff, m_bg & 0xff, 255));
  LICE_Clear(m_lbm, LICE_RGBA((m_bg >> 16) & 0xff, (m_bg >> 8) & 0xff, m_bg & 0xff, 255));
  BitBlt(dc, r.left, r.top, w, 7, m_tbm->getDC(), 0, 0, SRCCOPY);
  BitBlt(dc, r.left, r.top, 7, h, m_lbm->getDC(), 0, 0, SRCCOPY);
  EndPaint(m_hwnd, &ps);
}

INT_PTR PHO_MainWnd::OnNotify(WPARAM wparam, LPARAM lparam)
{
  //switch (LOWORD(wparam))
  {

  }

  return 0;
}

void PHO_MainWnd::OnContextMenu(WPARAM wparam, LPARAM lparam)
{
  POINT pt;
  pt.x = GET_X_LPARAM(lparam);
  pt.y = GET_Y_LPARAM(lparam);
  HANDLE hwnd = (HANDLE)wparam;

  if (hwnd == m_hwnd)
  {
    ScreenToClient(m_hwnd, (LPPOINT)&pt);

  }
}

void PHO_MainWnd::OnLButtonDown(WPARAM wparam, LPARAM lparam)
{
  POINT pt;
  pt.x = GET_X_LPARAM(lparam);
  pt.y = GET_Y_LPARAM(lparam);

}

void PHO_MainWnd::OnLButtonUp(WPARAM wparam, LPARAM lparam)
{

}

void PHO_MainWnd::OnMouseMove(WPARAM wparam, LPARAM lparam)
{
  POINT pt;
  pt.x = GET_X_LPARAM(lparam);
  pt.y = GET_Y_LPARAM(lparam);

}

INT_PTR PHO_MainWnd::OnCtrlColorEdit(UINT msg, WPARAM wparam, LPARAM lparam)
{
  if (0)//if((HWND)lparam == GetDlgItem(m_hwnd, IDC_EDIT1))
  {
    int bk = g_preferences->GetBackgroundColor();
    int fg = g_preferences->GetTextColor();
    SetBkMode((HDC)wparam, RGB((bk >> 16) & 0xFF,(bk >> 8) & 0xFF, bk & 0xFF));
    SetTextColor((HDC)wparam, RGB((fg >> 16) & 0xFF,(fg >> 8) & 0xFF, fg & 0xFF));
    if (!m_brush) m_brush = CreateSolidBrush(RGB((bk >> 16) & 0xFF,(bk >> 8) & 0xFF, bk & 0xFF));
    return (INT_PTR)m_brush;
  }
  else
  {
    return DefWindowProc(m_hwnd, msg, wparam, lparam);
  }
}

void PHO_MainWnd::OnActivateApp(WPARAM wparam, LPARAM lparam)
{

}

void PHO_MainWnd::OnNCLButtonDown(WPARAM wparam, LPARAM lparam)
{

}

WDL_DLGRET PHO_MainWnd::NewTreeProc(HWND hwnd, UINT msg, WPARAM wparam, LPARAM lparam)
{
  switch (msg)
  {
  case WM_MOUSEWHEEL:
  case WM_MOUSEHWHEEL:
    if (GetAsyncKeyState(VK_SHIFT) & 0x8000)
    {
      return 0;
    }
    break;
  case WM_SYSKEYUP:
  case WM_KEYUP:
    {
      switch(wparam)
      {
      case VK_UP:
        {
          HTREEITEM sel = TreeView_GetSelection(s_tree);
          if (!sel) return 0;

          if (sel == TreeView_GetRoot(s_tree) && s_istreeroot)
          {
            s_istreeroot = false;
            SetFocus(s_edit);
          }
          else if (sel == TreeView_GetRoot(s_tree))
          {
            s_istreeroot = true;
          }
          else
          {
            s_istreeroot = false;
          }
        }
        break;
      }
    }
    break;
  }

  return CallWindowProc(PrevTreeProc, hwnd, msg, wparam, lparam);
}

WDL_DLGRET PHO_MainWnd::NewListProc(HWND hwnd, UINT msg, WPARAM wparam, LPARAM lparam)
{
  switch (msg)
  {
  case WM_MOUSEWHEEL:
  case WM_MOUSEHWHEEL:
    if (GetAsyncKeyState(VK_SHIFT) & 0x8000)
    {
      return 0;
    }
    break;
  case WM_SETFOCUS:
    {
      int one_ping_only = ListView_GetNextItem(s_list, -1, LVNI_SELECTED);
      if (one_ping_only == -1)
      {
        ListView_SetItemState(s_list, 0,
          LVNI_SELECTED|LVNI_FOCUSED,
          LVNI_SELECTED|LVNI_FOCUSED)
      }
    }
    break;
  }

  return CallWindowProc(PrevListProc, hwnd, msg, wparam, lparam);
}

WDL_DLGRET PHO_MainWnd::NewEditProc(HWND hwnd, UINT msg, WPARAM wparam, LPARAM lparam)
{
  switch (msg)
  {
  case WM_MOUSEWHEEL:
  case WM_MOUSEHWHEEL:
    if (GetAsyncKeyState(VK_SHIFT) & 0x8000)
    {
      return 0;
    }
    break;
  case WM_SYSKEYDOWN:
  case WM_KEYDOWN:
    {
      switch(wparam)
      {
      case VK_HOME:
        {

        } break;
      case VK_END:
        {

        } break;
      case VK_PRIOR:
        {

        } break;
      case VK_NEXT:
        {

        } break;
      case VK_UP:
        {

        } break;
      case VK_DOWN:
        {

        }
        break;
      case VK_BACK:
        {
          if ((GetAsyncKeyState(VK_CONTROL) & 0x8000))
          {

          }
        }
        break;
      }
    }
    break;
  case WM_SETFOCUS:
    {

    } break;
  case WM_KILLFOCUS:
    {

    } break;
  }

  return CallWindowProc(PrevEditProc, hwnd, msg, wparam, lparam);
}

WDL_DLGRET PHO_FindWndProc(HWND hwnd, UINT msg, WPARAM wparam, LPARAM lparam)
{
  switch(msg)
  {
    case WM_INITDIALOG:
    {
      g_findwnd = hwnd;
      int x = GetPrivateProfileInt(PHO_NAME, "find_wnd_x", 0, g_inipath.Get());
      int y = GetPrivateProfileInt(PHO_NAME, "find_wnd_y", 50, g_inipath.Get());
      SetWindowPos(hwnd, NULL, x, y, 0, 0, SWP_NOSIZE | SWP_NOZORDER | SWP_NOACTIVATE);

      ShowWindow(hwnd, SW_SHOWDEFAULT);
    } break;
    case WM_DESTROY:
    {
      RECT r;
      GetWindowRect(hwnd, &r);

      int x = r.left;
      int y = r.top;

      WDL_FastString xstr, ystr;
      xstr.SetFormatted(32, "%d", x);
      ystr.SetFormatted(32, "%d", y);

      WritePrivateProfileString(PHO_NAME, "find_wnd_x", xstr.Get(), g_inipath.Get());
      WritePrivateProfileString(PHO_NAME, "find_wnd_y", ystr.Get(), g_inipath.Get());
      g_findwnd = hwnd = NULL;
    } break;
    case WM_COMMAND:
    {
      switch(LOWORD(wparam))
      {
      case IDOK:
        {
          char text[2048];
          int index = GetDlgItemText(hwnd, IDC_EDIT1, text, sizeof(text));
          if (index)
          {
            if (!(GetAsyncKeyState(VK_CONTROL) & 0x8000) &&
              !(GetAsyncKeyState(VK_SHIFT) & 0x8000) &&
              !(GetAsyncKeyState(VK_MENU) & 0x8000))
            {
              if (!GetCapture())
              {
                g_supereditor->FindText(text, false, true);
                DestroyWindow(hwnd);
              }
            }
            else if (!(GetAsyncKeyState(VK_CONTROL) & 0x8000) &&
              (GetAsyncKeyState(VK_SHIFT) & 0x8000) &&
              !(GetAsyncKeyState(VK_MENU) & 0x8000))
            {
              if (!GetCapture())
              {
                g_supereditor->FindText(text, false, false);
                DestroyWindow(hwnd);
              }
            }
          }
        } break;
      case IDCANCEL:
        {
          DestroyWindow(hwnd);
        } break;
      }
    } break;
  }

  return 0;
}

WDL_DLGRET PHO_ReplaceWndProc(HWND hwnd, UINT msg, WPARAM wparam, LPARAM lparam)
{
  switch(msg)
  {
    case WM_INITDIALOG:
    {
      g_replacewnd = hwnd;
      int x = GetPrivateProfileInt(PHO_NAME, "replace_wnd_x", 0, g_inipath.Get());
      int y = GetPrivateProfileInt(PHO_NAME, "replace_wnd_y", 50, g_inipath.Get());
      SetWindowPos(hwnd, NULL, x, y, 0, 0, SWP_NOSIZE | SWP_NOZORDER | SWP_NOACTIVATE);

      ShowWindow(hwnd, SW_SHOWDEFAULT);
    } break;
    case WM_DESTROY:
    {
      RECT r;
      GetWindowRect(hwnd, &r);

      int x = r.left;
      int y = r.top;

      WDL_FastString xstr, ystr;
      xstr.SetFormatted(32, "%d", x);
      ystr.SetFormatted(32, "%d", y);

      WritePrivateProfileString(PHO_NAME, "replace_wnd_x", xstr.Get(), g_inipath.Get());
      WritePrivateProfileString(PHO_NAME, "replace_wnd_y", ystr.Get(), g_inipath.Get());
      g_replacewnd = hwnd = NULL;
    } break;
    case WM_COMMAND:
    {
      switch(LOWORD(wparam))
      {
      case IDOK:
        {
          char texta[2048], textb[2048];
          int indexa = GetDlgItemText(hwnd, IDC_EDIT1, texta, sizeof(texta));
          int indexb = GetDlgItemText(hwnd, IDC_EDIT2, textb, sizeof(textb));
          if (indexa && indexb)
          {
            if (!(GetAsyncKeyState(VK_CONTROL) & 0x8000) &&
              !(GetAsyncKeyState(VK_SHIFT) & 0x8000) &&
              !(GetAsyncKeyState(VK_MENU) & 0x8000))
            {
              if (!GetCapture())
              {
                g_supereditor->ReplaceText(texta, textb, false, true, false);
                DestroyWindow(hwnd);
              }
            }
            else if (!(GetAsyncKeyState(VK_CONTROL) & 0x8000) &&
              (GetAsyncKeyState(VK_SHIFT) & 0x8000) &&
              !(GetAsyncKeyState(VK_MENU) & 0x8000))
            {
              if (!GetCapture())
              {
                g_supereditor->ReplaceText(texta, textb, false, false, false);
                DestroyWindow(hwnd);
              }
            }
            else if ((GetAsyncKeyState(VK_CONTROL) & 0x8000) &&
              !(GetAsyncKeyState(VK_SHIFT) & 0x8000) &&
              !(GetAsyncKeyState(VK_MENU) & 0x8000))
            {
              if (!GetCapture())
              {
                g_supereditor->ReplaceText(texta, textb, false, true, true);
                DestroyWindow(hwnd);
              }
            }
          }
        } break;
      case IDCANCEL:
        {
          DestroyWindow(hwnd);
        } break;
      }
    } break;
  }

  return 0;
}

WDL_DLGRET PHO_GotoLineWndProc(HWND hwnd, UINT msg, WPARAM wparam, LPARAM lparam)
{
  switch(msg)
  {
    case WM_INITDIALOG:
    {
      g_gotolinewnd = hwnd;
      int x = GetPrivateProfileInt(PHO_NAME, "goto_wnd_x", 0, g_inipath.Get());
      int y = GetPrivateProfileInt(PHO_NAME, "goto_wnd_y", 50, g_inipath.Get());
      SetWindowPos(hwnd, NULL, x, y, 0, 0, SWP_NOSIZE | SWP_NOZORDER | SWP_NOACTIVATE);

      ShowWindow(hwnd, SW_SHOWDEFAULT);
    } break;
    case WM_DESTROY:
    {
      RECT r;
      GetWindowRect(hwnd, &r);

      int x = r.left;
      int y = r.top;

      WDL_FastString xstr, ystr;
      xstr.SetFormatted(32, "%d", x);
      ystr.SetFormatted(32, "%d", y);

      WritePrivateProfileString(PHO_NAME, "goto_wnd_x", xstr.Get(), g_inipath.Get());
      WritePrivateProfileString(PHO_NAME, "goto_wnd_y", ystr.Get(), g_inipath.Get());
      g_gotolinewnd = hwnd = NULL;
    } break;
    case WM_COMMAND:
    {
      switch(LOWORD(wparam))
      {
      case IDOK:
        {
          int index = GetDlgItemInt(hwnd, IDC_EDIT1, NULL, TRUE);
          g_supereditor->GotoLine(index - 1);
          DestroyWindow(hwnd);
        } break;
      case IDCANCEL:
        {
          DestroyWindow(hwnd);
        } break;
      }
    } break;
  }

  return 0;
}

WDL_DLGRET PHO_GotoLineOffsetWndProc(HWND hwnd, UINT msg, WPARAM wparam, LPARAM lparam)
{
  switch(msg)
  {
    case WM_INITDIALOG:
    {
      g_gotolineoffsetwnd = hwnd;
      int x = GetPrivateProfileInt(PHO_NAME, "gotooffset_wnd_x", 0, g_inipath.Get());
      int y = GetPrivateProfileInt(PHO_NAME, "gotooffset_wnd_y", 50, g_inipath.Get());
      SetWindowPos(hwnd, NULL, x, y, 0, 0, SWP_NOSIZE | SWP_NOZORDER | SWP_NOACTIVATE);

      ShowWindow(hwnd, SW_SHOWDEFAULT);
    } break;
    case WM_DESTROY:
    {
      RECT r;
      GetWindowRect(hwnd, &r);

      int x = r.left;
      int y = r.top;

      WDL_FastString xstr, ystr;
      xstr.SetFormatted(32, "%d", x);
      ystr.SetFormatted(32, "%d", y);

      WritePrivateProfileString(PHO_NAME, "gotooffset_wnd_x", xstr.Get(), g_inipath.Get());
      WritePrivateProfileString(PHO_NAME, "gotooffset_wnd_y", ystr.Get(), g_inipath.Get());
      g_gotolineoffsetwnd = hwnd = NULL;
    } break;
    case WM_COMMAND:
    {
      switch(LOWORD(wparam))
      {
      case IDOK:
        {
          char text[2048]; char *end;
          int len = GetDlgItemText(hwnd, IDC_EDIT1, text, sizeof(text));
          int index = (int)strtol(text, &end, 10);
          g_supereditor->GotoLineOffset(index);
          DestroyWindow(hwnd);
        } break;
      case IDCANCEL:
        {
          DestroyWindow(hwnd);
        } break;
      }
    } break;
  }

  return 0;
}

WDL_DLGRET PHO_MainWnd::MainWndProc(HWND hwnd, UINT msg, WPARAM wparam, LPARAM lparam)
{
  PHO_MainWnd *self = (PHO_MainWnd *)GetWindowLongPtr(hwnd, GWLP_USERDATA);

  if (!self && msg == WM_INITDIALOG)
  {
    SetWindowLongPtr(hwnd, GWLP_USERDATA, lparam);
    self = (PHO_MainWnd *)lparam;
    self->m_hwnd = hwnd;
  }

  if (WDL_likely(self))
  {
    return self->MainWndLoop(msg, wparam, lparam);
  }
  else
  {
    return 0;
  }
}

WDL_DLGRET PHO_MainWnd::MainWndLoop(UINT msg, WPARAM wparam, LPARAM lparam)
{
#if defined(__linux__)
  if (msg == WM_KEYDOWN && LOWORD(wparam) == VK_RETURN)
  {
    SendMessage(m_hwnd, WM_COMMAND, MAKEWPARAM(IDOK, 0), 0);
    return 0;
  }
#endif

  switch(msg)
  {
    case WM_INITDIALOG: OnInitDialog(wparam, lparam); break;
    case WM_SYSCOMMAND: OnSysCommand(wparam, lparam); break;
    case WM_CLOSE: return OnClose(wparam, lparam); break;
    case WM_SIZE: OnSize(wparam, lparam); break;
    case WM_MOVE: OnMove(wparam, lparam); break;
    case WM_DESTROY: OnDestroy(wparam, lparam); break;
    case WM_COMMAND: OnCommand(wparam, lparam); break;
    case WM_TIMER: OnTimer(wparam, lparam); break;
    case WM_PAINT: OnPaint(wparam, lparam); break;
    case WM_ERASEBKGND: return 1; break;
    case WM_NOTIFY: return OnNotify(wparam, lparam); break;
    case WM_CONTEXTMENU: OnContextMenu(wparam, lparam); break;
    case WM_LBUTTONDOWN: OnLButtonDown(wparam, lparam); break;
    case WM_LBUTTONUP: OnLButtonUp(wparam, lparam); break;
    case WM_MOUSEMOVE: OnMouseMove(wparam, lparam); break;
    //case WM_CTLCOLOREDIT: return OnCtrlColorEdit(msg, wparam, lparam); break;
    case WM_ACTIVATEAPP: OnActivateApp(wparam, lparam); break;
    case WM_NCLBUTTONDOWN: OnNCLButtonDown(wparam, lparam); break;
  }

  return 0;
}

#include "WDL/eel2/ns-eel.h"
#include "WDL/lice/lice.h"
#include "WDL/wingui/virtwnd-controls.h"
#include "WDL/wingui/scrollbar/coolscroll.h"


int WDL_STYLE_WantGlobalButtonBorders() { return 0; }
bool WDL_STYLE_WantGlobalButtonBackground(int *col) { return false; }
bool WDL_STYLE_GetBackgroundGradient(double *gradstart, double *gradslope) { return false; }
LICE_IBitmap *WDL_STYLE_GetSliderBitmap2(bool vert) { return NULL; }
bool WDL_STYLE_AllowSliderMouseWheel() { return true; }
int WDL_STYLE_GetSliderDynamicCenterPos() { return 500; }


// TO BE IMPLEMENTED BY APP:
// implemented by calling app, can return a LICE_IBitmap **img for "scrollbar"
void *GetIconThemePointer(const char *name)
{
  if (!strcmp(name, "scrollbar"))
  {
    if (!g_scrollbar)
    {
      WDL_FastString sb(g_modpath);
      sb.Append("skin\\mask\\scrollbar.png");
      g_scrollbar = LICE_LoadPNG(sb.Get());
      return &g_scrollbar;
    }
  }

  return NULL;
}

// can be a passthrough to GetSysColor()
int CoolSB_GetSysColor(HWND m_hwnd, int val)
{
  if (!g_scrollbar)
  {
    return GetSysColor(val);
  }
  else
  {
    return RGB(51, 51, 51);
  }
}

#include <WDL/eel2/ns-eel.h>

void NSEEL_HOSTSTUB_EnterMutex()
{}

void NSEEL_HOSTSTUB_LeaveMutex()
{}
