// Copyright (c) 2021 Giorgos Vougioukas
//
// The license can be found in the LICENSE file.

#include "phoebe/plugin.h"

#if defined(_WIN32)
  #define PHO_PLUGIN_EXTENSION ".dll"
#elif defined(__APPLE__)
  #define PHO_PLUGIN_EXTENSION ".dylib"
#elif defined(__linux__)
  #define PHO_PLUGIN_EXTENSION ".so"
#else
  #error "Plugin extension not right"
#endif

#include "WDL/assocarray.h"
#include "WDL/ptrlist.h"
#include "WDL/dirscan.h"
#include "WDL/wdlcstring.h"

static PHO_PluginInfo plugin_info;
static WDL_StringKeyedArray<HINSTANCE> modules;

static int RegisterCalled(const char *name, void *info_struct)
{
  if (name == NULL || info_struct == NULL)
  {
    return 0;
  }

  g_pluginimport.Insert((INT_PTR)info_struct, strdup(name));

  return 1;
}

static void *GetFuncCalled(const char *name)
{
  if (!name) return NULL;
  return g_pluginexport.Get(name); // if not found returns 0
}

static void PrepareFuncCalls()
{}

void LoadPlugins()
{
  WDL_ASSERT(g_pluginimport.GetSize() == 0);
  WDL_ASSERT(g_pluginexport.GetSize() == 0);
  WDL_ASSERT(modules.GetSize() == 0);

  plugin_info.caller_version = 120;
  plugin_info.Register = &RegisterCalled;
  plugin_info.GetFunc = &GetFuncCalled;

  PrepareFuncCalls();

  WDL_DirScan dir;
  WDL_FastString fn;

  WDL_FastString path(g_modpath);
  path.Append("ext" WDL_DIRCHAR_STR);

  WDL_PtrList_DeleteOnDestroy<WDL_FastString> file_list;

  if (!dir.First(path.Get()))
  {
    do
    {
      if (!dir.GetCurrentIsDirectory())
      {
        dir.GetCurrentFullFN(&fn);

        if (!strcmp(fn.get_fileext(), PHO_PLUGIN_EXTENSION))
        {
          file_list.Add(new WDL_FastString(fn.Get()));
        }
      }
    }
    while (!dir.Next());
  }

  if (file_list.GetSize() == 0)
  {
    MessageBox(NULL, "Phoebe could not find any plugin. Press OK to exit Phoebe.",
      "Plugin error", MB_OK);
    exit(1);
  }

  for (int i = 0; i < file_list.GetSize(); i++)
  {
    WDL_FastString *file = file_list.Get(i);

    if (!file) continue;

    HINSTANCE inst = NULL;
    inst = LoadLibrary(file->Get());

    if (inst)
    {
      modules.Insert(file->Get(), inst);
    }
    else
    {
      WDL_FastString liberr;
      liberr.SetFormatted(1024, "Phoebe could not load the following library:\n\n%s",
        file->Get());
      MessageBox(NULL, liberr.Get(), "Library error", MB_OK);
    }
  }

  typedef int (*PluginEntry)(PHO_PLUGIN_HINSTANCE instance, PHO_PluginInfo *rec);

  WDL_FastString err;
  bool display_error = false;
  err.Set("Phoebe could not load the following plugin:\n\n");

  for (int i = 0; i < modules.GetSize(); i++)
  {
    PluginEntry plugin_entry = (PluginEntry)GetProcAddress(modules.Enumerate(i),
      PHO_PLUGIN_ENTRYPOINT_NAME);

#if defined(_WIN32)
    HINSTANCE inst = g_inst;
    //HINSTANCE inst = GetModuleHandle(NULL);
#else
    HINSTANCE inst = NULL;
#endif

    //int success = plugin_entry(inst, &plugin_info);
    int success = plugin_entry(modules.Enumerate(i), &plugin_info);

    if (!success)
    {
      display_error = true;
      err.AppendFormatted(2048, "%s\n",
        WDL_get_filepart(modules.ReverseLookup(modules.Enumerate(i))));
    }
  }

  if (display_error)
  {
    err.Append("\nPress OK to continue with limited functionality.");
    MessageBox(NULL, err.Get(), "Plugin error", MB_OK);
  }

  WDL_ASSERT(g_pluginimport.GetSize() > 0);
  WDL_ASSERT(g_pluginexport.GetSize() > 0);
}

void ReleasePlugins()
{
  for (int i = 0; i < modules.GetSize(); i++)
  {
    FreeLibrary(modules.Enumerate(i));
  }

  modules.DeleteAll();
  g_pluginimport.DeleteAll();
  g_pluginexport.DeleteAll();
}
