// Copyright (c) 2021 Giorgos Vougioukas
//
// The license can be found in the LICENSE file.

#include "phoebe/preferences.h"
#include "phoebe/definitions.h"
#include "phoebe/phoebe_plugin.h"
#include "phoebe/app_info.h"
#include "phoebe/main_wnd.h"

#include "WDL/fileread.h"
#include "WDL/filewrite.h"
#include "WDL/aggarray.h"

static const char *default_preferences[] =
{
  "//",
  "// Welcome to " PHO_NAME_MARKETING " " PHO_ARCH " Text Editor, Version " PHO_NAKED_VERSION,
  "//",
  "// " PHO_NAME_MARKETING " preferences",
  "// Your changes will take effect the next time you relaunch " PHO_NAME_MARKETING "",
  "//",
  "",
  "{",
  "  // Display preferences with Curses interface",
  "  // (supports only ASCII for text input).",
  "  // If set to false it uses an external editor.",
  "  \"curses_preferences\": true,",
  "  \"curses_right_focus\": true,",
  "",
  "  // Monospace font selection.",
  "  // 1: Courier (Compatible with Windows XP)",
  "  // 2: Courier New",
  "  // 3: Consolas (Recommended for Windows 7+)",
  "  \"text_font\": 1,",
  "",
  "  // Monospace font height.",
  "  // The range is 7-24",
  "  \"text_font_height\": 14,",
  "",
  "  // Show whitespace.",
  "  \"show_whitespace\": true,",
  "",
  "  // Default TAB character (TAB or SPACE).",
  "  // Toggle between the two with F10",
  "  \"space\": true,",
  "",
  "  // Default TAB size.",
  "  // The range is 2-8",
  "  \"tab_size\": 2,",
  "",
  "  // Default end of line of text (CRLF or LF).",
  "  // Toggle between the two with F9",
  "  \"crlf\": true,",
  "",
  "  // File dialog.",
  "  \"preserve_directory\": true,",
  "",
  "  // Presentation.",
  "  // Colors in hexadecimal format (hex triplet).",
  "  \"background_color\": \"000000\",",
  "  \"active_text_color\": \"8080FF\",",
  "  \"text_color\": \"C0C0C0\"",
  "}",
};

static WDL_PtrArray<const char *, WDL_ARRAYCOUNT(default_preferences)> dpref(default_preferences);

PHO_Preferences::PHO_Preferences()
  : m_pf(NULL)
  , m_relaunch(false)
{}

PHO_Preferences::~PHO_Preferences()
{
  cJSON_Delete(m_pf);
}

void PHO_Preferences::Create()
{
  WDL_FastString defpref, usrpref;

  for (int i = 0; i < dpref.GetSize(); i++)
  {
    defpref.AppendFormatted(2048, "%s\n", dpref.Get()[i]);
  }

  const char *user_preferences[] =
  {
    "{",
    "  // Add your personal preferences here.",
    "}"
  };

  WDL_PtrArray<const char *, WDL_ARRAYCOUNT(user_preferences)> upref(user_preferences);

  for (int j = 0; j < upref.GetSize(); j++)
  {
    usrpref.AppendFormatted(2048, "%s\n", upref.Get()[j]);
  }

  WDL_FastString def(g_setpath), usr(g_setpath);
  def.Append("preferences-default.txt");
  usr.Append("preferences.txt");

  // Create default preferences file
  WDL_FileWrite *cdf = new WDL_FileWrite(def.Get());
  if (cdf && cdf->IsOpen())
  {
    cdf->Write(defpref.Get(), defpref.GetLength());
  }

  if (cdf) { delete cdf; cdf = NULL; }

  // Check if user preferences exist
  WDL_FileRead *uf = new WDL_FileRead(usr.Get());
  if (uf && !uf->IsOpen())
  {
    // Create user preferences file
    WDL_FileWrite *cuf = new WDL_FileWrite(usr.Get());
    if (cuf && cuf->IsOpen())
    {
      cuf->Write(usrpref.Get(), usrpref.GetLength());
    }

    if (cuf) { delete cuf; cuf = NULL; }
  }

  if (uf) { delete uf; uf = NULL; }
}

void PHO_Preferences::Open(bool default_file)
{
  WDL_FastString def(g_setpath), usr(g_setpath);
  def.Append("preferences-default.txt");
  usr.Append("preferences.txt");

  ShellExecute(g_mainwnd->Handle(), "", "notepad.exe", default_file ? def.Get() : usr.Get(), "", SW_SHOWNORMAL);
}

bool PHO_Preferences::Parse(bool default_file)
{
  WDL_FastString def(g_setpath), usr(g_setpath);
  def.Append("preferences-default.txt");
  usr.Append("preferences.txt");

  WDL_FileRead fr(default_file ? def.Get() : usr.Get());
  WDL_TypedBuf<char> strbuf;
  strbuf.Resize((int)fr.GetSize());
  fr.Read(strbuf.Get(), strbuf.GetSize());
  strbuf.Add('\0');

  cJSON_Minify(strbuf.Get());
  m_pf = cJSON_Parse(strbuf.Get());

  if (!m_pf)
  {
    const char *error_ptr = cJSON_GetErrorPtr();
    if (error_ptr)
    {
      m_err.SetFormatted(2048, "Parse error before: %s\n", error_ptr);
      wdl_log(m_err.Get());
    }

    return false;
  }

  return true;
}

bool PHO_Preferences::WantCurses() const
{
  const cJSON *curses = NULL;
  curses = cJSON_GetObjectItemCaseSensitive(m_pf, "curses_preferences");
  if (cJSON_IsTrue(curses)) return true;
  else if (cJSON_IsFalse(curses)) return false;
  else return true;
}

bool PHO_Preferences::WantCursesRightFocus() const
{
  const cJSON *curses_right_focus = NULL;
  curses_right_focus = cJSON_GetObjectItemCaseSensitive(m_pf, "curses_right_focus");
  if (cJSON_IsTrue(curses_right_focus)) return true;
  else if (cJSON_IsFalse(curses_right_focus)) return false;
  else return true;
}

int PHO_Preferences::GetBackgroundColor() const
{
  const cJSON *bg_color = NULL;
  bg_color = cJSON_GetObjectItemCaseSensitive(m_pf, "background_color");
  if (cJSON_IsString(bg_color) && (bg_color->valuestring != NULL))
  {
    int len = (int)strlen(bg_color->valuestring);
    if (len != 6) return 0x000000;
    int bgc = (int)strtol(bg_color->valuestring, NULL, 16);
    if (bgc < 0x000000 || bgc > 0xFFFFFF) return 0x000000;
    return bgc;
  }
  else return 0x000000;
}

int PHO_Preferences::GetActiveTextColor() const
{
  const cJSON *txt_color = NULL;
  txt_color = cJSON_GetObjectItemCaseSensitive(m_pf, "active_text_color");
  if (cJSON_IsString(txt_color) && (txt_color->valuestring != NULL))
  {
    int len = (int)strlen(txt_color->valuestring);
    if (len != 6) return 0x8080FF;
    int txtc = (int)strtol(txt_color->valuestring, NULL, 16);
    if (txtc < 0x000000 || txtc > 0xFFFFFF) return 0x8080FF;
    return txtc;
  }
  else return 0x8080FF;
}

int PHO_Preferences::GetTextColor() const
{
  const cJSON *txt_color = NULL;
  txt_color = cJSON_GetObjectItemCaseSensitive(m_pf, "text_color");
  if (cJSON_IsString(txt_color) && (txt_color->valuestring != NULL))
  {
    int len = (int)strlen(txt_color->valuestring);
    if (len != 6) return 0xC0C0C0;
    int txtc = (int)strtol(txt_color->valuestring, NULL, 16);
    if (txtc < 0x000000 || txtc > 0xFFFFFF) return 0xC0C0C0;
    return txtc;
  }
  else return 0xC0C0C0;
}

bool PHO_Preferences::WantPreserveDirectory() const
{
  const cJSON *preserve_directory = NULL;
  preserve_directory = cJSON_GetObjectItemCaseSensitive(m_pf, "preserve_directory");
  if (cJSON_IsTrue(preserve_directory)) return true;
  else if (cJSON_IsFalse(preserve_directory)) return false;
  else return true;
}

int PHO_Preferences::GetTextFont() const
{
  const cJSON *text_font = NULL;
  text_font = cJSON_GetObjectItemCaseSensitive(m_pf, "text_font");
  if (cJSON_IsNumber(text_font))
  {
    if (text_font->valueint >= 1 && text_font->valueint <= 3)
    {
      return text_font->valueint;
    }
    else return 1;
  }
  else return 1;
}

int PHO_Preferences::GetTextFontHeight() const
{
  const cJSON *text_font_height = NULL;
  text_font_height = cJSON_GetObjectItemCaseSensitive(m_pf, "text_font_height");
  if (cJSON_IsNumber(text_font_height))
  {
    if (text_font_height->valueint >= 7 && text_font_height->valueint <= 24)
    {
      return text_font_height->valueint;
    }
    else return 14;
  }
  else return 14;
}

bool PHO_Preferences::ShowWhitespace() const
{
  const cJSON *show_whitespace = NULL;
  show_whitespace = cJSON_GetObjectItemCaseSensitive(m_pf, "show_whitespace");
  if (cJSON_IsTrue(show_whitespace)) return true;
  else if (cJSON_IsFalse(show_whitespace)) return false;
  else return true;
}

bool PHO_Preferences::WantSpaceForTab() const
{
  const cJSON *space = NULL;
  space = cJSON_GetObjectItemCaseSensitive(m_pf, "space");
  if (cJSON_IsTrue(space)) return true;
  else if (cJSON_IsFalse(space)) return false;
  else return true;
}

int PHO_Preferences::GetTabSize() const
{
  const cJSON *tab_size = NULL;
  tab_size = cJSON_GetObjectItemCaseSensitive(m_pf, "tab_size");
  if (cJSON_IsNumber(tab_size))
  {
    if (tab_size->valueint >= 2 && tab_size->valueint <= 8)
    {
      return tab_size->valueint;
    }
    else return 2;
  }
  else return 2;
}

bool PHO_Preferences::WantCRLF() const
{
  const cJSON *crlf = NULL;
  crlf = cJSON_GetObjectItemCaseSensitive(m_pf, "crlf");
  if (cJSON_IsTrue(crlf)) return true;
  else if (cJSON_IsFalse(crlf)) return false;
  else return true;
}
