// Copyright (c) 2023 Giorgos Vougioukas
//
// The license can be found in the LICENSE file.

#include "phoebe/switch_wnd.h"
#include "phoebe/app_info.h"
#include "phoebe/preferences.h"

static BOOL CALLBACK PHO_EnumerateWindowCallback(HWND hwnd, LPARAM lparam)
{
  WDL_PtrList<PHO_SwitchWindowInfo> *m_swi = (WDL_PtrList<PHO_SwitchWindowInfo> *)lparam;

  if (m_swi)
  {
    char text[2048];
    GetClassName(hwnd, text, sizeof(text));

    if (!strcmp(text, "phoebe_superhost"))
    {
      int len = GetWindowText(hwnd, text, sizeof(text));
      PHO_SwitchWindowInfo *swi = new PHO_SwitchWindowInfo;

      if (swi)
      {
        int start = 0;
        for (int i = 0; i < len - 2; i++)
        {
          if (text[i] == '-' && text[i + 1] == ' ')
          {
            start = i + 2; break;
          }
        }
        swi->hwnd = hwnd;
        WDL_ASSERT(start < len);
        swi->text.Set(&text[start]);
      }

      m_swi->Add(swi);
    }
  }

  return TRUE;
}

PHO_SwitchWnd::PHO_SwitchWnd()
  : m_hwnd(NULL)
  , m_list(NULL)
{}

PHO_SwitchWnd::~PHO_SwitchWnd()
{
  DestroyWindow(m_hwnd);
  m_swi.Empty(true);
}

void PHO_SwitchWnd::GotoSelectedWindow()
{
  WDL_ASSERT(m_swi.GetSize() == ListView_GetItemCount(m_list));
  int pos = ListView_GetNextItem(m_list, -1, LVIS_SELECTED|LVIS_FOCUSED);
  if (pos >= 0)
  {
    PHO_SwitchWindowInfo *swi = m_swi.Get(pos);
    if (swi)
    {
      SetForegroundWindow(swi->hwnd);
    }
  }
}

void PHO_SwitchWnd::OnInitDialog(WPARAM wparam, LPARAM lparam)
{
  m_list = GetDlgItem(m_hwnd, IDC_LIST1);
  WDL_UTF8_HookListView(m_list);
  m_resize.init(m_hwnd);
  m_resize.init_itemhwnd(m_list, 0.0f, 0.0f, 1.0f, 1.0f);
  EnumWindows(PHO_EnumerateWindowCallback, (LPARAM)&m_swi);
  qsort(m_swi.GetList(), m_swi.GetSize(),
    sizeof(PHO_SwitchWindowInfo *), windowtextcmp);
  int x = GetPrivateProfileInt(PHO_NAME, "switch_wnd_x", 0, g_inipath.Get());
  int y = GetPrivateProfileInt(PHO_NAME, "switch_wnd_y", 50, g_inipath.Get());
  int w = GetPrivateProfileInt(PHO_NAME, "switch_wnd_w", 400, g_inipath.Get());
  int h = GetPrivateProfileInt(PHO_NAME, "switch_wnd_h", 900, g_inipath.Get());
  SetWindowPos(m_hwnd, NULL, x, y, w, h, SWP_NOZORDER | SWP_NOACTIVATE);

#if defined(_WIN32)
  ListView_SetExtendedListViewStyleEx(m_list,
    LVS_EX_HEADERDRAGDROP | LVS_EX_FULLROWSELECT | LVS_EX_DOUBLEBUFFER,
    LVS_EX_HEADERDRAGDROP | LVS_EX_FULLROWSELECT | LVS_EX_DOUBLEBUFFER);
#else
  ListView_SetExtendedListViewStyleEx(m_list,
    LVS_EX_HEADERDRAGDROP | LVS_EX_FULLROWSELECT,
    LVS_EX_HEADERDRAGDROP | LVS_EX_FULLROWSELECT);
#endif

  int bk = g_preferences->GetBackgroundColor();
  int fg = g_preferences->GetTextColor();
  ListView_SetBkColor(m_list, RGB((bk >> 16) & 0xFF,(bk >> 8) & 0xFF, bk & 0xFF));
  ListView_SetTextBkColor(m_list, RGB((bk >> 16) & 0xFF,(bk >> 8) & 0xFF, bk & 0xFF));
  ListView_SetTextColor(m_list, RGB((fg >> 16) & 0xFF,(fg >> 8) & 0xFF, fg & 0xFF));

  LVCOLUMN lvc = { 0, };
  lvc.mask = LVCF_FMT | LVCF_TEXT | LVCF_WIDTH;

  RECT r;
  GetClientRect(m_list, &r);

  lvc.cx = r.right - r.left - 7;
  lvc.fmt = LVCFMT_LEFT;
  lvc.pszText = (char *)"";
  ListView_InsertColumn(m_list, 0, &lvc);

  for (int i = 0; i < m_swi.GetSize(); i++)
  {
    PHO_SwitchWindowInfo *swi = m_swi.Get(i);

    if (swi)
    {
      LVITEM lvi = { 0, };
      lvi.mask = LVIF_TEXT | LVIF_PARAM;
      lvi.lParam = (LPARAM)i;
      lvi.iItem = i;
      lvi.iSubItem = 0;
      lvi.pszText = (char *)swi->text.Get();
      lvi.cchTextMax = swi->text.GetLength();
      ListView_InsertItem(m_list, &lvi);
    }
  }

  ListView_SetItemState(m_list, 0, LVNI_SELECTED | LVNI_FOCUSED, LVNI_SELECTED | LVNI_FOCUSED);
}

void PHO_SwitchWnd::OnDestroy(WPARAM wparam, LPARAM lparam)
{
  RECT r;
  GetWindowRect(m_hwnd, &r);

  WDL_FastString rleft, rtop, rwidth, rheight;
  rleft.SetFormatted(32, "%d", r.left);
  rtop.SetFormatted(32, "%d", r.top);
  rwidth.SetFormatted(32, "%d", r.right - r.left);
  rheight.SetFormatted(32, "%d", r.bottom - r.top);

  WritePrivateProfileString(PHO_NAME,"switch_wnd_x", rleft.Get(), g_inipath.Get());
  WritePrivateProfileString(PHO_NAME,"switch_wnd_y", rtop.Get(), g_inipath.Get());
  WritePrivateProfileString(PHO_NAME,"switch_wnd_w", rwidth.Get(), g_inipath.Get());
  WritePrivateProfileString(PHO_NAME,"switch_wnd_h", rheight.Get(), g_inipath.Get());
}

void PHO_SwitchWnd::OnSize(WPARAM wparam, LPARAM lparam)
{
  if (wparam != SIZE_MINIMIZED)
  {
    m_resize.onResize();

    RECT r;
    GetClientRect(m_list, &r);
    int cw = r.right - r.left - 7;
    ListView_SetColumnWidth(m_list, 0, cw);
  }
}

void PHO_SwitchWnd::OnCommand(WPARAM wparam, LPARAM lparam)
{
  switch(LOWORD(wparam))
  {
  case IDOK:
    {
      GotoSelectedWindow();
      if (g_switchwnd) { delete g_switchwnd; g_switchwnd = NULL; }
    } break;
  case IDCANCEL:
    {
      if (g_switchwnd) { delete g_switchwnd; g_switchwnd = NULL; }
    } break;
  }
}

INT_PTR PHO_SwitchWnd::OnNotify(WPARAM wparam, LPARAM lparam)
{
  switch (LOWORD(wparam))
  {
  case IDC_LIST1:
    {
      switch (((LPNMHDR)lparam)->code)
      {
      case NM_DBLCLK:
        {
          GotoSelectedWindow();
          if (g_switchwnd) { delete g_switchwnd; g_switchwnd = NULL; }
        } break;
      }
    } break;
  }

  return 0;
}

WDL_DLGRET PHO_SwitchWnd::SwitchWndProc(HWND hwnd, UINT msg, WPARAM wparam, LPARAM lparam)
{
  PHO_SwitchWnd *self = (PHO_SwitchWnd *)GetWindowLongPtr(hwnd, GWLP_USERDATA);

  if (!self && msg == WM_INITDIALOG)
  {
    SetWindowLongPtr(hwnd, GWLP_USERDATA, lparam);
    self = (PHO_SwitchWnd *)lparam;
    self->m_hwnd = hwnd;
  }

  if (WDL_likely(self))
  {
    return self->SwitchWndLoop(msg, wparam, lparam);
  }
  else
  {
    return 0;
  }
}

WDL_DLGRET PHO_SwitchWnd::SwitchWndLoop(UINT msg, WPARAM wparam, LPARAM lparam)
{
  switch (msg)
  {
    case WM_INITDIALOG: OnInitDialog(wparam, lparam); break;
    case WM_DESTROY: OnDestroy(wparam, lparam); break;
    case WM_SIZE: OnSize(wparam, lparam); break;
    case WM_COMMAND: OnCommand(wparam, lparam); break;
    case WM_NOTIFY: return OnNotify(wparam, lparam); break;
  }

  return 0;
}
