// Copyright (c) 2023 Giorgos Vougioukas
//
// The license can be found in the LICENSE file.

#include "phoebe/cursor.h"
#include "phoebe/super_editor_wnd.h"

PHO_Cursor::PHO_Cursor()
  : m_x(0)
  , m_y(0)
  , m_w(0)
  , m_h(0)
  , m_vx(0)
  , m_vy(0)
  , m_vis(false)
  , m_bt(0)
{}

PHO_Cursor::~PHO_Cursor()
{}

void PHO_Cursor::SingleLetter(const RECT *r)
{
  m_sl.left = r->left;
  m_sl.top = r->top;
  m_sl.right = r->right;
  m_sl.bottom = r->bottom;
  m_w = m_sl.right - m_sl.left;
  m_h = m_sl.bottom - m_sl.top;
}

void PHO_Cursor::ViewSurface(const RECT *r)
{
  m_bt += PHO_UPDATE_CURSOR_MS;

  if (m_bt > 500)
  {
    m_vis = !m_vis;
    m_bt = 0;
  }

  m_vs.left = r->left;
  m_vs.top = r->top;
  m_vs.right = r->right;
  m_vs.bottom = r->bottom;
}

int PHO_Cursor::GetViewSurfaceLines() const
{
  return m_h > 0 ? (m_vs.bottom - m_vs.top) / (m_h + 1) : 0;
}

int PHO_Cursor::GetViewSurfaceColumns() const
{
  return m_w > 0 ? (m_vs.right - m_vs.left) / (m_w) : 0;
}

void PHO_Cursor::GetCursor(bool *visible, RECT *r) const
{
  *visible = m_vis;
  r->left = m_vx * m_w;
  r->top = m_vy * (m_h + 1);
  r->right = r->left + m_w;
  r->bottom = r->top + m_h;
}

void PHO_Cursor::GetPosition(int *x, int *y) const
{
  *x = m_x;
  *y = m_y;
}

void PHO_Cursor::GetDeltaPosition(int *x, int *y) const
{
  *x = m_x - m_vx;
  *y = m_y - m_vy;
}

void PHO_Cursor::SetPosition(int x, int y)
{
  SetX(x);
  SetY(y);
}

int PHO_Cursor::GetX() const
{
  return m_x;
}

void PHO_Cursor::SetX(int x)
{
  HideMouseCursor();

  m_vis = true;
  int right = GetRight();
  int left = GetLeft();
  int delta = x - m_x;

  if (delta > 0)
  {
    m_vx = wdl_min(m_vx + delta, right);
  }
  else if (delta < 0)
  {
    m_vx = wdl_max(m_vx + delta, 0);
  }

  m_x = x;
}

int PHO_Cursor::GetY() const
{
  return m_y;
}

void PHO_Cursor::SetY(int y)
{
  HideMouseCursor();

  m_vis = true;
  int bot = GetBottom();
  int top = GetTop();
  int delta = y - m_y;

  if (delta > 0)
  {
    m_vy = wdl_min(m_vy + delta, bot);
  }
  else if (delta < 0)
  {
    m_vy = wdl_max(m_vy + delta, 0);
  }

  m_y = y;
}

int PHO_Cursor::GetWidth() const
{
  return m_w;
}

int PHO_Cursor::GetHeight() const
{
  return m_h;
}

void PHO_Cursor::MoveToTop()
{
  m_vy = 0;
}

void PHO_Cursor::MoveToX(int x)
{
  m_vx = -x;
}

void PHO_Cursor::MoveToY(int y)
{
  m_vy = -y;
}

int PHO_Cursor::GetBottom() const
{
  return GetViewSurfaceLines() - 4;
}

int PHO_Cursor::GetTop() const
{
  return 0;
}

int PHO_Cursor::GetRight() const
{
  return GetViewSurfaceColumns();
}

int PHO_Cursor::GetLeft() const
{
  return 0;
}

void PHO_Cursor::HideMouseCursor()
{
  if (!GetCapture())
  {
    POINT pt;
    GetCursorPos(&pt);

    HWND hwnd = g_supereditorwnd->Handle();
    ScreenToClient(hwnd, &pt);

    RECT r;
    GetClientRect(hwnd, &r);

    if (pt.x >= r.left && pt.x <= r.right &&
      pt.y >= r.top && pt.y <= r.bottom)
    {
      SetCursor(NULL);
    }
  }
}
