// Copyright (c) 2023 Giorgos Vougioukas
//
// The license can be found in the LICENSE file.

#include "phoebe/super_editor.h"
#include "phoebe/cursor.h"
#include "phoebe/main_wnd.h"

#include "WDL/aggarray.h"
#include "WDL/fpcmp.h"
#include "WDL/filewrite.h"
#include "WDL/wdlutf8.h"

static LOGFONT lf =
{
#if defined(_WIN32)
  14, 0, 0, 0, FW_NORMAL, FALSE, FALSE, FALSE, DEFAULT_CHARSET,
  OUT_DEFAULT_PRECIS, CLIP_DEFAULT_PRECIS, DEFAULT_QUALITY, DEFAULT_PITCH,
  "Arial"
#else
  11, 0, 0, 0, FW_NORMAL, FALSE, FALSE, FALSE, DEFAULT_CHARSET,
  OUT_DEFAULT_PRECIS, CLIP_DEFAULT_PRECIS, DEFAULT_QUALITY, DEFAULT_PITCH,
  "Arial"
#endif
};

PHO_SuperEditor::PHO_SuperEditor()
  : m_killthread(false)
  , m_running(false)
  , m_offsetlines(false)
  , m_lines(false)
  , m_seltext(false)
  , m_headcurx(0)
  , m_headcury(0)
  , m_startcurx(0)
  , m_startcury(0)
  , m_endcurx(0)
  , m_endcury(0)
  , m_crlf(true)
  , m_space(true)
  , m_tabsize(0)
  , m_textchanged(false)
  , m_replacetext(false)
{
  m_crlf = g_preferences->WantCRLF();
  m_space = g_preferences->WantSpaceForTab();
  m_tabsize = g_preferences->GetTabSize();
  //StartThread();
}

PHO_SuperEditor::~PHO_SuperEditor()
{
  //if (IsRunning()) StopThread();
  m_text.Empty(true, free);
  m_buffer.Empty(true, free);
}

void PHO_SuperEditor::New()
{
  m_fn.Set("");
  m_text.Empty(true, free);
  m_textchanged = false;
  m_text.Add(strdup(""));
  g_cursor->SetPosition(0, 0);
  m_crlf = g_preferences->WantCRLF();
  m_space = g_preferences->WantSpaceForTab();
  m_strbuf.SetFormatted(2048, "%s - *", g_mainwnd->GetOrigTitle());
  SetWindowText(g_mainwnd->Handle(), m_strbuf.Get());
}

void PHO_SuperEditor::Open(const char *filename)
{
  WDL_TypedArray<char, 65536> str;
  FILE *fp = fopen(filename, "r");

  if (fp)
  {
    m_fn.Set(filename);
    m_text.Empty(true, free);
    m_textchanged = false;
    g_cursor->SetPosition(0, 0);
    m_crlf = g_preferences->WantCRLF();
    m_space = g_preferences->WantSpaceForTab();
    m_strbuf.SetFormatted(2048, "%s - %s",
      g_mainwnd->GetOrigTitle(), m_fn.get_filepart());
    SetWindowText(g_mainwnd->Handle(), m_strbuf.Get());

    while (fgets(str.Get(), str.GetSize(), fp) != NULL)
    {
      m_strbuf.Set(str.Get());
      const int cr = m_strbuf.GetLength() - 2;
      const int lf = m_strbuf.GetLength() - 1;

      if (m_strbuf.Get()[cr] == '\r' && m_strbuf.Get()[lf] == '\n')
      {
        m_strbuf.DeleteSub(cr, 2);
      }
      else if (m_strbuf.Get()[lf] == '\n')
      {
        m_strbuf.DeleteSub(lf, 1);
      }

      m_text.Add(strdup(m_strbuf.Get()));
    }

    g_cursor->SetPosition(0, 0);
    fclose(fp);
  }
}

bool PHO_SuperEditor::IsOpen() const
{
  return m_fn.GetLength() > 0;
}

void PHO_SuperEditor::Save()
{
  WDL_ASSERT(m_fn.GetLength());
  WDL_FileWrite fw(m_fn.Get());
  if (fw.IsOpen())
  {
    for (int i = 0; i < m_text.GetSize(); i++)
    {
      const char *ln = m_text.Get(i);
      if (ln)
      {
        m_strbuf.Set(ln);
        m_strbuf.Append(m_crlf ? "\r\n" : "\n");
        fw.Write(m_strbuf.Get(), m_strbuf.GetLength());
      }
    }

    WDL_FastString str;
    str.SetFormatted(2048, "%s - %s", g_mainwnd->GetOrigTitle(), m_fn.get_filepart());
    SetWindowText(g_mainwnd->Handle(), str.Get());
    m_textchanged = false;
  }
}

void PHO_SuperEditor::SaveAs(const char *filename)
{
  m_fn.Set(filename); Save();
}

void PHO_SuperEditor::MoveLeft()
{
  int curx = wdl_max(g_cursor->GetX() - 1, 0);
  g_cursor->SetX(curx);
}

void PHO_SuperEditor::MoveRight()
{
  int curx = g_cursor->GetX();
  int cury = g_cursor->GetY();
  if (cury < m_text.GetSize())
  {
    const char *ln = m_text.Get(cury);
    if (ln) curx = wdl_min(curx + 1, WDL_utf8_get_charlen(ln));
    g_cursor->SetX(curx);
  }
}

void PHO_SuperEditor::MoveUp()
{
  int curx = g_cursor->GetX();
  int cury = g_cursor->GetY();
  cury = wdl_max(cury - 1, 0);
  g_cursor->SetY(cury);

  if (cury < m_text.GetSize())
  {
    const char *ln = m_text.Get(cury);
    if (ln) curx = wdl_min(curx, WDL_utf8_get_charlen(ln));
    g_cursor->SetX(curx);
  }
}

void PHO_SuperEditor::MoveDown()
{
  int curx = g_cursor->GetX();
  int cury = g_cursor->GetY();
  if (m_text.GetSize())
  {
    cury = wdl_min(cury + 1, m_text.GetSize() - 1);
    g_cursor->SetY(cury);
  }

  if (cury < m_text.GetSize())
  {
    const char *ln = m_text.Get(cury);
    if (ln) curx = wdl_min(curx, WDL_utf8_get_charlen(ln));
    g_cursor->SetX(curx);
  }
}

void PHO_SuperEditor::MoveStart()
{
  int curx = g_cursor->GetX();
  int cury = g_cursor->GetY();
  const int oldcurx = curx;
  curx = 0;
  const char *ln = m_text.Get(cury);
  if (ln)
  {
    int i = 0;
    while (ln[i] == ' ' || ln[i] == '\t') i++;
    if (ln[i] && i != oldcurx) curx = i;
    g_cursor->SetX(curx);
  }
}

void PHO_SuperEditor::MoveEnd()
{
  int curx = g_cursor->GetX();
  int cury = g_cursor->GetY();
  if (cury < m_text.GetSize())
  {
    const char *ln = m_text.Get(cury);
    if (ln) curx = WDL_utf8_get_charlen(ln);
    g_cursor->SetX(curx);
  }
}

void PHO_SuperEditor::MoveTop()
{
  g_cursor->SetPosition(0, 0);
}

void PHO_SuperEditor::MoveBottom()
{
  int curx = 0;
  int cury = m_text.GetSize() - 1;
  const char *ln = m_text.Get(cury);
  if (ln) curx = WDL_utf8_get_charlen(ln);
  g_cursor->SetPosition(curx, cury);
}

void PHO_SuperEditor::MovePageUp()
{
  int curx = g_cursor->GetX();
  int cury = g_cursor->GetY();
  int lns = g_cursor->GetViewSurfaceLines();
  cury = wdl_max(cury - lns, 0);
  g_cursor->SetY(cury);

  if (cury < m_text.GetSize())
  {
    const char *ln = m_text.Get(cury);
    if (ln) curx = wdl_min(curx, WDL_utf8_get_charlen(ln));
    g_cursor->SetX(curx);
  }
}

void PHO_SuperEditor::MovePageDown()
{
  int curx = g_cursor->GetX();
  int cury = g_cursor->GetY();
  int lns = g_cursor->GetViewSurfaceLines();
  cury = wdl_min(cury + lns, m_text.GetSize() - 1);
  g_cursor->SetY(cury);
  g_cursor->MoveToTop();

  if (cury < m_text.GetSize())
  {
    const char *ln = m_text.Get(cury);
    if (ln) curx = wdl_min(curx, WDL_utf8_get_charlen(ln));
    g_cursor->SetX(curx);
  }
}

void PHO_SuperEditor::TextSelection(bool active)
{
  if (active && !m_seltext)
  {
    m_seltext = true;
    if (GetCapture())
    {
      m_headcurx = m_startcurx;
      m_headcury = m_startcury;
    }
    else
    {
      m_headcurx = g_cursor->GetX();
      m_headcury = g_cursor->GetY();
    }
  }

  if (active)
  {
    m_seltext = true;
    int tailcurx;
    int tailcury;

    if (GetCapture())
    {
      tailcurx = m_endcurx;
      tailcury = m_endcury;
    }
    else
    {
      tailcurx = g_cursor->GetX();
      tailcury = g_cursor->GetY();
    }

    if (tailcury < m_headcury)
    {
      m_startcury = tailcury;
      m_startcurx = tailcurx;
      m_endcury = m_headcury;
      m_endcurx = m_headcurx;
    }
    else if (tailcury == m_headcury &&
      tailcurx < m_headcurx)
    {
      m_startcury = tailcury;
      m_startcurx = tailcurx;
      m_endcury = m_headcury;
      m_endcurx = m_headcurx;
    }
    else
    {
      m_startcury = m_headcury;
      m_startcurx = m_headcurx;
      m_endcury = tailcury;
      m_endcurx = tailcurx;
    }
  }
  else
  {
    m_seltext = false;
    m_headcurx = 0;
    m_headcury = 0;
    m_startcurx = 0;
    m_startcury = 0;
    m_endcurx = 0;
    m_endcury = 0;
  }
}

bool PHO_SuperEditor::HasSelectedText() const
{
  return m_seltext;
}

void PHO_SuperEditor::GetSelectStart(int *curx, int *cury) const
{
  *curx = m_startcurx;
  *cury = m_startcury;
}

void PHO_SuperEditor::GetSelectEnd(int *curx, int *cury) const
{
  *curx = m_endcurx;
  *cury = m_endcury;
}

void PHO_SuperEditor::SetSelectStart(int curx, int cury)
{
  m_startcurx = curx;
  m_startcury = cury;
}

void PHO_SuperEditor::SetSelectEnd(int curx, int cury)
{
  m_endcurx = curx;
  m_endcury = cury;
}

void PHO_SuperEditor::MoveTextUp()
{
  int curx = g_cursor->GetX();
  int cury = g_cursor->GetY();
  cury = cury-- > 0 ? cury : 0;

  g_cursor->SetY(cury);
  g_cursor->SetX(0);
  g_cursor->MoveToY(0);
  g_cursor->MoveToX(0);
}

void PHO_SuperEditor::MoveTextDown()
{
  int curx = g_cursor->GetX();
  int cury = g_cursor->GetY();
  int page = g_cursor->GetBottom();
  int bot = m_text.GetSize() - page;

  if (bot > page)
  {
    cury = cury++ < bot ? cury : bot;
  }
  else
  {
    cury = cury++ < page ? cury : page;
  }

  g_cursor->SetY(cury);
  g_cursor->SetX(0);
  g_cursor->MoveToY(0);
  g_cursor->MoveToX(0);
}

const char *PHO_SuperEditor::GetTextRegion(int line, int column) const
{
  if (line < m_text.GetSize())
  {
    const char *ln = m_text.Get(line);
    if (ln && column < (int)strlen(ln)) return &ln[column];
  }

  return "";
}

void PHO_SuperEditor::InitialState()
{
  TextSelection(false);
  m_offsetlines = false;
  m_lines = false;
  m_textfind.SetLen(0, true);
  m_textreplace.SetLen(0, true);
}

void PHO_SuperEditor::ShowOffsetLines()
{
  m_offsetlines = true;
  m_lines = false;
}

void PHO_SuperEditor::ShowLines()
{
  m_offsetlines = false;
  m_lines = true;
}

bool PHO_SuperEditor::HasOffsetLines() const
{
  return m_offsetlines;
}

bool PHO_SuperEditor::HasLines() const
{
  return m_lines;
}

void PHO_SuperEditor::GotoLine(int index)
{
  InitialState();
  int cury = g_cursor->GetY();
  cury = wdl_clamp(index, 0, m_text.GetSize() - 1);
  g_cursor->SetY(cury);
}

void PHO_SuperEditor::GotoLineOffset(int index)
{
  InitialState();
  int cury = g_cursor->GetY();
  cury = wdl_clamp(cury + index, 0, m_text.GetSize() - 1);
  g_cursor->SetY(cury);
}

int PHO_SuperEditor::GetMaxLine() const
{
  return m_text.GetSize();
}

int PHO_SuperEditor::GetMaxColumn() const
{
  int mxcol = 0;

  for (int i = 0; i < m_text.GetSize(); i++)
  {
    const char *ln = m_text.Get(i);
    if (ln) mxcol = wdl_max(mxcol, WDL_utf8_get_charlen(ln));
  }

  return mxcol;
}

int PHO_SuperEditor::GetMaxLength(int cury) const
{
  int mxcol = 0;

  if (cury < m_text.GetSize())
  {
    const char *ln = m_text.Get(cury);
    if (ln) mxcol = WDL_utf8_get_charlen(ln);
  }

  return mxcol;
}

void PHO_SuperEditor::ToggleCRLF()
{
  m_crlf = !m_crlf;
}

void PHO_SuperEditor::ToggleSpace()
{
  m_space = !m_space;
}

bool PHO_SuperEditor::WantCRLF() const
{
  return m_crlf;
}

bool PHO_SuperEditor::WantSpaceForTab() const
{
  return m_space;
}

void PHO_SuperEditor::InsertCharacter(const char *str)
{
  bool del = false;
  if (HasSelectedText())
  {
    DeleteSelection();
    del = true;
  }

  int x = g_cursor->GetX();
  int y = g_cursor->GetY();
  if (y < m_text.GetSize())
  {
    const char *ln = m_text.Get(y);
    int len = WDL_utf8_get_charlen(ln);
    if (ln && x <= len)
    {
      m_strbuf.Set(ln);
      int pos = WDL_utf8_charpos_to_bytepos(ln, x);

      if (!del && m_replacetext && x + 1 <= len)
      {
        int epos = WDL_utf8_charpos_to_bytepos(ln, x + 1);
        m_strbuf.DeleteSub(pos, epos - pos);
      }

      m_strbuf.Insert(str, pos);
      m_text.Delete(y, true, free);
      m_text.Insert(y, strdup(m_strbuf.Get()));
      int len = WDL_utf8_get_charlen(str);
      g_cursor->SetX(x + len);
      TextChanged();
    }
  }
}

void PHO_SuperEditor::DeleteCharacter()
{
  int x = g_cursor->GetX();
  int y = g_cursor->GetY();
  if (y < m_text.GetSize())
  {
    if (HasSelectedText())
    {
      DeleteSelection();
    }
    else
    {
      const char *ln = m_text.Get(y);
      if (ln && x < WDL_utf8_get_charlen(ln))
      {
        int tail = WDL_utf8_charpos_to_bytepos(ln, x + 1);
        int head = WDL_utf8_charpos_to_bytepos(ln, x);
        int len = tail - head;
        m_strbuf.Set(ln);
        m_strbuf.DeleteSub(head, len);
        m_text.Delete(y, true, free);
        m_text.Insert(y, strdup(m_strbuf.Get()));
        TextChanged();
      }
      else if (ln && x == WDL_utf8_get_charlen(ln) && y + 1 < m_text.GetSize())
      {
        const char *nextln = m_text.Get(y + 1);
        if (nextln)
        {
          m_strbuf.Set(ln);
          m_strbuf.Append(nextln);
          m_text.Delete(y, true, free);
          m_text.Insert(y, strdup(m_strbuf.Get()));
          m_text.Delete(y + 1, true, free);
          TextChanged();
        }
      }
    }
  }
}

void PHO_SuperEditor::DeleteCharacterBackspace()
{
  int x = g_cursor->GetX();
  int y = g_cursor->GetY();
  if (y < m_text.GetSize())
  {
    if (HasSelectedText())
    {
      DeleteSelection();
    }
    else
    {
      const char *ln = m_text.Get(y);
      if (ln && x > 0 && x <= WDL_utf8_get_charlen(ln))
      {
        int tail = WDL_utf8_charpos_to_bytepos(ln, x);
        int head = WDL_utf8_charpos_to_bytepos(ln, x - 1);
        int len = tail - head;
        m_strbuf.Set(ln);
        m_strbuf.DeleteSub(head, len);
        m_text.Delete(y, true, free);
        m_text.Insert(y, strdup(m_strbuf.Get()));
        g_cursor->SetX(x - 1);
        TextChanged();
      }
      else if (ln && x == 0 && y > 0)
      {
        const char *prevln = m_text.Get(y - 1);
        if (prevln)
        {
          int newx = WDL_utf8_bytepos_to_charpos(prevln, (int)strlen(prevln));
          m_strbuf.Set(prevln);
          m_strbuf.Append(ln);
          m_text.Delete(y - 1, true, free);
          m_text.Insert(y - 1, strdup(m_strbuf.Get()));
          m_text.Delete(y, true, free);
          g_cursor->SetPosition(newx, y - 1);
          TextChanged();
        }
      }
    }
  }
}

void PHO_SuperEditor::DeleteWord()
{
  int x = g_cursor->GetX();
  int y = g_cursor->GetY();
  if (y < m_text.GetSize())
  {
    if (HasSelectedText())
    {
      DeleteSelection();
    }
    else
    {
      const char *ln = m_text.Get(y);
      if (ln && x < WDL_utf8_get_charlen(ln))
      {
        int tail = WDL_utf8_get_charlen(ln);
        int head = WDL_utf8_charpos_to_bytepos(ln, x);
        for (int i = head; i < WDL_utf8_get_charlen(ln); i++)
        {
          int pos = WDL_utf8_charpos_to_bytepos(ln, i);
          if (ln[pos] == ' ' || ln[pos] == '\t')
          {
            tail = pos + 1; break;
          }
        }
        int len = tail - head;
        m_strbuf.Set(ln);
        m_strbuf.DeleteSub(head, len);
        m_text.Delete(y, true, free);
        m_text.Insert(y, strdup(m_strbuf.Get()));
        TextChanged();
      }
      else if (ln && x == WDL_utf8_get_charlen(ln) && y + 1 < m_text.GetSize())
      {
        const char *nextln = m_text.Get(y + 1);
        if (nextln)
        {
          m_strbuf.Set(ln);
          m_strbuf.Append(nextln);
          m_text.Delete(y, true, free);
          m_text.Insert(y, strdup(m_strbuf.Get()));
          m_text.Delete(y + 1, true, free);
          TextChanged();
        }
      }
    }
  }
}

void PHO_SuperEditor::DeleteWordBackspace()
{
  int x = g_cursor->GetX();
  int y = g_cursor->GetY();
  if (y < m_text.GetSize())
  {
    if (HasSelectedText())
    {
      DeleteSelection();
    }
    else
    {
      const char *ln = m_text.Get(y);
      if (ln && x > 0 && x <= WDL_utf8_get_charlen(ln))
      {
        int tail = WDL_utf8_charpos_to_bytepos(ln, x);
        int head = 0;
        int idx = x - 1;
        for (; idx >= 0;)
        {
          int pos = WDL_utf8_charpos_to_bytepos(ln, idx);
          if (ln[pos] == ' ' || ln[pos] == '\t')
          {
            head = pos; break;
          }
          else idx--;
        }
        int len = tail - head;
        m_strbuf.Set(ln);
        m_strbuf.DeleteSub(head, len);
        m_text.Delete(y, true, free);
        m_text.Insert(y, strdup(m_strbuf.Get()));
        g_cursor->SetX(idx >= 0 ? idx : 0);
        TextChanged();
      }
      else if (ln && x == 0 && y > 0)
      {
        const char *prevln = m_text.Get(y - 1);
        if (prevln)
        {
          int newx = WDL_utf8_bytepos_to_charpos(prevln, (int)strlen(prevln));
          m_strbuf.Set(prevln);
          m_strbuf.Append(ln);
          m_text.Delete(y - 1, true, free);
          m_text.Insert(y - 1, strdup(m_strbuf.Get()));
          m_text.Delete(y, true, free);
          g_cursor->SetPosition(newx, y - 1);
          TextChanged();
        }
      }
    }
  }
}

void PHO_SuperEditor::JumpWordLeft()
{
  int x = g_cursor->GetX();
  int y = g_cursor->GetY();
  if (y < m_text.GetSize())
  {
    const char *ln = m_text.Get(y);
    if (ln && x >= 0 && x <= WDL_utf8_get_charlen(ln))
    {
      for (int i = x; i >= 0; i--)
      {
        int pos = WDL_utf8_charpos_to_bytepos(ln, i);
        if (i == 0)
        {
          if (i != g_cursor->GetX())
          {
            g_cursor->SetX(0); return;
          }
        }
        else if (ln[pos] == ' ' || ln[pos] == '\t' &&
          pos <= WDL_utf8_get_charlen(ln))
        {
          if (ln[pos + 1] != ' ' && ln[pos + 1] != '\t')
          {
            if (i + 1 == g_cursor->GetX()) continue;
            g_cursor->SetX(i + 1); return;
          }
        }
      }
      if (y > 0)
      {
        g_cursor->SetY(y - 1);
        g_cursor->SetX(WDL_utf8_get_charlen(m_text.Get(y - 1)));
        JumpWordLeft();
      }
    }
  }
}

void PHO_SuperEditor::JumpWordRight()
{
  int x = g_cursor->GetX();
  int y = g_cursor->GetY();
  if (y < m_text.GetSize())
  {
    int len = 0;
    const char *ln = m_text.Get(y);
    if (ln) len = WDL_utf8_get_charlen(ln);
    if (ln && x >= 0 && x <= len)
    {
      for (int i = x; i <= len; i++)
      {
        int pos = WDL_utf8_charpos_to_bytepos(ln, i);
        if (i == len)
        {
          if (i != g_cursor->GetX())
          {
            g_cursor->SetX(len); return;
          }
        }
        else if (ln[pos] == ' ' || ln[pos] == '\t' &&
          i + 1 <= len)
        {
          if (ln[pos + 1] != ' ' && ln[pos + 1] != '\t')
          {
            g_cursor->SetX(i + 1); return;
          }
        }
      }
      if (y + 1 < m_text.GetSize())
      {
        g_cursor->SetY(y + 1);
        g_cursor->SetX(0);
        JumpWordRight();
      }
    }
  }
}

void PHO_SuperEditor::NewLine()
{
  int x = g_cursor->GetX();
  int y = g_cursor->GetY();
  if (y < m_text.GetSize())
  {
    const char *ln = m_text.Get(y);
    if (ln && x <= WDL_utf8_get_charlen(ln))
    {
      WDL_FastString newln;
      int pos = WDL_utf8_charpos_to_bytepos(ln, x);
      newln.Set(&ln[pos]);
      m_strbuf.Set(ln);
      m_strbuf.DeleteSub(pos, m_strbuf.GetLength() - pos);
      m_text.Delete(y, true, free);
      m_text.Insert(y, strdup(m_strbuf.Get()));
      m_text.Insert(y + 1, strdup(newln.Get()));
      g_cursor->SetPosition(0, y + 1);
      TextChanged();
    }
  }
}

void PHO_SuperEditor::InsertTab()
{
  if (!HasSelectedText())
  {
    for (int i = 0; i < m_tabsize; i++)
    {
      InsertCharacter(m_space ? " " : "\t");
    }
  }
  else
  {
    int sx, sy, ex, ey;
    GetSelectStart(&sx, &sy);
    GetSelectEnd(&ex, &ey);

    if (sy == ey)
    {
      WDL_ASSERT(sy < m_text.GetSize());
      for (int i = 0; i < m_tabsize; i++)
      {
        InsertCharacter(m_space ? " " : "\t");
      }
    }
    else if (sy < ey)
    {
      WDL_ASSERT(sy < m_text.GetSize() && ey < m_text.GetSize());
      for (int i = sy; i <= ey; i++)
      {
        const char *ln = m_text.Get(i);
        if (ln)
        {
          m_strbuf.Set(ln);
          for (int j = 0; j < m_tabsize; j++)
          {
            m_strbuf.Insert(m_space ? " " : "\t", 0);
          }
          m_text.Delete(i, true, free);
          m_text.Insert(i, strdup(m_strbuf.Get()));
        }
      }
    }
  }
}

void PHO_SuperEditor::RemoveTab()
{
  if (!HasSelectedText())
  {
    for (int i = 0; i < m_tabsize; i++)
    {
      int x = g_cursor->GetX();
      int y = g_cursor->GetY();
      if (y < m_text.GetSize())
      {
        const char *ln = m_text.Get(y);
        if (ln)
        {
          if (x - 1 >= 0)
          {
            int pos = WDL_utf8_charpos_to_bytepos(ln, x - 1);
            if (ln[pos] == ' ' || ln[pos] == '\t')
            {
              DeleteCharacterBackspace();
            }
          }
        }
      }
    }
  }
  else
  {
    int sx, sy, ex, ey;
    GetSelectStart(&sx, &sy);
    GetSelectEnd(&ex, &ey);

    if (sy == ey)
    {
      for (int i = 0; i < m_tabsize; i++)
      {
        int x = sx;
        int y = sy;
        if (y < m_text.GetSize())
        {
          const char *ln = m_text.Get(y);
          if (ln)
          {
            if (x - 1 >= 0)
            {
              int pos = WDL_utf8_charpos_to_bytepos(ln, x - 1);
              if (ln[pos] == ' ' || ln[pos] == '\t')
              {
                m_strbuf.Set(ln);
                m_strbuf.DeleteSub(pos, 1);
                m_text.Delete(y, true, free);
                m_text.Insert(y, strdup(m_strbuf.Get()));
                int curx = g_cursor->GetX();
                sx--; ex--; curx--;
                g_cursor->SetX(curx);
                m_startcurx = sx;
                m_startcury = sy;
                m_endcurx = ex;
                m_endcury = ey;
              }
            }
          }
        }
      }
    }
    else if (sy < ey)
    {
      WDL_ASSERT(sy < m_text.GetSize() && ey < m_text.GetSize());
      for (int i = sy; i <= ey; i++)
      {
        const char *ln = m_text.Get(i);
        if (ln)
        {
          m_strbuf.Set(ln);
          for (int j = 0; j < m_tabsize; j++)
          {
            if (m_strbuf.Get()[0] == ' ' ||
              m_strbuf.Get()[0] == '\t')
            {
              m_strbuf.DeleteSub(0, 1);
            }
          }
          m_text.Delete(i, true, free);
          m_text.Insert(i, strdup(m_strbuf.Get()));
        }
      }
    }
  }
}

void PHO_SuperEditor::CutSelectedText()
{
  CopySelectedText();
  DeleteCharacter();
}

void PHO_SuperEditor::CopySelectedText()
{
  if (HasSelectedText())
  {
    int sx, sy, ex, ey;
    g_supereditor->GetSelectStart(&sx, &sy);
    g_supereditor->GetSelectEnd(&ex, &ey);

    m_buffer.Empty(true, free);

    if (sy == ey)
    {
      if (sy < m_text.GetSize())
      {
        const char *ln = m_text.Get(sy);
        if (ln && sx <= WDL_utf8_get_charlen(ln))
        {
          WDL_String strbuf(ln);
          int sxpos = WDL_utf8_charpos_to_bytepos(ln, sx);
          int expos = WDL_utf8_charpos_to_bytepos(ln, ex);
          int len = expos - sxpos;
          strbuf.SetLen(len);
          memcpy(strbuf.Get(), &ln[sxpos], len * sizeof(char));
          m_buffer.Add(strdup(strbuf.Get()));
        }
      }
    }
    else if (sy < ey)
    {
      if (sy < m_text.GetSize())
      {
        const char *ln = m_text.Get(sy);
        if (ln && sx <= WDL_utf8_get_charlen(ln))
        {
          WDL_String strbuf(ln);
          int sxpos = WDL_utf8_charpos_to_bytepos(ln, sx);
          int len = strbuf.GetLength() - sxpos;
          strbuf.SetLen(len);
          memcpy(strbuf.Get(), &ln[sxpos], len * sizeof(char));
          m_buffer.Add(strdup(strbuf.Get()));
        }
      }

      for (int i = sy + 1; i < ey; i++)
      {
        const char *ln = m_text.Get(i);
        if (ln)
        {
          m_buffer.Add(strdup(ln));
        }
      }

      if (ey < m_text.GetSize())
      {
        const char *ln = m_text.Get(ey);
        if (ln && ex <= WDL_utf8_get_charlen(ln))
        {
          WDL_String strbuf(ln);
          int expos = WDL_utf8_charpos_to_bytepos(ln, ex);
          strbuf.SetLen(expos);
          memcpy(strbuf.Get(), ln, expos * sizeof(char));
          m_buffer.Add(strdup(strbuf.Get()));
        }
      }
    }
  }
}

void PHO_SuperEditor::PasteSelectedText()
{
  if (m_buffer.GetSize())
  {
    int x, y;
    g_cursor->GetPosition(&x, &y);

    if (y < m_text.GetSize())
    {
      const char *ln = m_text.Get(y);
      if (ln)
      {
        m_strbuf.Set(ln);
        int pos = WDL_utf8_charpos_to_bytepos(ln, x);
        m_strbuf.Insert(m_buffer.Get(0), pos);
        m_text.Delete(y, true, free);
        m_text.Insert(y, strdup(m_strbuf.Get()));

        int ny = y;
        for (int i = 1; i < m_buffer.GetSize(); i++, ny++)
        {
          m_text.Insert(y + i, strdup(m_buffer.Get(i)));
        }

        int len = WDL_utf8_get_charlen(m_buffer.Get(m_buffer.GetSize() - 1));
        g_cursor->SetPosition(ny == y ? x + len : len, ny);
      }
    }
  }
}

void PHO_SuperEditor::ToggleReplaceCharacter()
{
  m_replacetext = !m_replacetext;
}

bool PHO_SuperEditor::HasReplaceCharacter() const
{
  return m_replacetext;
}

void PHO_SuperEditor::FindText(const char *a, bool reverse, bool sof)
{
  int x = g_cursor->GetX();
  int y = g_cursor->GetY();

  if (!reverse)
  {
    if (y < m_text.GetSize())
    {
      m_textfind.Set(a);
      m_textreplace.Set("");
      if (sof) y = 0;
      for (int i = y; i < m_text.GetSize(); i++)
      {
        const char *ln = m_text.Get(i);
        if (ln)
        {
          int len = WDL_utf8_get_charlen(a);
          for (int j = i == y ? x : 0; j <= WDL_utf8_get_charlen(ln) - len; j++)
          {
            int pos = WDL_utf8_charpos_to_bytepos(ln, j);
            if (!strncmp(a, &ln[pos], len))
            {
              TextSelection(true);
              SetSelectStart(j, i);
              SetSelectEnd(j + len, i);
              g_cursor->SetPosition(j, i);
              return;
            }
          }
        }
      }
    }

    if (HasSelectedText())
    {
      int ex, ey;
      GetSelectEnd(&ex, &ey);
      g_cursor->SetPosition(ex, ey);
    }
  }
  else
  {
    if (y >= 0)
    {
      m_textfind.Set(a);
      m_textreplace.Set("");
      for (int i = y; i >= 0; i--)
      {
        const char *ln = m_text.Get(i);
        if (ln)
        {
          int len = WDL_utf8_get_charlen(a);
          int start = WDL_utf8_get_charlen(ln) - len;
          for (int j = i == y ? x : start; j >= 0; j--)
          {
            int pos = WDL_utf8_charpos_to_bytepos(ln, j);
            if (!strncmp(a, &ln[pos], len))
            {
              TextSelection(true);
              SetSelectStart(j, i);
              SetSelectEnd(j + len, i);
              g_cursor->SetPosition(j, i);
              return;
            }
          }
        }
      }
    }

    if (HasSelectedText())
    {
      int sx, sy;
      GetSelectStart(&sx, &sy);
      g_cursor->SetPosition(sx, sy);
    }
  }
}

void PHO_SuperEditor::FindNext()
{
  if (m_textfind.GetLength())
  {
    int len = WDL_utf8_get_charlen(m_textfind.Get());
    int x = g_cursor->GetX();
    int y = g_cursor->GetY();
    if (y < m_text.GetSize())
    {
      const char *ln = m_text.Get(y);
      if (ln)
      {
        x = wdl_min(x + len, WDL_utf8_get_charlen(ln));
        g_cursor->SetX(x);
        FindText(m_textfind.Get(), false, false);
      }
    }
  }
}

void PHO_SuperEditor::FindPrevious()
{
  if (m_textfind.GetLength())
  {
    int len = WDL_utf8_get_charlen(m_textfind.Get());
    int x = g_cursor->GetX();
    int y = g_cursor->GetY();
    if (y < m_text.GetSize())
    {
      x = wdl_max(x - len, 0);
      g_cursor->SetX(x);
      FindText(m_textfind.Get(), true, false);
    }
  }
}

void PHO_SuperEditor::ReplaceText(const char *a, const char *b, bool reverse, bool sof, bool all)
{
  int x = g_cursor->GetX();
  int y = g_cursor->GetY();

  if (!reverse)
  {
    if (y < m_text.GetSize())
    {
      m_textfind.Set(a);
      m_textreplace.Set(b);
      if (sof) y = 0;
      for (int i = y; i < m_text.GetSize(); i++)
      {
        const char *ln = m_text.Get(i);
        if (ln)
        {
          int len = WDL_utf8_get_charlen(a);
          for (int j = i == y ? x : 0; j <= WDL_utf8_get_charlen(ln) - len; j++)
          {
            int pos = WDL_utf8_charpos_to_bytepos(ln, j);
            if (!strncmp(a, &ln[pos], len))
            {
              TextSelection(true);
              SetSelectStart(j, i);
              SetSelectEnd(j + len, i);
              g_cursor->SetPosition(j, i);

              DeleteCharacter();
              WDL_FastString character;
              for (int k = 0; k < WDL_utf8_get_charlen(m_textreplace.Get()); k++)
              {
                int start = WDL_utf8_charpos_to_bytepos(m_textreplace.Get(), k);
                int end = WDL_utf8_charpos_to_bytepos(m_textreplace.Get(), k + 1);
                character.SetRaw(&m_textreplace.Get()[start], end - start);
                InsertCharacter(character.Get());
              }

              if (!all) return;
            }
          }
        }
      }
    }
  }
  else
  {
    if (y >= 0)
    {
      m_textfind.Set(a);
      m_textreplace.Set(b);
      for (int i = y; i >= 0; i--)
      {
        const char *ln = m_text.Get(i);
        if (ln)
        {
          int len = WDL_utf8_get_charlen(a);
          int start = WDL_utf8_get_charlen(ln) - len;
          for (int j = i == y ? x : start; j >= 0; j--)
          {
            int pos = WDL_utf8_charpos_to_bytepos(ln, j);
            if (!strncmp(a, &ln[pos], len))
            {
              TextSelection(true);
              SetSelectStart(j, i);
              SetSelectEnd(j + len, i);
              g_cursor->SetPosition(j, i);

              DeleteCharacter();
              WDL_FastString character;
              for (int k = 0; k < WDL_utf8_get_charlen(m_textreplace.Get()); k++)
              {
                int start = WDL_utf8_charpos_to_bytepos(m_textreplace.Get(), k);
                int end = WDL_utf8_charpos_to_bytepos(m_textreplace.Get(), k + 1);
                character.SetRaw(&m_textreplace.Get()[start], end - start);
                InsertCharacter(character.Get());
              }

              if (!all) return;
            }
          }
        }
      }
    }
  }
}

void PHO_SuperEditor::ReplaceNext()
{
  if (m_textfind.GetLength() && m_textreplace.GetLength())
  {
    ReplaceText(m_textfind.Get(), m_textreplace.Get(), false, false, false);
  }
}

void PHO_SuperEditor::ReplacePrevious()
{
  if (m_textfind.GetLength() && m_textreplace.GetLength())
  {
    ReplaceText(m_textfind.Get(), m_textreplace.Get(), true, false, false);
  }
}

void PHO_SuperEditor::ReplaceAll()
{
  if (m_textfind.GetLength() && m_textreplace.GetLength())
  {
    g_cursor->SetPosition(0, 0);
    ReplaceText(m_textfind.Get(), m_textreplace.Get(), true, true, true);
  }
}

bool PHO_SuperEditor::IsFindEnabled() const
{
  return m_textfind.GetLength() > 0;
}

bool PHO_SuperEditor::IsReplaceEnabled() const
{
  return m_textreplace.GetLength() > 0;
}

void PHO_SuperEditor::TextChanged()
{
  if (!m_textchanged)
  {
    WDL_FastString str;
    str.SetFormatted(2048, "%s - *%s", g_mainwnd->GetOrigTitle(), m_fn.get_filepart());
    SetWindowText(g_mainwnd->Handle(), str.Get());
    m_textchanged = true;
  }
}

void PHO_SuperEditor::DeleteSelection()
{
  int sx, sy, ex, ey;
  GetSelectStart(&sx, &sy);
  GetSelectEnd(&ex, &ey);

  if (sy == ey)
  {
    WDL_ASSERT(sy < m_text.GetSize());
    const char *ln = m_text.Get(sy);
    if (ln)
    {
      WDL_ASSERT(ex > sx);
      const int pos = sx > 0 ? WDL_utf8_charpos_to_bytepos(ln, sx) : 0;
      const int len = WDL_utf8_charpos_to_bytepos(ln, ex) - pos;
      WDL_ASSERT((pos + len) <= (int)strlen(ln));
      m_strbuf.Set(ln);
      m_strbuf.DeleteSub(pos, len);
      m_text.Delete(sy, true, free);
      m_text.Insert(sy, strdup(m_strbuf.Get()));
      g_cursor->SetPosition(sx, sy);
      TextSelection(false);
      TextChanged();
    }
  }
  else if (sy < ey)
  {
    WDL_ASSERT(sy < m_text.GetSize() && ey < m_text.GetSize());
    const char *sln = m_text.Get(sy);
    if (sln)
    {
      const int pos = sx > 0 ? WDL_utf8_charpos_to_bytepos(sln, sx) : 0;
      const int len = (int)strlen(sln) - pos;
      WDL_ASSERT((pos + len) <= (int)strlen(sln));
      m_strbuf.Set(sln);
      m_strbuf.DeleteSub(pos, len);
      m_text.Delete(sy, true, free);
      m_text.Insert(sy, strdup(m_strbuf.Get()));
    }

    const int delta = (ey - 1) - (sy + 1);

    for (int i = 0; i < delta; i++)
    {
      m_text.Delete(sy + 1, true, free); ey--;
    }

    const char *eln = m_text.Get(ey);
    if (eln)
    {
      const int len = ex > 0 ? WDL_utf8_charpos_to_bytepos(eln, ex) : 0;
      m_strbuf.Set(m_text.Get(sy));
      m_strbuf.Append(&eln[len]);
      m_text.Delete(sy, true, free);
      m_text.Insert(sy, strdup(m_strbuf.Get()));
      m_text.Delete(ey, true, free);
    }

    g_cursor->SetPosition(sx, sy);
    TextSelection(false);
    TextChanged();
  }
}

void PHO_SuperEditor::StartThread()
{
  WDL_ASSERT(!m_threads.GetSize());

  m_killthread = false;

  m_threads.Resize(1);
  //m_threads.Resize(RHEA_GetCPUCores());
  for (int i = 0; i < m_threads.GetSize(); i++)
  {
    ThreadDetails *rt = m_threads.Get();
    rt[i].thread = (HANDLE)_beginthreadex(NULL, 0,
      ThreadFunction, (void *)this, 0, &rt[i].id);

    SetThreadPriority(rt[i].thread, THREAD_PRIORITY_NORMAL);
  }

  m_running = true;
}

void PHO_SuperEditor::StopThread()
{
  m_killthread = true;
  for (int i = 0; i < m_threads.GetSize(); i++)
  {
    ThreadDetails *rt = m_threads.Get();
    WaitForSingleObject(rt[i].thread, INFINITE);
    CloseHandle(rt[i].thread);
    m_threads.Resize(0);
  }

  m_running = false;
}

bool PHO_SuperEditor::IsRunning() const
{
  return m_running;
}

int PHO_SuperEditor::Run()
{
  return 1;
}

unsigned int WINAPI PHO_SuperEditor::ThreadFunction(void *arg)
{
#if defined(_WIN32)
  CoInitialize(NULL);
#endif

  PHO_SuperEditor *self = (PHO_SuperEditor *)arg;

  if (WDL_NORMALLY(self))
  {
    self->m_killthread = false;

    while (!self->m_killthread)
    {
      self->m_mutex.Enter();
      while (!self->Run());
      self->m_mutex.Leave();
      Sleep(50);
    }
  }

#if defined(_WIN32)
  CoUninitialize();
#endif

  return 0;
}
