// Copyright (c) 2023 Giorgos Vougioukas
//
// The license can be found in the LICENSE file.

#ifndef _PHOEBE_SUPER_EDITOR_H_
#define _PHOEBE_SUPER_EDITOR_H_

#include "phoebe/definitions.h"
#include "phoebe/preferences.h"

#include "WDL/mutex.h"
#include "WDL/heapbuf.h"
#include "WDL/ptrlist.h"
#include "WDL/wdlstring.h"
#include "WDL/lice/lice.h"
#include "WDL/lice/lice_text.h"

class PHO_SuperEditor
{
public:
  PHO_SuperEditor();
  ~PHO_SuperEditor();

  void New();
  void Open(const char *filename);
  bool IsOpen() const;
  void Save();
  void SaveAs(const char *filename);

  void MoveLeft();
  void MoveRight();
  void MoveUp();
  void MoveDown();
  void MoveStart();
  void MoveEnd();
  void MoveTop();
  void MoveBottom();
  void MovePageUp();
  void MovePageDown();
  void TextSelection(bool active);
  bool HasSelectedText() const;
  void GetSelectStart(int *curx, int *cury) const;
  void GetSelectEnd(int *curx, int *cury) const;
  void SetSelectStart(int curx, int cury);
  void SetSelectEnd(int curx, int cury);
  void MoveTextUp();
  void MoveTextDown();

  const char *GetTextRegion(int line, int column) const;

  void InitialState();
  void ShowOffsetLines();
  void ShowLines();
  bool HasOffsetLines() const;
  bool HasLines() const;

  void GotoLine(int index);
  void GotoLineOffset(int index);

  int GetMaxLine() const;
  int GetMaxColumn() const;
  int GetMaxLength(int cury) const;

  void ToggleCRLF();
  void ToggleSpace();
  bool WantCRLF() const;
  bool WantSpaceForTab() const;

  void InsertCharacter(const char *str);
  void DeleteCharacter();
  void DeleteCharacterBackspace();
  void DeleteWord();
  void DeleteWordBackspace();
  void JumpWordLeft();
  void JumpWordRight();
  void NewLine();
  void InsertTab();
  void RemoveTab();
  void CutSelectedText();
  void CopySelectedText();
  void PasteSelectedText();
  void ToggleReplaceCharacter();
  bool HasReplaceCharacter() const;

  void FindText(const char *a, bool reverse, bool sof);
  void FindNext();
  void FindPrevious();
  void ReplaceText(const char *a, const char *b, bool reverse, bool sof, bool all);
  void ReplaceNext();
  void ReplacePrevious();
  void ReplaceAll();
  bool IsFindEnabled() const;
  bool IsReplaceEnabled() const;

private:
  void TextChanged();
  void DeleteSelection();
  void StartThread();
  void StopThread();
  bool IsRunning() const;
  static unsigned int WINAPI ThreadFunction(void *arg);
  int Run();

  struct ThreadDetails
  {
    unsigned int id;
    HANDLE thread;
  };

  WDL_TypedBuf<ThreadDetails> m_threads;
  bool m_killthread;
  WDL_Mutex m_mutex;
  bool m_running;

  WDL_FastString m_strbuf;

  WDL_PtrList<char> m_text;
  WDL_PtrList<char> m_buffer;

  bool m_offsetlines;
  bool m_lines;

  bool m_seltext;
  int m_headcurx;
  int m_headcury;
  int m_startcurx;
  int m_startcury;
  int m_endcurx;
  int m_endcury;
  bool m_crlf;
  bool m_space;
  int m_tabsize;

  WDL_FastString m_fn;
  bool m_textchanged;
  bool m_replacetext;

  WDL_FastString m_textfind;
  WDL_FastString m_textreplace;
};

#endif // _PHOEBE_SUPER_EDITOR_H_
