// Copyright (c) 2023 Giorgos Vougioukas
//
// The license can be found in the LICENSE file.

#include "phoebe/super_editor_wnd.h"
#include "phoebe/super_editor.h"
#include "phoebe/preferences.h"
#include "phoebe/main_wnd.h"
#include "phoebe/cursor.h"

#include "WDL/wingui/scrollbar/coolscroll.h"
#include "WDL/wdlutf8.h"

static const char *s_courier = "Courier";
static const char *s_couriernew = "Courier New";
static const char *s_consolas = "Consolas";

static LOGFONT lf =
{
#if defined(_WIN32)
  14, 0, 0, 0, FW_NORMAL, FALSE, FALSE, FALSE, DEFAULT_CHARSET,
  OUT_DEFAULT_PRECIS, CLIP_DEFAULT_PRECIS, DEFAULT_QUALITY, DEFAULT_PITCH,
  "Consolas"
#else
  11, 0, 0, 0, FW_NORMAL, FALSE, FALSE, FALSE, DEFAULT_CHARSET,
  OUT_DEFAULT_PRECIS, CLIP_DEFAULT_PRECIS, DEFAULT_QUALITY, DEFAULT_PITCH,
  "Arial"
#endif
};

static WNDPROC PrevSuperEditorWndProc = NULL;
static HWND s_supereditorwnd = NULL;

PHO_SuperEditorWnd::PHO_SuperEditorWnd(HWND hwnd)
  : m_hwnd(hwnd)
  , m_bm(NULL)
  , m_font(NULL)
  , m_textheight(0)
  , m_textpage(0)
  , m_letterheight(0)
  , m_showwhitespace(false)
  , m_lastx(0)
  , m_lasty(0)
  , m_lastoffx(0)
  , m_lastoffy(0)
  , m_lastmx(0)
  , m_topln(0)
  , m_bg(0)
  , m_fg(0)
  , m_at(0)
{}

PHO_SuperEditorWnd::~PHO_SuperEditorWnd()
{
  if (m_font) delete m_font;
  if (m_bm) delete m_bm;
}

void PHO_SuperEditorWnd::ToggleWhitespace()
{
  m_showwhitespace = !m_showwhitespace; AskForFullRender();
}

void PHO_SuperEditorWnd::AskForFullRender()
{
  m_lastx = 0;
  m_lasty = 0;
  m_lastoffx = 0;
  m_lastoffy = 0;
  m_lastmx = 0;
}

void PHO_SuperEditorWnd::OnCreate(WPARAM wparam, LPARAM lparam)
{
  m_showwhitespace = g_preferences->ShowWhitespace();
  int text_font = g_preferences->GetTextFont();
  int text_font_height = g_preferences->GetTextFontHeight();
  m_bg = g_preferences->GetBackgroundColor();
  m_fg = g_preferences->GetTextColor();
  m_at = g_preferences->GetActiveTextColor();

  switch (text_font)
  {
  case 1:
    {
      int mx = (int)strlen(s_courier);
      WDL_ASSERT(mx > 0 && mx < 32);
      for (int i = 0; i < mx; i++)
      {
        lf.lfFaceName[i] = s_courier[i];
      }
      lf.lfFaceName[mx] = '\0';
      lf.lfHeight = text_font_height;
    } break;
  case 2:
    {
      int mx = (int)strlen(s_couriernew);
      WDL_ASSERT(mx > 0 && mx < 32);
      for (int i = 0; i < mx; i++)
      {
        lf.lfFaceName[i] = s_couriernew[i];
      }
      lf.lfFaceName[mx] = '\0';
      lf.lfHeight = text_font_height;
    } break;
  case 3:
    {
      int mx = (int)strlen(s_consolas);
      WDL_ASSERT(mx > 0 && mx < 32);
      for (int i = 0; i < mx; i++)
      {
        lf.lfFaceName[i] = s_consolas[i];
      }
      lf.lfFaceName[mx] = '\0';
      lf.lfHeight = text_font_height;
    } break;
  }

  m_bm = new LICE_SysBitmap();
  m_font = new LICE_CachedFont();
  m_font->SetFromHFont(CreateFontIndirect(&lf), LICE_FONT_FLAG_OWNS_HFONT);

  InitializeCoolSB(m_hwnd);

  RECT r;
  GetWindowRect(m_hwnd, &r);
  ScreenToClient(m_hwnd, (LPPOINT)&r);
  ScreenToClient(m_hwnd, ((LPPOINT)&r)+1);

  m_bm->resize(r.right - r.left, r.bottom - r.top);

  SCROLLINFO vsi =
  {
    sizeof(vsi),
    SIF_PAGE | SIF_POS | SIF_RANGE,
    0,
    m_textheight,
    m_textpage,
    0,
  };

  CoolSB_SetScrollInfo(m_hwnd, SB_VERT, &vsi, TRUE);
  CoolSB_ShowScrollBar(m_hwnd, SB_VERT, FALSE);

  PrevSuperEditorWndProc = (WNDPROC)SetWindowLongPtrW(m_hwnd,
    GWLP_WNDPROC, (LONG_PTR)NewSuperEditorWndProc);
    s_supereditorwnd = m_hwnd;

  SetTimer(m_hwnd, PHO_UPDATE_CURSOR, PHO_UPDATE_CURSOR_MS, NULL);
}

void PHO_SuperEditorWnd::OnDestroy(WPARAM wparam, LPARAM lparam)
{
  KillTimer(m_hwnd, PHO_UPDATE_CURSOR);
  UninitializeCoolSB(m_hwnd);
}

void PHO_SuperEditorWnd::OnSize(WPARAM wparam, LPARAM lparam)
{
  if (wparam != SIZE_MINIMIZED)
  {
    m_resize.onResize();

    RECT r;
    GetClientRect(m_hwnd, &r);
    int w = r.right - r.left;
    int h = r.bottom - r.top;
    m_bm->resize(w, h);

    if (!CoolSB_IsCoolScrollEnabled(m_hwnd))
    {
      CoolSB_ShowScrollBar(m_hwnd, SB_VERT, TRUE);
    }

    AskForFullRender();
    InvalidateRect(m_hwnd, NULL, FALSE);
  }
}

void PHO_SuperEditorWnd::OnCommand(WPARAM wparam, LPARAM lparam)
{}

void PHO_SuperEditorWnd::OnPaint(WPARAM wparam, LPARAM lparam)
{
  RECT r;
  GetClientRect(m_hwnd, &r);
  int w = r.right - r.left;
  int h = r.bottom - r.top;

  PAINTSTRUCT ps;
  HDC dc = BeginPaint(m_hwnd, &ps);

  if (m_bm && g_supereditor)
  {
    LICE_Clear(m_bm, LICE_RGBA((m_bg >> 16) & 0xff, (m_bg >> 8) & 0xff, m_bg & 0xff, 255));

    RECT let;
    m_font->DrawText(NULL, "V", 1, &let, DT_CALCRECT);
    g_cursor->SingleLetter(&let);
    g_cursor->ViewSurface(&r);

    m_textheight = g_supereditor->GetMaxLine();
    m_textpage = g_cursor->GetViewSurfaceLines();
    m_letterheight = 1;

    int pos = CoolSB_GetScrollPos(m_hwnd, SB_VERT);
    SCROLLINFO vsi =
    {
      sizeof(vsi),
      SIF_PAGE|SIF_POS|SIF_RANGE,
      0,
      m_textheight,
      m_textpage,
      pos,
    };

    CoolSB_SetScrollInfo(m_hwnd, SB_VERT, &vsi, TRUE);

    bool vis; RECT cur;
    g_cursor->GetCursor(&vis, &cur);

    int curx, cury;
    g_cursor->GetDeltaPosition(&curx, &cury);
    m_view.CursorPosition(curx, cury);

    if (vis)
    {
      if (WDL_unlikely(g_supereditor->HasReplaceCharacter()))
      {
        LICE_FillRect(m_bm, cur.left, cur.top, cur.right - cur.left, cur.bottom - cur.top,
          LICE_RGBA((m_at >> 16) & 0xff, (m_at >> 8) & 0xff, m_at & 0xff, 255), 0.6f);
      }
      else
      {
        LICE_Line(m_bm, cur.left, cur.top, cur.left, cur.bottom,
          LICE_RGBA((m_at >> 16) & 0xff, (m_at >> 8) & 0xff, m_at & 0xff, 255));
      }
    }

    int lw = g_cursor->GetWidth();
    int lh = g_cursor->GetHeight();

    RECT lnr;
    lnr.left = 0;
    lnr.top = 0;
    lnr.right = r.right;
    lnr.bottom = lh;

    const int offx = m_view.GetOffsetX();
    const int offy = m_view.GetOffsetY();
    const int lines = g_cursor->GetViewSurfaceLines();
    const int offsetstart = offy - g_cursor->GetY();
    const bool offln = g_supereditor->HasOffsetLines();
    const bool regln = g_supereditor->HasLines();
    m_topln = offy;

    const bool samepos = m_lastx == g_cursor->GetX() &&
      m_lasty == g_cursor->GetY() && m_lastoffx == offx &&
      m_lastoffy == offy && m_lastmx == m_textheight &&
      m_lastmx > 1;
    const int bottomline = lines - 1;
    int currentline = 0;
    bool drawtext = true;

    for (int i = 0, j = offsetstart; i < lines; i++, j++)
    {
      if (samepos && j && !offln && !regln && i < bottomline)
      {
        drawtext = false;
      }
      else if (samepos && !j && !offln && !regln && i < bottomline)
      {
        drawtext = true; currentline = i;
      }
      else
      {
        drawtext = true;
      }

      if (drawtext)
      {
        m_font->SetBkMode(TRANSPARENT);
        m_font->SetTextColor(LICE_RGBA((m_fg >> 16) & 0xff, (m_fg >> 8) & 0xff, m_fg & 0xff, 255));
        m_strbuf.Set(g_supereditor->GetTextRegion(i + offy, offx));

        for (int k = 0; k < WDL_utf8_get_charlen(m_strbuf.Get()); k++)
        {
          const int pos = WDL_utf8_charpos_to_bytepos(m_strbuf.Get(), k);
          char *character = m_strbuf.Get();
          if (character[pos] == '\t') character[pos] = ' ';
        }

        m_font->DrawText(m_bm, m_strbuf.Get(), m_strbuf.GetLength(), &lnr, 0);
      }

      if (m_showwhitespace && drawtext)
      {
        m_strbuf.Set(g_supereditor->GetTextRegion(i + offy, offx));

        RECT sc;
        sc.left = lnr.left;
        sc.top = lnr.top;
        sc.right = sc.left + lw;
        sc.bottom = sc.top + lh;

        for (int k = 0; k < WDL_utf8_get_charlen(m_strbuf.Get()); k++)
        {
          const int pos = WDL_utf8_charpos_to_bytepos(m_strbuf.Get(), k);
          const char *character = m_strbuf.Get();

          if (character[pos] == ' ')
          {
            int pixx = sc.left + lw / 2;
            int pixy = sc.top + lh / 2;
            LICE_PutPixel(m_bm, pixx, pixy,
              LICE_RGBA((m_at >> 16) & 0xff, (m_at >> 8) & 0xff, m_at & 0xff, 255),
              0.6f, LICE_BLIT_MODE_COPY);
            //m_font->SetTextColor(LICE_RGBA(128, 128, 255, 153));
            //m_font->DrawText(m_bm, "\xc2\xb7", -1, &sc, 0);
          }
          else if (character[pos] == '\t')
          {
            m_font->SetTextColor(LICE_RGBA((m_at >> 16) & 0xff,
              (m_at >> 8) & 0xff, m_at & 0xff, 153));
            m_font->DrawText(m_bm, "\xe2\x86\x92", -1, &sc, 0);
          }

          sc.left += lw;
          sc.right += lw;
        }
      }

      if (offln || regln)
      {
        if (offln)
        {
          if (j > 0) m_strbuf.SetFormatted(64, "%+8d  ", j);
          else m_strbuf.SetFormatted(64, "%8d  ", j);
        }
        else if (regln)
        {
          m_strbuf.SetFormatted(64, "%8d  ", i + offy + 1);
        }

        RECT clnr;
        m_font->DrawText(m_bm, m_strbuf.Get(), m_strbuf.GetLength(), &clnr, DT_CALCRECT);

        int linew = lnr.right - lnr.left;
        int numw = clnr.right - clnr.left;
        clnr.left =  (linew / 2) - (numw / 2);
        clnr.top = lnr.top;
        clnr.right = clnr.left + numw;
        clnr.bottom = lnr.bottom;

        m_font->SetBkMode(OPAQUE);
        m_font->SetBkColor(LICE_RGBA(0, 0, 0, 255));
        m_font->SetTextColor(LICE_RGBA(128, 128, 255, 255));
        if (offln && !j) m_font->SetTextColor(LICE_RGBA(255, 0, 0, 255));
        else if (regln && !j) m_font->SetTextColor(LICE_RGBA(255, 0, 0, 255));
        m_font->DrawText(m_bm, m_strbuf.Get(), m_strbuf.GetLength(), &clnr, 0);
        LICE_Line(m_bm, lnr.left, lnr.bottom, lnr.right, lnr.bottom,
          LICE_RGBA((m_at >> 16) & 0xff, (m_at >> 8) & 0xff, m_at & 0xff, 255), 0.2f);
      }
      else if (i == lines - 1)
      {
        RECT geom;
        bool crlf = g_supereditor->WantCRLF();
        bool space = g_supereditor->WantSpaceForTab();
        m_strbuf.SetFormatted(64, "  %d : %d %s %s  ", g_cursor->GetX() + 1,
          g_cursor->GetY() + 1, crlf ? "CRLF" : "LF", space ? "SPACE" : "TAB");
        m_font->DrawText(m_bm, m_strbuf.Get(), m_strbuf.GetLength(), &geom, DT_CALCRECT);

        int geomw = geom.right - geom.left;
        int geomh = geom.bottom - geom.top;
        geom.left = r.right - geomw;
        geom.top = lnr.top;
        geom.right = geom.left + geomw;
        geom.bottom = geom.top + geomh;
        m_font->SetBkMode(OPAQUE);
        m_font->SetBkColor(LICE_RGBA((m_bg >> 16) & 0xff, (m_bg >> 8) & 0xff, m_bg & 0xff, 255));
        m_font->SetTextColor(LICE_RGBA((m_at >> 16) & 0xff, (m_at >> 8) & 0xff, m_at & 0xff, 255));
        m_font->DrawText(m_bm, m_strbuf.Get(), m_strbuf.GetLength(), &geom, 0);
      }

      lnr.left = 0;
      lnr.top += lh + 1;
      lnr.right = r.right;
      lnr.bottom = lnr.top + lh;
    }

    if (g_supereditor->HasSelectedText())
    {
      int sx, sy, ex, ey;
      g_supereditor->GetSelectStart(&sx, &sy);
      g_supereditor->GetSelectEnd(&ex, &ey);

      sx -= offx;
      sy -= offy;
      ex -= offx;
      ey -= offy;

      if (sy == ey)
      {
        LICE_FillRect(m_bm, sx * lw, sy * (lh + 1), ex * lw - sx * lw, lnr.bottom - lnr.top,
          LICE_RGBA((m_at >> 16) & 0xff, (m_at >> 8) & 0xff, m_at & 0xff, 255), 0.4f);
      }
      else if (sy < ey)
      {
        LICE_FillRect(m_bm, sx * lw, sy * (lh + 1), lnr.right - sx * lw, lnr.bottom - lnr.top,
          LICE_RGBA((m_at >> 16) & 0xff, (m_at >> 8) & 0xff, m_at & 0xff, 255), 0.4f);

        for (int i = sy + 1; i < ey; i++)
        {
          LICE_FillRect(m_bm, 0, i * (lh + 1), lnr.right - lnr.left, lnr.bottom - lnr.top,
            LICE_RGBA((m_at >> 16) & 0xff, (m_at >> 8) & 0xff, m_at & 0xff, 255), 0.4f);
        }

        LICE_FillRect(m_bm, 0, ey * (lh + 1), ex * lw - lnr.left, lnr.bottom - lnr.top,
          LICE_RGBA((m_at >> 16) & 0xff, (m_at >> 8) & 0xff, m_at & 0xff, 255), 0.4f);
      }
    }

    if (samepos && !offln && !regln)
    {
      BitBlt(dc, r.left, currentline * (lh + 1), w, lh + 1, m_bm->getDC(), 0, currentline * (lh + 1), SRCCOPY);
      BitBlt(dc, r.left, bottomline * (lh + 1), w, lh + 1, m_bm->getDC(), 0, bottomline * (lh + 1), SRCCOPY);
    }
    else
    {
      BitBlt(dc, r.left, r.top, w, h, m_bm->getDC(), 0, 0, SRCCOPY);

      if (offln || regln)
      {
        AskForFullRender();
      }
      else
      {
        if (g_supereditor->HasSelectedText())
        {
          AskForFullRender();
        }
        else
        {
          m_lastx = g_cursor->GetX();
          m_lasty = g_cursor->GetY();
          m_lastoffx = m_view.GetOffsetX();
          m_lastoffy = m_view.GetOffsetY();
          m_lastmx = g_supereditor->GetMaxLine();
        }
      }
    }
  }
  EndPaint(m_hwnd, &ps);
}

void PHO_SuperEditorWnd::OnLButtonDown(WPARAM wparam, LPARAM lparam)
{
  g_supereditor->TextSelection(false);
  AskForFullRender();
  InvalidateRect(m_hwnd, NULL, FALSE);

  SetCapture(m_hwnd);

  POINT pt;
  pt.x = GET_X_LPARAM(lparam);
  pt.y = GET_Y_LPARAM(lparam);

  bool vis; RECT c;
  g_cursor->GetCursor(&vis, &c);
  int w = c.right - c.left;
  int h = c.bottom - c.top;

  int mxln = g_supereditor->GetMaxLine();
  int y = wdl_clamp(((int)(pt.y / (h + 1)) + m_topln), 0, mxln);
  int x = wdl_clamp((int)(pt.x / w), 0,
    g_supereditor->GetMaxLength(y));

  g_supereditor->SetSelectStart(x, y);
  g_supereditor->SetSelectEnd(x, y);
  g_supereditor->TextSelection(true);
  g_cursor->SetPosition(x, y);
}

void PHO_SuperEditorWnd::OnLButtonUp(WPARAM wparam, LPARAM lparam)
{
  POINT pt;
  pt.x = GET_X_LPARAM(lparam);
  pt.y = GET_Y_LPARAM(lparam);

  if (GetCapture())
  {
    bool vis; RECT c;
    g_cursor->GetCursor(&vis, &c);
    int w = c.right - c.left;
    int h = c.bottom - c.top;

    int mxln = g_supereditor->GetMaxLine();
    int mxcl = g_supereditor->GetMaxColumn();

    int y = wdl_clamp(((int)(pt.y / (h + 1)) + m_topln), 0, mxln);
    int x = wdl_clamp((int)(pt.x / w), 0,
      g_supereditor->GetMaxLength(y));

    g_supereditor->SetSelectEnd(x, y);
    g_supereditor->TextSelection(true);

    int orisx, orisy, oriex, oriey;
    g_supereditor->GetSelectStart(&orisx, &orisy);
    g_supereditor->GetSelectEnd(&oriex, &oriey);
    g_cursor->SetPosition(x, y);

    if (orisy == oriey && orisx == oriex)
    {
      g_supereditor->TextSelection(false);
      AskForFullRender();
      InvalidateRect(m_hwnd, NULL, FALSE);
    }

    ReleaseCapture();
  }
}

void PHO_SuperEditorWnd::OnMouseMove(WPARAM wparam, LPARAM lparam)
{
  SetCursor(LoadCursor(NULL, IDC_IBEAM));

  if (GetCapture())
  {
    POINT pt;
    pt.x = GET_X_LPARAM(lparam);
    pt.y = GET_Y_LPARAM(lparam);

    bool vis; RECT c;
    g_cursor->GetCursor(&vis, &c);
    int w = c.right - c.left;
    int h = c.bottom - c.top;

    int mxln = g_supereditor->GetMaxLine();
    int mxcl = g_supereditor->GetMaxColumn();

    int ln = wdl_clamp((int)(pt.y / (h + 1)), 0, mxln);
    int y = wdl_clamp(ln + m_topln, 0, mxln);
    int x = wdl_clamp((int)(pt.x / w), 0,
      g_supereditor->GetMaxLength(y));

    g_supereditor->SetSelectEnd(x, y);
    g_supereditor->TextSelection(true);
    g_cursor->SetPosition(x, y);
    AskForFullRender();
    InvalidateRect(m_hwnd, NULL, FALSE);
  }
}

void PHO_SuperEditorWnd::OnTimer(WPARAM wparam, LPARAM lparam)
{
  if (wparam == PHO_UPDATE_CURSOR)
  {
    InvalidateRect(m_hwnd, NULL, FALSE);
  }
}

void PHO_SuperEditorWnd::OnVScroll(WPARAM wparam, LPARAM lparam)
{
  SCROLLINFO vsi = { sizeof(SCROLLINFO), SIF_POS|SIF_PAGE|SIF_RANGE|SIF_TRACKPOS, 0 };
  CoolSB_GetScrollInfo(m_hwnd, SB_VERT, &vsi);
  switch (LOWORD(wparam))
  {
  case SB_THUMBTRACK:
    vsi.nPos = vsi.nTrackPos;
    break;
  case SB_LINEUP:
    vsi.nPos -= m_letterheight;
    break;
  case SB_LINEDOWN:
    vsi.nPos += m_letterheight;
    break;
  case SB_PAGEUP:
    vsi.nPos -= (int)vsi.nPage;
    break;
  case SB_PAGEDOWN:
    vsi.nPos += (int)vsi.nPage;
    break;
  case SB_TOP:
    vsi.nPos = 0;
    break;
  case SB_BOTTOM:
    vsi.nPos = vsi.nMax-(int)vsi.nPage;
    break;
  }
  if (vsi.nPos > vsi.nMax-(int)vsi.nPage) vsi.nPos = vsi.nMax-(int)vsi.nPage;
  if (vsi.nPos < 0) vsi.nPos = 0;

  g_cursor->SetY(vsi.nPos);
  g_cursor->SetX(0);
  g_cursor->MoveToY(0);
  g_cursor->MoveToX(0);

  CoolSB_SetScrollPos(m_hwnd, SB_VERT, vsi.nPos, TRUE);
  AskForFullRender();
  InvalidateRect(m_hwnd, NULL, FALSE);
}

void PHO_SuperEditorWnd::OnKeyDown(WPARAM wparam, LPARAM lparam)
{
  switch (wparam)
  {
  case VK_HOME:
    {
      CoolSB_SetScrollPos(m_hwnd, SB_VERT, 0, TRUE);
      InvalidateRect(m_hwnd, NULL, FALSE);
    }
    break;
  case VK_END:
    {
      SCROLLINFO vsi = { sizeof(SCROLLINFO), SIF_POS|SIF_PAGE|SIF_RANGE|SIF_TRACKPOS, 0 };
      CoolSB_GetScrollInfo(m_hwnd, SB_VERT, &vsi);

      CoolSB_SetScrollPos(m_hwnd, SB_VERT, vsi.nMax, TRUE);
      InvalidateRect(m_hwnd, NULL, FALSE);
    }
    break;
  case VK_PRIOR:
    {
      SCROLLINFO vsi = { sizeof(SCROLLINFO), SIF_POS|SIF_PAGE|SIF_RANGE|SIF_TRACKPOS, 0 };
      CoolSB_GetScrollInfo(m_hwnd, SB_VERT, &vsi);

      int pos = vsi.nPos - vsi.nPage;
      if (pos < 0) pos = 0;

      CoolSB_SetScrollPos(m_hwnd, SB_VERT, pos, TRUE);
      InvalidateRect(m_hwnd, NULL, FALSE);
    }
    break;
  case VK_NEXT:
    {
      SCROLLINFO vsi = { sizeof(SCROLLINFO), SIF_POS|SIF_PAGE|SIF_RANGE|SIF_TRACKPOS, 0 };
      CoolSB_GetScrollInfo(m_hwnd, SB_VERT, &vsi);

      int pos = vsi.nPos + vsi.nPage;
      if (pos > vsi.nMax) pos = vsi.nMax;

      CoolSB_SetScrollPos(m_hwnd, SB_VERT, pos, TRUE);
      InvalidateRect(m_hwnd, NULL, FALSE);
    }
    break;
  case VK_UP:
    {
      SCROLLINFO vsi = { sizeof(SCROLLINFO), SIF_POS|SIF_PAGE|SIF_RANGE|SIF_TRACKPOS, 0 };
      CoolSB_GetScrollInfo(m_hwnd, SB_VERT, &vsi);

      int pos = vsi.nPos - m_letterheight;
      if (pos < 0) pos = 0;

      CoolSB_SetScrollPos(m_hwnd, SB_VERT, pos, TRUE);
      InvalidateRect(m_hwnd, NULL, FALSE);
    }
    break;
  case VK_DOWN:
    {
      SCROLLINFO vsi = { sizeof(SCROLLINFO), SIF_POS|SIF_PAGE|SIF_RANGE|SIF_TRACKPOS, 0 };
      CoolSB_GetScrollInfo(m_hwnd, SB_VERT, &vsi);

      int pos = vsi.nPos + m_letterheight;
      if (pos > vsi.nMax) pos = vsi.nMax;

      CoolSB_SetScrollPos(m_hwnd, SB_VERT, pos, TRUE);
      InvalidateRect(m_hwnd, NULL, FALSE);
    }
    break;
  }
}

void PHO_SuperEditorWnd::OnMouseWheel(WPARAM wparam, LPARAM lparam)
{
  if (GetCapture()) return;
  POINT p = { GET_X_LPARAM(lparam), GET_Y_LPARAM(lparam) };
  ScreenToClient(m_hwnd, &p);

  int l = (short)HIWORD(wparam);
  float pos = -l / 120.0f;

  SCROLLINFO vsi = { sizeof(SCROLLINFO), SIF_POS|SIF_PAGE|SIF_RANGE|SIF_TRACKPOS, 0 };
  CoolSB_GetScrollInfo(m_hwnd, SB_VERT, &vsi);

  pos *= 10; //(int)vsi.nPage;
  pos += vsi.nPos;

  const float mv = (float)(vsi.nMax - 10 /* (int)vsi.nPage */);
  if (pos > mv) pos = mv;
  if (pos < 0) pos = 0;

  CoolSB_SetScrollPos(m_hwnd, SB_VERT, (int)(pos + 0.5), TRUE);

  g_cursor->SetY((int)(pos + 0.5));
  g_cursor->SetX(0);
  g_cursor->MoveToY(0);
  g_cursor->MoveToX(0);

  AskForFullRender();
  InvalidateRect(m_hwnd, NULL, FALSE);
}

WDL_DLGRET PHO_SuperEditorWnd::NewSuperEditorWndProc(HWND hwnd, UINT msg, WPARAM wparam, LPARAM lparam)
{
  switch (msg)
  {
  case WM_KEYDOWN:
    {
      switch(wparam)
      {
      case VK_DELETE:
        {
          if (!(GetAsyncKeyState(VK_CONTROL) & 0x8000) &&
            !(GetAsyncKeyState(VK_SHIFT) & 0x8000) &&
            !(GetAsyncKeyState(VK_MENU) & 0x8000))
          {
            if (!GetCapture())
            {
              g_supereditor->DeleteCharacter();
            }
          }
        }
        break;
      case VK_INSERT:
        {
          if (!(GetAsyncKeyState(VK_CONTROL) & 0x8000) &&
            !(GetAsyncKeyState(VK_SHIFT) & 0x8000) &&
            !(GetAsyncKeyState(VK_MENU) & 0x8000))
          {
            if (!GetCapture())
            {
              g_supereditor->ToggleReplaceCharacter();
            }
          }
        }
        break;
      }
    }
    break;
  case WM_CHAR:
    {
      switch(wparam)
      {
      case VK_BACK:
        {
          if (!(GetAsyncKeyState(VK_CONTROL) & 0x8000) &&
            !(GetAsyncKeyState(VK_SHIFT) & 0x8000) &&
            !(GetAsyncKeyState(VK_MENU) & 0x8000))
          {
            if (!GetCapture())
            {
              g_supereditor->DeleteCharacterBackspace();
            }
          }
        }
        break;
      default:
        {
          wchar_t wc[2];
          wc[0] = (wchar_t)wparam;
          wc[1] = L'\0';
          char mb[32];
          WDL_WideToMBStr(mb, wc, sizeof(mb));
          g_supereditor->InsertCharacter(mb);
        }
        break;
      }
    }
    break;
  }

  return CallWindowProc(PrevSuperEditorWndProc, hwnd, msg, wparam, lparam);
}

WDL_DLGRET PHO_SuperEditorWnd::SuperEditorWndProc(HWND hwnd, UINT msg, WPARAM wparam, LPARAM lparam)
{
  g_supereditorwnd = (PHO_SuperEditorWnd *)GetWindowLongPtr(hwnd, GWLP_USERDATA);

#ifdef _WIN32
  if (!g_supereditorwnd && msg == WM_NCCREATE)
#else
  if (!g_supereditorwnd && msg == WM_CREATE)
#endif
  {
    g_supereditorwnd = new PHO_SuperEditorWnd(hwnd);
    SetWindowLongPtr(hwnd, GWLP_USERDATA, (INT_PTR)g_supereditorwnd);
#ifndef _WIN32
    //SWELL_SetClassName(hwnd, "waveform_superhost");
#endif
    if (g_supereditorwnd) g_supereditorwnd->OnCreate(wparam, lparam);
    return (g_supereditorwnd != NULL);
  }

#ifdef _WIN32
  if (g_supereditorwnd && msg == WM_NCDESTROY)
#else
  if (g_supereditorwnd && msg == WM_DESTROY)
#endif
  {
    g_supereditorwnd->OnDestroy(wparam, lparam);
    delete g_supereditorwnd; g_supereditorwnd = NULL;
  }

  if (WDL_likely(g_supereditorwnd))
  {
    return g_supereditorwnd->SuperEditorWndLoop(msg, wparam, lparam);
  }
  else
  {
    return 0;
  }
}

WDL_DLGRET PHO_SuperEditorWnd::SuperEditorWndLoop(UINT msg, WPARAM wparam, LPARAM lparam)
{
  switch (msg)
  {
#ifdef _WIN32
    case WM_NCCREATE: OnCreate(wparam, lparam); break;
    case WM_NCDESTROY: OnDestroy(wparam, lparam); break;
#else
    case WM_CREATE: OnCreate(wparam, lparam); break;
    case WM_DESTROY: OnDestroy(wparam, lparam); break;
#endif
    case WM_SIZE: OnSize(wparam, lparam); break;
    case WM_COMMAND: OnCommand(wparam, lparam); break;
    case WM_PAINT: OnPaint(wparam, lparam); break;
    case WM_LBUTTONDOWN: OnLButtonDown(wparam, lparam); break;
    case WM_LBUTTONUP: OnLButtonUp(wparam, lparam); break;
    case WM_MOUSEMOVE: OnMouseMove(wparam, lparam); break;
    case WM_TIMER: OnTimer(wparam, lparam); break;
    case WM_VSCROLL: OnVScroll(wparam, lparam); break;
    case WM_KEYDOWN: OnKeyDown(wparam, lparam); break;
    case WM_MOUSEWHEEL: OnMouseWheel(wparam, lparam); break;
  }

  return DefWindowProc(m_hwnd, msg, wparam, lparam);
}
