// Copyright (c) 2023 Giorgos Vougioukas
//
// The license can be found in the LICENSE file.

#include "rhea/antidote_input.h"
#include "rhea/preferences.h"
#include "rhea/media_input.h"
#include "rhea/hestia.h"

#include <math.h>

#include "WDL/assocarray.h"
#include "WDL/pcmfmtcvt.h"
#include "WDL/dirscan.h"
#include "WDL/fpcmp.h"

class RHEA_AntidoteRef
{
public:
  RHEA_AntidoteRef()
  {}

  ~RHEA_AntidoteRef()
  {
    WDL_ASSERT(!m_fc.GetSize());

    if (m_cache.GetLength())
    {
      RemoveAllFiles(m_cache.Get());

      for (int i = m_dirlist.GetSize() - 1; i >= 0; i--)
      {
        RemoveDirectory(m_dirlist.Get(i)->Get());
      }

      RemoveDirectory(m_cache.Get());

      m_dirlist.Empty(true);
      m_dellist.Empty(true);
    }
  }

  void Add(const char *filename)
  {
    if (!m_cache.GetLength())
    {
      m_cache.Set(g_setpath.Get());
      m_cache.Append("cache" WDL_DIRCHAR_STR);
    }

    if (m_fc.Exists(filename))
    {
      int cnt = m_fc.Get(filename);
      m_fc.Insert(filename, ++cnt);
    }
    else
    {
      m_fc.Insert(filename, 1);
    }
  }

  void Remove(const char *filename)
  {
    if (m_fc.Exists(filename))
    {
      int cnt = m_fc.Get(filename);
      if (!--cnt)
      {
        m_fc.Delete(filename);

        m_dellist.Add(new WDL_FastString(filename));

        bool loading = false;

        for (int i = 0; i < g_hestia->DeckCount(); i++)
        {
          if (g_mediainput->IsLoading(true, i)) loading = true;
        }

        for (int i = 0; i < g_hestia->SamplerCount(); i++)
        {
          if (g_mediainput->IsLoading(false, i)) loading = true;
        }

        if (!loading)
        {
          for (int i = 0; i < m_dellist.GetSize(); i++)
          {
            WDL_FastString *s = m_dellist.Get(i);

            if (s && !m_fc.Exists(s->Get()))
            {
              WDL_FileRead *fr = new WDL_NEW WDL_FileRead(s->Get());
              if (fr)
              {
                if (fr->IsOpen())
                {
                  delete fr; fr = NULL;
                  DeleteFile(s->Get());
                }
                else
                {
                  delete fr; fr = NULL;
                }
              }
#ifdef _DEBUG
              fr = new WDL_NEW WDL_FileRead(s->Get());
              WDL_ASSERT(fr && !fr->IsOpen());
              delete fr; fr = NULL;
#endif
            }
          }

          m_dellist.Empty(true);
        }
      }
      else
      {
        m_fc.Insert(filename, cnt);
      }
    }
  }

private:
  void RemoveAllFiles(const char *path)
  {
    WDL_DirScan dir;
    WDL_FastString fn;

    if (!dir.First(path))
    {
      do
      {
        if (dir.GetCurrentIsDirectory())
        {
          dir.GetCurrentFullFN(&fn);

          if (strcmp(fn.get_filepart(), ".") &&
              strcmp(fn.get_filepart(), ".."))
          {
            m_dirlist.Add(new WDL_FastString(fn.Get()));
            RemoveAllFiles(fn.Get());
          }
        }
        else
        {
          dir.GetCurrentFullFN(&fn);
          DeleteFile(fn.Get());
        }
      }
      while (!dir.Next());
    }
  }

  WDL_PtrList<WDL_FastString> m_dellist;
  WDL_PtrList<WDL_FastString> m_dirlist;
  WDL_StringKeyedArray<int> m_fc;
  WDL_FastString m_cache;
};

static RHEA_AntidoteRef s_antidoteref;

RHEA_AntidoteInput::RHEA_AntidoteInput(
  const char *filename,
  WDL_FileRead *fileread,
  const char *input_filename,
  int channels, double orig_srate)
  : m_fn(filename)
  , m_fr(fileread)
  , m_bitdepth(g_preferences->GetAntidoteBitDepth())
  , m_srate(g_preferences->GetAudioDeviceSamplerate())
  , m_ifn(input_filename)
  , m_ebur128(false)
  , m_ebur128downwardonly(false)
  , m_ebur128reference(0.0)
  , m_ebur128mode(0)
  , m_ebur128momentary(-HUGE_VAL)
  , m_ebur128shortterm(-HUGE_VAL)
  , m_ebur128momentarywindow(400)
  , m_ebur128shorttermwindow(3000)
  , m_ebur128integrated(-HUGE_VAL)
  , m_ebur128range(-HUGE_VAL)
  , m_ebur128samplepeak(-HUGE_VAL)
  , m_ebur128truepeak(-HUGE_VAL)
  , m_ebur128gain(-HUGE_VAL)
  , m_momentarywindow(0)
  , m_shorttermwindow(0)
  , m_eof(false)
  , m_position(0.0)
  , m_rev(false)
  , m_nch(channels)
  , m_orig_srate(orig_srate)
{
  WDL_ASSERT(m_fr->IsOpen());
  s_antidoteref.Add(m_fn.Get());
}

RHEA_AntidoteInput::~RHEA_AntidoteInput()
{
  if (m_fr) delete m_fr;
  s_antidoteref.Remove(m_fn.Get());
}

void RHEA_AntidoteInput::SetTitle(const char *title)
{
  m_title.Set(title);
}

void RHEA_AntidoteInput::SetArtist(const char *artist)
{
  m_artist.Set(artist);
}

const char *RHEA_AntidoteInput::GetTitle() const
{
  return m_title.Get();
}

const char *RHEA_AntidoteInput::GetArtist() const
{
  return m_artist.Get();
}

void RHEA_AntidoteInput::ebur128info(
  bool ebur128,
  bool ebur128downwardonly,
  double ebur128reference,
  int ebur128mode,
  double ebur128momentary,
  double ebur128shortterm,
  int ebur128momentarywindow,
  int ebur128shorttermwindow,
  double ebur128integrated,
  double ebur128range,
  double ebur128samplepeak,
  double ebur128truepeak,
  double ebur128gain,
  int momentarywindow,
  int shorttermwindow)
{
  m_ebur128 = ebur128;
  m_ebur128downwardonly = ebur128downwardonly;
  m_ebur128reference = ebur128reference;
  m_ebur128mode = ebur128mode;
  m_ebur128momentary = ebur128momentary;
  m_ebur128shortterm = ebur128shortterm;
  m_ebur128momentarywindow = ebur128momentarywindow;
  m_ebur128shorttermwindow = ebur128shorttermwindow;
  m_ebur128integrated = ebur128integrated;
  m_ebur128range = ebur128range;
  m_ebur128samplepeak = ebur128samplepeak;
  m_ebur128truepeak = ebur128truepeak;
  m_ebur128gain = ebur128gain;
  m_momentarywindow = momentarywindow;
  m_shorttermwindow = shorttermwindow;
}

const char *RHEA_AntidoteInput::GetType()
{
  return "ANTIDOTE";
}

const char *RHEA_AntidoteInput::GetFileName()
{
  return m_fn.Get();
}

int RHEA_AntidoteInput::GetChannels()
{
  return m_nch;
}

double RHEA_AntidoteInput::GetSampleRate()
{
  return m_srate;
}

double RHEA_AntidoteInput::GetLength()
{
  double len = 0.0;

  if (m_fr)
  {
    len = m_fr->GetSize() / (GetBitsPerSample() / 8)  / GetSampleRate() / GetChannels();
  }

  return len;
}

int RHEA_AntidoteInput::GetBitsPerSample()
{
  int bps = 0;

  switch (m_bitdepth)
  {
    case 0: bps = 16; break;
    case 1: bps = 24; break;
    case 2: bps = 32; break;
    case 3: bps = 32; break;
    case 4: bps = 64; break;
  }

  return bps;
}

double RHEA_AntidoteInput::GetPosition()
{
  return m_position;
}

void RHEA_AntidoteInput::Seek(double time)
{
  double st = time;
  double du = GetLength();

  if (WDL_DefinitelyLessThan(st, 0.0)) { st = 0.0; m_eof = false; }
  else if (WDL_DefinitelyGreaterThan(st, du)) { st = du; m_eof = true; }
  else { m_eof = false; }

  m_buffer_queue.Clear();

  WDL_INT64 timems = (WDL_INT64)(time * 1000);
  int csrate = (int)GetSampleRate() / 1000;
  WDL_INT64 pos = (WDL_INT64)(timems * (csrate * GetChannels() * (GetBitsPerSample() / 8)));
  m_fr->SetPosition(pos);

  m_position = time;
}

bool RHEA_AntidoteInput::IsReverse() const
{
  return m_rev;
}

void RHEA_AntidoteInput::SetReverse(bool state)
{
  m_rev = state;
}

int RHEA_AntidoteInput::GetSamples(SAM *buffer, int length)
{
  int ret = 0;

  while (m_buffer_queue.Available() < length && !m_eof)
  {
    int read = 0;
    switch (m_bitdepth)
    {
    case 0:
      {
        if (WDL_likely(!m_rev))
        {
          m_rawbuf.Resize(4096 * sizeof(short));
          read = m_fr->Read(m_rawbuf.Get(), m_rawbuf.GetSize());
          m_buffer.Resize(read / sizeof(short));
          pcmToDoubles(m_rawbuf.Get(), m_buffer.GetSize(), 16, 1, m_buffer.Get(), 1);
        }
        else
        {
          m_rawbuf.Resize(4096 * sizeof(short));
          if (m_fr->GetPosition() < m_rawbuf.GetSize()) m_rawbuf.Resize((int)m_fr->GetPosition());
          m_fr->SetPosition(m_fr->GetPosition() - m_rawbuf.GetSize());
          read = m_fr->Read(m_rawbuf.Get(), m_rawbuf.GetSize());
          m_fr->SetPosition(m_fr->GetPosition() - read);
          m_buffer.Resize(read / sizeof(short));
          pcmToDoubles(m_rawbuf.Get(), m_buffer.GetSize(), 16, 1, m_buffer.Get(), 1);
        }
      } break;
    case 1:
      {
        if (WDL_likely(!m_rev))
        {
          m_rawbuf.Resize(4096 * sizeof(char) * 3);
          read = m_fr->Read(m_rawbuf.Get(), m_rawbuf.GetSize());
          m_buffer.Resize(read / (sizeof(char) * 3));
          pcmToDoubles(m_rawbuf.Get(), m_buffer.GetSize(), 24, 1, m_buffer.Get(), 1);
        }
        else
        {
          m_rawbuf.Resize(4096 * sizeof(char) * 3);
          if (m_fr->GetPosition() < m_rawbuf.GetSize()) m_rawbuf.Resize((int)m_fr->GetPosition());
          m_fr->SetPosition(m_fr->GetPosition() - m_rawbuf.GetSize());
          read = m_fr->Read(m_rawbuf.Get(), m_rawbuf.GetSize());
          m_fr->SetPosition(m_fr->GetPosition() - read);
          m_buffer.Resize(read / (sizeof(char) * 3));
          pcmToDoubles(m_rawbuf.Get(), m_buffer.GetSize(), 24, 1, m_buffer.Get(), 1);
        }
      } break;
    case 2:
      {
        if (WDL_likely(!m_rev))
        {
          m_rawbuf.Resize(4096 * sizeof(int));
          read = m_fr->Read(m_rawbuf.Get(), m_rawbuf.GetSize());
          m_buffer.Resize(read / sizeof(int));
          pcmToDoubles(m_rawbuf.Get(), m_buffer.GetSize(), 32, 1, m_buffer.Get(), 1);
        }
        else
        {
          m_rawbuf.Resize(4096 * sizeof(int));
          if (m_fr->GetPosition() < m_rawbuf.GetSize()) m_rawbuf.Resize((int)m_fr->GetPosition());
          m_fr->SetPosition(m_fr->GetPosition() - m_rawbuf.GetSize());
          read = m_fr->Read(m_rawbuf.Get(), m_rawbuf.GetSize());
          m_fr->SetPosition(m_fr->GetPosition() - read);
          m_buffer.Resize(read / sizeof(int));
          pcmToDoubles(m_rawbuf.Get(), m_buffer.GetSize(), 32, 1, m_buffer.Get(), 1);
        }
      } break;
    case 3:
      {
        if (WDL_likely(!m_rev))
        {
          m_rawbuf.Resize(4096 * sizeof(float));
          read = m_fr->Read(m_rawbuf.Get(), m_rawbuf.GetSize());
          m_buffer.Resize(read / sizeof(float));
          float *flt = (float *)m_rawbuf.Get();
          for (int i = 0; i < m_buffer.GetSize(); i++)
          {
            m_buffer.Get()[i] = (SAM)flt[i];
          }
        }
        else
        {
          m_rawbuf.Resize(4096 * sizeof(float));
          if (m_fr->GetPosition() < m_rawbuf.GetSize()) m_rawbuf.Resize((int)m_fr->GetPosition());
          m_fr->SetPosition(m_fr->GetPosition() - m_rawbuf.GetSize());
          read = m_fr->Read(m_rawbuf.Get(), m_rawbuf.GetSize());
          m_fr->SetPosition(m_fr->GetPosition() - read);
          m_buffer.Resize(read / sizeof(float));
          float *flt = (float *)m_rawbuf.Get();
          for (int i = 0; i < m_buffer.GetSize(); i++)
          {
            m_buffer.Get()[i] = (SAM)flt[i];
          }
        }
      } break;
    case 4:
      {
        if (WDL_likely(!m_rev))
        {
          m_rawbuf.Resize(4096 * sizeof(double));
          read = m_fr->Read(m_rawbuf.Get(), m_rawbuf.GetSize());
          m_buffer.Resize(read / sizeof(double));
          double *dbl = (double *)m_rawbuf.Get();
          for (int i = 0; i < m_buffer.GetSize(); i++)
          {
            m_buffer.Get()[i] = (SAM)dbl[i];
          }
        }
        else
        {
          m_rawbuf.Resize(4096 * sizeof(double));
          if (m_fr->GetPosition() < m_rawbuf.GetSize()) m_rawbuf.Resize((int)m_fr->GetPosition());
          m_fr->SetPosition(m_fr->GetPosition() - m_rawbuf.GetSize());
          read = m_fr->Read(m_rawbuf.Get(), m_rawbuf.GetSize());
          m_fr->SetPosition(m_fr->GetPosition() - read);
          m_buffer.Resize(read / sizeof(double));
          double *dbl = (double *)m_rawbuf.Get();
          for (int i = 0; i < m_buffer.GetSize(); i++)
          {
            m_buffer.Get()[i] = (SAM)dbl[i];
          }
        }
      } break;
    }

    if (WDL_unlikely(m_rev))
    {
      int nch = GetChannels();

      for (int ch = 0; ch < nch; ch++)
      {
        for (int lo = ch, hi = m_buffer.GetSize() - nch + ch; lo < hi; lo += nch, hi -= nch)
        {
          SAM tmp = m_buffer.Get()[lo];
          m_buffer.Get()[lo] = m_buffer.Get()[hi];
          m_buffer.Get()[hi] = tmp;
        }
      }
    }

    m_buffer_queue.Add(m_buffer.Get(), m_buffer.GetSize());
    if (m_fr->GetPosition() == m_fr->GetSize() && !m_rev) m_eof = true;
    if (m_fr->GetPosition() == 0 && m_rev) return 0;
  }

  if (m_buffer_queue.Available() > length)
  {
    memcpy(buffer, m_buffer_queue.Get(), length * sizeof(SAM));
    ret = length;
    m_buffer_queue.Advance(length);
    m_buffer_queue.Compact();
  }
  else
  {
    memcpy(buffer, m_buffer_queue.Get(), m_buffer_queue.Available() * sizeof(SAM));
    ret = m_buffer_queue.Available();
    m_buffer_queue.Clear();
  }

  if (m_eof /* && !m_rev */ && (m_buffer_queue.GetSize() == 0))
  {
    m_position = GetLength();
    m_rev = false;
  }
  else if (m_eof && m_rev && (m_buffer_queue.GetSize() == 0))
  {
    //m_position = 0;
  }
  else
  {
    if (WDL_likely(!m_rev))
    {
      m_position += ret / GetSampleRate() / GetChannels();
    }
    else
    {
      m_position -= ret / GetSampleRate() / GetChannels();
    }
  }

  return ret;
}

bool RHEA_AntidoteInput::IsStreaming()
{
  return false;
}

int RHEA_AntidoteInput::Extended(int call, void *parm1, void *parm2, void *parm3)
{
  return 0;
}
