// Copyright (c) 2023 Giorgos Vougioukas
//
// The license can be found in the LICENSE file.

#include "rhea/hestia.h"
#include "rhea/preferences.h"
//#include "rhea/concurrency.h"

#include "WDL/fpcmp.h"

RHEA_Hestia::RHEA_Hestia()
  : m_dkillthread(false)
  , m_skillthread(false)
  , m_running(false)
  , m_dcb(NULL)
  , m_scb(NULL)
  , m_dwait(0)
  , m_swait(0)
  , m_ffrew(false)
  , m_shift(g_preferences->GetShift())
  , m_tempo(g_preferences->GetTempo())
  , m_altshift(g_preferences->GetAltShift())
  , m_alttempo(g_preferences->GetAltTempo())
{
  for (int i = 0; i < 4; i++)
  {
    m_decks.Add(new RHEA_Track);
  }

  for (int i = 0; i < 8; i++)
  {
    m_samplers.Add(new RHEA_Track);
  }

  StartThread();
}

RHEA_Hestia::~RHEA_Hestia()
{
  if (IsRunning()) StopThread();
  m_decks.Empty(true);
  m_samplers.Empty(true);
}

void RHEA_Hestia::Start(RHEA_MediaInputDeckCallback dcb, RHEA_MediaInputSamplerCallback scb)
{
  m_dcb = dcb;
  m_scb = scb;
}

void RHEA_Hestia::Stop()
{
  m_dcb = NULL;
  m_scb = NULL;
}

void RHEA_Hestia::AudioData(WDL_TypedBuf<SAM> *output, int frame_count, int nch)
{
  //WDL_MutexLock lock(&m_mutex);
  memset(output->Get(), 0, frame_count * sizeof(SAM));

  m_mixer.Setup(output, frame_count, nch);

  wdl_atomic_incr(&m_dwait);
  for (int i = 0; i < m_decks.GetSize(); i++)
  {
    RHEA_Track *trk = m_decks.Get(i);
    m_mixer.DeckTime(trk, i);
    m_mixer.DeckToOutput(trk, i);
  }
  wdl_atomic_decr(&m_dwait);

  wdl_atomic_incr(&m_swait);
  for (int i = 0; i < m_samplers.GetSize(); i++)
  {
    RHEA_Track *trk = m_samplers.Get(i);
    m_mixer.SamplerTime(trk, i);
    m_mixer.SamplerToOutput(trk, i);
  }
  wdl_atomic_decr(&m_swait);
}

int RHEA_Hestia::DeckCount() const { return m_decks.GetSize(); }
int RHEA_Hestia::SamplerCount() const { return m_samplers.GetSize(); }

double RHEA_Hestia::GetTimeSec(bool deck, int index) const
{
  if (deck)
  {
    WDL_ASSERT(index >= 0 && index < m_decks.GetSize());
    return m_decks.Get(index)->GetTimeSec();
  }
  else
  {
    WDL_ASSERT(index >= 0 && index < m_samplers.GetSize());
    return m_samplers.Get(index)->GetTimeSec();
  }
}

const char *RHEA_Hestia::GetTime(bool deck, int index) const
{
  if (deck)
  {
    WDL_ASSERT(index >= 0 && index < m_decks.GetSize());
    return m_decks.Get(index)->GetTime();
  }
  else
  {
    WDL_ASSERT(index >= 0 && index < m_samplers.GetSize());
    return m_samplers.Get(index)->GetTime();
  }
}

const char *RHEA_Hestia::GetTitle(bool deck, int index) const
{
  if (deck)
  {
    WDL_ASSERT(index >= 0 && index < m_decks.GetSize());
    return m_decks.Get(index)->GetTitle();
  }
  else
  {
    WDL_ASSERT(index >= 0 && index < m_samplers.GetSize());
    return m_samplers.Get(index)->GetTitle();
  }
}

const char *RHEA_Hestia::GetArtist(bool deck, int index) const
{
  if (deck)
  {
    WDL_ASSERT(index >= 0 && index < m_decks.GetSize());
    return m_decks.Get(index)->GetArtist();
  }
  else
  {
    WDL_ASSERT(index >= 0 && index < m_samplers.GetSize());
    return m_samplers.Get(index)->GetArtist();
  }
}

void RHEA_Hestia::ToggleActivate(bool deck, int index)
{
  if (deck)
  {
    WDL_ASSERT(index >= 0 && index < m_decks.GetSize());
    RHEA_Track *trk = m_decks.Get(index);
    if (trk)
    {
      bool act = !trk->IsActive();
      if (trk->IsLoaded())
      {
        trk->Active(act);
      }
    }
  }
  else
  {
    WDL_ASSERT(index >= 0 && index < m_samplers.GetSize());
    RHEA_Track *trk = m_samplers.Get(index);
    if (trk)
    {
      bool act = !trk->IsActive();
      if (trk->IsLoaded())
      {
        trk->Active(act);
      }
    }
  }
}

bool RHEA_Hestia::IsActive(bool deck, int index) const
{
  if (deck)
  {
    WDL_ASSERT(index >= 0 && index < m_decks.GetSize());
    return m_decks.Get(index)->IsActive();
  }
  else
  {
    WDL_ASSERT(index >= 0 && index < m_samplers.GetSize());
    return m_samplers.Get(index)->IsActive();
  }
}

void RHEA_Hestia::Eject(bool deck, int index)
{
  if (deck)
  {
    WDL_ASSERT(index >= 0 && index < m_decks.GetSize());
    RHEA_Track *trk = m_decks.Get(index);
    if (trk)
    {
      if (!trk->IsActive() && trk->IsLoaded())
      {
        trk->Eject(true);
      }
    }
  }
  else
  {
    WDL_ASSERT(index >= 0 && index < m_samplers.GetSize());
    RHEA_Track *trk = m_samplers.Get(index);
    if (trk)
    {
      if (!trk->IsActive() && trk->IsLoaded())
      {
        trk->Eject(true);
      }
    }
  }
}

void RHEA_Hestia::Rewind(bool deck, int index)
{
  if (deck)
  {
    WDL_ASSERT(index >= 0 && index < m_decks.GetSize());
    RHEA_Track *trk = m_decks.Get(index);
    if (trk)
    {
      if (!trk->IsActive() && !m_ffrew)
      {
        trk->SetShift(m_shift);
        trk->SetTempo(m_tempo);
        double tm = trk->GetTimeSec();
        trk->m_bq.Flush();
        trk->Seek(tm);
        trk->Reverse(true);
        trk->Active(true);
        m_ffrew = true;
      }
    }
  }
  else
  {
    WDL_ASSERT(index >= 0 && index < m_samplers.GetSize());
    RHEA_Track *trk = m_samplers.Get(index);
    if (trk)
    {
      if (!trk->IsActive() && !m_ffrew)
      {
        trk->SetShift(m_shift);
        trk->SetTempo(m_tempo);
        double tm = trk->GetTimeSec();
        trk->m_bq.Flush();
        trk->Seek(tm);
        trk->Reverse(true);
        trk->Active(true);
        m_ffrew = true;
      }
    }
  }
}

void RHEA_Hestia::FastForward(bool deck, int index)
{
  if (deck)
  {
    WDL_ASSERT(index >= 0 && index < m_decks.GetSize());
    RHEA_Track *trk = m_decks.Get(index);
    if (trk)
    {
      if (!trk->IsActive() && !m_ffrew)
      {
        trk->SetShift(m_shift);
        trk->SetTempo(m_tempo);
        double tm = trk->GetTimeSec();
        trk->m_bq.Flush();
        trk->Seek(tm);
        trk->Reverse(false);
        trk->Active(true);
        m_ffrew = true;
      }
    }
  }
  else
  {
    WDL_ASSERT(index >= 0 && index < m_samplers.GetSize());
    RHEA_Track *trk = m_samplers.Get(index);
    if (trk)
    {
      if (!trk->IsActive() && !m_ffrew)
      {
        trk->SetShift(m_shift);
        trk->SetTempo(m_tempo);
        double tm = trk->GetTimeSec();
        trk->m_bq.Flush();
        trk->Seek(tm);
        trk->Reverse(false);
        trk->Active(true);
        m_ffrew = true;
      }
    }
  }
}

void RHEA_Hestia::StopRewind(bool deck, int index)
{
  if (deck)
  {
    WDL_ASSERT(index >= 0 && index < m_decks.GetSize());
    RHEA_Track *trk = m_decks.Get(index);
    if (trk)
    {
      if (m_ffrew)
      {
        trk->SetShift(1.0);
        trk->SetTempo(1.0);
        trk->Active(false);
        trk->Reverse(false);
        double tm = trk->GetTimeSec();
        trk->m_bq.Flush();
        trk->Seek(tm);
        m_ffrew = false;
      }
    }
  }
  else
  {
    WDL_ASSERT(index >= 0 && index < m_samplers.GetSize());
    RHEA_Track *trk = m_samplers.Get(index);
    if (trk)
    {
      if (m_ffrew)
      {
        trk->SetShift(1.0);
        trk->SetTempo(1.0);
        trk->Active(false);
        trk->Reverse(false);
        double tm = trk->GetTimeSec();
        trk->m_bq.Flush();
        trk->Seek(tm);
        m_ffrew = false;
      }
    }
  }
}

void RHEA_Hestia::StopFastForward(bool deck, int index)
{
  if (deck)
  {
    WDL_ASSERT(index >= 0 && index < m_decks.GetSize());
    RHEA_Track *trk = m_decks.Get(index);
    if (trk)
    {
      if (m_ffrew)
      {
        trk->SetShift(1.0);
        trk->SetTempo(1.0);
        trk->Active(false);
        trk->Reverse(false);
        double tm = trk->GetTimeSec();
        trk->m_bq.Flush();
        trk->Seek(tm);
        m_ffrew = false;
      }
    }
  }
  else
  {
    WDL_ASSERT(index >= 0 && index < m_samplers.GetSize());
    RHEA_Track *trk = m_samplers.Get(index);
    if (trk)
    {
      if (m_ffrew)
      {
        trk->SetShift(1.0);
        trk->SetTempo(1.0);
        trk->Active(false);
        trk->Reverse(false);
        double tm = trk->GetTimeSec();
        trk->m_bq.Flush();
        trk->Seek(tm);
        m_ffrew = false;
      }
    }
  }
}

void RHEA_Hestia::AltRewind(bool deck, int index)
{
  if (deck)
  {
    WDL_ASSERT(index >= 0 && index < m_decks.GetSize());
    RHEA_Track *trk = m_decks.Get(index);
    if (trk)
    {
      if (!trk->IsActive() && !m_ffrew)
      {
        trk->SetShift(m_altshift);
        trk->SetTempo(m_alttempo);
        double tm = trk->GetTimeSec();
        trk->m_bq.Flush();
        trk->Seek(tm);
        trk->Reverse(true);
        trk->Active(true);
        m_ffrew = true;
      }
    }
  }
  else
  {
    WDL_ASSERT(index >= 0 && index < m_samplers.GetSize());
    RHEA_Track *trk = m_samplers.Get(index);
    if (trk)
    {
      if (!trk->IsActive() && !m_ffrew)
      {
        trk->SetShift(m_altshift);
        trk->SetTempo(m_alttempo);
        double tm = trk->GetTimeSec();
        trk->m_bq.Flush();
        trk->Seek(tm);
        trk->Reverse(true);
        trk->Active(true);
        m_ffrew = true;
      }
    }
  }
}

void RHEA_Hestia::AltFastForward(bool deck, int index)
{
  if (deck)
  {
    WDL_ASSERT(index >= 0 && index < m_decks.GetSize());
    RHEA_Track *trk = m_decks.Get(index);
    if (trk)
    {
      if (!trk->IsActive() && !m_ffrew)
      {
        trk->SetShift(m_altshift);
        trk->SetTempo(m_alttempo);
        double tm = trk->GetTimeSec();
        trk->m_bq.Flush();
        trk->Seek(tm);
        trk->Reverse(false);
        trk->Active(true);
        m_ffrew = true;
      }
    }
  }
  else
  {
    WDL_ASSERT(index >= 0 && index < m_samplers.GetSize());
    RHEA_Track *trk = m_samplers.Get(index);
    if (trk)
    {
      if (!trk->IsActive() && !m_ffrew)
      {
        trk->SetShift(m_altshift);
        trk->SetTempo(m_alttempo);
        double tm = trk->GetTimeSec();
        trk->m_bq.Flush();
        trk->Seek(tm);
        trk->Reverse(false);
        trk->Active(true);
        m_ffrew = true;
      }
    }
  }
}

void RHEA_Hestia::GotoStart(bool deck, int index)
{
  if (deck)
  {
    WDL_ASSERT(index >= 0 && index < m_decks.GetSize());
    RHEA_Track *trk = m_decks.Get(index);
    if (trk)
    {
      if (!trk->IsActive())
      {
        trk->Seek(0.0);
        trk->m_bq.Flush();
        trk->Reverse(false);
      }
    }
  }
  else
  {
    WDL_ASSERT(index >= 0 && index < m_samplers.GetSize());
    RHEA_Track *trk = m_samplers.Get(index);
    if (trk)
    {
      if (!trk->IsActive())
      {
        trk->Seek(0.0);
        trk->m_bq.Flush();
        trk->Reverse(false);
      }
    }
  }
}

void RHEA_Hestia::GotoEnd(bool deck, int index)
{
  if (deck)
  {
    WDL_ASSERT(index >= 0 && index < m_decks.GetSize());
    RHEA_Track *trk = m_decks.Get(index);
    if (trk)
    {
      if (!trk->IsActive())
      {
        trk->Seek(trk->GetLength());
        trk->m_bq.Flush();
        trk->Reverse(true);
      }
    }
  }
  else
  {
    WDL_ASSERT(index >= 0 && index < m_samplers.GetSize());
    RHEA_Track *trk = m_samplers.Get(index);
    if (trk)
    {
      if (!trk->IsActive())
      {
        trk->Seek(trk->GetLength());
        trk->m_bq.Flush();
        trk->Reverse(true);
      }
    }
  }
}

bool RHEA_Hestia::IsReverse(bool deck, int index) const
{
  if (deck)
  {
    WDL_ASSERT(index >= 0 && index < m_decks.GetSize());
    return m_decks.Get(index)->IsReverse();
  }
  else
  {
    WDL_ASSERT(index >= 0 && index < m_samplers.GetSize());
    return m_samplers.Get(index)->IsReverse();
  }
}

void RHEA_Hestia::ToggleReverse(bool deck, int index)
{
  if (deck)
  {
    WDL_ASSERT(index >= 0 && index < m_decks.GetSize());
    double tm = m_decks.Get(index)->GetTimeSec();
    double len = m_decks.Get(index)->GetLength();
    if (WDL_DefinitelyGreaterThan(tm, 1.0)
      && WDL_DefinitelyLessThan(tm, len - 1.0))
    {
      bool rev = !m_decks.Get(index)->IsReverse();
      m_decks.Get(index)->Reverse(rev);
    }
  }
  else
  {
    WDL_ASSERT(index >= 0 && index < m_samplers.GetSize());
    double tm = m_samplers.Get(index)->GetTimeSec();
    double len = m_samplers.Get(index)->GetLength();
    if (WDL_DefinitelyGreaterThan(tm, 1.0)
      && WDL_DefinitelyLessThan(tm, len - 1.0))
    {
      bool rev = !m_samplers.Get(index)->IsReverse();
      m_samplers.Get(index)->Reverse(rev);
    }
  }
}

int RHEA_Hestia::RepeatCount(bool deck, int index) const
{
  if (deck)
  {
    WDL_ASSERT(index >= 0 && index < m_decks.GetSize());
    return m_decks.Get(index)->RepeatCount();
  }
  else
  {
    WDL_ASSERT(index >= 0 && index < m_samplers.GetSize());
    return m_samplers.Get(index)->RepeatCount();
  }
}

void RHEA_Hestia::Repeat(bool deck, int index, bool infinite)
{
  if (deck)
  {
    WDL_ASSERT(index >= 0 && index < m_decks.GetSize());
    m_decks.Get(index)->Repeat(infinite);
  }
  else
  {
    WDL_ASSERT(index >= 0 && index < m_samplers.GetSize());
    m_samplers.Get(index)->Repeat(infinite);
  }
}

void RHEA_Hestia::ClearRepeat(bool deck, int index)
{
  if (deck)
  {
    WDL_ASSERT(index >= 0 && index < m_decks.GetSize());
    m_decks.Get(index)->ClearRepeat();
  }
  else
  {
    WDL_ASSERT(index >= 0 && index < m_samplers.GetSize());
    m_samplers.Get(index)->ClearRepeat();
  }
}

double RHEA_Hestia::GetLength(bool deck, int index) const
{
  if (deck)
  {
    WDL_ASSERT(index >= 0 && index < m_decks.GetSize());
    return m_decks.Get(index)->GetLength();
  }
  else
  {
    WDL_ASSERT(index >= 0 && index < m_samplers.GetSize());
    return m_samplers.Get(index)->GetLength();
  }

  return 0.0;
}

void RHEA_Hestia::StartThread()
{
  WDL_ASSERT(!m_dthreads.GetSize());
  WDL_ASSERT(!m_sthreads.GetSize());

  int priority = g_preferences->GetRenderSystemPriority();

  m_dkillthread = false;
  m_dthreads.Resize(1);
  //m_dthreads.Resize(RHEA_GetCPUCores());
  for (int i = 0; i < m_dthreads.GetSize(); i++)
  {
    ThreadDetails *rt = m_dthreads.Get();
    rt[i].thread = (HANDLE)_beginthreadex(NULL, 0,
      ThreadFunctionForDecks, (void *)this, 0, &rt[i].id);
    SetThreadPriority(rt[i].thread, priority);
  }

  m_skillthread = false;
  m_sthreads.Resize(1);
  //m_sthreads.Resize(RHEA_GetCPUCores());
  for (int i = 0; i < m_sthreads.GetSize(); i++)
  {
    ThreadDetails *rt = m_sthreads.Get();
    rt[i].thread = (HANDLE)_beginthreadex(NULL, 0,
      ThreadFunctionForSamplers, (void *)this, 0, &rt[i].id);
    SetThreadPriority(rt[i].thread, priority);
  }

  m_running = true;
}

void RHEA_Hestia::StopThread()
{
  m_dkillthread = true;
  for (int i = 0; i < m_dthreads.GetSize(); i++)
  {
    ThreadDetails *rt = m_dthreads.Get();
    WaitForSingleObject(rt[i].thread, INFINITE);
    CloseHandle(rt[i].thread);
    m_dthreads.Resize(0);
  }

  m_skillthread = true;
  for (int i = 0; i < m_sthreads.GetSize(); i++)
  {
    ThreadDetails *rt = m_sthreads.Get();
    WaitForSingleObject(rt[i].thread, INFINITE);
    CloseHandle(rt[i].thread);
    m_sthreads.Resize(0);
  }

  m_running = false;
}

bool RHEA_Hestia::IsRunning() const
{
  return m_running;
}

int RHEA_Hestia::RunDecks()
{
  if (m_dcb) m_dcb(&m_decks);

  for (int i = 0; i < m_decks.GetSize(); i++)
  {
    RHEA_Track *trk = m_decks.Get(i);

    if (trk)
    {
      if (trk->IsDrained())
      {

      }

      while (trk->WantMore() && !m_dwait)
      {
        //if (deck->track.IsPlaybackReverse())
        //{
        //  if (!deck->track.DoBufferingReverse()) break;
        //}
        //else
        {
          if (!trk->DoBuffering()) break;
        }
      }
    }
  }

  return 1;
}

int RHEA_Hestia::RunSamplers()
{
  if (m_scb) m_scb(&m_samplers);

  for (int i = 0; i < m_samplers.GetSize(); i++)
  {
    RHEA_Track *trk = m_samplers.Get(i);

    if (trk)
    {
      if (trk->IsDrained())
      {

      }

      while (trk->WantMore() && !m_swait)
      {
        //if (deck->track.IsPlaybackReverse())
        //{
        //  if (!deck->track.DoBufferingReverse()) break;
        //}
        //else
        {
          if (!trk->DoBuffering()) break;
        }
      }
    }
  }

  return 1;
}

unsigned int WINAPI RHEA_Hestia::ThreadFunctionForDecks(void *arg)
{
#if defined(_WIN32)
  CoInitialize(NULL);
#endif

  RHEA_Hestia *self = (RHEA_Hestia *)arg;

  if (WDL_NORMALLY(self))
  {
    int sleepstep = g_preferences->GetRenderSystemSleepStep();

    self->m_dkillthread = false;

    while (!self->m_dkillthread)
    {
      self->m_dmutex.Enter();
      while (!self->RunDecks());
      self->m_dmutex.Leave();
      Sleep(sleepstep);
    }
  }

#if defined(_WIN32)
  CoUninitialize();
#endif

  return 0;
}

unsigned int WINAPI RHEA_Hestia::ThreadFunctionForSamplers(void *arg)
{
#if defined(_WIN32)
  CoInitialize(NULL);
#endif

  RHEA_Hestia *self = (RHEA_Hestia *)arg;

  if (WDL_NORMALLY(self))
  {
    int sleepstep = g_preferences->GetRenderSystemSleepStep();

    self->m_skillthread = false;

    while (!self->m_skillthread)
    {
      self->m_smutex.Enter();
      while (!self->RunSamplers());
      self->m_smutex.Leave();
      Sleep(sleepstep);
    }
  }

#if defined(_WIN32)
  CoUninitialize();
#endif

  return 0;
}
