// Copyright (c) 2021 Giorgos Vougioukas
//
// The license can be found in the LICENSE file.

#include "rhea/main_wnd.h"

#define RHEA_MAIN_SCREEN_UPDATE 300
#define RHEA_MAIN_SCREEN_UPDATE_MS 50
#define RHEA_CLEAR_EDIT 310
#define RHEA_CLEAR_EDIT_MS 10
#define RHEA_CLEAR_INFO 320
#define RHEA_CLEAR_INFO_MS 1000
#define RHEA_ORIGINAL_TITLE 330
#define RHEA_ORIGINAL_TITLE_MS 1200
#define RHEA_DISABLE_CONTROLS 340
#define RHEA_DISABLE_CONTROLS_MS 50

#include "rhea/app_info.h"
#include "rhea/plugin.h"
#include "rhea/about_wnd.h"
#include "rhea/changelog_wnd.h"
#include "rhea/license_wnd.h"
#include "rhea/preferences.h"
#include "rhea/preferences_wnd.h"
#include "rhea/search_wnd.h"
#include "rhea/media_input.h"
#include "rhea/waveform_wnd.h"
#include "rhea/hestia.h"
#include "rhea/peaks.h"
#include "rhea/database.h"
#include "rhea/query.h"

#include "WDL/fpcmp.h"
#include "WDL/wdlstring.h"
#include "WDL/time_precise.h"
#include "WDL/filebrowse.h"
#include "WDL/lice/lice.h"
#include "WDL/wingui/virtwnd.h"
#include "WDL/wingui/scrollbar/coolscroll.h"
#include "WDL/wdlutf8.h"
#include "WDL/time_precise.h"
#include "WDL/dirscan.h"

#define RHEA_OPEN_MEDIA_FILES \
  "All supported media files\0*.flac;*.mp3;*.ogg;*.wv;*.wav;*.aiff;*.opus\0" \
  "FLAC audio files (*.flac)\0*.flac\0" \
  "MPEG audio files (*.mp3)\0*.mp3\0" \
  "Vorbis audio files (*.ogg)\0*.ogg\0" \
  "WavPack audio files (*.wv)\0*.wv\0" \
  "WAV audio files (*.wav)\0*.wav\0" \
  "AIFF audio files (*.aiff)\0*.aiff\0" \
  "Opus audio files (*.opus)\0*.opus\0" \
  "All files (*.*)\0*.*\0\0"
#define RHEA_OPEN_MEDIA_FILE_DEFEXT "mp3"

static LOGFONT lf =
{
#if defined(_WIN32)
  14, 0, 0, 0, FW_NORMAL, FALSE, FALSE, FALSE, DEFAULT_CHARSET,
  OUT_DEFAULT_PRECIS, CLIP_DEFAULT_PRECIS, DEFAULT_QUALITY, DEFAULT_PITCH,
  "Arial"
#else
  11, 0, 0, 0, FW_NORMAL, FALSE, FALSE, FALSE, DEFAULT_CHARSET,
  OUT_DEFAULT_PRECIS, CLIP_DEFAULT_PRECIS, DEFAULT_QUALITY, DEFAULT_PITCH,
  "Arial"
#endif
};

RHEA_MainWnd::RHEA_MainWnd()
  : m_hwnd(NULL)
  , m_x(0)
  , m_y(0)
  , m_w(0)
  , m_h(0)
  , m_main_menu(NULL)
  , m_scandb(false)
{}

RHEA_MainWnd::~RHEA_MainWnd()
{
  m_deckwnds.Empty(true);
  m_samplerwnds.Empty(true);
  DestroyWindow(m_hwnd);
}

void RHEA_MainWnd::LoadMedia(bool deck, int index)
{
  if (g_hestia->IsActive(deck, index))
  {
    WDL_FastString inf;

    if (deck)
    {
      inf.SetFormatted(512, "D%d is active. Do you want to continue?", index + 1);
    }
    else
    {
      inf.SetFormatted(512, "S%d is active. Do you want to continue?", index + 1);
    }

    int res = MessageBox(m_hwnd, inf.Get(), "Load confirmation", MB_YESNO);
    if (res == IDYES)
    {
      char *fl = NULL;
      WDL_FastString act;

      bool preservecwd = g_preferences->WantPreserveDirectory();
      if (deck) act.SetFormatted(512, "Load track to deck %d", index + 1);
      else act.SetFormatted(512, "Load track to sampler %d", index + 1);
      fl = WDL_ChooseFileForOpen(m_hwnd, act.Get(), NULL, NULL,
        RHEA_OPEN_MEDIA_FILES, RHEA_OPEN_MEDIA_FILE_DEFEXT, preservecwd, false);

      if (fl)
      {
        if (g_hestia->IsActive(deck, index))
        {
          g_hestia->ToggleActivate(deck, index);
        }
        g_hestia->Eject(deck, index);
        g_mediainput->Open(fl, deck, index); free(fl);
      }
    }
  }
  else
  {
    if (!g_mediainput->IsLoading(deck, index))
    {
      char *fl = NULL;
      WDL_FastString act;

      bool preservecwd = g_preferences->WantPreserveDirectory();
      if (deck) act.SetFormatted(512, "Load track to deck %d", index + 1);
      else act.SetFormatted(512, "Load track to sampler %d", index + 1);
      fl = WDL_ChooseFileForOpen(m_hwnd, act.Get(), NULL, NULL,
        RHEA_OPEN_MEDIA_FILES, RHEA_OPEN_MEDIA_FILE_DEFEXT, preservecwd, false);

      if (fl)
      {
        if (g_hestia->IsActive(deck, index))
        {
          g_hestia->ToggleActivate(deck, index);
        }
        g_hestia->Eject(deck, index);
        g_mediainput->Open(fl, deck, index); free(fl);
      }
    }
  }
}

void RHEA_MainWnd::Select(bool deck, int index)
{
  WDL_ASSERT((deck && index >= 0 && index < m_deckwnds.GetSize())
    || (!deck && index >= 0 && index < m_samplerwnds.GetSize()));

  for (int i = 0; i < m_deckwnds.GetSize(); i++)
  {
    RHEA_TrackWnd *trkwnd = m_deckwnds.Get(i);
    if (trkwnd)
    {
      if (deck && i == index)
      {
        trkwnd->Select(true);
        m_sel.act = true;
        m_sel.deck = true;
        m_sel.idx = index;
        g_waveformwnd->EnsureVisible(true, index);
      }
      else trkwnd->Select(false);
    }
  }

  for (int i = 0; i < m_samplerwnds.GetSize(); i++)
  {
    RHEA_TrackWnd *trkwnd = m_samplerwnds.Get(i);
    if (trkwnd)
    {
      if (!deck && i == index)
      {
        trkwnd->Select(true);
        m_sel.act = true;
        m_sel.deck = false;
        m_sel.idx = index;
        g_waveformwnd->EnsureVisible(false, index);
      }
      else trkwnd->Select(false);
    }
  }
}

void RHEA_MainWnd::SelectNone()
{
  m_sel.act = false;

  for (int i = 0; i < m_deckwnds.GetSize(); i++)
  {
    RHEA_TrackWnd *trkwnd = m_deckwnds.Get(i);
    if (trkwnd) trkwnd->Select(false);
  }

  for (int i = 0; i < m_samplerwnds.GetSize(); i++)
  {
    RHEA_TrackWnd *trkwnd = m_samplerwnds.Get(i);
    if (trkwnd) trkwnd->Select(false);
  }
}

void RHEA_MainWnd::ToggleActivate()
{
  if (m_sel.act)
  {
    g_hestia->ToggleActivate(m_sel.deck, m_sel.idx);
  }
}

void RHEA_MainWnd::Eject()
{
  if (m_sel.act)
  {
    g_hestia->Eject(m_sel.deck, m_sel.idx);
  }
}

void RHEA_MainWnd::Rewind()
{
  if (m_sel.act)
  {
    g_hestia->Rewind(m_sel.deck, m_sel.idx);
  }
}

void RHEA_MainWnd::FastForward()
{
  if (m_sel.act)
  {
    g_hestia->FastForward(m_sel.deck, m_sel.idx);
  }
}

void RHEA_MainWnd::StopRewind()
{
  if (m_sel.act)
  {
    g_hestia->StopRewind(m_sel.deck, m_sel.idx);
  }
}

void RHEA_MainWnd::StopFastForward()
{
  if (m_sel.act)
  {
    g_hestia->StopFastForward(m_sel.deck, m_sel.idx);
  }
}

void RHEA_MainWnd::AltRewind()
{
  if (m_sel.act)
  {
    g_hestia->AltRewind(m_sel.deck, m_sel.idx);
  }
}

void RHEA_MainWnd::AltFastForward()
{
  if (m_sel.act)
  {
    g_hestia->AltFastForward(m_sel.deck, m_sel.idx);
  }
}

void RHEA_MainWnd::GotoStart()
{
  if (m_sel.act)
  {
    g_hestia->GotoStart(m_sel.deck, m_sel.idx);
  }
}

void RHEA_MainWnd::GotoEnd()
{
  if (m_sel.act)
  {
    g_hestia->GotoEnd(m_sel.deck, m_sel.idx);
  }
}

void RHEA_MainWnd::ToggleReverse()
{
  if (m_sel.act)
  {
    g_hestia->ToggleReverse(m_sel.deck, m_sel.idx);
  }
}

void RHEA_MainWnd::Repeat(bool infinite)
{
  if (m_sel.act)
  {
    if (!m_sel.deck)
    {
      g_hestia->Repeat(m_sel.deck, m_sel.idx, infinite);
    }
  }
}

void RHEA_MainWnd::ClearRepeat()
{
  if (m_sel.act)
  {
    if (!m_sel.deck)
    {
      g_hestia->ClearRepeat(m_sel.deck, m_sel.idx);
    }
  }
}

void RHEA_MainWnd::GetSelectedTrack(bool *act, bool *deck, int *index)
{
  *act = m_sel.act;
  *deck = m_sel.deck;
  *index = m_sel.idx;
}

void RHEA_MainWnd::Search()
{
  if (!g_searchwnd)
  {
    g_searchwnd = new RHEA_SearchWnd;
    CreateDialogParam(g_inst, MAKEINTRESOURCE(IDD_SEARCH), m_hwnd,
      RHEA_SearchWnd::SearchWndProc, (LPARAM)g_searchwnd);
    ShowWindow(g_searchwnd->Handle(), SW_SHOW);
  }
  else
  {
    SetFocus(g_searchwnd->Handle());
  }
}

void RHEA_MainWnd::OnInitDialog(WPARAM wparam, LPARAM lparam)
{
  double wall0 = time_precise();

  if (IsStableRelease())
  {
    m_titleorig.Set(RHEA_NAME_MARKETING);
  }
  else
  {
    m_titleorig.Set(RHEA_FULL_VERSION " (" RHEA_ARCH ")");
  }
  SetWindowText(m_hwnd, m_titleorig.Get());

  RECT r;
  GetClientRect(m_hwnd, &r);

  int w = r.right - r.left;
  int h = r.bottom - r.top;

  for (int i = 0; i < 4; i++)
  {
    RHEA_TrackWnd *d = new RHEA_TrackWnd(true, i);
    m_deckwnds.Add(d);

    CreateDialogParam(g_inst, MAKEINTRESOURCE(IDD_TRACK),
      m_hwnd, RHEA_TrackWnd::TrackWndProc, (LPARAM)d);
  }

  for (int i = 0; i < 8; i++)
  {
    RHEA_TrackWnd *s = new RHEA_TrackWnd(false, i);
    m_samplerwnds.Add(s);

    CreateDialogParam(g_inst, MAKEINTRESOURCE(IDD_TRACK),
      m_hwnd, RHEA_TrackWnd::TrackWndProc, (LPARAM)s);
  }

  m_resize.init(m_hwnd);

  m_main_menu = LoadMenu(g_inst, MAKEINTRESOURCE(IDR_MAIN_MENUBAR));
  SetMenu(m_hwnd, m_main_menu);

  m_x = GetPrivateProfileInt(RHEA_NAME, "main_wnd_x", 50, g_inipath.Get());
  m_y = GetPrivateProfileInt(RHEA_NAME, "main_wnd_y", 50, g_inipath.Get());
  m_w = GetPrivateProfileInt(RHEA_NAME, "main_wnd_w", 1120, g_inipath.Get());
  m_h = GetPrivateProfileInt(RHEA_NAME, "main_wnd_h", 900, g_inipath.Get());
  SetWindowPos(m_hwnd, NULL, m_x, m_y, m_w, m_h, SWP_NOACTIVATE);

  int maximized = GetPrivateProfileInt(RHEA_NAME, "main_wnd_maximized", 0, g_inipath.Get());

  if (maximized)
  {
    ShowWindow(m_hwnd, SW_SHOWMAXIMIZED);
  }
  else
  {
    ShowWindow(m_hwnd, SW_SHOWNORMAL);
  }

  g_audiostreamer = CreateAudioStreamer();

  if (g_audiostreamer)
  {
    if (!g_audiostreamer->Open())
    {
      int fmt = g_preferences->GetAudioDeviceBitDepth();
      int srate = (int)g_preferences->GetAudioDeviceSamplerate();
      int nch = g_preferences->GetAudioDeviceOutputChannels();

      WDL_FastString fmtstr;

      switch (fmt)
      {
      case 0: fmtstr.Set("16-bit"); break;
      case 1: fmtstr.Set("24-bit"); break;
      case 2: fmtstr.Set("32-bit"); break;
      case 3: fmtstr.Set("32-bit FP"); break;
      default: fmtstr.Set("Unknown"); break;
      }

      WDL_FastString err;
      err.SetFormatted(1024,
        RHEA_NAME_MARKETING " cannot open the audio device with the following settings:\n\n"
        "Audio bit depth: %s\n"
        "Audio sample rate: %d\n"
        "Audio output channels: %d\n\n"
        "Make sure that the related settings are correct in " RHEA_NAME_MARKETING " preferences.",
        fmtstr.Get(), srate, nch);

      MessageBox(m_hwnd, err.Get(), "Audio device error", MB_OK);
    }
    else
    {

    }
  }
  else
  {
    MessageBox(m_hwnd,
      RHEA_NAME_MARKETING " cannot find a valid audio system.\n"
      "Press OK to close " RHEA_NAME_MARKETING ".",
      "Audio system error", MB_OK);
    DestroyWindow(m_hwnd);
  }

  g_mediainput = new RHEA_MediaInput;
  g_hestia = new RHEA_Hestia;
  g_hestia->Start(RHEA_OnMediaInputDeck, RHEA_OnMediaInputSampler);

  WDL_ASSERT(g_hestia->DeckCount() == m_deckwnds.GetSize());
  WDL_ASSERT(g_hestia->SamplerCount() == m_samplerwnds.GetSize());

  g_audiostreamer->Start(RHEA_OnAudioData);

  double sec = g_preferences->GetPreAllocatePeaks();
  if (WDL_DefinitelyGreaterThan(sec, 0.0))
  {
    WDL_ASSERT(g_peaks);
    g_peaks->PreAllocate(sec);
  }

  g_database = new RHEA_Database;
  g_database->Open();
  g_query = new RHEA_Query;
  g_query->Open();

  double wall1 = time_precise();

  if (WDL_unlikely(!IsStableRelease()))
  {
    const char warn_msg[] =
    {
      "Welcome to " RHEA_NAME_MARKETING " (the \"Software\").\n\n"
      "THIS IS AN UNSTABLE RELEASE STILL IN DEVELOPEMENT PHASE\n"
      "AND IS PROVIDED ON AN \"AS IS\" BASIS AND IS BELIEVED\n"
      "TO CONTAIN DEFECTS AND A PRIMARY PURPOSE OF THIS\n"
      "RELEASE IS TO OBTAIN FEEDBACK ON SOFTWARE PERFORMANCE\n"
      "AND THE IDENTIFICATION OF DEFECTS.\n\n"
      "IT IS ADVISED TO USE CAUTION AND NOT TO RELY IN ANY WAY\n"
      "ON THE CORRECT FUNCTIONING OF THIS RELEASE.\n\n"
      RHEA_NAME_MARKETING " version: " RHEA_NAKED_VERSION " (" RHEA_ARCH ")\n\n"
      "For the latest production release visit: " RHEA_WEBSITE_URL "\n\n"
      "Startup time: %.3f sec\n\n"
      RHEA_COPYRIGHT
    };

    WDL_FastString tmp;

    tmp.SetFormatted(1024, warn_msg, wall1 - wall0);

    MessageBox(m_hwnd, tmp.Get(), "Release for testing purposes", MB_OK);
  }

  SetTimer(m_hwnd, RHEA_MAIN_SCREEN_UPDATE, RHEA_MAIN_SCREEN_UPDATE_MS, NULL);
}

void RHEA_MainWnd::OnSysCommand(WPARAM wparam, LPARAM lparam)
{
  if (wparam == SC_CLOSE)
  {
    SendMessage(m_hwnd, WM_COMMAND, ID_FILE_EXIT, 0);
  }
}

INT_PTR RHEA_MainWnd::OnClose(WPARAM wparam, LPARAM lparam)
{
  return 0;
}

void RHEA_MainWnd::OnSize(WPARAM wparam, LPARAM lparam)
{
  if (wparam != SIZE_MINIMIZED)
  {
    m_resize.onResize();

    RECT r;
    GetClientRect(m_hwnd, &r);

    int w = r.right - r.left;
    int h = r.bottom - r.top;

    for (int i = 0, dw = w / m_deckwnds.GetSize(),
      lb = 0; i < m_deckwnds.GetSize(); i++, lb += dw)
    {
      RHEA_TrackWnd *d = m_deckwnds.Get(i);
      SetWindowPos(d->Handle(), NULL, lb, 0, dw,
        110, SWP_NOZORDER | SWP_NOACTIVATE);
      InvalidateRect(d->Handle(), NULL, FALSE);
    }

    for (int i = 0, dw = w / m_samplerwnds.GetSize(),
      lb = 0; i < m_samplerwnds.GetSize(); i++, lb += dw)
    {
      RHEA_TrackWnd *s = m_samplerwnds.Get(i);
      SetWindowPos(s->Handle(), NULL, lb, 110, dw,
        110, SWP_NOZORDER | SWP_NOACTIVATE);
      InvalidateRect(s->Handle(), NULL, FALSE);
    }

    HWND cstm = GetDlgItem(m_hwnd, IDC_CUSTOM1);
    SetWindowPos(cstm, NULL, 0, 220, w,
      h - 220, SWP_NOZORDER | SWP_NOACTIVATE);
    InvalidateRect(cstm, NULL, FALSE);
  }

  if (wparam != SIZE_MINIMIZED && wparam != SIZE_MAXIMIZED)
  {
    RECT r;
    GetWindowRect(m_hwnd, &r);

    m_x = r.left;
    m_y = r.top;
    m_w = r.right - r.left;
    m_h = r.bottom - r.top;
  }
}

void RHEA_MainWnd::OnMove(WPARAM wparam, LPARAM lparam)
{
  int xpos = (int)(short) LOWORD(lparam); // horizontal position
  int ypos = (int)(short) HIWORD(lparam); // vertical position

  if (xpos >= 0 && ypos >= 0 && GetCapture() == m_hwnd)
  {
    RECT r;
    GetWindowRect(m_hwnd, &r);

    m_x = r.left;
    m_y = r.top;
    m_w = r.right - r.left;
    m_h = r.bottom - r.top;
  }
}

void RHEA_MainWnd::OnDestroy(WPARAM wparam, LPARAM lparam)
{
  KillTimer(m_hwnd, RHEA_MAIN_SCREEN_UPDATE);

#ifdef _WIN32
  WINDOWPLACEMENT wp = { sizeof(wp) };
  GetWindowPlacement(m_hwnd, &wp);

  WDL_FastString xstr, ystr, wstr, hstr;
  xstr.SetFormatted(32, "%d", m_x);
  ystr.SetFormatted(32, "%d", m_y);
  wstr.SetFormatted(32, "%d", m_w);
  hstr.SetFormatted(32, "%d", m_h);

  if (wp.showCmd == SW_SHOWMAXIMIZED)
  {
    WritePrivateProfileString(RHEA_NAME, "main_wnd_maximized", "1", g_inipath.Get());
  }
  else
  {
    WritePrivateProfileString(RHEA_NAME, "main_wnd_maximized", "0", g_inipath.Get());
    WritePrivateProfileString(RHEA_NAME, "main_wnd_x", xstr.Get(), g_inipath.Get());
    WritePrivateProfileString(RHEA_NAME, "main_wnd_y", ystr.Get(), g_inipath.Get());
    WritePrivateProfileString(RHEA_NAME, "main_wnd_w", wstr.Get(), g_inipath.Get());
    WritePrivateProfileString(RHEA_NAME, "main_wnd_h", hstr.Get(), g_inipath.Get());
  }

#else
  RECT r;
  GetWindowRect(m_hwnd, &r);

  WDL_FastString xstr, ystr, wstr, hstr;
  xstr.SetFormatted(32, "%d", r.left);
  ystr.SetFormatted(32, "%d", r.top);
  wstr.SetFormatted(32, "%d", r.right - r.left);
  hstr.SetFormatted(32, "%d", r.bottom - r.top);

  if (r.left >= 0 && r.top >= 0)
  {
    WritePrivateProfileString(RHEA_NAME, "main_wnd_x", xstr.Get(), g_inipath.Get());
    WritePrivateProfileString(RHEA_NAME, "main_wnd_y", ystr.Get(), g_inipath.Get());
    WritePrivateProfileString(RHEA_NAME, "main_wnd_w", wstr.Get(), g_inipath.Get());
    WritePrivateProfileString(RHEA_NAME, "main_wnd_h", hstr.Get(), g_inipath.Get());
  }

  #ifdef __APPLE__
  SWELL_PostQuitMessage(g_mainwnd);
  #endif
  g_hasrequestedquit = true;
#endif

  if (g_audiostreamer->IsRunning())
  {
    g_audiostreamer->Close();
  }

  if (g_audiostreamer) delete g_audiostreamer;
  if (g_hestia) delete g_hestia;
  if (g_mediainput) delete g_mediainput;
  if (g_scrollbar) delete g_scrollbar;
  if (g_query) delete g_query;
  if (g_database) delete g_database;

  PostQuitMessage(0);
}

void RHEA_MainWnd::OnCommand(WPARAM wparam, LPARAM lparam)
{
  switch (LOWORD(wparam))
  {
  //case IDOK:
  //  {
  //    SetDlgItemText(m_hwnd, IDC_EDIT1, "");
  //  }
  //  break;
  case ID_FILE_LOADTRACKTODECK1:
    {
      LoadMedia(true, 0);
    }
    break;
  case ID_FILE_LOADTRACKTODECK2:
    {
      LoadMedia(true, 1);
    }
    break;
  case ID_FILE_LOADTRACKTODECK3:
    {
      LoadMedia(true, 2);
    }
    break;
  case ID_FILE_LOADTRACKTODECK4:
    {
      LoadMedia(true, 3);
    }
    break;
  case ID_FILE_LOADTRACKTOSAMPLER1:
    {
      LoadMedia(false, 0);
    }
    break;
  case ID_FILE_LOADTRACKTOSAMPLER2:
    {
      LoadMedia(false, 1);
    }
    break;
  case ID_FILE_LOADTRACKTOSAMPLER3:
    {
      LoadMedia(false, 2);
    }
    break;
  case ID_FILE_LOADTRACKTOSAMPLER4:
    {
      LoadMedia(false, 3);
    }
    break;
  case ID_FILE_LOADTRACKTOSAMPLER5:
    {
      LoadMedia(false, 4);
    }
    break;
  case ID_FILE_LOADTRACKTOSAMPLER6:
    {
      LoadMedia(false, 5);
    }
    break;
  case ID_FILE_LOADTRACKTOSAMPLER7:
    {
      LoadMedia(false, 6);
    }
    break;
  case ID_FILE_LOADTRACKTOSAMPLER8:
    {
      LoadMedia(false, 7);
    }
    break;
  case ID_FILE_EXIT:
    {
      bool act = false;

      for (int i = 0; i < g_hestia->DeckCount(); i++)
      {
        if (g_hestia->IsActive(true, i)) act = true;
      }

      for (int i = 0; i < g_hestia->SamplerCount(); i++)
      {
        if (g_hestia->IsActive(false, i)) act = true;
      }

      if (act)
      {
        int res = MessageBox(m_hwnd, "Rhea is active. Do you want to exit now?", "Exit confirmation", MB_YESNO);
        if (res == IDYES)
        {
          DestroyWindow(g_mainwnd->Handle());
        }
      }
      else
      {
        DestroyWindow(g_mainwnd->Handle());
      }
    }
    break;
  case ID_OPTIONS_SCANMEDIALIBRARYPATHS:
    {
      if (!g_database->IsScanInProgress())
      {
        g_database->ScanFiles();
      }
    }
    break;
  case ID_OPTIONS_PREFERENCES:
    {
      if (!g_preferences->WantCurses())
      {
        g_preferences->Open();
      }
      else
      {
        if (!g_prefwnd)
        {
          g_prefwnd = new RHEA_PreferencesWnd;
          CreateDialogParam(g_inst, MAKEINTRESOURCE(IDD_PREFERENCES), m_hwnd,
            RHEA_PreferencesWnd::PreferencesWndProc, (LPARAM)g_prefwnd);
          ShowWindow(g_prefwnd->Handle(), SW_SHOW);
        }
        else
        {
          SetFocus(g_prefwnd->Handle());
        }
      }
    }
    break;
  case ID_HELP_DOCUMENTATION:
    {
      WDL_FastString doc;
      doc.Set(g_modpath.Get());
      doc.Append("documentation.txt");

      ShellExecute(m_hwnd, "", "notepad.exe", doc.Get(), "", SW_SHOWNORMAL);
    }
    break;
  case ID_HELP_ABOUT:
    {
      if (!g_aboutwnd)
      {
        g_aboutwnd = new RHEA_AboutWnd;
        CreateDialogParam(g_inst, MAKEINTRESOURCE(IDD_ABOUT), m_hwnd,
          RHEA_AboutWnd::AboutWndProc, (LPARAM)g_aboutwnd);
        ShowWindow(g_aboutwnd->Handle(), SW_SHOW);
      }
      else
      {
        SetFocus(g_aboutwnd->Handle());
      }
    }
    break;
  case ID_HELP_CHANGELOG:
    {
      if (!g_changelogwnd)
      {
        g_changelogwnd = new RHEA_ChangelogWnd;
        CreateDialogParam(g_inst, MAKEINTRESOURCE(IDD_CHANGELOG), m_hwnd,
          RHEA_ChangelogWnd::ChangelogWndProc, (LPARAM)g_changelogwnd);
        ShowWindow(g_changelogwnd->Handle(), SW_SHOW);
      }
      else
      {
        SetFocus(g_changelogwnd->Handle());
      }
    }
    break;
  case ID_HELP_VERSION:
    {
      const char version_str[] =
      {
        RHEA_NAME_MARKETING " version: " RHEA_NAKED_VERSION " (" RHEA_ARCH ")\n\n"
        "Build timestamp: " RHEA_TIMESTAMP "\n"
        "Git commit: " RHEA_GIT_SHA
      };

      MessageBox(m_hwnd, version_str, "Version details", MB_OK);
    }
    break;
  case ID_HELP_OFFICIALWEBSITE:
    {
      ShellExecute(m_hwnd, "", RHEA_WEBSITE_URL, "", "", SW_SHOWNORMAL);
    }
    break;
  case ID_HELP_LICENSE:
    {
      if (!g_licwnd)
      {
        g_licwnd = new RHEA_LicenseWnd;
        CreateDialogParam(g_inst, MAKEINTRESOURCE(IDD_LICENSE), m_hwnd,
          RHEA_LicenseWnd::LicenseWndProc, (LPARAM)g_licwnd);
        ShowWindow(g_licwnd->Handle(), SW_SHOW);
      }
      else
      {
        SetFocus(g_licwnd->Handle());
      }
    }
    break;
  }
}

void RHEA_MainWnd::OnTimer(WPARAM wparam, LPARAM lparam)
{
  if (wparam == RHEA_MAIN_SCREEN_UPDATE)
  {
    if (g_database->IsScanInProgress())
    {
      if (!m_scandb)
      {
        m_scandb = true;
        WDL_FastString tit(m_titleorig);
        tit.Append(" | Scanning media library paths");
        SetWindowText(m_hwnd, tit.Get());
        EnableMenuItem(GetMenu(m_hwnd), ID_OPTIONS_SCANMEDIALIBRARYPATHS, MF_GRAYED);
      }
    }
    else
    {
      if (m_scandb)
      {
        m_scandb = false;
        SetWindowText(m_hwnd, m_titleorig.Get());
        EnableMenuItem(GetMenu(m_hwnd), ID_OPTIONS_SCANMEDIALIBRARYPATHS, MF_ENABLED);
      }
    }
  }
}

void RHEA_MainWnd::OnPaint(WPARAM wparam, LPARAM lparam)
{
  RECT r;
  GetClientRect(m_hwnd, &r);

  int w = r.right - r.left;
  int h = r.bottom - r.top;

  m_painter.PaintBegin(m_hwnd, RGB(51, 51, 51));
  int xo, yo;
  LICE_IBitmap *bm = m_painter.GetBuffer(&xo, &yo);

  m_painter.PaintEnd();

  //HDC hdc;
  //PAINTSTRUCT ps;
  //hdc = GetDC(m_list);//BeginPaint(m_list, &ps);
  //HPEN pen = CreatePen(PS_SOLID, 2, RGB(0, 0, 255));
  //SelectObject(hdc, pen);
  //MoveToEx(hdc, 0, 100, NULL);
  //LineTo(hdc, 100, 100);
  //DeleteObject(pen);
  //EndPaint(m_list, &ps);
}

INT_PTR RHEA_MainWnd::OnNotify(WPARAM wparam, LPARAM lparam)
{
  //switch (LOWORD(wparam))
  //{}

  return 0;
}

void RHEA_MainWnd::OnContextMenu(WPARAM wparam, LPARAM lparam)
{
  POINT pt;
  pt.x = GET_X_LPARAM(lparam);
  pt.y = GET_Y_LPARAM(lparam);
  HANDLE hwnd = (HANDLE)wparam;
}

void RHEA_MainWnd::OnLButtonDown(WPARAM wparam, LPARAM lparam)
{
  POINT pt;
  pt.x = GET_X_LPARAM(lparam);
  pt.y = GET_Y_LPARAM(lparam);
}

void RHEA_MainWnd::OnLButtonUp(WPARAM wparam, LPARAM lparam)
{

}

void RHEA_MainWnd::OnMouseMove(WPARAM wparam, LPARAM lparam)
{

}

void RHEA_MainWnd::OnActivateApp(WPARAM wparam, LPARAM lparam)
{

}

void RHEA_MainWnd::OnNCLButtonDown(WPARAM wparam, LPARAM lparam)
{

}

WDL_DLGRET RHEA_MainWnd::MainWndProc(HWND hwnd, UINT msg, WPARAM wparam, LPARAM lparam)
{
  RHEA_MainWnd *self = (RHEA_MainWnd *)GetWindowLongPtr(hwnd, GWLP_USERDATA);

  if (!self && msg == WM_INITDIALOG)
  {
    SetWindowLongPtr(hwnd, GWLP_USERDATA, lparam);
    self = (RHEA_MainWnd *)lparam;
    self->m_hwnd = hwnd;
  }

  if (WDL_likely(self))
  {
    return self->MainWndLoop(msg, wparam, lparam);
  }
  else
  {
    return 0;
  }
}

WDL_DLGRET RHEA_MainWnd::MainWndLoop(UINT msg, WPARAM wparam, LPARAM lparam)
{
#if defined(__linux__)
  if (msg == WM_KEYDOWN && LOWORD(wparam) == VK_RETURN)
  {
    SendMessage(m_hwnd, WM_COMMAND, MAKEWPARAM(IDOK, 0), 0);
    return 0;
  }
#endif

  switch(msg)
  {
    case WM_INITDIALOG: OnInitDialog(wparam, lparam); break;
    case WM_SYSCOMMAND: OnSysCommand(wparam, lparam); break;
    case WM_CLOSE: return OnClose(wparam, lparam); break;
    case WM_SIZE: OnSize(wparam, lparam); break;
    case WM_MOVE: OnMove(wparam, lparam); break;
    case WM_DESTROY: OnDestroy(wparam, lparam); break;
    case WM_COMMAND: OnCommand(wparam, lparam); break;
    case WM_TIMER: OnTimer(wparam, lparam); break;
    case WM_PAINT: OnPaint(wparam, lparam); break;
    case WM_ERASEBKGND: return 1; break;
    case WM_NOTIFY: return OnNotify(wparam, lparam); break;
    case WM_CONTEXTMENU: OnContextMenu(wparam, lparam); break;
    case WM_LBUTTONDOWN: OnLButtonDown(wparam, lparam); break;
    case WM_LBUTTONUP: OnLButtonUp(wparam, lparam); break;
    case WM_MOUSEMOVE: OnMouseMove(wparam, lparam); break;
    //case WM_CTLCOLOREDIT: return OnCtrlColorEdit(msg, wparam, lparam); break;
    case WM_ACTIVATEAPP: OnActivateApp(wparam, lparam); break;
    case WM_NCLBUTTONDOWN: OnNCLButtonDown(wparam, lparam); break;
  }

  return 0;
}

#include "WDL/eel2/ns-eel.h"
#include "WDL/lice/lice.h"
#include "WDL/wingui/virtwnd-controls.h"
#include "WDL/wingui/scrollbar/coolscroll.h"


int WDL_STYLE_WantGlobalButtonBorders() { return 0; }
bool WDL_STYLE_WantGlobalButtonBackground(int *col) { return false; }
bool WDL_STYLE_GetBackgroundGradient(double *gradstart, double *gradslope) { return false; }
LICE_IBitmap *WDL_STYLE_GetSliderBitmap2(bool vert) { return NULL; }
bool WDL_STYLE_AllowSliderMouseWheel() { return true; }
int WDL_STYLE_GetSliderDynamicCenterPos() { return 500; }


// TO BE IMPLEMENTED BY APP:
// implemented by calling app, can return a LICE_IBitmap **img for "scrollbar"
void *GetIconThemePointer(const char *name)
{
  if (!strcmp(name, "scrollbar"))
  {
    if (!g_scrollbar)
    {
      WDL_FastString sb(g_modpath);
      sb.Append("skin\\mask\\scrollbar.png");
      g_scrollbar = LICE_LoadPNG(sb.Get());
      return &g_scrollbar;
    }
  }

  return NULL;
}

// can be a passthrough to GetSysColor()
int CoolSB_GetSysColor(HWND m_hwnd, int val)
{
  return GetSysColor(val);
}

#include <WDL/eel2/ns-eel.h>

void NSEEL_HOSTSTUB_EnterMutex()
{}

void NSEEL_HOSTSTUB_LeaveMutex()
{}
