// Copyright (c) 2023 Giorgos Vougioukas
//
// The license can be found in the LICENSE file.

#include "rhea/media_input.h"
#include "rhea/track.h"
#include "rhea/peaks.h"
//#include "rhea/concurrency.h"

#include "WDL/sha.h"
#include "WDL/fileread.h"
#include "WDL/pcmfmtcvt.h"

RHEA_MediaInput::RHEA_MediaInput()
  : m_killthread(false)
  , m_running(false)
  , m_cancel(false)
{
  StartThread();
}

RHEA_MediaInput::~RHEA_MediaInput()
{
  m_cancel = true;

  if (IsRunning()) StopThread();

  m_fileio.Empty(true);

  for (int i = 0; i < m_decks.GetSize(); i++)
  {
    RHEA_AntidoteInput *ai = m_decks.Enumerate(i);
    if (ai) delete ai;
  }

  for (int i = 0; i < m_samplers.GetSize(); i++)
  {
    RHEA_AntidoteInput *ai = m_samplers.Enumerate(i);
    if (ai) delete ai;
  }
}

bool RHEA_MediaInput::Open(const char *filename, bool deck, int index)
{
  //WDL_MutexLock lock(&m_mutex);
  RHEA_IFileInput *fi = CreateFileInput(filename);

  if (fi)
  {
    RHEA_IFileTag *ft = CreateFileTag(filename);

    WDL_FastString fn(g_setpath.Get());
    fn.Append("cache" WDL_DIRCHAR_STR);
    CreateDirectory(fn.Get(), NULL);

    WDL_SHA1 sha;
    char hash[WDL_SHA1SIZE];
    sha.reset();
    sha.add(filename, (int)strlen(filename));
    sha.result(hash);

    fn.AppendFormatted(64, "%02x", (unsigned char)(hash[0] & 0xff));
    fn.Append(WDL_DIRCHAR_STR);
    CreateDirectory(fn.Get(), NULL);

    for (int i = 1; i < WDL_SHA1SIZE; i++)
    {
      fn.AppendFormatted(64, "%02x", (unsigned char)(hash[i] & 0xff));
    }

    bool fileexists = false;
    WDL_FileRead *fr = new WDL_NEW WDL_FileRead(fn.Get());
    if (fr)
    {
      fileexists = fr->IsOpen();
      delete fr; fr = NULL;
    }

    int wmode, wbufsize, wminbufs, wmaxbufs;
    g_preferences->GetDiskWriteMode(&wmode, &wbufsize, &wminbufs, &wmaxbufs);
    WDL_FileWrite *fw = NULL;

    if (!fileexists) fw = new WDL_NEW WDL_FileWrite(fn.Get(), wmode, wbufsize, wminbufs, wmaxbufs);

    if (fileexists || (fw && fw->IsOpen()))
    {
      FileIO *fio = new WDL_NEW FileIO;
      if (fio)
      {
        fio->fi = fi;
        fio->ft = ft;
        fio->fw = fw ? fw : NULL;
        fio->fifn.Set(filename);
        fio->fwfn.Set(fn.Get());
        fio->deck = deck;
        fio->idx = index;

        m_fileio.Add(fio);
        m_antidote = g_preferences->GetAntidoteBitDepth();
        m_srate = g_preferences->GetAudioDeviceSamplerate();

        return true;
      }
    }

    if (fi) delete fi;
    if (ft) delete ft;
    if (fw)
    {
      delete fw;

      fileexists = false;
      fr = new WDL_NEW WDL_FileRead(fn.Get());
      if (fr)
      {
        fileexists = fr->IsOpen();
        delete fr;
      }

      if (fileexists) DeleteFile(fn.Get());
    }
  }

  return false;
}

void RHEA_MediaInput::DeckTrackInputs(WDL_PtrList<RHEA_Track> *decks)
{
  if (m_fileio.GetSize() || !m_decks.GetSize()) return;
  WDL_MutexLock lock(&m_mutex);

  WDL_TypedBuf<int> deckid;

  for (int i = 0; i < m_decks.GetSize(); i++)
  {
    int idx;
    RHEA_AntidoteInput *ai = m_decks.Enumerate(i, &idx);
    if (ai)
    {
      WDL_ASSERT(idx < decks->GetSize());
      RHEA_Track *trk = decks->Get(idx);
      if (trk)
      {
        if (trk->IsLoaded())
        {
          trk->Eject(false);
        }
        g_peaks->PeaksReady(true, idx);
        trk->Load(ai, true, idx);
        deckid.Add(idx);
      }
    }
  }

  for (int i = 0; i < deckid.GetSize(); i++)
  {
    m_decks.Delete(deckid.Get()[i]);
  }
}

void RHEA_MediaInput::SamplerTrackInputs(WDL_PtrList<RHEA_Track> *samplers)
{
  if (m_fileio.GetSize() || !m_samplers.GetSize()) return;
  WDL_MutexLock lock(&m_mutex);

  WDL_TypedBuf<int> samplerid;

  for (int i = 0; i < m_samplers.GetSize(); i++)
  {
    int idx;
    RHEA_AntidoteInput *ai = m_samplers.Enumerate(i, &idx);
    if (ai)
    {
      WDL_ASSERT(idx < samplers->GetSize());
      RHEA_Track *trk = samplers->Get(idx);
      if (trk)
      {
        if (trk->IsLoaded())
        {
          trk->Eject(false);
        }
        g_peaks->PeaksReady(false, idx);
        trk->Load(ai, false, idx);
        samplerid.Add(idx);
      }
    }
  }

  for (int i = 0; i < samplerid.GetSize(); i++)
  {
    m_samplers.Delete(samplerid.Get()[i]);
  }
}

bool RHEA_MediaInput::IsLoading(bool deck, int index) const
{
  for (int i = 0; i < m_fileio.GetSize(); i++)
  {
    FileIO *fio = m_fileio.Get(i);

    if (fio && deck == fio->deck && index == fio->idx)
    {
      return true;
    }
  }

  if (deck)
  {
    if (m_decks.Exists(index)) return true;
  }
  else
  {
    if (m_samplers.Exists(index)) return true;
  }

  return false;
}

bool RHEA_MediaInput::StorePeaks(RHEA_AntidoteInput *ai, bool deck, int index)
{
  int nread = 0;
  int samidx = 0;
  float peaks[2];
  WDL_TypedBuf<SAM> buf;
  const int every = ((int)ai->GetSampleRate() / 400);
  const double len = floor(ai->GetLength()) + 10.0;
  const double srate = ai->GetSampleRate() / 400;
  const int nch = ai->GetChannels();
  const size_t request = (size_t)(len * 400 * nch * sizeof(float));

  if (!g_peaks->Allocate(request, deck, index))
  {
    return false;
  }

  do
  {
    buf.Resize(4096);
    nread = ai->GetSamples(buf.Get(), buf.GetSize());
    if (nread)
    {
      buf.Resize(nread, false);
      if (WDL_unlikely(nch == 1))
      {
        for (int i = 0; i < buf.GetSize(); i += nch)
        {
          if (samidx == every - 1)
          {
            WDL_ASSERT(i + 1 < buf.GetSize());
            peaks[0] = (float)wdl_clamp(wdl_abs(buf.Get()[i]), 0.0, 1.0);
            peaks[1] = (float)wdl_clamp(wdl_abs(buf.Get()[i]), 0.0, 1.0);
            g_peaks->AddPeaks(peaks, 2, deck, index);
            samidx = 0;
          }
          else
          {
            samidx++;
          }
        }
      }
      else
      {
        for (int i = 0; i < buf.GetSize(); i += nch)
        {
          if (samidx == every - 1)
          {
            WDL_ASSERT(i + 1 < buf.GetSize());
            peaks[0] = (float)wdl_clamp(wdl_abs(buf.Get()[i]), 0.0, 1.0);
            peaks[1] = (float)wdl_clamp(wdl_abs(buf.Get()[i + 1]), 0.0, 1.0);
            g_peaks->AddPeaks(peaks, 2, deck, index);
            samidx = 0;
          }
          else
          {
            samidx++;
          }
        }
      }
    }
  } while (nread > 0);

  ai->Seek(0.0);
  return true;
}

void RHEA_MediaInput::StartThread()
{
  WDL_ASSERT(!m_threads.GetSize());

  int priority;
  m_killthread = false;

  m_threads.Resize(1);
  //m_threads.Resize(RHEA_GetCPUCores());
  for (int i = 0; i < m_threads.GetSize(); i++)
  {
    ThreadDetails *rt = m_threads.Get();
    rt[i].thread = (HANDLE)_beginthreadex(NULL, 0,
      ThreadFunction, (void *)this, 0, &rt[i].id);

    priority = g_preferences->GetDiskIOPriority();
    SetThreadPriority(rt[i].thread, priority);
  }

  m_running = true;
}

void RHEA_MediaInput::StopThread()
{
  m_killthread = true;
  for (int i = 0; i < m_threads.GetSize(); i++)
  {
    ThreadDetails *rt = m_threads.Get();
    WaitForSingleObject(rt[i].thread, INFINITE);
    CloseHandle(rt[i].thread);
    m_threads.Resize(0);
  }

  m_running = false;
}

bool RHEA_MediaInput::IsRunning() const
{
  return m_running;
}

int RHEA_MediaInput::Run()
{
  for (int i = 0; i < m_fileio.GetSize(); i++)
  {
    int nread = 0;
    WDL_HeapBuf rawbuf;
    WDL_TypedBuf<SAM> buf;
    buf.Resize(4096);

    FileIO *fio = m_fileio.Get(i);
    if (WDL_NOT_NORMALLY(!fio))
    {
      m_fileio.Delete(i, false);
      return 1;
    }

    if (!fio->stt && fio->ebur128)
    {
      fio->stt = ebur128_init((unsigned int)fio->fi->GetChannels(),
        (unsigned long)fio->fi->GetSampleRate(), fio->ebur128mode);
      ebur128_set_channel(fio->stt, 0, EBUR128_LEFT);
      ebur128_set_channel(fio->stt, 1, EBUR128_RIGHT);
    }

    buf.Resize(4096);

    nread = fio->fi->GetSamples(buf.Get(), buf.GetSize());

    if (m_cancel) nread = 0;

    if (nread > 0)
    {
      buf.Resize(nread, false);

      if (fio->ebur128)
      {
        size_t frames = (size_t)(buf.GetSize() / fio->fi->GetChannels());
        ebur128_add_frames_double(fio->stt, buf.Get(), frames);

        fio->momentarywindow += (int)(frames / fio->fi->GetSampleRate() * 1000);
        fio->shorttermwindow += (int)(frames / fio->fi->GetSampleRate() * 1000);

        if ((fio->ebur128mode & EBUR128_MODE_M) == EBUR128_MODE_M &&
          fio->momentarywindow >= fio->ebur128momentary)
        {
          double tmp;
          int res = ebur128_loudness_momentary(fio->stt, &tmp);
          if (res != EBUR128_SUCCESS) fio->ebur128 = false;
          fio->ebur128momentary = wdl_max(fio->ebur128momentary, tmp);
          fio->momentarywindow = 0;
        }

        if ((fio->ebur128mode & EBUR128_MODE_S) == EBUR128_MODE_S &&
          fio->shorttermwindow >= fio->ebur128shorttermwindow)
        {
          double tmp;
          int res = ebur128_loudness_shortterm(fio->stt, &tmp);
          if (res != EBUR128_SUCCESS) fio->ebur128 = false;
          fio->ebur128shortterm = wdl_max(fio->ebur128shortterm, tmp);
          fio->shorttermwindow = 0;
        }

        if ((fio->ebur128mode & EBUR128_MODE_SAMPLE_PEAK) == EBUR128_MODE_SAMPLE_PEAK)
        {
          for (int i = 0; i < fio->fi->GetChannels(); i++)
          {
            double tmp;
            int res = ebur128_sample_peak(fio->stt, (unsigned int)i, &tmp);
            if (res != EBUR128_SUCCESS) fio->ebur128 = false;
            fio->ebur128samplepeak = wdl_max(fio->ebur128samplepeak, tmp);
          }
        }

        if ((fio->ebur128mode & EBUR128_MODE_TRUE_PEAK) == EBUR128_MODE_TRUE_PEAK)
        {
          for (int i = 0; i < fio->fi->GetChannels(); i++)
          {
            double tmp;
            int res = ebur128_true_peak(fio->stt, (unsigned int)i, &tmp);
            if (res != EBUR128_SUCCESS) fio->ebur128 = false;
            fio->ebur128truepeak = wdl_max(fio->ebur128truepeak, tmp);
          }
        }
      }

      switch (m_antidote)
      {
      case 0:
        {
          rawbuf.Resize(buf.GetSize() * sizeof(short));
          doublesToPcm(buf.Get(), 1, buf.GetSize(), rawbuf.Get(), 16, 1);
        } break;
      case 1:
        {
          rawbuf.Resize(buf.GetSize() * sizeof(char) * 3);
          doublesToPcm(buf.Get(), 1, buf.GetSize(), rawbuf.Get(), 24, 1);
        } break;
      case 2:
        {
          rawbuf.Resize(buf.GetSize() * sizeof(int));
          doublesToPcm(buf.Get(), 1, buf.GetSize(), rawbuf.Get(), 32, 1);
        } break;
      case 3:
        {
          rawbuf.Resize(buf.GetSize() * sizeof(float));
          float *flt = (float *)rawbuf.Get();
          for (int i = 0; i < buf.GetSize(); i++)
          {
            flt[i] = (float)buf.Get()[i];
          }
        } break;
      case 4:
        {
          rawbuf.Resize(buf.GetSize() * sizeof(double));
          double *dbl = (double *)rawbuf.Get();
          for (int i = 0; i < buf.GetSize(); i++)
          {
            dbl[i] = (double)buf.Get()[i];
          }
        } break;
      }

      if (fio->fw) fio->fw->Write(rawbuf.Get(), rawbuf.GetSize());

      return 0;
    }
    else
    {
      bool deck = fio->deck;
      int index = fio->idx;
      WDL_FastString fn(fio->fwfn.Get());
      WDL_FastString ifn(fio->fifn.Get());
      int channels = fio->fi->GetChannels();
      double orig_srate = fio->fi->GetSampleRate();

      if (fio->ebur128 && (fio->ebur128mode & EBUR128_MODE_M) == EBUR128_MODE_M)
      {
        fio->ebur128gain = fio->ebur128reference - fio->ebur128momentary;
      }

      if (fio->ebur128 && (fio->ebur128mode & EBUR128_MODE_S) == EBUR128_MODE_S)
      {
        fio->ebur128gain = fio->ebur128reference - fio->ebur128shortterm;
      }

      if (fio->ebur128 && (fio->ebur128mode & EBUR128_MODE_I) == EBUR128_MODE_I)
      {
        int res = ebur128_loudness_global(fio->stt, &fio->ebur128integrated);
        if (res == EBUR128_SUCCESS)
        {
          fio->ebur128gain = fio->ebur128reference - fio->ebur128integrated;
        }
        else
        {
          fio->ebur128 = false;
        }
      }

      if (fio->ebur128 && (fio->ebur128mode & EBUR128_MODE_LRA) == EBUR128_MODE_LRA)
      {
        int res = ebur128_loudness_range(fio->stt, &fio->ebur128range);
        if (res != EBUR128_SUCCESS)
        {
          fio->ebur128 = false;
        }
      }

      if (fio->fw) { delete fio->fw; fio->fw = NULL; }

      int rmode, rbufsize, rnbufs;
      g_preferences->GetDiskReadMode(&rmode, &rbufsize, &rnbufs);
      WDL_FileRead *fr = new WDL_NEW WDL_FileRead(fn.Get(), rmode, rbufsize, rnbufs);
      if (fr)
      {
        if (fr->IsOpen())
        {
          RHEA_AntidoteInput *ai =
            new WDL_NEW RHEA_AntidoteInput(
            fn.Get(), fr, ifn.Get(),
            channels, orig_srate);

          if (ai)
          {

            if (fio->ft)
            {
              if (strlen(fio->ft->GetTitle()))
              {
                ai->SetTitle(fio->ft->GetTitle());
              }
              else
              {
                WDL_FastString tit;
                tit.Set(fio->ft->GetFileName());
                tit.remove_fileext();
                ai->SetTitle(tit.Get());
              }

              ai->SetArtist(fio->ft->GetArtist());
            }

            ai->ebur128info(fio->ebur128,
              fio->ebur128downwardonly,
              fio->ebur128reference,
              fio->ebur128mode,
              fio->ebur128momentary,
              fio->ebur128shortterm,
              fio->ebur128momentarywindow,
              fio->ebur128shorttermwindow,
              fio->ebur128integrated,
              fio->ebur128range,
              fio->ebur128samplepeak,
              fio->ebur128truepeak,
              fio->ebur128gain,
              fio->momentarywindow,
              fio->shorttermwindow);

            if (StorePeaks(ai, deck, index))
            {
              if (deck) m_decks.Insert(index, ai);
              else m_samplers.Insert(index, ai);
            }
            else
            {
              delete ai;
            }
          }
        }
        else
        {
          delete fr;
        }
      }
      m_fileio.Delete(i, true);
    }
  }

  return 1;
}

unsigned int WINAPI RHEA_MediaInput::ThreadFunction(void *arg)
{
#if defined(_WIN32)
  CoInitialize(NULL);
#endif

  RHEA_MediaInput *self = (RHEA_MediaInput *)arg;

  if (WDL_NOT_NORMALLY(!self)) return 0;

  int sleepstep = g_preferences->GetDiskIOSleepStep();

  self->m_killthread = false;

  while (!self->m_killthread)
  {
    self->m_mutex.Enter();
    while (!self->Run());
    self->m_mutex.Leave();
    Sleep(sleepstep);
  }

#if defined(_WIN32)
  CoUninitialize();
#endif

  return 0;
}
