// Copyright (c) 2023 Giorgos Vougioukas
//
// The license can be found in the LICENSE file.

#ifndef _RHEA_MEDIA_INPUT_H_
#define _RHEA_MEDIA_INPUT_H_

#include "rhea/definitions.h"
#include "rhea/plugin.h"
#include "rhea/antidote_input.h"
#include "rhea/track.h"
#include "rhea/preferences.h"

#include "third_party/libebur128/ebur128/ebur128.h"

#include <math.h>

#include "WDL/mutex.h"
#include "WDL/heapbuf.h"
#include "WDL/ptrlist.h"
#include "WDL/wdlstring.h"
#include "WDL/filewrite.h"
#include "WDL/assocarray.h"

class RHEA_MediaInput
{
public:
  RHEA_MediaInput();
  ~RHEA_MediaInput();

  bool Open(const char *filename, bool deck, int index);
  void DeckTrackInputs(WDL_PtrList<RHEA_Track> *decks);
  void SamplerTrackInputs(WDL_PtrList<RHEA_Track> *samplers);

  bool IsLoading(bool deck, int index) const;

private:
  bool StorePeaks(RHEA_AntidoteInput *ai, bool deck, int index);
  void StartThread();
  void StopThread();
  bool IsRunning() const;
  static unsigned int WINAPI ThreadFunction(void *arg);
  int Run();

  struct ThreadDetails
  {
    unsigned int id;
    HANDLE thread;
  };

  WDL_TypedBuf<ThreadDetails> m_threads;
  bool m_killthread;
  WDL_Mutex m_mutex;
  bool m_running;

  class FileIO
  {
  public:
    FileIO()
      : fi(NULL)
      , fw(NULL)
      , ft(NULL)
      , deck(false)
      , idx(0)
      , stt(NULL)
      , ebur128(false)
      , ebur128downwardonly(false)
      , ebur128reference(0.0)
      , ebur128mode(0)
      , ebur128momentary(-HUGE_VAL)
      , ebur128shortterm(-HUGE_VAL)
      , ebur128momentarywindow(400)
      , ebur128shorttermwindow(3000)
      , ebur128integrated(-HUGE_VAL)
      , ebur128range(-HUGE_VAL)
      , ebur128samplepeak(-HUGE_VAL)
      , ebur128truepeak(-HUGE_VAL)
      , ebur128gain(-HUGE_VAL)
      , momentarywindow(0)
      , shorttermwindow(0)
    {
      ebur128 = g_preferences->WantEBUR128();
      ebur128reference = g_preferences->GetEBUR128LUFS();
      ebur128mode = g_preferences->GetEBUR128Mode();
      ebur128downwardonly = g_preferences->WantDownwardOnly();
    }

    ~FileIO()
    {
      if (fi) delete fi;
      if (fw) delete fw;
      if (ft) delete ft;
      if (stt) ebur128_destroy(&stt);
    }

    RHEA_IFileInput *fi;
    WDL_FastString fifn;
    WDL_FileWrite *fw;
    WDL_FastString fwfn;
    RHEA_IFileTag *ft;
    bool deck;
    int idx;

    ebur128_state *stt;
    bool ebur128;
    bool ebur128downwardonly;
    double ebur128reference;
    int ebur128mode;
    double ebur128momentary;
    double ebur128shortterm;
    int ebur128momentarywindow;
    int ebur128shorttermwindow;
    double ebur128integrated;
    double ebur128range;
    double ebur128samplepeak;
    double ebur128truepeak;
    double ebur128gain;
    int momentarywindow;
    int shorttermwindow;
  };

  WDL_PtrList<FileIO> m_fileio;
  WDL_IntKeyedArray<RHEA_AntidoteInput *> m_decks;
  WDL_IntKeyedArray<RHEA_AntidoteInput *> m_samplers;
  int m_antidote;
  double m_srate;
  bool m_cancel;
};

#endif // _RHEA_MEDIA_INPUT_H_
