// Copyright (c) 2020 Giorgos Vougioukas
//
// The license can be found in the LICENSE file.

#include "rhea/query.h"
#include "rhea/app_info.h"
#include "rhea/preferences.h"
#include "rhea/main_wnd.h"

#include "WDL/fpcmp.h"

#define RHEA_QUERY_NO_ACTION 0
#define RHEA_QUERY_SEARCH_DB 1

RHEA_Query::RHEA_Query()
  : m_open(false)
  , m_db(NULL)
  , m_stmt(NULL)
  , m_errmsg(NULL)
  , m_step(0)
  , m_querylimit(0)
{
  m_querylimit = g_preferences->GetDatabaseSearchQueryLimit();
}

RHEA_Query::~RHEA_Query()
{
  if (IsOpen()) Close();
  m_matches.Empty(true);
}

bool RHEA_Query::Open()
{
  int rc = 0;
  char *errmsg = NULL;

  m_fn.Set(g_setpath.Get());
  m_fn.Append(RHEA_NAME ".db");

  rc = sqlite3_open(m_fn.Get(), &m_db);

  if (WDL_NOT_NORMALLY(rc))
  {
    wdl_log("cannot open sqlite database\n");
    sqlite3_close(m_db);
    m_db = NULL;
    return false;
  }

  const char *sql_create_table =
    "CREATE TABLE IF NOT EXISTS metadata ("
    "type TEXT NOT NULL,"
    "title TEXT NOT NULL,"
    "artist TEXT NOT NULL,"
    "album TEXT NOT NULL,"
    "length REAL DEFAULT 0.0,"
    "genre TEXT NOT NULL,"
    "samplerate REAL DEFAULT 44100.0,"
    "file_name TEXT NOT NULL,"
    "file_path TEXT NOT NULL,"
    "file_ext TEXT NOT NULL,"
    "file_size INTEGER DEFAULT 0,"
    "comment TEXT NOT NULL);";

  rc = sqlite3_exec(m_db, sql_create_table, 0, 0, &errmsg);

  if (WDL_NOT_NORMALLY(rc != SQLITE_OK))
  {
    wdl_log("cannot execute SQL command (%s)\n", errmsg);
    sqlite3_free(errmsg);
    sqlite3_close(m_db);
    m_db = NULL;
    return false;
  }

  m_open = true;

  return true;
}

bool RHEA_Query::IsOpen() const
{
  return m_open;
}

void RHEA_Query::Close()
{
  if (m_stmt)
  {
    sqlite3_finalize(m_stmt);
    m_stmt = NULL;
  }

  if (WDL_NORMALLY(m_db))
  {
    sqlite3_close(m_db);
    m_db = NULL;
  }

  m_open = false;
}

bool RHEA_Query::Query(const char *text)
{
  WDL_ASSERT(!m_stmt);
  m_querytext.Set(text);
  m_querysql.Set(text);
  m_querysql.Insert("%", 0);
  m_querysql.Append("%");

  m_matches.Empty(true);

  const char *sql =
    "SELECT * FROM metadata WHERE title LIKE ?1"
    " OR artist LIKE ?2 OR album LIKE ?3 LIMIT ?4;";

  int rc = sqlite3_prepare_v2(m_db, sql, -1, &m_stmt, NULL);
  if (WDL_NOT_NORMALLY(rc != SQLITE_OK))
  {
    wdl_log("cannot execute sqlite3_prepare_v2\n");
  }

  rc = sqlite3_bind_text(m_stmt, 1, m_querysql.Get(), -1, NULL);
  if (WDL_NOT_NORMALLY(rc != SQLITE_OK))
  {
    wdl_log("cannot bind value %s\n", sqlite3_errmsg(m_db));
  }

  rc = sqlite3_bind_text(m_stmt, 2, m_querysql.Get(), -1, NULL);
  if (WDL_NOT_NORMALLY(rc != SQLITE_OK))
  {
    wdl_log("cannot bind value %s\n", sqlite3_errmsg(m_db));
  }

  rc = sqlite3_bind_text(m_stmt, 3, m_querysql.Get(), -1, NULL);
  if (WDL_NOT_NORMALLY(rc != SQLITE_OK))
  {
    wdl_log("cannot bind value %s\n", sqlite3_errmsg(m_db));
  }

  rc = sqlite3_bind_int(m_stmt, 4, m_querylimit);
  if (WDL_NOT_NORMALLY(rc != SQLITE_OK))
  {
    wdl_log("cannot bind value %s\n", sqlite3_errmsg(m_db));
  }

  m_step = sqlite3_step(m_stmt);
  bool row = (m_step == SQLITE_ROW);

  while (HasRow())
  {
    PlayListEntry *ple = new PlayListEntry;
    ple->type.Set(GetColumnText(0));
    ple->title.Set(GetColumnText(1));
    ple->artist.Set(GetColumnText(2));
    ple->album.Set(GetColumnText(3));
    ple->length = GetColumnDouble(4);
    ple->genre.Set(GetColumnText(5));
    ple->srate = GetColumnDouble(6);
    ple->filename.Set(GetColumnText(7));
    ple->filepath.Set(GetColumnText(8));
    ple->fileext.Set(GetColumnText(9));
    ple->filesize = GetColumnInt64(10);
    ple->comment.Set(GetColumnText(11));
    m_matches.Add(ple);
    NextStep();
  }

  sqlite3_finalize(m_stmt); m_stmt = NULL;

  return row;
}

const char *RHEA_Query::QueryText() const
{
  return m_querytext.Get();
}

WDL_PtrList<PlayListEntry> *RHEA_Query::GetMatches()
{
  WDL_ASSERT(!m_stmt);
  return &m_matches;
}

bool RHEA_Query::HasRow()
{
  bool row = (m_step == SQLITE_ROW);

  if (!row)
  {
    sqlite3_finalize(m_stmt); m_stmt = NULL;
  }

  return row;
}

const char *RHEA_Query::GetColumnText(int col)
{
  if (m_stmt)
  {
    return (const char *)sqlite3_column_text(m_stmt, col);
  }

  return "";
}

double RHEA_Query::GetColumnDouble(int col)
{
  if (m_stmt)
  {
    return sqlite3_column_double(m_stmt, col);
  }

  return 0.0;
}

WDL_INT64 RHEA_Query::GetColumnInt64(int col)
{
  if (m_stmt)
  {
    return sqlite3_column_int64(m_stmt, col);
  }

  return WDL_INT64_CONST(0);
}

void RHEA_Query::NextStep()
{
  m_step = sqlite3_step(m_stmt);
}
