// Copyright (c) 2023 Giorgos Vougioukas
//
// The license can be found in the LICENSE file.

#include "rhea/search_wnd.h"
#include "rhea/main_wnd.h"
#include "rhea/app_info.h"
#include "rhea/hestia.h"
#include "rhea/query.h"
#include "rhea/media_input.h"

#include "WDL/fpcmp.h"

#define RHEA_CLEAR_EDIT 310
#define RHEA_CLEAR_EDIT_MS 10

static WNDPROC PrevListProc = NULL;
static HWND s_list = NULL;
static WNDPROC PrevEditProc = NULL;
static HWND s_edit = NULL;

RHEA_SearchWnd::RHEA_SearchWnd()
  : m_hwnd(NULL)
  , m_x(0)
  , m_y(0)
  , m_w(0)
  , m_h(0)
  , m_edit(NULL)
  , m_list(NULL)
  , m_column_menu(NULL)
  , m_matches_menu(NULL)
{}

RHEA_SearchWnd::~RHEA_SearchWnd()
{
  DestroyWindow(m_hwnd);
}

void RHEA_SearchWnd::Search()
{
  char text[2048];
  int len = GetWindowText(m_edit, text, sizeof(text));
  if (len)
  {
    g_query->Query(text);

    ListView_DeleteAllItems(m_list);

    WDL_PtrList<PlayListEntry> *matches = g_query->GetMatches();

    for (int i = 0; i < matches->GetSize(); i++)
    {
      PlayListEntry *ple = matches->Get(i);

      LVITEM lvi = { 0, };
      int si = 0;
      lvi.mask = LVIF_TEXT | LVIF_PARAM;
      lvi.lParam = (LPARAM)i;
      lvi.iItem = i;
      lvi.iSubItem = si++; // type
      lvi.pszText = (char *)ple->type.Get();
      lvi.cchTextMax = ple->type.GetLength();
      ListView_InsertItem(m_list, &lvi);

      lvi.mask = LVIF_TEXT;
      lvi.iSubItem = si++; // index
      m_strbuf.SetFormatted(128, "%d", i + 1);
      lvi.pszText = (char *)m_strbuf.Get();
      lvi.cchTextMax = m_strbuf.GetLength();
      ListView_SetItem(m_list, &lvi);

      lvi.mask = LVIF_TEXT;
      lvi.iSubItem = si++; // title
      lvi.pszText = (char *)ple->title.Get();
      lvi.cchTextMax = ple->title.GetLength();
      ListView_SetItem(m_list, &lvi);

      lvi.mask = LVIF_TEXT;
      lvi.iSubItem = si++; // artist
      lvi.pszText = (char *)ple->artist.Get();
      lvi.cchTextMax = ple->artist.GetLength();
      ListView_SetItem(m_list, &lvi);

      lvi.mask = LVIF_TEXT;
      lvi.iSubItem = si++; // album
      lvi.pszText = (char *)ple->album.Get();
      lvi.cchTextMax = ple->album.GetLength();
      ListView_SetItem(m_list, &lvi);

      lvi.mask = LVIF_TEXT;
      lvi.iSubItem = si++; // length
      int minutes = 0;
      int seconds = 0;
      double time = 0.0;
      time = ple->length;
      if (WDL_DefinitelyGreaterThan(time, 0.0))
      {
        minutes = (int)time / 60;
        seconds = (int)time % 60;
      }
      m_strbuf.SetFormatted(128, "%01d:%02d", minutes, seconds);
      lvi.pszText = (char *)m_strbuf.Get();
      lvi.cchTextMax = m_strbuf.GetLength();
      ListView_SetItem(m_list, &lvi);

      lvi.mask = LVIF_TEXT;
      lvi.iSubItem = si++; // genre
      lvi.pszText = (char *)ple->genre.Get();
      lvi.cchTextMax = ple->genre.GetLength();
      ListView_SetItem(m_list, &lvi);

      lvi.mask = LVIF_TEXT;
      lvi.iSubItem = si++; // samplerate
      m_strbuf.SetFormatted(128,
        "%.0f Hz", ple->srate);
      lvi.pszText = (char *)m_strbuf.Get();
      lvi.cchTextMax = m_strbuf.GetLength();
      ListView_SetItem(m_list, &lvi);

      lvi.mask = LVIF_TEXT;
      lvi.iSubItem = si++; // filename
      lvi.pszText = (char *)ple->filename.Get();
      lvi.cchTextMax = ple->filename.GetLength();
      ListView_SetItem(m_list, &lvi);

      lvi.mask = LVIF_TEXT;
      lvi.iSubItem = si++; // filepath
      lvi.pszText = (char *)ple->filepath.Get();
      lvi.cchTextMax = ple->filepath.GetLength();
      ListView_SetItem(m_list, &lvi);

      lvi.mask = LVIF_TEXT;
      lvi.iSubItem = si++; // fileext
      lvi.pszText = (char *)ple->fileext.Get();
      lvi.cchTextMax = ple->fileext.GetLength();
      ListView_SetItem(m_list, &lvi);

      lvi.mask = LVIF_TEXT;
      lvi.iSubItem = si++; // filesize
      double mb = (double)ple->filesize / 1024 / 1024;
      m_strbuf.SetFormatted(128, "%.2f MB", mb);
      lvi.pszText = (char *)m_strbuf.Get();
      lvi.cchTextMax = m_strbuf.GetLength();
      ListView_SetItem(m_list, &lvi);

      lvi.mask = LVIF_TEXT;
      lvi.iSubItem = si++; // comment
      lvi.pszText = (char *)ple->comment.Get();
      lvi.cchTextMax = ple->comment.GetLength();
      ListView_SetItem(m_list, &lvi);
    }
  }
  else
  {
    ListView_DeleteAllItems(m_list);
  }
}

void RHEA_SearchWnd::LoadMedia(bool deck, int index)
{
  if (g_hestia->IsActive(deck, index))
  {
    WDL_FastString inf;

    if (deck)
    {
      inf.SetFormatted(512, "D%d is active. Do you want to continue?", index + 1);
    }
    else
    {
      inf.SetFormatted(512, "S%d is active. Do you want to continue?", index + 1);
    }

    int res = MessageBox(m_hwnd, inf.Get(), "Load confirmation", MB_YESNO);
    if (res == IDYES)
    {
      int pos = ListView_GetNextItem(m_list, -1, LVIS_SELECTED|LVIS_FOCUSED);
      if (pos >= 0)
      {
        int subitem = 0;
        char text[2048];
        ColumnHeader *ch = m_column_header.Get();

        for (int i = 0; i < m_column_header.GetSize(); i++)
        {
          if (ch[i].id == ID_LIBRARY_HEADER_FILEPATH)
          {
            subitem = i;
            break;
          }
        }

        ListView_GetItemText(m_list, pos, subitem, text, sizeof(text));

        if (g_hestia->IsActive(deck, index))
        {
          g_hestia->ToggleActivate(deck, index);
        }
        g_hestia->Eject(deck, index);
        g_mediainput->Open(text, deck, index);
      }
    }
  }
  else
  {
    if (!g_mediainput->IsLoading(deck, index))
    {
      int pos = ListView_GetNextItem(m_list, -1, LVIS_SELECTED|LVIS_FOCUSED);
      if (pos >= 0)
      {
        int subitem = 0;
        char text[2048];
        ColumnHeader *ch = m_column_header.Get();

        for (int i = 0; i < m_column_header.GetSize(); i++)
        {
          if (ch[i].id == ID_LIBRARY_HEADER_FILEPATH)
          {
            subitem = i;
            break;
          }
        }

        ListView_GetItemText(m_list, pos, subitem, text, sizeof(text));

        if (g_hestia->IsActive(deck, index))
        {
          g_hestia->ToggleActivate(deck, index);
        }
        g_hestia->Eject(deck, index);
        g_mediainput->Open(text, deck, index);
      }
    }
  }
}

void RHEA_SearchWnd::OnInitDialog(WPARAM wparam, LPARAM lparam)
{
  m_list = GetDlgItem(m_hwnd, IDC_LIST1);
  WDL_UTF8_HookListView(m_list);

#if defined(_WIN32)
  ListView_SetExtendedListViewStyleEx(m_list,
    LVS_EX_HEADERDRAGDROP | LVS_EX_FULLROWSELECT | LVS_EX_DOUBLEBUFFER,
    LVS_EX_HEADERDRAGDROP | LVS_EX_FULLROWSELECT | LVS_EX_DOUBLEBUFFER);
#else
  ListView_SetExtendedListViewStyleEx(m_list,
    LVS_EX_HEADERDRAGDROP | LVS_EX_FULLROWSELECT,
    LVS_EX_HEADERDRAGDROP | LVS_EX_FULLROWSELECT);
#endif

  ColumnHeader default_column_header[] =
  {
    // The alignment of the leftmost column is
    // always LVCFMT_LEFT; it cannot be changed.
    { ID_LIBRARY_HEADER_TYPE, "Type", 50, LVCFMT_LEFT, 1, 0 },
    { ID_LIBRARY_HEADER_INDEX, "#", 30, LVCFMT_RIGHT, 1, 1 },
    { ID_LIBRARY_HEADER_TITLE, "Title", 200, LVCFMT_LEFT, 1, 0 },
    { ID_LIBRARY_HEADER_ARTIST, "Artist", 200, LVCFMT_LEFT, 1, 0 },
    { ID_LIBRARY_HEADER_ALBUM, "Album", 200, LVCFMT_LEFT, 1, 0 },
    { ID_LIBRARY_HEADER_LENGTH, "Length", 50, LVCFMT_RIGHT, 1, 1 },
    { ID_LIBRARY_HEADER_GENRE, "Genre", 80, LVCFMT_LEFT, 1, 0 },
    { ID_LIBRARY_HEADER_SAMPLERATE, "Sample rate", 80, LVCFMT_RIGHT, 1, 1 },
    { ID_LIBRARY_HEADER_FILENAME, "File name", 150, LVCFMT_LEFT, 1, 0 },
    { ID_LIBRARY_HEADER_FILEPATH, "File path", 150, LVCFMT_LEFT, 1 , 0 },
    { ID_LIBRARY_HEADER_FILEEXTENSION, "File extension", 80, LVCFMT_LEFT, 1, 0 },
    { ID_LIBRARY_HEADER_FILESIZE, "File size", 100, LVCFMT_RIGHT, 1, 1 },
    { ID_LIBRARY_HEADER_COMMENT, "Comment", 100, LVCFMT_LEFT, 1, 0 }
  };

  m_column_header.Resize(sizeof(default_column_header) /
    sizeof(default_column_header[0]));

  for (int i = 0; i < m_column_header.GetSize(); i++)
  {
    ColumnHeader *ch = m_column_header.Get();
    ch[i] = default_column_header[i];
  }

  LVCOLUMN lvc = { 0, };
  lvc.mask = LVCF_FMT | LVCF_TEXT | LVCF_WIDTH;

  int w = 0;
  WDL_FastString s;

  ColumnHeader *ch = m_column_header.Get();

  for (int i = 0; i < m_column_header.GetSize(); i++)
  {
    s.SetFormatted(128, "library_column_width_%02d", i);
    w = GetPrivateProfileInt(RHEA_NAME, s.Get(), ch[i].width, g_inipath.Get());
    lvc.cx = w;
    lvc.fmt = ch[i].fmt;
    lvc.pszText = (char *)ch[i].text;
    ListView_InsertColumn(m_list, i, &lvc);
  }

  WDL_TypedBuf<int> order_array;
  order_array.Resize(Header_GetItemCount(ListView_GetHeader(m_list)));
  int *oa = order_array.Get();

  for (int i = 0; i < order_array.GetSize(); i++)
  {
    s.SetFormatted(128, "library_column_order_%02d", i);
    oa[i] = GetPrivateProfileInt(RHEA_NAME, s.Get(), i, g_inipath.Get());
  }

  ListView_SetColumnOrderArray(m_list, order_array.GetSize(), order_array.Get());

  int defcol = GetPrivateProfileInt(RHEA_NAME, "library_defcol", 1, g_inipath.Get());
  if (defcol)
  {
    SendMessage(m_hwnd, WM_COMMAND, ID_LIBRARY_HEADER_DEFAULT, 0);
  }

  m_column_menu = (HMENU)GetSubMenu(LoadMenu(g_inst, MAKEINTRESOURCE(IDR_LIBRARY_CONTEXT)), 0);
  m_matches_menu = (HMENU)GetSubMenu(LoadMenu(g_inst, MAKEINTRESOURCE(IDR_LIBRARY_CONTEXT)), 1);

  m_edit = GetDlgItem(m_hwnd, IDC_EDIT1);

  PrevListProc = (WNDPROC)SetWindowLongPtrW(GetDlgItem(m_hwnd, IDC_LIST1),
  GWLP_WNDPROC, (LONG_PTR)NewListProc);
  s_list = m_list;

  PrevEditProc = (WNDPROC)SetWindowLongPtrW(GetDlgItem(m_hwnd, IDC_EDIT1),
    GWLP_WNDPROC, (LONG_PTR)NewEditProc);
  s_edit = m_edit;

  m_resize.init(m_hwnd);
  m_resize.init_item(IDC_EDIT1, 0.0f, 0.0f, 1.0f, 0.0f);
  m_resize.init_item(IDC_LIST1, 0.0f, 0.0f, 1.0f, 1.0f);

  m_x = GetPrivateProfileInt(RHEA_NAME, "search_wnd_x", 50, g_inipath.Get());
  m_y = GetPrivateProfileInt(RHEA_NAME, "search_wnd_y", 50, g_inipath.Get());
  m_w = GetPrivateProfileInt(RHEA_NAME, "search_wnd_w", 900, g_inipath.Get());
  m_h = GetPrivateProfileInt(RHEA_NAME, "search_wnd_h", 600, g_inipath.Get());
  SetWindowPos(m_hwnd, NULL, m_x, m_y, m_w, m_h, SWP_NOACTIVATE);
}

void RHEA_SearchWnd::OnDestroy(WPARAM wparam, LPARAM lparam)
{
  WDL_FastString xstr, ystr, wstr, hstr;
  xstr.SetFormatted(32, "%d", m_x);
  ystr.SetFormatted(32, "%d", m_y);
  wstr.SetFormatted(32, "%d", m_w);
  hstr.SetFormatted(32, "%d", m_h);

  WritePrivateProfileString(RHEA_NAME, "search_wnd_x", xstr.Get(), g_inipath.Get());
  WritePrivateProfileString(RHEA_NAME, "search_wnd_y", ystr.Get(), g_inipath.Get());
  WritePrivateProfileString(RHEA_NAME, "search_wnd_w", wstr.Get(), g_inipath.Get());
  WritePrivateProfileString(RHEA_NAME, "search_wnd_h", hstr.Get(), g_inipath.Get());

  WDL_FastString s;
  WDL_TypedBuf<int> order_array;
  WDL_FastString str;

  order_array.Resize(Header_GetItemCount(ListView_GetHeader(m_list)));
  ListView_GetColumnOrderArray(m_list, order_array.GetSize(), order_array.Get());

  int *oa = order_array.Get();

  for (int i = 0; i < order_array.GetSize(); i++)
  {
    s.SetFormatted(128, "library_column_order_%02d", i);
    str.SetFormatted(32, "%d", oa[i]);
    WritePrivateProfileString(RHEA_NAME, s.Get(), str.Get(), g_inipath.Get());
  }

  for (int i = 0; i < Header_GetItemCount(ListView_GetHeader(m_list)); i++)
  {
    s.SetFormatted(128, "library_column_width_%02d", i);
    str.SetFormatted(32, "%d", ListView_GetColumnWidth(m_list, i));
    WritePrivateProfileString(RHEA_NAME, s.Get(), str.Get(), g_inipath.Get());
  }
}

void RHEA_SearchWnd::OnSize(WPARAM wparam, LPARAM lparam)
{
  if (wparam != SIZE_MINIMIZED)
  {
    m_resize.onResize();
  }

  if (wparam != SIZE_MINIMIZED && wparam != SIZE_MAXIMIZED)
  {
    RECT r;
    GetWindowRect(m_hwnd, &r);

    m_x = r.left;
    m_y = r.top;
    m_w = r.right - r.left;
    m_h = r.bottom - r.top;
  }
}

void RHEA_SearchWnd::OnMove(WPARAM wparam, LPARAM lparam)
{
  int xpos = (int)(short) LOWORD(lparam); // horizontal position
  int ypos = (int)(short) HIWORD(lparam); // vertical position

  if (xpos >= 0 && ypos >= 0 && GetCapture() == m_hwnd)
  {
    RECT r;
    GetWindowRect(m_hwnd, &r);

    m_x = r.left;
    m_y = r.top;
    m_w = r.right - r.left;
    m_h = r.bottom - r.top;
  }
}

void RHEA_SearchWnd::OnCommand(WPARAM wparam, LPARAM lparam)
{
  switch(LOWORD(wparam))
  {
    case IDCANCEL:
    {
      if (g_searchwnd) { delete g_searchwnd; g_searchwnd = NULL; }
    } break;
    case ID_LIBRARY_HEADER_DEFAULT:
    {
      int defzero[] =
      {
        ID_LIBRARY_HEADER_TYPE,
        ID_LIBRARY_HEADER_GENRE,
        ID_LIBRARY_HEADER_FILENAME,
        ID_LIBRARY_HEADER_FILEPATH,
        ID_LIBRARY_HEADER_FILEEXTENSION,
        ID_LIBRARY_HEADER_COMMENT
      };

      int defstd[] =
      {
        ID_LIBRARY_HEADER_INDEX,
        ID_LIBRARY_HEADER_TITLE,
        ID_LIBRARY_HEADER_ARTIST,
        ID_LIBRARY_HEADER_ALBUM,
        ID_LIBRARY_HEADER_LENGTH,
        ID_LIBRARY_HEADER_SAMPLERATE,
        ID_LIBRARY_HEADER_FILESIZE
      };

      for (int i = 0, it = 0; i < (sizeof(defzero) / sizeof(defzero[0])); i++)
      {
        for (int j = 0; j < m_column_header.GetSize(); j++)
        {
          if (m_column_header.Get()[j].id == defzero[i])
          {
            it = j;
            break;
          }
        }

        ListView_SetColumnWidth(m_list, it, 0);
      }

      for (int i = 0, it = 0, w = 0; i < (sizeof(defstd) / sizeof(defstd[0])); i++)
      {
        for (int j = 0; j < m_column_header.GetSize(); j++)
        {
          if (m_column_header.Get()[j].id == defstd[i])
          {
            it = j;
            w = m_column_header.Get()[j].width;
            break;
          }
        }

        ListView_SetColumnWidth(m_list, it, w);
      }
    } break;
    case ID_LIBRARY_HEADER_TYPE:
    case ID_LIBRARY_HEADER_TITLE:
    case ID_LIBRARY_HEADER_ARTIST:
    case ID_LIBRARY_HEADER_ALBUM:
    case ID_LIBRARY_HEADER_LENGTH:
    case ID_LIBRARY_HEADER_GENRE:
    case ID_LIBRARY_HEADER_SAMPLERATE:
    case ID_LIBRARY_HEADER_FILENAME:
    case ID_LIBRARY_HEADER_FILEPATH:
    case ID_LIBRARY_HEADER_FILEEXTENSION:
    case ID_LIBRARY_HEADER_FILESIZE:
    case ID_LIBRARY_HEADER_INDEX:
    case ID_LIBRARY_HEADER_COMMENT:
    {
      for (int i = 0; i < m_column_header.GetSize(); i++)
      {
        ColumnHeader *ch = m_column_header.Get();

        if (ch[i].id == LOWORD(wparam))
        {
          MENUITEMINFO mii = { 0 };
          mii.cbSize = sizeof(mii);
          mii.fMask = MIIM_STATE;

          if (GetMenuItemInfo(m_column_menu, ch[i].id, FALSE, &mii))
          {
            if (!(mii.fState & MFS_CHECKED))
            {
              ListView_SetColumnWidth(m_list, i, ch[i].width);
            }
            else if (mii.fState & MFS_CHECKED)
            {
              ListView_SetColumnWidth(m_list, i, 0);
            }

            //mii.fState ^= MFS_CHECKED;
            //SetMenuItemInfo(m_column_menu, pl_column_info[i].id, FALSE, &mii);
            WDL_FastString str;
            str.SetFormatted(32, "%d", 0);
            WritePrivateProfileString(RHEA_NAME, "library_defcol", str.Get(), g_inipath.Get());
          }
        }
      }
    } break;
    case ID_SEARCH_LOADTRACKTODECK1: LoadMedia(true, 0); break;
    case ID_SEARCH_LOADTRACKTODECK2: LoadMedia(true, 1); break;
    case ID_SEARCH_LOADTRACKTODECK3: LoadMedia(true, 2); break;
    case ID_SEARCH_LOADTRACKTODECK4: LoadMedia(true, 3); break;
    case ID_SEARCH_LOADTRACKTOSAMPLER1: LoadMedia(false, 0); break;
    case ID_SEARCH_LOADTRACKTOSAMPLER2: LoadMedia(false, 1); break;
    case ID_SEARCH_LOADTRACKTOSAMPLER3: LoadMedia(false, 2); break;
    case ID_SEARCH_LOADTRACKTOSAMPLER4: LoadMedia(false, 3); break;
    case ID_SEARCH_LOADTRACKTOSAMPLER5: LoadMedia(false, 4); break;
    case ID_SEARCH_LOADTRACKTOSAMPLER6: LoadMedia(false, 5); break;
    case ID_SEARCH_LOADTRACKTOSAMPLER7: LoadMedia(false, 6); break;
    case ID_SEARCH_LOADTRACKTOSAMPLER8: LoadMedia(false, 7); break;
  }
}

void RHEA_SearchWnd::OnSysCommand(WPARAM wparam, LPARAM lparam)
{
  if (LOWORD(wparam) == SC_CLOSE)
  {
    if (g_searchwnd) { delete g_searchwnd; g_searchwnd = NULL; }
  }
}

void RHEA_SearchWnd::OnTimer(WPARAM wparam, LPARAM lparam)
{
  if (wparam == RHEA_CLEAR_EDIT)
  {
    SetWindowText(m_edit, "");
    KillTimer(m_hwnd, RHEA_CLEAR_EDIT);
  }
}

void RHEA_SearchWnd::OnPaint(WPARAM wparam, LPARAM lparam)
{}

void RHEA_SearchWnd::OnContextMenu(WPARAM wparam, LPARAM lparam)
{
  POINT pt;
  pt.x = GET_X_LPARAM(lparam);
  pt.y = GET_Y_LPARAM(lparam);
  HANDLE hwnd = (HANDLE)wparam;

  if (hwnd == m_list && pt.x == -1 && pt.y == -1)
  {
    int pos = ListView_GetNextItem(m_list, -1, LVNI_FOCUSED|LVNI_SELECTED);
    if (pos >= 0)
    {
      RECT r;
      ListView_EnsureVisible(m_list, pos, FALSE);
      ListView_GetItemRect(m_list, pos, &r, LVIR_BOUNDS);
      pt.x = r.left;
      pt.y = r.bottom;

      GetClientRect(m_list, &r);
      if (pt.x < r.left)
      {
        pt.x = r.left;
      }

      ClientToScreen(m_list, (LPPOINT)&pt);
      TrackPopupMenu(m_matches_menu, TPM_LEFTALIGN, pt.x, pt.y, 0, m_hwnd, NULL);
    }
  }

  if (hwnd == m_list)
  {
    //POINT pt;
    //pt.x = GET_X_LPARAM(lparam);
    //pt.y = GET_Y_LPARAM(lparam);
    ScreenToClient(m_list, (LPPOINT)&pt);

    HWND h = ListView_GetHeader(GetDlgItem(m_hwnd, IDC_LIST1));
    RECT r;
    GetWindowRect(h, &r);
    ScreenToClient(m_list, (LPPOINT)&r);
    ScreenToClient(m_list, ((LPPOINT)&r)+1);

    ColumnHeader *ch = m_column_header.Get();

    EnableMenuItem(m_column_menu, ID_LIBRARY_HEADER_INFO, MF_GRAYED);

#if defined(__APPLE__) || defined(__linux__)
    r.bottom = r.top + SWELL_GetListViewHeaderHeight(m_list);
#endif

    if (pt.x >= r.left && pt.y >= r.top && pt.x <= r.right && pt.y <= r.bottom)
    {
      for (int i = 0; i < m_column_header.GetSize(); i++)
      {
        if (ListView_GetColumnWidth(m_list, i))
        {
          CheckMenuItem(m_column_menu, ch[i].id, MF_CHECKED);
        }
        else
        {
          CheckMenuItem(m_column_menu, ch[i].id, MF_UNCHECKED);
        }
      }

      ClientToScreen(m_list, (LPPOINT)&pt);
      TrackPopupMenu(m_column_menu, TPM_LEFTALIGN, pt.x, pt.y, 0, m_hwnd, NULL);
    }
  }
}

INT_PTR RHEA_SearchWnd::OnNotify(WPARAM wparam, LPARAM lparam)
{
  switch (LOWORD(wparam))
  {
  case IDC_LIST1:
    {
      switch (((LPNMHDR)lparam)->code)
      {
      case NM_RCLICK:
        {
          int pos = ListView_GetNextItem(m_list, -1, LVNI_FOCUSED|LVNI_SELECTED);
          if (pos >= 0)
          {
            POINT pt;
            GetCursorPos(&pt);
            TrackPopupMenu(m_matches_menu, TPM_LEFTALIGN, pt.x, pt.y, 0, m_hwnd, NULL);
          }
        } break;
      }
    } break;
  }

  return 0;
}

WDL_DLGRET RHEA_SearchWnd::NewListProc(HWND hwnd, UINT msg, WPARAM wparam, LPARAM lparam)
{
  switch (msg)
  {
  case WM_MOUSEWHEEL:
  case WM_MOUSEHWHEEL:
    if (GetAsyncKeyState(VK_SHIFT) & 0x8000)
    {
      return 0;
    }
    break;
  case WM_KEYDOWN:
    {
      switch(wparam)
      {
      case VK_UP:
        {
          int pos = ListView_GetNextItem(s_list, -1, LVIS_FOCUSED|LVIS_SELECTED);
          if (pos == 0)
          {
            SetFocus(s_edit);
          }
        }
        break;
      }
    }
    break;
  }

  return CallWindowProc(PrevListProc, hwnd, msg, wparam, lparam);
}

WDL_DLGRET RHEA_SearchWnd::NewEditProc(HWND hwnd, UINT msg, WPARAM wparam, LPARAM lparam)
{
  switch (msg)
  {
  case WM_MOUSEWHEEL:
  case WM_MOUSEHWHEEL:
    if (GetAsyncKeyState(VK_SHIFT) & 0x8000)
    {
      return 0;
    }
    break;
  case WM_SYSKEYDOWN:
  case WM_KEYDOWN:
    {
      switch(wparam)
      {
      case VK_DOWN:
        {
          if (ListView_GetItemCount(s_list)) SetFocus(s_list);
        }
        break;
      case VK_BACK:
        {
          if ((GetAsyncKeyState(VK_CONTROL) & 0x8000))
          {
            SetTimer(g_searchwnd->Handle(), RHEA_CLEAR_EDIT,
              RHEA_CLEAR_EDIT_MS, NULL);
          }
        }
        break;
      }
    }
    break;
  }

  return CallWindowProc(PrevEditProc, hwnd, msg, wparam, lparam);
}

WDL_DLGRET RHEA_SearchWnd::SearchWndProc(HWND hwnd, UINT msg, WPARAM wparam, LPARAM lparam)
{
  RHEA_SearchWnd *self = (RHEA_SearchWnd *)GetWindowLongPtr(hwnd, GWLP_USERDATA);

  if (!self && msg == WM_INITDIALOG)
  {
    SetWindowLongPtr(hwnd, GWLP_USERDATA, lparam);
    self = (RHEA_SearchWnd *)lparam;
    self->m_hwnd = hwnd;
  }

  if (WDL_likely(self))
  {
    return self->SearchWndLoop(msg, wparam, lparam);
  }
  else
  {
    return 0;
  }
}

WDL_DLGRET RHEA_SearchWnd::SearchWndLoop(UINT msg, WPARAM wparam, LPARAM lparam)
{
  switch (msg)
  {
    case WM_INITDIALOG: OnInitDialog(wparam, lparam); break;
    case WM_DESTROY: OnDestroy(wparam, lparam); break;
    case WM_SIZE: OnSize(wparam, lparam); break;
    case WM_MOVE: OnMove(wparam, lparam); break;
    case WM_COMMAND: OnCommand(wparam, lparam); break;
    case WM_SYSCOMMAND: OnSysCommand(wparam, lparam); break;
    case WM_TIMER: OnTimer(wparam, lparam); break;
    case WM_PAINT: OnPaint(wparam, lparam); break;
    case WM_CONTEXTMENU: OnContextMenu(wparam, lparam); break;
    case WM_NOTIFY: return OnNotify(wparam, lparam); break;
  }

  return 0;
}
