// Copyright (c) 2022 Giorgos Vougioukas
//
// The license can be found in the LICENSE file.

#include "rhea/track.h"
#include "rhea/preferences.h"
#include "rhea/peaks.h"

#include "WDL/db2val.h"
#include "WDL/fpcmp.h"

RHEA_Track::RHEA_Track()
  : m_eof(false)
  , m_bqms(1200)
  , m_input(NULL)
  , m_length(0.0)
  , m_nch(0)
  , m_timesec(0.0)
  , m_act(false)
  , m_ps(NULL)
  , m_shift(1.0)
  , m_tempo(1.0)
  , m_deck(false)
  , m_idx(0)
  , m_rpt(0)
{
  m_bqms = g_preferences->GetMediaBufferSize();
}

RHEA_Track::~RHEA_Track()
{
  if (m_input) delete m_input;
  if (m_ps) delete m_ps;
}

void RHEA_Track::Load(RHEA_AntidoteInput *ai, bool deck, int index)
{
  WDL_ASSERT(!IsLoaded());
  WDL_ASSERT(ai && !m_ps);
  m_ps = CreatePitchShift("soundtouch");
  m_input = ai;
  m_deck = deck;
  m_idx = index;
  m_nch = ai->GetChannels();
  m_length = ai->GetLength();
  m_time.Set("");
  m_timesec = 0.0;
  m_eof = false;
}

bool RHEA_Track::IsLoaded() const
{
  return m_input != NULL;
}

void RHEA_Track::Eject(bool clear_peaks)
{
  WDL_ASSERT(IsLoaded());

  m_eof = true;

  if (m_input)
  {
    delete m_input;
    m_input = NULL;
  }

  if (m_ps)
  {
    delete m_ps;
    m_ps = NULL;
  }

  m_time.Set("");
  m_bq.Flush();

  if (clear_peaks)
  {
    g_peaks->ClearPeaks(m_deck, m_idx);
  }
}

void RHEA_Track::Active(bool active)
{
  m_act = m_eof ? false : active;
}

bool RHEA_Track::IsActive() const
{
  return m_act;
}

bool RHEA_Track::IsDrained() const
{
  return m_eof;
}

bool RHEA_Track::WantMore() const
{
  if (m_input && !m_eof)
  {
    double srate = RHEA_GetAudioDeviceSamplerate();
    // duration = filesize / (srate * nch * (bps / 8))
    int duration = (int)(((double)m_bq.GetSize() / (srate * m_nch * sizeof(SAM))) * 1000);
    if (duration < m_bqms) return true;
  }

  return false;
}

bool RHEA_Track::DoBuffering()
{
  if (m_input && !m_eof)
  {
    m_buffer.Resize(2048 * m_input->GetChannels());
    int nsam = m_input->GetSamples(m_buffer.Get(), m_buffer.GetSize());
    if (nsam > 0)
    {
      m_buffer.Resize(nsam, false);

      if (m_input->get_ebur128())
      {
        if (!m_input->get_ebur128downwardonly())
        {
          for (int i = 0; i < m_buffer.GetSize(); i++)
          {
            m_buffer.Get()[i] *= DB2VAL(m_input->get_ebur128gain());
          }
        }
        else if (m_input->get_ebur128downwardonly() &&
          WDL_DefinitelyLessThan(m_input->get_ebur128gain(), 0.0))
        {
          for (int i = 0; i < m_buffer.GetSize(); i++)
          {
            m_buffer.Get()[i] *= DB2VAL(m_input->get_ebur128gain());
          }
        }
      }

      if (m_ps)
      {
        m_ps->set_srate(m_input->GetSampleRate());
        m_ps->set_nch(m_input->GetChannels());
        m_ps->set_shift(m_shift);
        m_ps->set_tempo(m_tempo);

        int spl = m_buffer.GetSize() / m_input->GetChannels();

        SAM *sam = m_ps->GetBuffer(spl);
        memcpy(sam, m_buffer.Get(), m_buffer.GetSize() * sizeof(SAM));
        m_ps->BufferDone(spl);

        int want = (int)((nsam / m_input->GetChannels()) * (1.0 / m_tempo));
        if (want)
        {
          m_buffer.Resize(want * m_input->GetChannels());
          int rd = m_ps->GetSamples(want, m_buffer.Get());
          if (rd) m_buffer.Resize(rd * m_input->GetChannels());
        }
      }

      m_bq.AddBlock(&m_buffer, m_input->GetPosition());
      return true;
    }
    else
    {
      if (m_input->IsReverse())
      {
        if (!m_rpt)
        {
          m_act = false;
          m_input->SetReverse(false);
          m_input->Seek(0.0);
          m_bq.Flush();
        }
        else
        {
          if (m_rpt > 0)
          {
            m_rpt = wdl_clamp(m_rpt - 1, 0, 99);
          }
          m_input->Seek(m_input->GetLength());
          m_bq.Flush();
        }
      }
      else
      {
        if (!m_rpt)
        {
          //m_eof = true;
          m_act = false;

          if (m_deck)
          {
            m_input->SetReverse(true);
            m_input->Seek(m_input->GetLength());
            m_bq.Flush();
          }
          else
          {
            m_input->SetReverse(false);
            m_input->Seek(0.0);
            m_bq.Flush();
          }
        }
        else
        {
          if (m_rpt > 0)
          {
            m_rpt = wdl_clamp(m_rpt - 1, 0, 99);
          }
          m_input->Seek(0.0);
          m_bq.Flush();
        }
      }
    }
  }

  return false;
}

void RHEA_Track::SetTime(const char *time, double timesec)
{
  m_time.Set(time);
  m_timesec = timesec;
}

double RHEA_Track::GetTimeSec() const
{
  return m_timesec;
}

const char *RHEA_Track::GetTime() const
{
  return m_time.Get();
}

const char *RHEA_Track::GetTitle() const
{
  if (m_input) return m_input->GetTitle();
  return "";
}

const char *RHEA_Track::GetArtist() const
{
  if (m_input) return m_input->GetArtist();
  return "";
}

void RHEA_Track::SetShift(double shift)
{
  m_shift = shift;
}

void RHEA_Track::SetTempo(double tempo)
{
  m_tempo = tempo;
}

void RHEA_Track::Reverse(bool state)
{
  if (m_input) m_input->SetReverse(state);
}

bool RHEA_Track::IsReverse() const
{
  if (m_input) return m_input->IsReverse();
  return false;
}

int RHEA_Track::RepeatCount() const
{
  return m_rpt;
}

void RHEA_Track::Repeat(bool infinite)
{
  if (infinite)
  {
    m_rpt = -1;
  }
  else
  {
    if (m_rpt < 0) m_rpt = 0;
    m_rpt = wdl_clamp(m_rpt + 1, 0, 99);
  }
}

void RHEA_Track::ClearRepeat()
{
  m_rpt = 0;
}

void RHEA_Track::Seek(double time)
{
  if (m_input)
  {
    m_bq.Flush();
    m_eof = false;
    m_input->Seek(time);
  }
}
