// Copyright (c) 2023 Giorgos Vougioukas
//
// The license can be found in the LICENSE file.

#include "rhea/track_wnd.h"
#include "rhea/media_input.h"
#include "rhea/hestia.h"

#define RHEA_LOADING 120
#define RHEA_LOADING_MS 333
#define RHEA_TIME 121
#define RHEA_TIME_MS 50

static LOGFONT lf =
{
#if defined(_WIN32)
  14, 0, 0, 0, FW_NORMAL, FALSE, FALSE, FALSE, DEFAULT_CHARSET,
  OUT_DEFAULT_PRECIS, CLIP_DEFAULT_PRECIS, DEFAULT_QUALITY, DEFAULT_PITCH,
  "Arial"
#else
  11, 0, 0, 0, FW_NORMAL, FALSE, FALSE, FALSE, DEFAULT_CHARSET,
  OUT_DEFAULT_PRECIS, CLIP_DEFAULT_PRECIS, DEFAULT_QUALITY, DEFAULT_PITCH,
  "Arial"
#endif
};

static LOGFONT biglf =
{
#if defined(_WIN32)
  20, 0, 0, 0, FW_NORMAL, FALSE, FALSE, FALSE, DEFAULT_CHARSET,
  OUT_DEFAULT_PRECIS, CLIP_DEFAULT_PRECIS, DEFAULT_QUALITY, DEFAULT_PITCH,
  "Arial"
#else
  17, 0, 0, 0, FW_NORMAL, FALSE, FALSE, FALSE, DEFAULT_CHARSET,
  OUT_DEFAULT_PRECIS, CLIP_DEFAULT_PRECIS, DEFAULT_QUALITY, DEFAULT_PITCH,
  "Arial"
#endif
};

RHEA_TrackWnd::RHEA_TrackWnd(bool deck, int index)
  : m_hwnd(NULL)
  , m_deck(deck)
  , m_idx(index)
  , m_medialoading(false)
  , m_bm(NULL)
  , m_font(NULL)
  , m_bigfont(NULL)
  , m_info(NULL)
  , m_time(NULL)
  , m_title(NULL)
  , m_artist(NULL)
{}

RHEA_TrackWnd::~RHEA_TrackWnd()
{
  if (m_font) delete m_font;
  if (m_bigfont) delete m_bigfont;
  m_vwnd.RemoveAllChildren();
}

void RHEA_TrackWnd::Select(bool enable)
{
  if (enable)
  {
    m_info->SetColors(LICE_RGBA(255, 0, 0, 255), LICE_RGBA(51, 51, 51, 255));
  }
  else
  {
    m_info->SetColors(LICE_RGBA(128, 128, 255, 255), LICE_RGBA(51, 51, 51, 255));
  }

  RECT r;
  m_info->GetPosition(&r);
  InvalidateRect(m_hwnd, &r, FALSE);
}

void RHEA_TrackWnd::OnInitDialog(WPARAM wparam, LPARAM lparam)
{
  RECT r;
  GetWindowRect(m_hwnd, &r);
  ScreenToClient(m_hwnd, (LPPOINT)&r);
  ScreenToClient(m_hwnd, ((LPPOINT)&r)+1);

  m_info = new WDL_VirtualStaticText;
  m_time = new WDL_VirtualStaticText;
  m_title = new WDL_VirtualStaticText;
  m_artist = new WDL_VirtualStaticText;
  m_font = new LICE_CachedFont;
  m_font->SetFromHFont(CreateFontIndirect(&lf), LICE_FONT_FLAG_OWNS_HFONT);
  m_font->SetTextColor(LICE_RGBA(0, 255, 255, 255));
  m_font->SetBkMode(TRANSPARENT);
  m_bigfont = new LICE_CachedFont;
  m_bigfont->SetFromHFont(CreateFontIndirect(&biglf), LICE_FONT_FLAG_OWNS_HFONT);
  m_bigfont->SetTextColor(LICE_RGBA(0, 255, 255, 255));
  m_bigfont->SetBkMode(TRANSPARENT);

  RECT t;
  t.left = r.left;
  t.top = r.top;
  t.right = r.right;
  t.bottom = t.top + 20;

  m_info->SetPosition(&t);
  m_info->SetColors(LICE_RGBA(128, 128, 255, 255), LICE_RGBA(51, 51, 51, 255));
  if (m_deck) m_infostr.SetFormatted(512, "D%d", m_idx + 1);
  else m_infostr.SetFormatted(512, "S%d", m_idx + 1);
  m_info->SetText(m_infostr.Get());
  m_info->SetAlign(0);
  m_info->SetID(INFO);
  m_info->SetFont(m_bigfont);
  m_info->SetRealParent(m_hwnd);

  t.top = t.bottom + 10;
  t.bottom = t.top + 20;

  m_time->SetPosition(&t);
  m_time->SetColors(LICE_RGBA(255, 255, 255, 255), LICE_RGBA(51, 51, 51, 255));
  m_time->SetText("00:00");
  m_time->SetAlign(0);
  m_time->SetID(TIME);
  m_time->SetFont(m_bigfont);
  m_time->SetRealParent(m_hwnd);

  t.top = t.bottom + 10;
  t.bottom = t.top + 20;

  m_title->SetPosition(&t);
  m_title->SetColors(LICE_RGBA(255, 255, 255, 255), LICE_RGBA(51, 51, 51, 255));
  m_title->SetText("");
  m_title->SetAlign(0);
  m_title->SetID(TITLE);
  m_title->SetFont(m_font);
  m_title->SetRealParent(m_hwnd);

  t.top = t.bottom;
  t.bottom = t.top + 20;

  m_artist->SetPosition(&t);
  m_artist->SetColors(LICE_RGBA(255, 255, 255, 255), LICE_RGBA(51, 51, 51, 255));
  m_artist->SetText("");
  m_artist->SetAlign(0);
  m_artist->SetID(ARTIST);
  m_artist->SetFont(m_font);
  m_artist->SetRealParent(m_hwnd);

  m_vwnd.SetPosition(&r);
  m_vwnd.SetRealParent(m_hwnd);
  m_vwnd.AddChild(m_info);
  m_vwnd.AddChild(m_time);
  m_vwnd.AddChild(m_title);
  m_vwnd.AddChild(m_artist);
  m_vwnd.RequestRedraw(NULL);

  InvalidateRect(m_hwnd, NULL, FALSE);

  m_resize.init(m_hwnd);
  m_resize.init_itemvirt(&m_vwnd, 0.0f, 0.0f, 1.0f, 1.0f);
  m_resize.init_itemvirt(m_info, 0.0f, 0.0f, 1.0f, 0.0f);
  m_resize.init_itemvirt(m_time, 0.0f, 0.0f, 1.0f, 0.0f);
  m_resize.init_itemvirt(m_title, 0.0f, 0.0f, 1.0f, 0.0f);
  m_resize.init_itemvirt(m_artist, 0.0f, 0.0f, 1.0f, 0.0f);
  SetTimer(m_hwnd, RHEA_LOADING, RHEA_LOADING_MS, NULL);
  SetTimer(m_hwnd, RHEA_TIME, RHEA_TIME_MS, NULL);
  ShowWindow(m_hwnd, SW_SHOW);
}

void RHEA_TrackWnd::OnDestroy(WPARAM wparam, LPARAM lparam)
{
  KillTimer(m_hwnd, RHEA_LOADING);
  KillTimer(m_hwnd, RHEA_TIME);
}

void RHEA_TrackWnd::OnSize(WPARAM wparam, LPARAM lparam)
{
  if (wparam != SIZE_MINIMIZED)
  {
    m_resize.onResize();
  }
}

void RHEA_TrackWnd::OnCommand(WPARAM wparam, LPARAM lparam)
{}

void RHEA_TrackWnd::OnTimer(WPARAM wparam, LPARAM lparam)
{
  if (wparam == RHEA_LOADING)
  {
    bool loading;

    if (m_deck)
    {
      loading = g_mediainput->IsLoading(true, m_idx);
    }
    else
    {
      loading = g_mediainput->IsLoading(false, m_idx);
    }

    if (loading && !m_medialoading)
    {
      WDL_FastString str(m_infostr);
      str.Append(" - loading");
      m_info->SetText(str.Get());
      m_medialoading = true;
      RECT r;
      m_info->GetPosition(&r);
      m_vwnd.RequestRedraw(&r);
      InvalidateRect(m_hwnd, &r, FALSE);
    }
    else if (!loading && m_medialoading)
    {
      m_info->SetText(m_infostr.Get());
      m_medialoading = false;
      RECT r;
      m_info->GetPosition(&r);
      m_vwnd.RequestRedraw(&r);
      InvalidateRect(m_hwnd, &r, FALSE);
    }
  }

  if (wparam == RHEA_TIME)
  {
    if (m_deck)
    {
      m_time->SetText(g_hestia->GetTime(true, m_idx));

      m_strbuf.Set(g_hestia->GetTitle(true, m_idx));
      m_strbuf.Ellipsize(50, 55);
      m_title->SetText(m_strbuf.Get());

      m_strbuf.Set(g_hestia->GetArtist(true, m_idx));
      m_strbuf.Ellipsize(50, 55);
      m_artist->SetText(m_strbuf.Get());
    }
    else
    {
      m_time->SetText(g_hestia->GetTime(false, m_idx));

      m_strbuf.Set(g_hestia->GetTitle(false, m_idx));
      m_strbuf.Ellipsize(26, 30);
      m_title->SetText(m_strbuf.Get());

      m_strbuf.Set(g_hestia->GetArtist(false, m_idx));
      m_strbuf.Ellipsize(26, 30);
      m_artist->SetText(m_strbuf.Get());
    }
  }
}

void RHEA_TrackWnd::OnPaint(WPARAM wparam, LPARAM lparam)
{
  RECT r;
  GetClientRect(m_hwnd, &r);
  //m_vwnd.SetPosition(&r);

  int w = r.right - r.left;
  int h = r.bottom - r.top;

  int xo, yo;
  m_painter.PaintBegin(m_hwnd, RGB(51, 51, 51));
  m_bm = m_painter.GetBuffer(&xo, &yo);

  m_painter.PaintVirtWnd(&m_vwnd);
  m_painter.PaintEnd();
}

WDL_DLGRET RHEA_TrackWnd::TrackWndProc(HWND hwnd, UINT msg, WPARAM wparam, LPARAM lparam)
{
  RHEA_TrackWnd *self = (RHEA_TrackWnd *)GetWindowLongPtr(hwnd, GWLP_USERDATA);

  if (!self && msg == WM_INITDIALOG)
  {
    SetWindowLongPtr(hwnd, GWLP_USERDATA, lparam);
    self = (RHEA_TrackWnd *)lparam;
    self->m_hwnd = hwnd;
  }

  if (WDL_likely(self))
  {
    return self->TrackWndLoop(msg, wparam, lparam);
  }
  else
  {
    return 0;
  }
}

WDL_DLGRET RHEA_TrackWnd::TrackWndLoop(UINT msg, WPARAM wparam, LPARAM lparam)
{
  switch (msg)
  {
    case WM_INITDIALOG: OnInitDialog(wparam, lparam); break;
    case WM_DESTROY: OnDestroy(wparam, lparam); break;
    case WM_SIZE: OnSize(wparam, lparam); break;
    case WM_COMMAND: OnCommand(wparam, lparam); break;
    case WM_TIMER: OnTimer(wparam, lparam); break;
    case WM_PAINT: OnPaint(wparam, lparam); break;
  }

  return 0;
}
