// Copyright (c) 2023 Giorgos Vougioukas
//
// The license can be found in the LICENSE file.

#include "rhea/waveform.h"
#include "rhea/preferences.h"
#include "rhea/main_wnd.h"
#include "rhea/hestia.h"
#include "rhea/peaks.h"

#include "WDL/fpcmp.h"

static LOGFONT lf =
{
#if defined(_WIN32)
  14, 0, 0, 0, FW_NORMAL, FALSE, FALSE, FALSE, DEFAULT_CHARSET,
  OUT_DEFAULT_PRECIS, CLIP_DEFAULT_PRECIS, DEFAULT_QUALITY, DEFAULT_PITCH,
  "Arial"
#else
  11, 0, 0, 0, FW_NORMAL, FALSE, FALSE, FALSE, DEFAULT_CHARSET,
  OUT_DEFAULT_PRECIS, CLIP_DEFAULT_PRECIS, DEFAULT_QUALITY, DEFAULT_PITCH,
  "Arial"
#endif
};

RHEA_Waveform::RHEA_Waveform()
  : m_killthread(false)
  , m_running(false)
  , m_bm(NULL)
{
  g_peaks = new RHEA_Peaks;
  m_bm = new LICE_SysBitmap;
  m_label.SetFromHFont(CreateFontIndirect(&lf), LICE_FONT_FLAG_OWNS_HFONT);
  m_label.SetBkMode(TRANSPARENT);
  m_repeat.SetFromHFont(CreateFontIndirect(&lf), LICE_FONT_FLAG_OWNS_HFONT);
  m_repeat.SetBkMode(TRANSPARENT);
  StartThread();
}

RHEA_Waveform::~RHEA_Waveform()
{
  if (IsRunning()) StopThread();
  if (g_peaks) delete g_peaks;
  if (m_bm) delete m_bm;
}

void RHEA_Waveform::DrawWaveForm(float *pk, size_t sz, int sector,
  int width, double time, double length, bool selected, bool reverse,
  int repeat, const char *label)
{
  int h = 100;
  int hh = h / 2;
  int w = width;

  float avgpk = 0.0f;
  size_t pt = sz / 2 / w;
  size_t idx = 0;
  int lineindex = 0;
  const int spc = 2;

  for (size_t j = 0; j < sz; j += 2)
  {
    avgpk += pk[j];
    if (idx == pt)
    {
      avgpk /= pt;
      int lineheight = wdl_clamp((int)(avgpk * hh), 0, hh);
      int y1 = (hh - lineheight) / 2;
      int y2 = y1 + lineheight;

      if (selected && reverse)
      {
        LICE_Line(m_bm, lineindex, y1 + sector, lineindex, y2 + sector,
          LICE_RGBA(255, 255, 128, 255));
      }
      else if (selected)
      {
        LICE_Line(m_bm, lineindex, y1 + sector, lineindex, y2 + sector,
          LICE_RGBA(192, 192, 192, 255));
      }
      else
      {
        LICE_Line(m_bm, lineindex, y1 + sector, lineindex, y2 + sector,
          LICE_RGBA(128, 128, 128, 255));
      }
      lineindex++;
      idx = 0;
    }
    else
    {
      idx++;
    }
  }

  LICE_DashedLine(m_bm, 0, sector, w, sector, 1, 8, LICE_RGBA(128, 128, 128, 255), 0.40f);
  LICE_DashedLine(m_bm, 0, sector + h, w, sector + h, 1, 8, LICE_RGBA(128, 128, 128, 255), 0.40f);

  lineindex = 0 ; idx = 0; avgpk = 0.0f;
  for (size_t j = 1; j < sz; j += 2)
  {
    avgpk += pk[j];
    if (idx == pt)
    {
      avgpk /= pt;
      int lineheight = wdl_clamp((int)(avgpk * hh), 0, hh);
      int y1 = (hh - lineheight) / 2;
      int y2 = y1 + lineheight;

      if (selected && reverse)
      {
        LICE_Line(m_bm, lineindex, y1 + hh + 1 + spc + sector, lineindex,
          y2 + hh + 1 + spc + sector, LICE_RGBA(255, 255, 128, 255));
      }
      else if (selected)
      {
        LICE_Line(m_bm, lineindex, y1 + hh + 1 + spc + sector, lineindex,
          y2 + hh + 1 + spc + sector, LICE_RGBA(192, 192, 192, 255));
      }
      else
      {
        LICE_Line(m_bm, lineindex, y1 + hh + 1 + spc + sector, lineindex,
          y2 + hh + 1 + spc + sector, LICE_RGBA(128, 128, 128, 255));
      }
      lineindex++;
      idx = 0;
    }
    else
    {
      idx++;
    }
  }

  int timept = 0;
  if (WDL_DefinitelyGreaterThan(length, 0.0))
  {
    timept = (int)(time * lineindex / length);
  }

  RECT r;
  int fw;
  int fh;

  if (selected)
  {
    m_strbuf.Set(label);
    m_label.DrawText(m_bm, m_strbuf.Get(), m_strbuf.GetLength(), &r, DT_CALCRECT);
    fw = r.right - r.left;
    fh = r.bottom - r.top;

    LICE_Line(m_bm, timept, sector, timept, hh * 2 + sector, LICE_RGBA(255, 0, 0, 255));
    if (timept <= fw)
    {
      r.left = timept + 2;
      r.top = sector;
      r.right = r.left + fw;
      r.bottom = r.top + fh;
      m_label.SetTextColor(LICE_RGBA(255, 0, 0, 255));
      m_label.DrawText(m_bm, m_strbuf.Get(), m_strbuf.GetLength(), &r, 0);
    }
    else
    {
      r.left = timept - fw - 2;
      r.top = sector;
      r.right = r.left + fw;
      r.bottom = r.top + fh;
      m_label.SetTextColor(LICE_RGBA(255, 0, 0, 255));
      m_label.DrawText(m_bm, m_strbuf.Get(), m_strbuf.GetLength(), &r, 0);
    }
  }
  else
  {
    m_strbuf.Set(label);
    m_label.DrawText(m_bm, m_strbuf.Get(), m_strbuf.GetLength(), &r, DT_CALCRECT);
    fw = r.right - r.left;
    fh = r.bottom - r.top;

    LICE_Line(m_bm, timept, sector, timept, hh * 2 + sector, LICE_RGBA(128, 128, 255, 255));
    if (timept <= fw)
    {
      r.left = timept + 2;
      r.top = sector;
      r.right = r.left + fw;
      r.bottom = r.top + fh;
      m_label.SetTextColor(LICE_RGBA(128, 128, 255, 255));
      m_label.DrawText(m_bm, m_strbuf.Get(), m_strbuf.GetLength(), &r, 0);
    }
    else
    {
      r.left = timept - fw - 2;
      r.top = sector;
      r.right = r.left + fw;
      r.bottom = r.top + fh;
      m_label.SetTextColor(LICE_RGBA(128, 128, 255, 255));
      m_label.DrawText(m_bm, m_strbuf.Get(), m_strbuf.GetLength(), &r, 0);
    }
  }

  if (selected && reverse && repeat > 0)
  {
    m_strbuf.SetFormatted(128, "R%d", repeat);
    m_repeat.DrawText(m_bm, m_strbuf.Get(), m_strbuf.GetLength(), &r, DT_CALCRECT);
    fw = r.right - r.left;
    fh = r.bottom - r.top;
    r.left = w / 2 - fw;
    r.top = sector + hh - (fh / 2);
    r.right = r.left + fw;
    r.bottom = r.top + fh;
    m_repeat.SetTextColor(LICE_RGBA(255, 255, 128, 255));
    m_repeat.DrawText(m_bm, m_strbuf.Get(), m_strbuf.GetLength(), &r, 0);
  }
  else if (selected && repeat > 0)
  {
    m_strbuf.SetFormatted(128, "R%d", repeat);
    m_repeat.DrawText(m_bm, m_strbuf.Get(), m_strbuf.GetLength(), &r, DT_CALCRECT);
    fw = r.right - r.left;
    fh = r.bottom - r.top;
    r.left = w / 2 - fw;
    r.top = sector + hh - (fh / 2);
    r.right = r.left + fw;
    r.bottom = r.top + fh;
    m_repeat.SetTextColor(LICE_RGBA(192, 192, 192, 255));
    m_repeat.DrawText(m_bm, m_strbuf.Get(), m_strbuf.GetLength(), &r, 0);
  }
  else if (repeat > 0)
  {
    m_strbuf.SetFormatted(128, "R%d", repeat);
    m_repeat.DrawText(m_bm, m_strbuf.Get(), m_strbuf.GetLength(), &r, DT_CALCRECT);
    fw = r.right - r.left;
    fh = r.bottom - r.top;
    r.left = w / 2 - fw;
    r.top = sector + hh - (fh / 2);
    r.right = r.left + fw;
    r.bottom = r.top + fh;
    m_repeat.SetTextColor(LICE_RGBA(128, 128, 128, 255));
    m_repeat.DrawText(m_bm, m_strbuf.Get(), m_strbuf.GetLength(), &r, 0);
  }
  if (selected && reverse && repeat < 0)
  {
    m_strbuf.Set("R\xe2\x88\x9e");
    m_repeat.DrawText(m_bm, m_strbuf.Get(), m_strbuf.GetLength(), &r, DT_CALCRECT);
    fw = r.right - r.left;
    fh = r.bottom - r.top;
    r.left = w / 2 - fw;
    r.top = sector + hh - (fh / 2);
    r.right = r.left + fw;
    r.bottom = r.top + fh;
    m_repeat.SetTextColor(LICE_RGBA(255, 255, 128, 255));
    m_repeat.DrawText(m_bm, m_strbuf.Get(), m_strbuf.GetLength(), &r, 0);
  }
  else if (selected && repeat < 0)
  {
    m_strbuf.Set("R\xe2\x88\x9e");
    m_repeat.DrawText(m_bm, m_strbuf.Get(), m_strbuf.GetLength(), &r, DT_CALCRECT);
    fw = r.right - r.left;
    fh = r.bottom - r.top;
    r.left = w / 2 - fw;
    r.top = sector + hh - (fh / 2);
    r.right = r.left + fw;
    r.bottom = r.top + fh;
    m_repeat.SetTextColor(LICE_RGBA(192, 192, 192, 255));
    m_repeat.DrawText(m_bm, m_strbuf.Get(), m_strbuf.GetLength(), &r, 0);
  }
  else if (repeat < 0)
  {
    m_strbuf.Set("R\xe2\x88\x9e");
    m_repeat.DrawText(m_bm, m_strbuf.Get(), m_strbuf.GetLength(), &r, DT_CALCRECT);
    fw = r.right - r.left;
    fh = r.bottom - r.top;
    r.left = w / 2 - fw;
    r.top = sector + hh - (fh / 2);
    r.right = r.left + fw;
    r.bottom = r.top + fh;
    m_repeat.SetTextColor(LICE_RGBA(128, 128, 128, 255));
    m_repeat.DrawText(m_bm, m_strbuf.Get(), m_strbuf.GetLength(), &r, 0);
  }
}

LICE_IBitmap *RHEA_Waveform::GetBitMap(int width, int height)
{
  if (g_peaks && g_hestia)
  {
    int sector = 0;
    m_bm->resize(width, 12 * 100 /* height */);

    LICE_Clear(m_bm, LICE_RGBA(31, 31, 31, 255));

    bool act, deck; int index;
    g_mainwnd->GetSelectedTrack(&act, &deck, &index);

    for (int i = 0; i < g_peaks->DeckCount(); i++, sector += 100)
    {
      if (g_peaks->HasPeaks(true, i))
      {
        float *pk = g_peaks->GetPeaks(true, i);
        size_t sz = g_peaks->GetPeaksSize(true, i);

        if (pk && sz)
        {
          double time = g_hestia->GetTimeSec(true, i);
          double length = g_hestia->GetLength(true, i);
          bool rev = g_hestia->IsReverse(true, i);
          int rpt = g_hestia->RepeatCount(true, i);
          m_strbuf.SetFormatted(128, "D%d", i + 1);
          if (act && deck && index == i)
          {
            DrawWaveForm(pk, sz, sector, width, time, length, true, rev, rpt, m_strbuf.Get());
          }
          else
          {
            DrawWaveForm(pk, sz, sector, width, time, length, false, rev, rpt, m_strbuf.Get());
          }
          g_peaks->DoneWithPeaks();
        }
      }
    }

    for (int i = 0; i < g_peaks->SamplerCount(); i++, sector += 100)
    {
      if (g_peaks->HasPeaks(false, i))
      {
        float *pk = g_peaks->GetPeaks(false, i);
        size_t sz = g_peaks->GetPeaksSize(false, i);

        if (pk && sz)
        {
          double time = g_hestia->GetTimeSec(false, i);
          double length = g_hestia->GetLength(false, i);
          bool rev = g_hestia->IsReverse(false, i);
          int rpt = g_hestia->RepeatCount(false, i);
          m_strbuf.SetFormatted(128, "S%d", i + 1);
          if (act && !deck && index == i)
          {
            DrawWaveForm(pk, sz, sector, width, time, length, true, rev, rpt, m_strbuf.Get());
          }
          else
          {
            DrawWaveForm(pk, sz, sector, width, time, length, false, rev, rpt, m_strbuf.Get());
          }
          g_peaks->DoneWithPeaks();
        }
      }
    }
  }

  return m_bm;
}

void RHEA_Waveform::StartThread()
{
  WDL_ASSERT(!m_threads.GetSize());

  int priority;
  m_killthread = false;

  m_threads.Resize(1);
  //m_threads.Resize(RHEA_GetCPUCores());
  for (int i = 0; i < m_threads.GetSize(); i++)
  {
    ThreadDetails *rt = m_threads.Get();
    rt[i].thread = (HANDLE)_beginthreadex(NULL, 0,
      ThreadFunction, (void *)this, 0, &rt[i].id);

    priority = g_preferences->GetDiskIOPriority();
    SetThreadPriority(rt[i].thread, priority);
  }

  m_running = true;
}

void RHEA_Waveform::StopThread()
{
  m_killthread = true;
  for (int i = 0; i < m_threads.GetSize(); i++)
  {
    ThreadDetails *rt = m_threads.Get();
    WaitForSingleObject(rt[i].thread, INFINITE);
    CloseHandle(rt[i].thread);
    m_threads.Resize(0);
  }

  m_running = false;
}

bool RHEA_Waveform::IsRunning() const
{
  return m_running;
}

int RHEA_Waveform::Run()
{


  return 1;
}

unsigned int WINAPI RHEA_Waveform::ThreadFunction(void *arg)
{
#if defined(_WIN32)
  CoInitialize(NULL);
#endif

  RHEA_Waveform *self = (RHEA_Waveform *)arg;

  if (WDL_NORMALLY(self))
  {
    int sleepstep = g_preferences->GetDiskIOSleepStep();

    self->m_killthread = false;

    while (!self->m_killthread)
    {
      self->m_mutex.Enter();
      while (!self->Run());
      self->m_mutex.Leave();
      Sleep(sleepstep);
    }
  }

#if defined(_WIN32)
  CoUninitialize();
#endif

  return 0;
}
