// Copyright (c) 2023 Giorgos Vougioukas
//
// The license can be found in the LICENSE file.

#include "rhea/peaks.h"

#include <string.h>
#include <math.h>

RHEA_Peaks::RHEA_Peaks()
{
  for (int i = 0; i < 4; i++)
  {
    m_dpeaks.Add(new PeaksBlock);
  }

  for (int i = 0; i < 8; i++)
  {
    m_speaks.Add(new PeaksBlock);
  }
}

RHEA_Peaks::~RHEA_Peaks()
{
  m_dpeaks.Empty(true);
  m_speaks.Empty(true);
}

void RHEA_Peaks::PreAllocate(double time)
{
  WDL_MutexLock lock(&m_mutex);
  const double newtime = wdl_max(300.0, floor(time) + 10.0);
  const size_t request = (size_t)(newtime * 400 * 2 * sizeof(float));

  for (int i = 0; i < m_dpeaks.GetSize(); i++)
  {
    Allocate(request, true, i);
  }

  for (int i = 0; i < m_speaks.GetSize(); i++)
  {
    Allocate(request, false, i);
  }
}

bool RHEA_Peaks::Allocate(size_t request, bool deck, int index)
{
  WDL_MutexLock lock(&m_mutex);
  if (deck)
  {
    WDL_ASSERT(index < m_dpeaks.GetSize());
    PeaksBlock *pb = m_dpeaks.Get(index);
    if (pb)
    {
      size_t cap = request / sizeof(float);

      if (!pb->pk)
      {
        pb->rdy = false;
        pb->pk = (float *)malloc(request);
        if (pb->pk)
        {
          pb->cp = cap;
          pb->sz = 0u;
          return true;
        }
        else
        {
          pb->cp = 0u;
          pb->sz = 0u;
          return false;
        }
      }
      else
      {
        if (cap > pb->cp)
        {
          pb->rdy = false;
          float *tmp = (float *)realloc(pb->pk, request);
          if (tmp)
          {
            pb->pk = tmp;
            pb->cp = cap;
            pb->sz = 0u;
            return true;
          }
          else
          {
            free(pb->pk);
            pb->pk = NULL;
            pb->cp = 0u;
            pb->sz = 0u;
            return false;
          }
        }
        else
        {
          pb->rdy = false;
          pb->sz = 0u;
          return true;
        }
      }
    }
  }
  else
  {
    WDL_ASSERT(index < m_speaks.GetSize());
    PeaksBlock *pb = m_speaks.Get(index);
    if (pb)
    {
      size_t cap = request / sizeof(float);

      if (!pb->pk)
      {
        pb->rdy = false;
        pb->pk = (float *)malloc(request);
        if (pb->pk)
        {
          pb->cp = cap;
          pb->sz = 0u;
          return true;
        }
        else
        {
          pb->cp = 0u;
          pb->sz = 0u;
          return false;
        }
      }
      else
      {
        if (cap > pb->cp)
        {
          pb->rdy = false;
          float *tmp = (float *)realloc(pb->pk, request);
          if (tmp)
          {
            pb->pk = tmp;
            pb->cp = cap;
            pb->sz = 0u;
            return true;
          }
          else
          {
            free(pb->pk);
            pb->pk = NULL;
            pb->cp = 0u;
            pb->sz = 0u;
            return false;
          }
        }
        else
        {
          pb->rdy = false;
          pb->sz = 0u;
          return true;
        }
      }
    }
  }

  return false;
}

void RHEA_Peaks::AddPeaks(float *peaks, size_t len, bool deck, int index)
{
  WDL_MutexLock lock(&m_mutex);
  if (deck)
  {
    WDL_ASSERT(index < m_dpeaks.GetSize());
    PeaksBlock *pb = m_dpeaks.Get(index);
    if (pb)
    {
      for (size_t i = 0; i < len; i++)
      {
        pb->pk[pb->sz + i] = peaks[i];
      }

      pb->sz += len;

      WDL_ASSERT(pb->sz <= pb->cp);
    }
  }
  else
  {
    WDL_ASSERT(index < m_speaks.GetSize());
    PeaksBlock *pb = m_speaks.Get(index);
    if (pb)
    {
      for (size_t i = 0; i < len; i++)
      {
        pb->pk[pb->sz + i] = peaks[i];
      }

      pb->sz += len;

      WDL_ASSERT(pb->sz <= pb->cp);
    }
  }
}

void RHEA_Peaks::PeaksReady(bool deck, int index)
{
  WDL_MutexLock lock(&m_mutex);
  if (deck)
  {
    WDL_ASSERT(index < m_dpeaks.GetSize());
    PeaksBlock *pb = m_dpeaks.Get(index);
    if (pb && pb->pk && pb->sz > 0u)
    {
      pb->rdy = true;
    }
  }
  else
  {
    WDL_ASSERT(index < m_speaks.GetSize());
    PeaksBlock *pb = m_speaks.Get(index);
    if (pb && pb->pk && pb->sz > 0u)
    {
      pb->rdy = true;
    }
  }
}

void RHEA_Peaks::ClearPeaks(bool deck, int index)
{
  WDL_MutexLock lock(&m_mutex);
  if (deck)
  {
    WDL_ASSERT(index < m_dpeaks.GetSize());
    PeaksBlock *pb = m_dpeaks.Get(index);
    if (pb && pb->pk)
    {
      pb->rdy = false;
      pb->sz = 0u;
    }
  }
  else
  {
    WDL_ASSERT(index < m_speaks.GetSize());
    PeaksBlock *pb = m_speaks.Get(index);
    if (pb && pb->pk)
    {
      pb->rdy = false;
      pb->sz = 0u;
    }
  }
}

bool RHEA_Peaks::HasPeaks(bool deck, int index) const
{
  if (deck)
  {
    WDL_ASSERT(index < m_dpeaks.GetSize());
    PeaksBlock *pb = m_dpeaks.Get(index);
    if (pb && pb->pk && pb->rdy)
    {
      return pb->sz > 0u;
    }
  }
  else
  {
    WDL_ASSERT(index < m_speaks.GetSize());
    PeaksBlock *pb = m_speaks.Get(index);
    if (pb && pb->pk && pb->rdy)
    {
      return pb->sz > 0u;
    }
  }

  return false;
}

float *RHEA_Peaks::GetPeaks(bool deck, int index)
{
  if (deck)
  {
    WDL_ASSERT(index < m_dpeaks.GetSize());
    PeaksBlock *pb = m_dpeaks.Get(index);
    if (pb && pb->pk && pb->rdy)
    {
      m_mutex.Enter();
      return pb->pk;
    }
  }
  else
  {
    WDL_ASSERT(index < m_speaks.GetSize());
    PeaksBlock *pb = m_speaks.Get(index);
    if (pb && pb->pk && pb->rdy)
    {
      m_mutex.Enter();
      return pb->pk;
    }
  }

  return NULL;
}

size_t RHEA_Peaks::GetPeaksSize(bool deck, int index) const
{
  if (deck)
  {
    WDL_ASSERT(index < m_dpeaks.GetSize());
    PeaksBlock *pb = m_dpeaks.Get(index);
    if (pb && pb->pk && pb->rdy)
    {
      return pb->sz;
    }
  }
  else
  {
    WDL_ASSERT(index < m_speaks.GetSize());
    PeaksBlock *pb = m_speaks.Get(index);
    if (pb && pb->pk && pb->rdy)
    {
      return pb->sz;
    }
  }

  return 0u;
}
