// Copyright (c) 2023 Giorgos Vougioukas
//
// The license can be found in the LICENSE file.

#include "rhea/waveform_wnd.h"
#include "rhea/main_wnd.h"

#include "WDL/wingui/scrollbar/coolscroll.h"

#define RHEA_WAVEFORMPOS 130
#define RHEA_WAVEFORMPOS_MS 50

RHEA_WaveformWnd::RHEA_WaveformWnd(HWND hwnd)
  : m_hwnd(hwnd)
{}

RHEA_WaveformWnd::~RHEA_WaveformWnd()
{}

void RHEA_WaveformWnd::EnsureVisible(bool deck, int index)
{
  SCROLLINFO si = { sizeof(SCROLLINFO), SIF_POS|SIF_PAGE|SIF_RANGE|SIF_TRACKPOS, 0 };
  CoolSB_GetScrollInfo(m_hwnd, SB_VERT, &si);
  int pos = CoolSB_GetScrollPos(m_hwnd, SB_VERT);

  if (deck)
  {
    int wfpos = 100 * index;
    if (wfpos + 100 > pos + (int)si.nPage || wfpos < pos)
    {
      si.nPos = wfpos;
    }
  }
  else
  {
    int wfpos = 400 + (100 * index);
    if (wfpos + 100 > pos + (int)si.nPage || wfpos < pos)
    {
      si.nPos = wfpos;
    }
  }
  CoolSB_SetScrollInfo(m_hwnd, SB_VERT, &si, TRUE);
}

void RHEA_WaveformWnd::OnCreate(WPARAM wparam, LPARAM lparam)
{
  InitializeCoolSB(m_hwnd);

  RECT r;
  GetWindowRect(m_hwnd, &r);
  ScreenToClient(m_hwnd, (LPPOINT)&r);
  ScreenToClient(m_hwnd, ((LPPOINT)&r)+1);

  SCROLLINFO si = 
  {
    sizeof(si),
    SIF_PAGE | SIF_POS | SIF_RANGE,
    0,
    r.bottom - r.top,
    r.bottom,
    0,
  };

  CoolSB_SetScrollInfo(m_hwnd, SB_VERT, &si, TRUE);
  CoolSB_ShowScrollBar(m_hwnd, SB_VERT, FALSE);
  SetTimer(m_hwnd, RHEA_WAVEFORMPOS, RHEA_WAVEFORMPOS_MS, NULL);
}

void RHEA_WaveformWnd::OnDestroy(WPARAM wparam, LPARAM lparam)
{
  KillTimer(m_hwnd, RHEA_WAVEFORMPOS);
  UninitializeCoolSB(m_hwnd);
}

void RHEA_WaveformWnd::OnSize(WPARAM wparam, LPARAM lparam)
{
  if (wparam != SIZE_MINIMIZED)
  {
    m_resize.onResize();

    if (!CoolSB_IsCoolScrollEnabled(m_hwnd))
    {
      CoolSB_ShowScrollBar(m_hwnd, SB_VERT, TRUE);
    }
  }
}

void RHEA_WaveformWnd::OnCommand(WPARAM wparam, LPARAM lparam)
{}

void RHEA_WaveformWnd::OnPaint(WPARAM wparam, LPARAM lparam)
{
  RECT r;
  GetClientRect(m_hwnd, &r);
  int w = r.right - r.left;
  int h = r.bottom - r.top;

  int xo, yo;
  m_painter.PaintBegin(m_hwnd, RGB(31, 31, 31));

  LICE_IBitmap *pnt = m_painter.GetBuffer(&xo, &yo);
  LICE_IBitmap *bm = m_wf.GetBitMap(w, h);
  if (pnt && bm)
  {
    int pos = CoolSB_GetScrollPos(m_hwnd, SB_VERT);
    SCROLLINFO si =
    {
      sizeof(si),
      SIF_PAGE|SIF_POS|SIF_RANGE,
      0,
      bm->getHeight(),
      h,
      pos,
    };
    CoolSB_SetScrollInfo(m_hwnd, SB_VERT, &si, TRUE);
    LICE_Blit(pnt, bm, 0, 0, 0, pos, bm->getWidth(), bm->getHeight(), 1.0f, LICE_BLIT_MODE_COPY);
  }

  m_painter.PaintEnd();
}

void RHEA_WaveformWnd::OnLButtonDown(WPARAM wparam, LPARAM lparam)
{
  //SetFocus(m_hwnd);
}

void RHEA_WaveformWnd::OnLButtonUp(WPARAM wparam, LPARAM lparam)
{}

void RHEA_WaveformWnd::OnMouseMove(WPARAM wparam, LPARAM lparam)
{}

void RHEA_WaveformWnd::OnTimer(WPARAM wparam, LPARAM lparam)
{
  if (wparam == RHEA_WAVEFORMPOS)
  {
    InvalidateRect(m_hwnd, NULL, FALSE);
  }
}

void RHEA_WaveformWnd::OnVScroll(WPARAM wparam, LPARAM lparam)
{
  SCROLLINFO si = { sizeof(SCROLLINFO), SIF_POS|SIF_PAGE|SIF_RANGE|SIF_TRACKPOS, 0 };
  CoolSB_GetScrollInfo(m_hwnd, SB_VERT, &si);
  switch (LOWORD(wparam))
  {
  case SB_THUMBTRACK:
    si.nPos = si.nTrackPos;
    break;
  case SB_LINEUP:
    si.nPos -= 50;
    break;
  case SB_LINEDOWN:
    si.nPos += 50;
    break;
  case SB_PAGEUP:
    si.nPos -= (int)si.nPage;
    break;
  case SB_PAGEDOWN:
    si.nPos += (int)si.nPage;
    break;
  case SB_TOP:
    si.nPos = 0;
    break;
  case SB_BOTTOM:
    si.nPos = si.nMax-(int)si.nPage;
    break;
  }
  if (si.nPos > si.nMax-(int)si.nPage) si.nPos = si.nMax-(int)si.nPage;
  if (si.nPos < 0) si.nPos = 0;

  CoolSB_SetScrollPos(m_hwnd, SB_VERT, si.nPos, TRUE);
  InvalidateRect(m_hwnd, NULL, FALSE);
}

void RHEA_WaveformWnd::OnKeyDown(WPARAM wparam, LPARAM lparam)
{
  switch (wparam)
  {
  case VK_HOME:
    {
      CoolSB_SetScrollPos(m_hwnd, SB_VERT, 0, TRUE);
      InvalidateRect(m_hwnd, NULL, FALSE);
    }
    break;
  case VK_END:
    {
      SCROLLINFO si = { sizeof(SCROLLINFO), SIF_POS|SIF_PAGE|SIF_RANGE|SIF_TRACKPOS, 0 };
      CoolSB_GetScrollInfo(m_hwnd, SB_VERT, &si);

      CoolSB_SetScrollPos(m_hwnd, SB_VERT, si.nMax, TRUE);
      InvalidateRect(m_hwnd, NULL, FALSE);
    }
    break;
  case VK_PRIOR:
    {
      SCROLLINFO si = { sizeof(SCROLLINFO), SIF_POS|SIF_PAGE|SIF_RANGE|SIF_TRACKPOS, 0 };
      CoolSB_GetScrollInfo(m_hwnd, SB_VERT, &si);

      int pos = si.nPos - si.nPage;
      if (pos < 0) pos = 0;

      CoolSB_SetScrollPos(m_hwnd, SB_VERT, pos, TRUE);
      InvalidateRect(m_hwnd, NULL, FALSE);
    }
    break;
  case VK_NEXT:
    {
      SCROLLINFO si = { sizeof(SCROLLINFO), SIF_POS|SIF_PAGE|SIF_RANGE|SIF_TRACKPOS, 0 };
      CoolSB_GetScrollInfo(m_hwnd, SB_VERT, &si);

      int pos = si.nPos + si.nPage;
      if (pos > si.nMax) pos = si.nMax;

      CoolSB_SetScrollPos(m_hwnd, SB_VERT, pos, TRUE);
      InvalidateRect(m_hwnd, NULL, FALSE);
    }
    break;
  case VK_UP:
    {
      SCROLLINFO si = { sizeof(SCROLLINFO), SIF_POS|SIF_PAGE|SIF_RANGE|SIF_TRACKPOS, 0 };
      CoolSB_GetScrollInfo(m_hwnd, SB_VERT, &si);

      int pos = si.nPos - 100;
      if (pos < 0) pos = 0;

      CoolSB_SetScrollPos(m_hwnd, SB_VERT, pos, TRUE);
      InvalidateRect(m_hwnd, NULL, FALSE);
    }
    break;
  case VK_DOWN:
    {
      SCROLLINFO si = { sizeof(SCROLLINFO), SIF_POS|SIF_PAGE|SIF_RANGE|SIF_TRACKPOS, 0 };
      CoolSB_GetScrollInfo(m_hwnd, SB_VERT, &si);

      int pos = si.nPos + 100;
      if (pos > si.nMax) pos = si.nMax;

      CoolSB_SetScrollPos(m_hwnd, SB_VERT, pos, TRUE);
      InvalidateRect(m_hwnd, NULL, FALSE);
    }
    break;
  }
}

void RHEA_WaveformWnd::OnMouseWheel(WPARAM wparam, LPARAM lparam)
{
  POINT p = { GET_X_LPARAM(lparam), GET_Y_LPARAM(lparam) };
  ScreenToClient(m_hwnd, &p);

  int l = (short)HIWORD(wparam);
  float pos = -l / 120.0f;

  SCROLLINFO si = { sizeof(SCROLLINFO), SIF_POS|SIF_PAGE|SIF_RANGE|SIF_TRACKPOS, 0 };
  CoolSB_GetScrollInfo(m_hwnd, SB_VERT, &si);

  pos *= 100 /* (int)si.nPage */;
  pos += si.nPos;

  const float mv = (float)(si.nMax - 100 /* (int)si.nPage) */);
  if (pos > mv) pos = mv;
  if (pos < 0) pos = 0;

  CoolSB_SetScrollPos(m_hwnd, SB_VERT, (int)(pos + 0.5), TRUE);
  InvalidateRect(m_hwnd, NULL, FALSE);
}

WDL_DLGRET RHEA_WaveformWnd::WaveformWndProc(HWND hwnd, UINT msg, WPARAM wparam, LPARAM lparam)
{
  g_waveformwnd = (RHEA_WaveformWnd *)GetWindowLongPtr(hwnd, GWLP_USERDATA);

#ifdef _WIN32
  if (!g_waveformwnd && msg == WM_NCCREATE)
#else
  if (!g_waveformwnd && msg == WM_CREATE)
#endif
  {
    g_waveformwnd = new RHEA_WaveformWnd(hwnd);
    SetWindowLongPtr(hwnd, GWLP_USERDATA, (INT_PTR)g_waveformwnd);
#ifndef _WIN32
    //SWELL_SetClassName(hwnd, "waveform_superhost");
#endif
    if (g_waveformwnd) g_waveformwnd->OnCreate(wparam, lparam);
    return (g_waveformwnd != NULL);
  }

#ifdef _WIN32
  if (g_waveformwnd && msg == WM_NCDESTROY)
#else
  if (g_waveformwnd && msg == WM_DESTROY)
#endif
  {
    g_waveformwnd->OnDestroy(wparam, lparam);
    delete g_waveformwnd; g_waveformwnd = NULL;
  }

  if (WDL_likely(g_waveformwnd))
  {
    return g_waveformwnd->WaveformWndLoop(msg, wparam, lparam);
  }
  else
  {
    return 0;
  }
}

WDL_DLGRET RHEA_WaveformWnd::WaveformWndLoop(UINT msg, WPARAM wparam, LPARAM lparam)
{
  switch (msg)
  {
#ifdef _WIN32
    case WM_NCCREATE: OnCreate(wparam, lparam); break;
    case WM_NCDESTROY: OnDestroy(wparam, lparam); break;
#else
    case WM_CREATE: OnCreate(wparam, lparam); break;
    case WM_DESTROY: OnDestroy(wparam, lparam); break;
#endif
    case WM_SIZE: OnSize(wparam, lparam); break;
    case WM_COMMAND: OnCommand(wparam, lparam); break;
    case WM_PAINT: OnPaint(wparam, lparam); break;
    case WM_LBUTTONDOWN: OnLButtonDown(wparam, lparam); break;
    case WM_LBUTTONUP: OnLButtonUp(wparam, lparam); break;
    case WM_MOUSEMOVE: OnMouseMove(wparam, lparam); break;
    case WM_TIMER: OnTimer(wparam, lparam); break;
    case WM_VSCROLL: OnVScroll(wparam, lparam); break;
    case WM_KEYDOWN: OnKeyDown(wparam, lparam); break;
    case WM_MOUSEWHEEL: OnMouseWheel(wparam, lparam); break;
    //case RHEA_ENSUREVISIBLE: EnsureVisible(wparam, lparam); break;
  }

  return DefWindowProc(m_hwnd, msg, wparam, lparam);
}
