// Copyright (c) 2022 Giorgos Vougioukas
//
// The license can be found in the LICENSE file.

#if defined(_WIN32) && defined(_DEBUG)
#include <vld.h>
#endif

//#define ENABLE_RHEA_OPUS_CONSOLE

#include "rhea/rhea_plugin_public.h"
#include "rhea_opus/opus_input.h"
#include "rhea_opus/opus_apic.h"
#include "rhea_opus/opus_tag.h"
#include "rhea_opus/opus_entry_point.h"

#include "WDL/wdlcstring.h"

double (*RHEA_GetAudioDeviceSamplerate)();
void (*RHEA_GetDiskReadMode)(int *rmode, int *rbufsize, int *rnbufs);
void (*RHEA_GetResampleMode)(bool *interp, int *filtercnt,
  bool *sinc, int *sinc_size, int *sinc_interpsize);

static HINSTANCE opus_instance;
static HWND opus_main_hwnd;

//opus_int64 tha_filelength(void *user_data)
//{
//  WDL_FileRead *fr = (WDL_FileRead *)user_data;
//  return (opus_int64)fr->GetSize();
//}

int tha_seek(void *user_data, opus_int64 offset, int whence)
{
  WDL_FileRead *fr = (WDL_FileRead *)user_data;

  switch (whence)
  {
  case SEEK_SET:
    {
      fr->SetPosition((WDL_INT64)offset);
    } break;
  case SEEK_CUR:
    {
      fr->SetPosition(fr->GetPosition() + (WDL_INT64)offset);
    } break;
  case SEEK_END:
    {
      fr->SetPosition(fr->GetSize() + (WDL_INT64)offset);
    } break;
  default: break;
  }

  return 0;
}

int tha_read(void *user_data, unsigned char *ptr, int count)
{
  WDL_FileRead *fr = (WDL_FileRead *)user_data;

  if (fr->GetPosition() + (WDL_INT64)count > fr->GetSize())
  {
    count = (int)(fr->GetSize() - fr->GetPosition());
  }

  fr->Read((unsigned char *)ptr, (int)count);
  return count;
}

//opus_int64 tha_write(const void *ptr, opus_int64 count, void *user_data)
//{
//  WDL_FileWrite *fw = (WDL_FileWrite *)user_data;
//  fw->Write(ptr, (int)count);
//  return count;
//}

opus_int64 tha_tell(void *user_data)
{
  WDL_FileRead *fr = (WDL_FileRead *)user_data;
  return (opus_int64)fr->GetPosition();
}

RHEA_IFileInput *CreateFromType(const char *type)
{
  if (!strcmp(type, "OPUS"))
  {
    return new RHEA_OpusInput;
  }

  return NULL;
}

RHEA_IFileInput *CreateFromFile(const char *filename)
{
  if (!wdl_filename_cmp(WDL_get_fileext(filename), ".opus"))
  {
    RHEA_OpusInput *p = new RHEA_OpusInput;

    if (p->Open(filename))
    {
      return p;
    }

    delete p;
  }

  return NULL;
}

struct RHEA_FileInputRegister opus_input_reg =
{
  CreateFromType,
  CreateFromFile,
};

RHEA_IFileTag *CreateOpusTag(const char *filename)
{
  if (!wdl_filename_cmp(WDL_get_fileext(filename), ".opus"))
  {
    RHEA_OpusTag *p = new RHEA_OpusTag;

    if (p && p->Open(filename))
    {
      return p;
    }
    else
    {
      delete p;
    }
  }

  return NULL;
}

RHEA_FileTagRegister opus_tag_reg =
{
  &CreateOpusTag
};

RHEA_IFilePic *CreateOpusAPIC(const char *filename)
{
  if (!wdl_filename_cmp(WDL_get_fileext(filename), ".opus"))
  {
    RHEA_OpusAPIC *p = new RHEA_OpusAPIC;

    if (p->Open(filename))
    {
      return p;
    }
    else
    {
      delete p;
    }
  }

  return NULL;
}

RHEA_FilePicRegister opus_apic_reg =
{
  &CreateOpusAPIC
};

extern "C"
{
  RHEA_PLUGIN_EXPORT int RHEA_PLUGIN_ENTRYPOINT(
    RHEA_PLUGIN_HINSTANCE instance, RHEA_PluginInfo *rec)
  {
#if defined(_WIN32) && defined(_DEBUG) && defined(ENABLE_RHEA_OPUS_CONSOLE)
    // The first will use the console only if
    // the application is started by the command 
    // line. The second one will always enable
    // the console.
    //if (AttachConsole(ATTACH_PARENT_PROCESS))
    if (AttachConsole(ATTACH_PARENT_PROCESS) || AllocConsole())
    {
        freopen("CONOUT$", "w", stdout);
        freopen("CONOUT$", "w", stderr);
    }
#endif

    opus_instance = instance;

    if (rec)
    {
      if (rec->caller_version != RHEA_PLUGIN_VERSION || !rec->GetFunc)
      {
        return 0;
      }

      opus_main_hwnd = rec->hwnd_main;

      *((void **)&RHEA_GetAudioDeviceSamplerate) = rec->GetFunc("RHEA_GetAudioDeviceSamplerate");
      *((void **)&RHEA_GetDiskReadMode) = rec->GetFunc("RHEA_GetDiskReadMode");
      *((void **)&RHEA_GetResampleMode) = rec->GetFunc("RHEA_GetResampleMode");

      if (!rec->Register || !RHEA_GetAudioDeviceSamplerate || !RHEA_GetDiskReadMode ||
        !RHEA_GetResampleMode)
      {
        return 0;
      }

      rec->Register("input", &opus_input_reg);
      rec->Register("tag", &opus_tag_reg);
      //rec->Register("apic", &opus_apic_reg);

      return 1;
    }
    else
    {
      return 0;
    }
  }
}
