// Copyright (c) 2022 Giorgos Vougioukas
//
// The license can be found in the LICENSE file.

#include "terpsichore/antidote.h"
#include "terpsichore/plugin.h"

#include <math.h>

#include "WDL/sha.h"
#include "WDL/lice/lice.h"
#include "WDL/assocarray.h"
#include "WDL/filewrite.h"
#include "WDL/fileread.h"
#include "WDL/pcmfmtcvt.h"
#include "WDL/heapbuf.h"
#include "WDL/queue.h"
#include "WDL/fpcmp.h"

RST_Antidote::RST_Antidote()
  : m_thread(NULL)
  , m_killthread(false)
  , m_cancel(false)
  , m_transcoding(false)
  , m_antidote(0)
  , m_transcodeprogress(0)
  , m_isdeck(false)
  , m_deck(0)
  , m_stt(NULL)
  , m_ebur128normalization(0)
  , m_ebur128downwardonly(0)
  , m_ebur128reference(0.0)
  , m_ebur128mode(0)
  , m_ebur128momentary(-HUGE_VAL)
  , m_ebur128shortterm(-HUGE_VAL)
  , m_ebur128momentarywindow(400)
  , m_ebur128shorttermwindow(3000)
  , m_ebur128integrated(-HUGE_VAL)
  , m_ebur128range(-HUGE_VAL)
  , m_ebur128samplepeak(-HUGE_VAL)
  , m_ebur128truepeak(-HUGE_VAL)
  , m_ebur128gain(-HUGE_VAL)
  , m_momentarywindow(0)
  , m_shorttermwindow(0)
  , m_peakrate(400)
{}

RST_Antidote::~RST_Antidote()
{
  StopThread();

  for (int i = 0; i < m_releaseref.GetSize(); i++)
  {
    const char *key;
    m_releaseref.Enumerate(i, &key);
    DeleteFile(key);

    WDL_FastString pkfn(key);
    pkfn.Append(".pk");
    DeleteFile(pkfn.Get());
  }

  m_releaseref.DeleteAll();

  if (m_stt) ebur128_destroy(&m_stt);
}

void RST_Antidote::Transcode(const char *filename, bool is_deck, int deck)
{
  m_ebur128normalization = g_ini_file->read_int("ebur128_normalization", 0, "preferences");
  m_ebur128downwardonly = g_ini_file->read_int("ebur128_downward_only", 1, "preferences");
  int ebur128reference = g_ini_file->read_int("ebur128_reference", 0, "preferences");
  int ebur128mode = g_ini_file->read_int("ebur128_mode", 2, "preferences");

  switch (ebur128reference)
  {
    case 0: m_ebur128reference = -23.0; break;
    case 1: m_ebur128reference = -18.0; break;
    case 2: m_ebur128reference = -16.0; break;
    case 3: m_ebur128reference = -14.0; break;
    case 4: m_ebur128reference = -11.0; break;
    default: m_ebur128reference = -23.0; break;
  }

  switch (ebur128mode)
  {
    case 0: m_ebur128mode = EBUR128_MODE_M; break;
    case 1: m_ebur128mode = EBUR128_MODE_M | EBUR128_MODE_S; break;
    case 2: m_ebur128mode = EBUR128_MODE_M | EBUR128_MODE_S | EBUR128_MODE_I; break;
    case 3: m_ebur128mode = EBUR128_MODE_M | EBUR128_MODE_S | EBUR128_MODE_I | EBUR128_MODE_LRA; break;
    case 4: m_ebur128mode = EBUR128_MODE_M | EBUR128_MODE_S | EBUR128_MODE_I |
              EBUR128_MODE_LRA | EBUR128_MODE_SAMPLE_PEAK; break;
    case 5: m_ebur128mode = EBUR128_MODE_M | EBUR128_MODE_S | EBUR128_MODE_I |
              EBUR128_MODE_LRA | EBUR128_MODE_SAMPLE_PEAK |
              EBUR128_MODE_TRUE_PEAK; break;
    //case 6: m_ebur128mode = EBUR128_MODE_M | EBUR128_MODE_S | EBUR128_MODE_I |
    //          EBUR128_MODE_LRA | EBUR128_MODE_SAMPLE_PEAK |
    //          EBUR128_MODE_TRUE_PEAK | EBUR128_MODE_HISTOGRAM; break;
    default: m_ebur128mode = EBUR128_MODE_M | EBUR128_MODE_S | EBUR128_MODE_I; break;
  }

  m_isdeck = is_deck;
  m_deck = deck;
  m_fn.Set(filename);
  m_tcfn.Set(g_settings_path.Get());

  m_tcfn.Append("cache" WDL_DIRCHAR_STR);
  CreateDirectory(m_tcfn.Get(), NULL);

  WDL_SHA1 sha;
  char hash[WDL_SHA1SIZE];
  sha.reset();
  sha.add(m_fn.Get(), m_fn.GetLength());
  sha.result(hash);

  m_tcfn.AppendFormatted(64, "%02x", (unsigned char)(hash[0] & 0xff));
  m_tcfn.Append(WDL_DIRCHAR_STR);
  CreateDirectory(m_tcfn.Get(), NULL);

  for (int i = 1; i < WDL_SHA1SIZE; i++)
  {
    m_tcfn.AppendFormatted(64, "%02x", (unsigned char)(hash[i] & 0xff));
  }

  if (!m_thread) StartThread();

  m_antidote = g_ini_file->read_int("antidote", 0, "preferences");

  int rr = m_releaseref.Get(m_tcfn.Get());

  WDL_ASSERT(rr >= 0);

  if (!rr)
  {
    WDL_FileRead *fr = new WDL_FileRead(m_tcfn.Get());
    bool fileexists = fr->IsOpen();
    delete fr;

    if (fileexists) DeleteFile(m_tcfn.Get());
    m_releaseref.Insert(m_tcfn.Get(), 1);
  }
  else
  {
    rr++;
    m_releaseref.Insert(m_tcfn.Get(), rr);
  }

  m_cancel = false;
  m_transcodeprogress = 0;
  m_transcoding = true;
}

bool RST_Antidote::IsTranscoding() const
{
  return m_transcoding;
}

int RST_Antidote::TranscodingProgress() const
{
  return m_transcodeprogress;
}

void RST_Antidote::AbortTranscoding()
{
  m_cancel = true;
  m_transcoding = false;
}

const char *RST_Antidote::GetFilename() const
{
  return m_tcfn.Get();
}

void RST_Antidote::ReleaseFilename(const char *filename)
{
  int rr = m_releaseref.Get(filename);
  if (rr)
  {
    rr--;

    WDL_ASSERT(rr >= 0);

    if (!rr)
    {
      m_releaseref.Delete(filename);
      DeleteFile(filename);

      WDL_FastString pkfn(filename);
      pkfn.Append(".pk");
      DeleteFile(pkfn.Get());
    }
    else
    {
      m_releaseref.Insert(filename, rr);
    }
  }
}

int RST_Antidote::Run()
{
  if (m_transcoding)
  {
    RST_IFileInput *fi = CreateAudioInput(m_fn.Get());

    if (fi)
    {
      int nread = 0;
      WDL_HeapBuf rawbuf;
      WDL_TypedBuf<SAM> buf;
      buf.Resize(4096);

      if (m_ebur128normalization)
      {
        WDL_ASSERT(m_stt == NULL);
        m_stt = ebur128_init((unsigned int)fi->GetChannels(),
          (unsigned long)fi->GetSampleRate(), m_ebur128mode);
        ebur128_set_channel(m_stt, 0, EBUR128_LEFT);
        ebur128_set_channel(m_stt, 1, EBUR128_RIGHT);
      }

      int readmode = g_ini_file->read_int("read_mode", 2, "preferences");
      int readbuffer = g_ini_file->read_int("read_buffer", 262144, "preferences");
      int readbuffers = g_ini_file->read_int("read_buffers", 3, "preferences");
      int writemode = g_ini_file->read_int("write_mode", 1, "preferences");
      int writebuffer = g_ini_file->read_int("write_buffer", 65536, "preferences");
      int minwritebuffers = g_ini_file->read_int("min_write_buffers", 16, "preferences");
      int maxwritebuffers = g_ini_file->read_int("max_write_buffers", 128, "preferences");

      WDL_FileRead *tcfr = new WDL_FileRead(m_tcfn.Get(), readmode, readbuffer, readbuffers);
      bool notcfile = !tcfr->IsOpen();
      delete tcfr;

      WDL_FastString pkfn(m_tcfn.Get());
      pkfn.Append(".pk");

      WDL_FileRead *pkfr = new WDL_FileRead(pkfn.Get(), readmode, readbuffer, readbuffers);
      bool nopkfile = !pkfr->IsOpen();
      delete pkfr;

      WDL_FileWrite *tcfw = NULL;
      if (notcfile) tcfw = new WDL_FileWrite(m_tcfn.Get(), writemode, writebuffer, minwritebuffers, maxwritebuffers);

      WDL_FileWrite *pkfw = NULL;
      if (nopkfile) pkfw = new WDL_FileWrite(pkfn.Get(), writemode, writebuffer, minwritebuffers, maxwritebuffers);

      const int sratepts = m_peakrate;
      const int nch = fi->GetChannels();
      const double trklen = floor(fi->GetLength()) + 2.0;
      const int sratestep = (int)(GetHardwareSampleRate() * nch / sratepts);
      const WDL_INT64 requestcount = (WDL_INT64)(trklen * sratepts * nch);
      const WDL_INT64 request = requestcount * sizeof(float);

      WDL_TypedQueue<float> stepsam;

      do
      {
        buf.Resize(4096);

        nread = fi->GetSamples(buf.Get(), buf.GetSize());

        buf.Resize(nread, false);

        if (m_ebur128normalization)
        {
          size_t frames = (size_t)(buf.GetSize() / fi->GetChannels());
          ebur128_add_frames_double(m_stt, buf.Get(), frames);

          m_momentarywindow += (int)(frames / fi->GetSampleRate() * 1000);
          m_shorttermwindow += (int)(frames / fi->GetSampleRate() * 1000);

          if ((m_ebur128mode & EBUR128_MODE_M) == EBUR128_MODE_M &&
            m_momentarywindow >= m_ebur128momentary)
          {
            double tmp;
            int res = ebur128_loudness_momentary(m_stt, &tmp);
            if (res != EBUR128_SUCCESS) m_ebur128normalization = 0;
            m_ebur128momentary = wdl_max(m_ebur128momentary, tmp);
            m_momentarywindow = 0;
          }

          if ((m_ebur128mode & EBUR128_MODE_S) == EBUR128_MODE_S &&
            m_shorttermwindow >= m_ebur128shorttermwindow)
          {
            double tmp;
            int res = ebur128_loudness_shortterm(m_stt, &tmp);
            if (res != EBUR128_SUCCESS) m_ebur128normalization = 0;
            m_ebur128shortterm = wdl_max(m_ebur128shortterm, tmp);
            m_shorttermwindow = 0;
          }

          if ((m_ebur128mode & EBUR128_MODE_SAMPLE_PEAK) == EBUR128_MODE_SAMPLE_PEAK)
          {
            for (int i = 0; i < fi->GetChannels(); i++)
            {
              double tmp;
              int res = ebur128_sample_peak(m_stt, (unsigned int)i, &tmp);
              if (res != EBUR128_SUCCESS) m_ebur128normalization = 0;
              m_ebur128samplepeak = wdl_max(m_ebur128samplepeak, tmp);
            }
          }

          if ((m_ebur128mode & EBUR128_MODE_TRUE_PEAK) == EBUR128_MODE_TRUE_PEAK)
          {
            for (int i = 0; i < fi->GetChannels(); i++)
            {
              double tmp;
              int res = ebur128_true_peak(m_stt, (unsigned int)i, &tmp);
              if (res != EBUR128_SUCCESS) m_ebur128normalization = 0;
              m_ebur128truepeak = wdl_max(m_ebur128truepeak, tmp);
            }
          }
        }

        switch (m_antidote)
        {
        case 0:
          {
            rawbuf.Resize(buf.GetSize() * sizeof(short));
            doublesToPcm(buf.Get(), 1, buf.GetSize(), rawbuf.Get(), 16, 1);
          } break;
        case 1:
          {
            rawbuf.Resize(buf.GetSize() * sizeof(char) * 3);
            doublesToPcm(buf.Get(), 1, buf.GetSize(), rawbuf.Get(), 24, 1);
          } break;
        case 2:
          {
            rawbuf.Resize(buf.GetSize() * sizeof(int));
            doublesToPcm(buf.Get(), 1, buf.GetSize(), rawbuf.Get(), 32, 1);
          } break;
        case 3:
          {
            rawbuf.Resize(buf.GetSize() * sizeof(float));
            float *flt = (float *)rawbuf.Get();
            for (int i = 0; i < buf.GetSize(); i++)
            {
              flt[i] = (float)buf.Get()[i];
            }
          } break;
        case 4:
          {
            rawbuf.Resize(buf.GetSize() * sizeof(double));
            double *dbl = (double *)rawbuf.Get();
            for (int i = 0; i < buf.GetSize(); i++)
            {
              dbl[i] = (double)buf.Get()[i];
            }
          } break;
        }

        m_transcodeprogress = 100 * (int)fi->GetPosition() / (int)fi->GetLength();

        if (tcfw && tcfw->IsOpen()) tcfw->Write(rawbuf.Get(), rawbuf.GetSize());

        WDL_TypedBuf<float> tmpsam;

        for (int i = 0; i < buf.GetSize(); i += nch)
        {
          tmpsam.Resize(nch);

          for (int j = 0; j < tmpsam.GetSize(); j++)
          {
            tmpsam.Get()[j] = (float)buf.Get()[i + j];
            tmpsam.Get()[j] = wdl_abs(tmpsam.Get()[j]);
            tmpsam.Get()[j] = wdl_clamp(tmpsam.Get()[j], 0.0f, 1.0f);
          }

          stepsam.Add(tmpsam.Get(), tmpsam.GetSize());

          if (stepsam.Available() >= sratestep)
          {
#if 0 // rms points
            WDL_TypedBuf<float> sqr;
            sqr.Resize(nch);
            sqr.SetToZero();

            for (int j = 0; j < stepsam.GetSize(); j += nch)
            {
              for (int k = 0; k < nch; k++)
              {
                sqr.Get()[k] += stepsam.Get()[j + k] * stepsam.Get()[j + k];
              }
            }

            for (int k = 0; k < nch; k++)
            {
              float mean = sqr.Get()[k] / (stepsam.GetSize() / nch);
              float rms = sqrtf(mean);
              if (pkfw && pkfw->IsOpen())
              {
                pkfw->Write(&rms, sizeof(float));
              }
            }
#endif
#if 0 // min/max points:
            WDL_TypedBuf<float> mn;
            mn.Resize(nch);
            mn.SetToZero();

            WDL_TypedBuf<float> mx;
            mx.Resize(nch);
            mx.SetToZero();

            for (int j = 0; j < stepsam.GetSize(); j += nch)
            {
              for (int k = 0; k < nch; k++)
              {
                float ss = stepsam.Get()[j + k];
                if (WDL_DefinitelyGreaterThan(ss, 0.0f))
                {
                  mx.Get()[k] = wdl_max(ss, mx.Get()[k]);
                  mx.Get()[k] = wdl_clamp(mx.Get()[k], 0.0f, 1.0f);
                }
                else if (WDL_DefinitelyLessThan(ss, 0.0f))
                {
                  mn.Get()[k] = wdl_min(ss, mn.Get()[k]);
                  mn.Get()[k] = wdl_clamp(mn.Get()[k], -1.0f, 0.0f);
                }
              }
            }

            for (int k = 0; k < nch; k++)
            {
              float mnmx = ((mx.Get()[k]) + (mn.Get()[k]));
              mnmx = wdl_abs(mnmx);
              mnmx = wdl_clamp(mnmx, 0.0f, 1.0f);
              if (pkfw && pkfw->IsOpen())
              {
                pkfw->Write(&mnmx, sizeof(float));
              }
            }

            // alternative implementation

            //WDL_TypedBuf<float> minmax;
            //int minmaxsize = 2; // 0: min 1: max
            //minmax.Resize(minmaxsize * nch);
            //minmax.SetToZero();

            //for (int j = 0; j < stepsam.GetSize(); j += nch)
            //{
            //  for (int k = 0; k < nch; k++)
            //  {
            //    float ss = wdl_clamp(stepsam.Get()[j + k], -1.0f, 1.0f);
            //    minmax.Get()[minmaxsize * k + 0] = wdl_min(ss, minmax.Get()[minmaxsize * k + 0]);
            //    minmax.Get()[minmaxsize * k + 1] = wdl_max(ss, minmax.Get()[minmaxsize * k + 1]);
            //  }
            //}

            //if (pkfw && pkfw->IsOpen())
            //{
            //  pkfw->Write(minmax.Get(), minmax.GetSize() * sizeof(float));
            //}
#endif
#if 1 // average points
            WDL_TypedBuf<float> avg;
            avg.Resize(nch);
            avg.SetToZero();

            for (int j = 0; j < stepsam.GetSize(); j += nch)
            {
              for (int k = 0; k < nch; k++)
              {
                avg.Get()[k] += stepsam.Get()[j + k];
              }
            }

            for (int k = 0; k < nch; k++)
            {
              avg.Get()[k] *= 2; // diff -1.0 - +1.0
              avg.Get()[k] /= stepsam.GetSize() / nch;
              if (pkfw && pkfw->IsOpen())
              {
                pkfw->Write(&avg.Get()[k], sizeof(float));
              }
            }
#endif
#if 0 // raw sample points
            WDL_TypedBuf<float> rawpt;
            rawpt.Resize(nch);
            rawpt.SetToZero();

            for (int k = 0; k < nch; k++)
            {
              rawpt.Get()[k] += stepsam.Get()[k];
            }

            for (int k = 0; k < nch; k++)
            {
              if (pkfw && pkfw->IsOpen())
              {
                pkfw->Write(&rawpt.Get()[k], sizeof(float));
              }
            }
#endif
            stepsam.Clear();
          }
        }
      } while (nread && !m_cancel);

      if (m_ebur128normalization && (m_ebur128mode & EBUR128_MODE_M) == EBUR128_MODE_M)
      {
        m_ebur128gain = m_ebur128reference - m_ebur128momentary;
      }

      if (m_ebur128normalization && (m_ebur128mode & EBUR128_MODE_S) == EBUR128_MODE_S)
      {
        m_ebur128gain = m_ebur128reference - m_ebur128shortterm;
      }

      if (m_ebur128normalization && (m_ebur128mode & EBUR128_MODE_I) == EBUR128_MODE_I)
      {
        int res = ebur128_loudness_global(m_stt, &m_ebur128integrated);
        if (res == EBUR128_SUCCESS)
        {
          m_ebur128gain = m_ebur128reference - m_ebur128integrated;
        }
        else
        {
          m_ebur128normalization = 0;
        }
      }

      if (m_ebur128normalization && (m_ebur128mode & EBUR128_MODE_LRA) == EBUR128_MODE_LRA)
      {
        int res = ebur128_loudness_range(m_stt, &m_ebur128range);
        if (res != EBUR128_SUCCESS)
        {
          m_ebur128normalization = 0;
        }
      }

      delete fi;
      if (tcfw) delete tcfw;
      if (pkfw) delete pkfw;
      if (m_stt) { ebur128_destroy(&m_stt); m_stt = NULL; }

      WDL_StringKeyedArray<double> ebur128inf;
      ebur128inf.Insert("ebur128normalization", (double)m_ebur128normalization);
      ebur128inf.Insert("ebur128downwardonly", (double)m_ebur128downwardonly);
      ebur128inf.Insert("ebur128reference", m_ebur128reference);
      ebur128inf.Insert("ebur128mode", (double)m_ebur128mode);
      ebur128inf.Insert("ebur128momentary", m_ebur128momentary);
      ebur128inf.Insert("ebur128shortterm", m_ebur128shortterm);
      ebur128inf.Insert("ebur128momentarywindow", (double)m_ebur128momentarywindow);
      ebur128inf.Insert("ebur128shorttermwindow", (double)m_ebur128shorttermwindow);
      ebur128inf.Insert("ebur128integrated", m_ebur128integrated);
      ebur128inf.Insert("ebur128range", m_ebur128range);
      ebur128inf.Insert("ebur128samplepeak", m_ebur128samplepeak);
      ebur128inf.Insert("ebur128truepeak", m_ebur128truepeak);
      ebur128inf.Insert("ebur128gain", m_ebur128gain);

      if (m_isdeck && !m_cancel)
      {
        switch (m_deck)
        {
        case 0:
          {
            g_main_loop_mutex.Enter();
            g_main_loop->LoadDeckA(m_fn.Get(), m_peakrate, &ebur128inf);
            g_main_loop_mutex.Leave();
          } break;
        case 1:
          {
            g_main_loop_mutex.Enter();
            g_main_loop->LoadDeckB(m_fn.Get(), m_peakrate, &ebur128inf);
            g_main_loop_mutex.Leave();
          } break;
        }
      }
    }

    if (m_cancel) ReleaseFilename(m_tcfn.Get());

    m_transcoding = false;
  }

  return 1;
}

void RST_Antidote::StartThread()
{
  WDL_ASSERT(m_thread == NULL);

  if (!m_thread)
  {
    unsigned int thread_id;
    m_thread = (HANDLE)_beginthreadex(NULL, 0, ThreadFunction, (void *)this, 0, &thread_id);
  }
}

void RST_Antidote::StopThread()
{
  m_cancel = true;
  m_killthread = true;

  if (m_thread)
  {
    WaitForSingleObject(m_thread, INFINITE);
    CloseHandle(m_thread);
    m_thread = NULL;
  }
}

unsigned int WINAPI RST_Antidote::ThreadFunction(void *arg)
{
#if defined(_WIN32)
  CoInitialize(NULL);
#endif

  RST_Antidote *self = (RST_Antidote *)arg;

  if (WDL_NORMALLY(self))
  {
    self->m_killthread = false;

    while (!self->m_killthread)
    {
      self->m_mutex.Enter();
      while (!self->Run());
      self->m_mutex.Leave();
      Sleep(50);
    }
  }

#if defined(_WIN32)
  CoUninitialize();
#endif

  return 0;
}
