// Copyright (c) 2017 Giorgos Vougioukas
//
// The license can be found in the LICENSE file.

#include "terpsichore/buffer_queue.h"

RST_BufferQueue::RST_BufferQueue()
  : m_size_bytes(0)
{}

RST_BufferQueue::~RST_BufferQueue()
{
  m_blocks.Empty(true);
  m_empty_blocks.Empty(true);
}

void RST_BufferQueue::AddBlock(WDL_TypedBuf<SAM> *block, double start_time)
{
  BQ_Block *b = NULL;

  WDL_MutexLock lock(&m_mutex);

  if (m_empty_blocks.GetSize() > 0)
  {
    int tail = m_empty_blocks.GetSize() - 1;
    b = m_empty_blocks.Get(tail);
    m_empty_blocks.Delete(tail);
    b->samq.Add(block->Get(), block->GetSize());
    b->stime = start_time;
    m_blocks.Add(b);
  }
  else
  {
    b = new BQ_Block;
    b->samq.Add(block->Get(), block->GetSize());
    b->stime = start_time;
    m_blocks.Add(b);
  }

  m_size_bytes += b->samq.GetSize() * sizeof(SAM);
}

bool RST_BufferQueue::GetBlock(BQ_Block **block)
{
  WDL_MutexLock lock(&m_mutex);

  if (m_blocks.GetSize() > 0)
  {
    *block = m_blocks.Get(0);
    m_blocks.Delete(0);

    m_size_bytes -= (*block)->samq.GetSize() * sizeof(SAM);

    return true;
  }

  return false;
}

void RST_BufferQueue::ReturnBlock(BQ_Block *block)
{
  WDL_MutexLock lock(&m_mutex);

  m_size_bytes += block->samq.GetSize() * sizeof(SAM);
  m_blocks.Insert(0, block);
}

void RST_BufferQueue::DisposeBlock(BQ_Block *block)
{
  WDL_MutexLock lock(&m_mutex);

  block->samq.Clear();
  m_empty_blocks.Add(block);
}

int RST_BufferQueue::GetSize() const
{
  //wdl_log("bq block size: %d, bq empty block size: %d\n",
  //  m_blocks.GetSize(), m_empty_blocks.GetSize());

  return m_size_bytes;
}

double RST_BufferQueue::GetNextRunTime()
{
  WDL_MutexLock lock(&m_mutex);

  if (m_blocks.GetSize())
  {
    double rt = m_blocks.Get(0)->stime;
    Flush();
    return rt;
  }

  return -1.0;
}

void RST_BufferQueue::Flush()
{
  WDL_MutexLock lock(&m_mutex);

  for (int i = 0; i < m_blocks.GetSize(); i++)
  {
    BQ_Block *block = m_blocks.Get(i);
    block->samq.Clear();
    m_empty_blocks.Add(block);
  }

  m_blocks.Empty();

  m_size_bytes = 0;
}

void RST_BufferQueue::Empty()
{
  WDL_MutexLock lock(&m_mutex);

  m_blocks.Empty(true);
  m_empty_blocks.Empty(true);

  m_size_bytes = 0;
}
