// Copyright (c) 2019 Giorgos Vougioukas
//
// The license can be found in the LICENSE file.

#include "terpsichore/file_tasks.h"
#include "terpsichore/main_wnd.h"

#include "WDL/wdltypes.h"

#define RST_OPEN_MEDIA_FILES \
  "All supported media files\0*.flac;*.mp3\0" \
  "FLAC files (*.FLAC)\0*.flac\0" \
  "MPEG audio files (*.MP3)\0*.mp3\0" \
  "All files (*.*)\0*.*\0\0"
#define RST_OPEN_MEDIA_FILE_DEFEXT "mp3"

#define RST_OPEN_PLAYLISTS \
  "All supported playlists\0*.tpl\0" \
  "Terpsichore playlists (*.TPL)\0*.tpl\0" \
  "All files (*.*)\0*.*\0\0"
#define RST_OPEN_PLAYLIST_DEFEXT "tpl"

#define RST_SAVE_PLAYLISTS \
  "Terpsichore playlists (*.TPL)\0*.tpl\0" \
  "All files (*.*)\0*.*\0\0"
#define RST_SAVE_PLAYLIST_DEFEXT "tpl"

static int sort_by_file_path(const void *a, const void *b)
{
  WDL_FastString *s1, *s2;
  s1 = *(WDL_FastString **)a;
  s2 = *(WDL_FastString **)b;
  return stricmp(s1->Get(), s2->Get());
}

RST_FileTasks::RST_FileTasks()
  : m_hwnd(NULL)
  , m_open_folder(false)
  , m_open_files(false)
  , m_add_folder(false)
  , m_add_files(false)
  , m_load_pl(false)
  , m_save_pl(false)
  , m_thread(NULL)
  , m_kill_thread(false)
  , m_want_progress(false)
  , m_scanning(false)
  , m_cancel(false)
  , m_idx(0)
  , m_selfiles(NULL)
{}

RST_FileTasks::~RST_FileTasks()
{
  StopThread();

  if (m_selfiles) free(m_selfiles);
}

void RST_FileTasks::SetParent(HWND parent)
{
  m_hwnd = parent;
  StartThread();
}

void RST_FileTasks::ScanFiles(const char *path, bool empty_file_cache)
{
  WDL_DirScan dir;
  WDL_FastString fn;

  m_scanning = true;

  m_lp.parse(g_media_ext.Get());

  if (empty_file_cache)
  {
    EmptyFileCache();
  }

  if (!dir.First(path))
  {
    do
    {
      if (dir.GetCurrentIsDirectory())
      {
        dir.GetCurrentFullFN(&fn);

        if (strcmp(fn.get_filepart(), ".") &&
            strcmp(fn.get_filepart(), ".."))
        {
          ScanFiles(fn.Get(), false);
        }
      }
      else
      {
        WDL_FastString *str = new WDL_FastString();

        if (str)
        {
          dir.GetCurrentFullFN(str);

          bool valid_ext = false;

          for (int i = 0; i < m_lp.getnumtokens(); i++)
          {
            if (wdl_filename_cmp(str->get_fileext(), m_lp.gettoken_str(i)) == 0)
            {
              valid_ext = true;
              break;
            }
          }

          if (valid_ext)
          {
            m_file_cache.Add(str);
          }
          else
          {
            delete str;
          }
        }
        else
        {
          delete str;
        }
      }
    }
    while (!dir.Next() && !m_cancel);
  }

  qsort(m_file_cache.GetList(), m_file_cache.GetSize(),
    sizeof(WDL_FastString *), sort_by_file_path);

  m_scanning = false;
}

void RST_FileTasks::OpenFolder()
{
  m_selected_dir.Resize(2048);

  if (WDL_ChooseDirectory(m_hwnd, "Open folder", NULL,
    m_selected_dir.Get(), m_selected_dir.GetSize(), true))
  {
    m_open_folder = true;
  }
}

void RST_FileTasks::OpenFiles()
{
  if (m_selfiles) { free(m_selfiles); m_selfiles = NULL; }

  m_selfiles = WDL_ChooseFileForOpen(m_hwnd, "Open files", NULL, NULL,
    RST_OPEN_MEDIA_FILES, RST_OPEN_MEDIA_FILE_DEFEXT, true, true);

  if (m_selfiles) m_open_files = true;
}

void RST_FileTasks::AddFolder()
{
  m_selected_dir.Resize(2048);

  if (WDL_ChooseDirectory(m_hwnd, "Add folder", NULL,
    m_selected_dir.Get(), m_selected_dir.GetSize(), true))
  {
    m_add_folder = true;
  }
}

void RST_FileTasks::AddFiles()
{
  if (m_selfiles) { free(m_selfiles); m_selfiles = NULL; }

  m_selfiles = WDL_ChooseFileForOpen(m_hwnd, "Add files", NULL, NULL,
    RST_OPEN_MEDIA_FILES, RST_OPEN_MEDIA_FILE_DEFEXT, true, true);

  if (m_selfiles) m_add_files = true;
}

void RST_FileTasks::LoadPlayList()
{
  if (m_selfiles) { free(m_selfiles); m_selfiles = NULL; }

  m_selfiles = WDL_ChooseFileForOpen(m_hwnd, "Load playlist", NULL, NULL,
    RST_OPEN_PLAYLISTS, RST_OPEN_PLAYLIST_DEFEXT, true, true);

  if (m_selfiles) m_load_pl = true;
}

void RST_FileTasks::SavePlayList()
{
  char fn[2048];

  if (WDL_ChooseFileForSave(m_hwnd, "Save playlist", NULL, NULL,
    RST_SAVE_PLAYLISTS, RST_SAVE_PLAYLIST_DEFEXT, true, fn, sizeof(fn)))
  {
    g_playlist_wnd->SavePlayList(fn);
  }

  //m_save_pl = true;
}

void RST_FileTasks::OnOpenFolder()
{
  WantProgress();
  ScanFiles(m_selected_dir.Get());

  if (m_cancel)
  {
    EmptyFileCache();
    return;
  }

  g_playlist_wnd->RemoveAll();

  SendMessage(GetDlgItem(g_playlist_wnd->Handle(), IDC_LIST1),
    WM_SETREDRAW, FALSE, 0);

  for (m_idx = 0; m_idx < m_file_cache.GetSize(); m_idx++)
  {
    m_curfn = m_file_cache.Get(m_idx);
    g_playlist_wnd->AddItem(m_curfn->Get());
    if (m_cancel) break;
  }

  SendMessage(GetDlgItem(g_playlist_wnd->Handle(), IDC_LIST1),
    WM_SETREDRAW, TRUE, 0);

  EmptyFileCache();
}

void RST_FileTasks::OnOpenFiles()
{
  const char *only_one = m_selfiles;

  while (*only_one++);

  if (*only_one == '\0' && *(only_one + 1) == '\0')
  {
    g_playlist_wnd->RemoveAll();
    g_playlist_wnd->AddItem(m_selfiles);
  }
  else
  {
    WantProgress();
    g_playlist_wnd->RemoveAll();

    WDL_FastString path;

    path.Set(m_selfiles);

    char *s = m_selfiles;
    while (*s++);

    do
    {
      WDL_FastString *fn = new WDL_FastString();

      if (fn)
      {
        fn->Set(path.Get());
        fn->Append(WDL_DIRCHAR_STR);
        fn->Append(s);
        m_file_cache.Add(fn);
      }

      while (*s++);
    } while (*s != '\0' && *(s + 1) != '\0');

    SendMessage(GetDlgItem(g_playlist_wnd->Handle(), IDC_LIST1),
      WM_SETREDRAW, FALSE, 0);

    for (m_idx = 0; m_idx < m_file_cache.GetSize(); m_idx++)
    {
      m_curfn = m_file_cache.Get(m_idx);
      g_playlist_wnd->AddItem(m_curfn->Get());
      if (m_cancel) break;
    }

    SendMessage(GetDlgItem(g_playlist_wnd->Handle(), IDC_LIST1),
      WM_SETREDRAW, TRUE, 0);

    EmptyFileCache();
  }
}

void RST_FileTasks::OnAddFolder()
{
  WantProgress();
  ScanFiles(m_selected_dir.Get());

  if (m_cancel)
  {
    EmptyFileCache();
    return;
  }

  SendMessage(GetDlgItem(g_playlist_wnd->Handle(), IDC_LIST1),
    WM_SETREDRAW, FALSE, 0);

  for (m_idx = 0; m_idx < m_file_cache.GetSize(); m_idx++)
  {
    m_curfn = m_file_cache.Get(m_idx);
    g_playlist_wnd->AddItem(m_curfn->Get());
    if (m_cancel) break;
  }

  SendMessage(GetDlgItem(g_playlist_wnd->Handle(), IDC_LIST1),
    WM_SETREDRAW, TRUE, 0);

  EmptyFileCache();
}

void RST_FileTasks::OnAddFiles()
{
  const char *only_one = m_selfiles;

  while (*only_one++);

  if (*only_one == '\0' && *(only_one + 1) == '\0')
  {
    g_playlist_wnd->AddItem(m_selfiles);
  }
  else
  {
    WantProgress();

    WDL_FastString path;

    path.Set(m_selfiles);

    char *s = m_selfiles;
    while (*s++);

    do
    {
      WDL_FastString *fn = new WDL_FastString();

      if (fn)
      {
        fn->Set(path.Get());
        fn->Append(WDL_DIRCHAR_STR);
        fn->Append(s);
        m_file_cache.Add(fn);
      }

      while (*s++);
    } while (*s != '\0' && *(s + 1) != '\0');

    SendMessage(GetDlgItem(g_playlist_wnd->Handle(), IDC_LIST1),
      WM_SETREDRAW, FALSE, 0);

    for (m_idx = 0; m_idx < m_file_cache.GetSize(); m_idx++)
    {
      m_curfn = m_file_cache.Get(m_idx);
      g_playlist_wnd->AddItem(m_curfn->Get());
      if (m_cancel) break;
    }

    SendMessage(GetDlgItem(g_playlist_wnd->Handle(), IDC_LIST1),
      WM_SETREDRAW, TRUE, 0);

    EmptyFileCache();
  }
}

void RST_FileTasks::OnLoadPlayList()
{
  const char *only_one = m_selfiles;

  while (*only_one++);

  if (*only_one == '\0' && *(only_one + 1) == '\0')
  {
    g_playlist_wnd->RemoveAll();

    if (!g_playlist_wnd->LoadPlayList(m_selfiles))
    {
      WDL_FastString err;

      err.SetFormatted(2048, "Terpsichore was unable to load:\n%s", m_selfiles);

      MessageBox(g_main_wnd->Handle(), err.Get(), "Terpsichore error", MB_OK);
    }
  }
  else
  {
    WantProgress();

    WDL_FastString path;

    path.Set(m_selfiles);

    char *s = m_selfiles;
    while (*s++);

    do
    {
      WDL_FastString *fn = new WDL_FastString();

      if (fn)
      {
        fn->Set(path.Get());
        fn->Append(WDL_DIRCHAR_STR);
        fn->Append(s);
        m_file_cache.Add(fn);
      }

      while (*s++);
    } while (*s != '\0' && *(s + 1) != '\0');

    for (m_idx = 0; m_idx < m_file_cache.GetSize(); m_idx++)
    {
      if (!m_idx)
      {
        g_playlist_wnd->RemoveAll();
      }

      m_curfn = m_file_cache.Get(m_idx);

      if (!g_playlist_wnd->LoadPlayList(m_curfn->Get()))
      {
        WDL_FastString err;

        err.SetFormatted(2048,
          "Terpsichore was unable to load:\n%s",
          m_curfn->Get());

        MessageBox(g_main_wnd->Handle(), err.Get(), "Terpsichore error", MB_OK);

        break;
      }

      if (m_cancel) break;
    }

    EmptyFileCache();
  }
}

void RST_FileTasks::OnSavePlayList()
{}

bool RST_FileTasks::WantProgressWindow() const
{
  return m_want_progress;
}

const char *RST_FileTasks::FileInProgress() const
{
  if (m_scanning) return "Preparing...";
  if (m_curfn) return m_curfn->get_filepart();

  return "";
}

int RST_FileTasks::TotalFiles() const
{
  return m_file_cache.GetSize();
}

int RST_FileTasks::CurrentIndex() const
{
  return m_idx;
}

void RST_FileTasks::AbortOperation()
{
  m_cancel = true;
}

void RST_FileTasks::EmptyFileCache()
{
  m_idx = 0;
  m_curfn = NULL;
  m_file_cache.Empty(true);
  m_want_progress = m_scanning ? true : false;
}

void RST_FileTasks::WantProgress()
{
  m_cancel = false;
  m_want_progress = true;
}

int RST_FileTasks::Run()
{
  if (m_open_folder)
  {
    OnOpenFolder();
    m_open_folder = false;
  }
  else if (m_open_files)
  {
    OnOpenFiles();
    m_open_files = false;
  }
  else if (m_add_folder)
  {
    OnAddFolder();
    m_add_folder = false;
  }
  else if (m_add_files)
  {
    OnAddFiles();
    m_add_files = false;
  }
  else if (m_load_pl)
  {
    OnLoadPlayList();
    m_load_pl = false;
  }
  else if (m_save_pl)
  {
    OnSavePlayList();
    m_save_pl = false;
  }

  return 1;
}

void RST_FileTasks::StartThread()
{
  WDL_ASSERT(m_thread == NULL);

  if (!m_thread)
  {
    unsigned int thread_id;
    m_thread = (HANDLE)_beginthreadex(NULL, 0, ThreadFunction, (void *)this, 0, &thread_id);
  }
}

void RST_FileTasks::StopThread()
{
  m_cancel = true;
  m_kill_thread = true;

  if (m_thread)
  {
    WaitForSingleObject(m_thread, INFINITE);
    CloseHandle(m_thread);
    m_thread = NULL;
  }
}

unsigned int WINAPI RST_FileTasks::ThreadFunction(void *arg)
{
#if defined(_WIN32)
  CoInitialize(NULL);
#endif

  RST_FileTasks *self = (RST_FileTasks *)arg;

  if (WDL_NORMALLY(self))
  {
    self->m_kill_thread = false;

    while (!self->m_kill_thread)
    {
      self->m_mutex.Enter();
      while (!self->Run());
      self->m_mutex.Leave();
      Sleep(50);
    }
  }

#if defined(_WIN32)
  CoUninitialize();
#endif

  return 0;
}
