// Copyright (c) 2019 Giorgos Vougioukas
//
// The license can be found in the LICENSE file.

#include "terpsichore/keyboard_map.h"
#include "terpsichore/definitions.h"

#include "WDL/aggarray.h"

static const KeyboardHotkey keyboard_hotkey[] =
{
  // Key, Ctrl, Menu, Shift, Action
  { VK_ESCAPE, 0, 0, 0, "EatEscape" },
  { VK_SPACE, 0, 0, 0, "ShowPlayList" },
  { 'Q', 1, 0, 0, "ExitTerpsichore" },
  { 'S', 1, 0, 0, "PL_SortMenu" },
  { 'R', 1, 0, 0, "PL_Reverse" },
  { 'R', 1, 0, 1, "PL_Randomize" },
  { 'G', 1, 0, 1, "FindInPL" },
  { 'W', 1, 0, 0, "CutTracks" },
  { 'W', 0, 1, 0, "CopyTracks" },
  { 'Y', 1, 0, 0, "PasteTracks" },
  //{ 'G', 1, 0, 0, "ML_Search" },
  //{ 'B', 1, 0, 1, "ML_Browser" },
  //{ VK_F5, 1, 0, 0, "ML_Scan" },
  { 'A', 1, 0, 0, "PL_SelectAll" },
  { 'N', 1, 0, 0, "PL_SelectNone" },
  { 'I', 1, 0, 0, "PL_InvertSelection" },
  { VK_DELETE, 0, 0, 0, "PL_Delete" },
  { VK_DELETE, 1, 0, 0, "PL_Crop" },
  { VK_DELETE, 1, 0, 1, "PL_Clear" },
  { VK_DELETE, 0, 1, 0, "PL_RemoveMissingFiles" },
  { VK_DELETE, 0, 0, 1, "PL_RemoveDuplicateEntries" },
  { 'F', 1, 0, 0, "OpenFolder" },
  { 'O', 1, 0, 0, "OpenFiles" },
  { 'F', 1, 0, 1, "AddFolder" },
  { 'O', 1, 0, 1, "AddFiles" },
  { 'L', 1, 0, 0, "LoadPlayList" },
  { 'S', 1, 0, 1, "SavePlayList" },
  { 'E', 0, 0, 0, "PL_SelectCurrent" },
  { 'A', 1, 0, 1, "PreviewArtwork" },
  { 'P', 1, 0, 0, "Preferences" },
  { '3', 0, 1, 0, "Properties" },
  { VK_UP, 0, 1, 0, "MoveUp" },
  { VK_DOWN, 0, 1, 0, "MoveDown" },
  { VK_F1, 1, 0, 0, "AboutTerpsichore" },
  { VK_F2, 1, 0, 0, "Changelog" },
  { VK_F3, 1, 0, 0, "Version" },
  { 'D', 0, 0, 0, "PlayDeckA" },
  { 'K', 0, 0, 0, "PlayDeckB" },
  { 'A', 0, 0, 1, "ToggleReverseDeckA" },
  { 'L', 0, 0, 1, "ToggleReverseDeckB" },
  { 'A', 0, 0, 0, "RewindDeckA" },
  { 'S', 0, 0, 0, "FastForwardDeckA" },
  { 'L', 0, 0, 0, "RewindDeckB" },
#if _WIN32
  { VK_OEM_1, 0, 0, 0, "FastForwardDeckB" },
#else
  { ';', 0, 0, 0, "FastForwardDeckB" },
#endif
  { 'F', 0, 0, 0, "EnableCueDeckA" },
  { 'J', 0, 0, 0, "EnableCueDeckB" },
  { 'D', 0, 0, 1, "SetCueDeckA" },
  { 'K', 0, 0, 1, "SetCueDeckB" },
  { 'Z', 0, 0, 0, "SetHotCue1DeckA" },
  { 'X', 0, 0, 0, "SetHotCue2DeckA" },
  { 'C', 0, 0, 0, "SetHotCue3DeckA" },
  { 'V', 0, 0, 0, "SetHotCue4DeckA" },
  { 'Z', 0, 0, 1, "ClearHotCue1DeckA" },
  { 'X', 0, 0, 1, "ClearHotCue2DeckA" },
  { 'C', 0, 0, 1, "ClearHotCue3DeckA" },
  { 'V', 0, 0, 1, "ClearHotCue4DeckA" },
  // VK_OEM_1: semicolon (0xBA)
  // VK_OEM_2: slash (0xBF)
  // VK_OEM_3: tilde (0xC0)
  // VK_OEM_4: left bracket (0xDB)
  // VK_OEM_5: backslash (0xDC)
  // VK_OEM_6: right bracket (0xDD)
  // VK_OEM_7: quote (0xDE)
  // VK_OEM_PLUS: equal (0xBB)
  // VK_OEM_MINUS: dash (0xBD)
  // VK_OEM_COMMA: comma (0xBC)
  // VK_OEM_PERIOD: period (0xBE)
#if _WIN32
  { 'M', 0, 0, 0, "SetHotCue1DeckB" },
  { VK_OEM_COMMA, 0, 0, 0, "SetHotCue2DeckB" },
  { VK_OEM_PERIOD, 0, 0, 0, "SetHotCue3DeckB" },
  { VK_OEM_2, 0, 0, 0, "SetHotCue4DeckB" },
  { 'M', 0, 0, 1, "ClearHotCue1DeckB" },
  { VK_OEM_COMMA, 0, 0, 1, "ClearHotCue2DeckB" },
  { VK_OEM_PERIOD, 0, 0, 1, "ClearHotCue3DeckB" },
  { VK_OEM_2, 0, 0, 1, "ClearHotCue4DeckB" },
#else
  { 'M', 0, 0, 0, "SetHotCue1DeckB" },
  { ',', 0, 0, 0, "SetHotCue2DeckB" },
  { '.', 0, 0, 0, "SetHotCue3DeckB" },
  { '/', 0, 0, 0, "SetHotCue4DeckB" },
  { 'M', 0, 0, 1, "ClearHotCue1DeckB" },
  { '<', 0, 0, 1, "ClearHotCue2DeckB" },
  { '>', 0, 0, 1, "ClearHotCue3DeckB" },
  { '?', 0, 0, 1, "ClearHotCue4DeckB" },
#endif
  { VK_F1, 0, 0, 0, "DecreaseShiftDeckA" },
  { VK_F2, 0, 0, 0, "IncreaseShiftDeckA" },
  { VK_F1, 0, 0, 1, "DecreaseShiftSmallDeckA" },
  { VK_F2, 0, 0, 1, "IncreaseShiftSmallDeckA" },
  { VK_F3, 0, 0, 0, "DecreaseTempoDeckA" },
  { VK_F4, 0, 0, 0, "IncreaseTempoDeckA" },
  { VK_F3, 0, 0, 1, "DecreaseTempoSmallDeckA" },
  { VK_F4, 0, 0, 1, "IncreaseTempoSmallDeckA" },
  { VK_F5, 0, 0, 0, "DecreaseShiftDeckB" },
  { VK_F6, 0, 0, 0, "IncreaseShiftDeckB" },
  { VK_F5, 0, 0, 1, "DecreaseShiftSmallDeckB" },
  { VK_F6, 0, 0, 1, "IncreaseShiftSmallDeckB" },
  { VK_F7, 0, 0, 0, "DecreaseTempoDeckB" },
  { VK_F8, 0, 0, 0, "IncreaseTempoDeckB" },
  { VK_F7, 0, 0, 1, "DecreaseTempoSmallDeckB" },
  { VK_F8, 0, 0, 1, "IncreaseTempoSmallDeckB" },
  { VK_F9, 0, 0, 0, "ResetShiftTempoDeckA" },
  { VK_F10, 0, 0, 0, "ResetShiftTempoDeckB" },
  { 'G', 0, 0, 0, "FaderDeckA" },
  { 'H', 0, 0, 0, "FaderDeckB" },
  { 'G', 0, 0, 1, "FaderSmallDeckA" },
  { 'H', 0, 0, 1, "FaderSmallDeckB" },
  { 'R', 0, 0, 0, "ResetFader" },
  { VK_LEFT, 0, 0, 1, "LoadDeckA" },
  { VK_RIGHT, 0, 0, 1, "LoadDeckB" },
  { VK_LEFT, 1, 0, 1, "EjectDeckA" },
  { VK_RIGHT, 1, 0, 1, "EjectDeckB" }
};

static WDL_PtrArray<const KeyboardHotkey, WDL_ARRAYCOUNT(keyboard_hotkey)> kbhotkey(keyboard_hotkey);

RST_KeyboardMap::RST_KeyboardMap()
  : m_keymap(WDL_assocarray_cmpstr)
{
  for (int i = 0; i < kbhotkey.GetSize(); i++)
  {
    m_keymap.AddUnsorted(kbhotkey.Get()[i].action, &kbhotkey.Get()[i]);
  }

  m_keymap.Resort();
}

RST_KeyboardMap::~RST_KeyboardMap()
{
  m_keymap.DeleteAll();
}

unsigned char RST_KeyboardMap::GetKey(const char *action)
{
  const KeyboardHotkey *k = m_keymap.Get(action);

  if (k)
  {
    return k->key;
  }

  return ' ';
}

bool RST_KeyboardMap::MatchModifier(const char *action)
{
  int c, m, s;
  c = m = s = 0;

  if (GetAsyncKeyState(VK_CONTROL) & 0x8000)
  {
    c = 1;
  }

  if (GetAsyncKeyState(VK_MENU) & 0x8000)
  {
    m = 1;
  }

  if (GetAsyncKeyState(VK_SHIFT) & 0x8000)
  {
    s = 1;
  }

  const KeyboardHotkey *k = m_keymap.Get(action);

  if (!k) return false;

  return ((k->ctrl == c) && (k->menu == m) &&
    (k->shift == s));
}
