// Copyright (c) 2020 Giorgos Vougioukas
//
// The license can be found in the LICENSE file.

#include "terpsichore/main_loop.h"
#include "terpsichore/plugin.h"

#include "WDL/fpcmp.h"
#include "WDL/db2val.h"

RST_MainLoop::RST_MainLoop()
  : m_kill_region(NULL)
  , m_play_lock_deck_a(false)
  , m_play_lock_deck_b(false)
{
  for (int i = 0; i < 2; i++)
  {
    RST_Deck *d = new RST_Deck;
    m_decks.Add(d);
  }
}

RST_MainLoop::~RST_MainLoop()
{
  if (m_kill_region) delete m_kill_region;
  m_decks.Empty(true);
}

bool RST_MainLoop::IsPlaying() const
{
  if (m_decks.Get(0)->IsActive()) return true;
  if (m_decks.Get(1)->IsActive()) return true;

  return false;
}

void RST_MainLoop::LoadDeckA(const char *filepath, int peakrate,
  WDL_StringKeyedArray<double> *ebur128_info)
{
  if (m_decks.Get(0)->IsActive()) return;
  EjectDeckA();
  m_decks.Get(0)->track.Open(filepath, peakrate, ebur128_info);
}

void RST_MainLoop::LoadDeckB(const char *filepath, int peakrate,
  WDL_StringKeyedArray<double> *ebur128_info)
{
  if (m_decks.Get(1)->IsActive()) return;
  EjectDeckB();
  m_decks.Get(1)->track.Open(filepath, peakrate, ebur128_info);
}

void RST_MainLoop::EjectDeckA()
{
  if (m_decks.Get(0)->IsActive()) return;
  RST_Deck *d = new RST_Deck;
  RST_Deck *p = m_decks.Get(0);
  m_decks.Set(0, d);
  if (p) delete p;
}

void RST_MainLoop::EjectDeckB()
{
  if (m_decks.Get(1)->IsActive()) return;
  RST_Deck *d = new RST_Deck;
  RST_Deck *p = m_decks.Get(1);
  m_decks.Set(1, d);
  if (p) delete p;
}

void RST_MainLoop::PlayDeckA()
{
  if (!m_decks.Get(0)->track.IsOpen() ||
    m_play_lock_deck_a) return;
  m_decks.Get(0)->SetState(!m_decks.Get(0)->IsPlayActive());
  m_play_lock_deck_a = true;
}

void RST_MainLoop::PlayDeckB()
{
  if (!m_decks.Get(1)->track.IsOpen() ||
    m_play_lock_deck_b) return;
  m_decks.Get(1)->SetState(!m_decks.Get(1)->IsPlayActive());
  m_play_lock_deck_b = true;
}

void RST_MainLoop::DisablePlayLockDeckA()
{
  m_play_lock_deck_a = false;
}

void RST_MainLoop::DisablePlayLockDeckB()
{
  m_play_lock_deck_b = false;
}

void RST_MainLoop::ToggleReverseDeckA()
{
  if (m_decks.Get(0)->track.IsOpen())
  {
    m_decks.Get(0)->track.ToggleReverse();
  }
}

void RST_MainLoop::ToggleReverseDeckB()
{
  if (m_decks.Get(1)->track.IsOpen())
  {
    m_decks.Get(1)->track.ToggleReverse();
  }
}

void RST_MainLoop::SeekBackwardDeckA()
{
  if (m_decks.Get(0)->track.IsOpen())
  {
    m_decks.Get(0)->track.SeekBackward();
  }
}

void RST_MainLoop::SeekForwardDeckA()
{
  if (m_decks.Get(0)->track.IsOpen())
  {
    m_decks.Get(0)->track.SeekForward();
  }
}

void RST_MainLoop::SeekBackwardDeckB()
{
  if (m_decks.Get(1)->track.IsOpen())
  {
    m_decks.Get(1)->track.SeekBackward();
  }
}

void RST_MainLoop::SeekForwardDeckB()
{
  if (m_decks.Get(1)->track.IsOpen())
  {
    m_decks.Get(1)->track.SeekForward();
  }
}

void RST_MainLoop::RewindDeckA(bool state)
{
  if (m_decks.Get(0)->track.IsOpen())
  {
    m_decks.Get(0)->track.Rewind(state);
  }
}

void RST_MainLoop::FastForwardDeckA(bool state)
{
  if (m_decks.Get(0)->track.IsOpen())
  {
    m_decks.Get(0)->track.FastForward(state);
  }
}

void RST_MainLoop::RewindDeckB(bool state)
{
  if (m_decks.Get(1)->track.IsOpen())
  {
    m_decks.Get(1)->track.Rewind(state);
  }
}

void RST_MainLoop::FastForwardDeckB(bool state)
{
  if (m_decks.Get(1)->track.IsOpen())
  {
    m_decks.Get(1)->track.FastForward(state);
  }
}

void RST_MainLoop::SetStateDeckA(bool state)
{
  m_decks.Get(0)->SetState(state);
}

void RST_MainLoop::SetStateDeckB(bool state)
{
  m_decks.Get(1)->SetState(state);
}

bool RST_MainLoop::IsActiveDeckA() const
{
  return m_decks.Get(0)->IsActive();
}

bool RST_MainLoop::IsActiveDeckB() const
{
  return m_decks.Get(1)->IsActive();
}

double RST_MainLoop::GetRunTimeDeckA() const
{
  return m_decks.Get(0)->track.GetRunTime();
}

double RST_MainLoop::GetRunTimeDeckB() const
{
  return m_decks.Get(1)->track.GetRunTime();
}

double RST_MainLoop::GetDurationDeckA() const
{
  return m_decks.Get(0)->track.GetDuration();
}

double RST_MainLoop::GetDurationDeckB() const
{
  return m_decks.Get(1)->track.GetDuration();
}

double RST_MainLoop::GetCueDeckA() const
{
  return m_decks.Get(0)->GetCue();
}

double RST_MainLoop::GetCueDeckB() const
{
  return m_decks.Get(1)->GetCue();
}

WDL_IntKeyedArray<double> *RST_MainLoop::GetHotCueDeckA() const
{
  return m_decks.Get(0)->GetHotCue();
}

WDL_IntKeyedArray<double> *RST_MainLoop::GetHotCueDeckB() const
{
  return m_decks.Get(1)->GetHotCue();
}

double RST_MainLoop::GetFaderDecibelDeckA() const
{
  return m_decks.Get(0)->track.GetFaderDecibel();
}

double RST_MainLoop::GetFaderDecibelDeckB() const
{
  return m_decks.Get(1)->track.GetFaderDecibel();
}

const char *RST_MainLoop::GetTitleDeckA() const
{
  return m_decks.Get(0)->track.GetTitle();
}

const char *RST_MainLoop::GetTitleDeckB() const
{
  return m_decks.Get(1)->track.GetTitle();
}

const char *RST_MainLoop::GetArtistDeckA() const
{
  return m_decks.Get(0)->track.GetArtist();
}

const char *RST_MainLoop::GetArtistDeckB() const
{
  return m_decks.Get(1)->track.GetArtist();
}

bool RST_MainLoop::IsPlayActiveDeckA() const
{
  return m_decks.Get(0)->IsPlayActive();
}

bool RST_MainLoop::IsPlayActiveDeckB() const
{
  return m_decks.Get(1)->IsPlayActive();
}

void RST_MainLoop::DecreaseShiftDeckA()
{
  m_decks.Get(0)->track.DecreaseShift();
}

void RST_MainLoop::DecreaseShiftDeckB()
{
  m_decks.Get(1)->track.DecreaseShift();
}

void RST_MainLoop::IncreaseShiftDeckA()
{
  m_decks.Get(0)->track.IncreaseShift();
}

void RST_MainLoop::IncreaseShiftDeckB()
{
  m_decks.Get(1)->track.IncreaseShift();
}

void RST_MainLoop::DecreaseShiftSmallDeckA()
{
  m_decks.Get(0)->track.DecreaseShiftSmall();
}

void RST_MainLoop::DecreaseShiftSmallDeckB()
{
  m_decks.Get(1)->track.DecreaseShiftSmall();
}

void RST_MainLoop::IncreaseShiftSmallDeckA()
{
  m_decks.Get(0)->track.IncreaseShiftSmall();
}

void RST_MainLoop::IncreaseShiftSmallDeckB()
{
  m_decks.Get(1)->track.IncreaseShiftSmall();
}

void RST_MainLoop::DecreaseTempoDeckA()
{
  m_decks.Get(0)->track.DecreaseTempo();
}

void RST_MainLoop::DecreaseTempoDeckB()
{
  m_decks.Get(1)->track.DecreaseTempo();
}

void RST_MainLoop::IncreaseTempoDeckA()
{
  m_decks.Get(0)->track.IncreaseTempo();
}

void RST_MainLoop::IncreaseTempoDeckB()
{
  m_decks.Get(1)->track.IncreaseTempo();
}

void RST_MainLoop::DecreaseTempoSmallDeckA()
{
  m_decks.Get(0)->track.DecreaseTempoSmall();
}

void RST_MainLoop::DecreaseTempoSmallDeckB()
{
  m_decks.Get(1)->track.DecreaseTempoSmall();
}

void RST_MainLoop::IncreaseTempoSmallDeckA()
{
  m_decks.Get(0)->track.IncreaseTempoSmall();
}

void RST_MainLoop::IncreaseTempoSmallDeckB()
{
  m_decks.Get(1)->track.IncreaseTempoSmall();
}

void RST_MainLoop::ResetShiftDeckA()
{
  m_decks.Get(0)->track.ResetShift();
}

void RST_MainLoop::ResetShiftDeckB()
{
  m_decks.Get(1)->track.ResetShift();
}

void RST_MainLoop::ResetTempoDeckA()
{
  m_decks.Get(0)->track.ResetTempo();
}

void RST_MainLoop::ResetTempoDeckB()
{
  m_decks.Get(1)->track.ResetTempo();
}

void RST_MainLoop::EnableCueDeckA()
{
  m_decks.Get(0)->EnableCue();
}

void RST_MainLoop::EnableCueDeckB()
{
  m_decks.Get(1)->EnableCue();
}

void RST_MainLoop::SetCueDeckA()
{
  m_decks.Get(0)->SetCue();
}

void RST_MainLoop::SetCueDeckB()
{
  m_decks.Get(1)->SetCue();
}

void RST_MainLoop::EnableHotCueDeckA(int idx)
{
  m_decks.Get(0)->EnableHotCue(idx);
}

void RST_MainLoop::EnableHotCueDeckB(int idx)
{
  m_decks.Get(1)->EnableHotCue(idx);
}

void RST_MainLoop::ClearHotCueDeckA(int idx)
{
  m_decks.Get(0)->ClearHotCue(idx);
}

void RST_MainLoop::ClearHotCueDeckB(int idx)
{
  m_decks.Get(1)->ClearHotCue(idx);
}

void RST_MainLoop::DisableCueDeckA()
{
  m_decks.Get(0)->DisableCue();
}

void RST_MainLoop::DisableCueDeckB()
{
  m_decks.Get(1)->DisableCue();
}

void RST_MainLoop::DisableHotCueDeckA(int idx)
{
  m_decks.Get(0)->DisableHotCue(idx);
}

void RST_MainLoop::DisableHotCueDeckB(int idx)
{
  m_decks.Get(1)->DisableHotCue(idx);
}

double RST_MainLoop::GetFaderDeckA() const
{
  return m_decks.Get(0)->track.GetFader();
}

double RST_MainLoop::GetFaderDeckB() const
{
  return m_decks.Get(1)->track.GetFader();
}

void RST_MainLoop::SetFaderDeckA(double percentage)
{
  m_decks.Get(0)->track.SetFader(percentage);
}

void RST_MainLoop::SetFaderDeckB(double percentage)
{
  m_decks.Get(1)->track.SetFader(percentage);
}

void RST_MainLoop::SetKillRegion(WDL_HeapBuf *kill_region)
{
  if (m_kill_region) { delete m_kill_region; m_kill_region = NULL; }

  m_kill_region = kill_region;
}

void RST_MainLoop::GetKillRegion(WDL_HeapBuf **kill_region)
{
  if (m_kill_region)
  {
    *kill_region = m_kill_region;
  }
  else
  {
    *kill_region = NULL;
  }
}

void RST_MainLoop::AudioOutput(WDL_TypedBuf<SAM> *output, int frame_count, int nch)
{
  //if (m_is_paused || !m_audio_input || IsDrained())
  //{
  //  memset(output->Get(), 0, frame_count * sizeof(SAM));
  //  return;
  //}

  memset(output->Get(), 0, frame_count * sizeof(SAM));
  if (!IsPlaying()) return;

  for (int i = 0; i < m_decks.GetSize(); i++)
  {
    int written = 0;
    BQ_Block *block = NULL;
    RST_Deck *d = m_decks.Get(i);

    if (!d->IsActive()) continue;

    if (d->track.IsDrained())
    {
      if (WDL_likely(!d->track.IsReverse()))
      {
        d->track.UpdateRunTime(d->track.GetDuration());
      }
      else
      {
        d->track.UpdateRunTime(0.0);
        d->track.ToggleReverse();
        d->track.Seek(0.0);
      }
      d->SetState(false); continue;
    }

    while (written < frame_count)
    {
      if (d->track.m_bq.GetBlock(&block))
      {
        double rt = block->stime;
        d->track.UpdateRunTime(rt);

        if (block->samq.Available() / 2 * nch > frame_count - written)
        {
          int sz = frame_count - written;

          for (int j = 0, k = 0; j < sz; j += nch, k += 2)
          {
            block->samq.Get()[k] *= DB2VAL(d->track.GetFaderDecibel());
            block->samq.Get()[k + 1] *= DB2VAL(d->track.GetFaderDecibel());
            output->Get()[written + j] += block->samq.Get()[k];
            output->Get()[written + j + 1] += block->samq.Get()[k + 1];
          }

          written += sz;

          block->samq.Advance(2 * sz / nch);
          block->samq.Compact();

          if (block->samq.Available()) d->track.m_bq.ReturnBlock(block);
          else d->track.m_bq.DisposeBlock(block);
        }
        else
        {
          int sz = block->samq.GetSize() / 2 * nch;

          for (int j = 0, k = 0; j < sz; j += nch, k += 2)
          {
            block->samq.Get()[k] *= DB2VAL(d->track.GetFaderDecibel());
            block->samq.Get()[k + 1] *= DB2VAL(d->track.GetFaderDecibel());
            output->Get()[written + j] += block->samq.Get()[k];
            output->Get()[written + j + 1] += block->samq.Get()[k + 1];
          }

          written += sz;

          block->samq.Clear();
          d->track.m_bq.DisposeBlock(block);
        }
      }
      else
      {
        break;
      }
    }
  }
}

int RST_MainLoop::Run()
{
  for (int i = 0; i < m_decks.GetSize(); i++)
  {
    RST_Deck *d = m_decks.Get(i);

    while (d->track.WantMore())
    {
      if (!d->track.DoBuffering()) break;
    }
  }

  //WDL_FastString ss;
  //ss.SetFormatted(64, "threadid: %d\n", GetCurrentThreadId());
  //OutputDebugString(ss.Get());

  //while (WantMore())
  //{
  //  if (!DoBuffering()) break;
  //}

  //if (IsDrained())
  //{
  //  int track_count = m_playlist->GetSize();

  //  if (m_has_stop_after_current)
  //  {
  //    Stop();
  //    return 1;
  //  }

  //  if (m_playback_queue.GetSize())
  //  {
  //    Play();
  //    return 1;
  //  }

  //  if (m_current_track + 1 < track_count || m_is_repeat_on || m_is_shuffle_on)
  //  {
  //    if (!m_is_shuffle_on)
  //    {
  //      if (m_current_track + 1 >= track_count) // repeat
  //      {
  //        m_current_track = 0;
  //      }
  //      else
  //      {
  //        m_current_track++;
  //      }
  //    }
  //    else
  //    {
  //      m_current_track = m_mt.randInt(track_count - 1);
  //    }

  //    Play(m_current_track);
  //  }
  //  else
  //  {
  //    Stop();
  //  }
  //}

  return 1;
}
