// Copyright (c) 2017 Giorgos Vougioukas
//
// The license can be found in the LICENSE file.

#include "terpsichore/main_wnd.h"

#include <string.h>

#include "terpsichore/app_info.h"
#include "terpsichore/plugin.h"
#include "terpsichore/theme.h"
#include "terpsichore/icon_theme.h"
#include "terpsichore/db2slider.h"
#include "terpsichore/concurrency.h"
#include "terpsichore/antidote.h"

#include "WDL/filebrowse.h"
#include "WDL/dirscan.h"
#include "WDL/fileread.h"
#include "WDL/heapbuf.h"
#include "WDL/time_precise.h"

#define RST_DISABLE_PLAYLIST_CONTROL 800
#define RST_DISABLE_PLAYLIST_CONTROL_MS 50
#define RST_MAIN_MENU_BAR 810
#define RST_MAIN_MENU_BAR_MS 500
#define RST_MAIN_TITLE_BAR 820
#define RST_MAIN_TITLE_BAR_MS 90
#define RST_CUT_COPY_PASTE_INFO 830
#define RST_CUT_COPY_PASTE_INFO_MS 90
#define RST_PROGRESS_WINDOW 840
#define RST_PROGRESS_WINDOW_MS 50
#define RST_RENDER_MAIN_DRAW 1900
#define RST_RENDER_MAIN_DRAW_MS 33 // 33ms 30Hz, 16ms 60Hz

static LOGFONT lf =
{
#if defined(_WIN32)
  20, 0, 0, 0, FW_NORMAL, FALSE, FALSE, FALSE, DEFAULT_CHARSET,
  OUT_DEFAULT_PRECIS, CLIP_DEFAULT_PRECIS, DEFAULT_QUALITY, DEFAULT_PITCH,
  "Arial"
#else
  17, 0, 0, 0, FW_NORMAL, FALSE, FALSE, FALSE, DEFAULT_CHARSET,
  OUT_DEFAULT_PRECIS, CLIP_DEFAULT_PRECIS, DEFAULT_QUALITY, DEFAULT_PITCH,
  "Arial"
#endif
};

static LOGFONT blf =
{
#if defined(_WIN32)
  24, 0, 0, 0, FW_NORMAL, FALSE, FALSE, FALSE, DEFAULT_CHARSET,
  OUT_DEFAULT_PRECIS, CLIP_DEFAULT_PRECIS, DEFAULT_QUALITY, DEFAULT_PITCH,
  "Arial"
#else
  21, 0, 0, 0, FW_NORMAL, FALSE, FALSE, FALSE, DEFAULT_CHARSET,
  OUT_DEFAULT_PRECIS, CLIP_DEFAULT_PRECIS, DEFAULT_QUALITY, DEFAULT_PITCH,
  "Arial"
#endif
};

unsigned WINAPI MainLoopThreadFunction(void *arg)
{
  while (!g_main_loop_done)
  {
    g_main_loop_mutex.Enter();
    while (!g_main_loop->Run());
    g_main_loop_mutex.Leave();

    Sleep(1);
  }

  return 0;
}

RST_MainWnd::RST_MainWnd()
  : m_hwnd(NULL)
  , m_title_capture(false)
  , m_x(0)
  , m_y(0)
  , m_w(0)
  , m_h(0)
  , m_cut_copy_paste_ms(0)
  , m_bgc(0)
  , m_txtc(0)
{}

RST_MainWnd::~RST_MainWnd()
{}

HWND RST_MainWnd::Handle() const
{
  return m_hwnd;
}

void RST_MainWnd::ShowPlayList()
{
#if 0
  WDL_WndSizer__rec *recd = m_resize.get_itembywnd(m_decks_wnd.Handle());
  WDL_WndSizer__rec *recp = m_resize.get_itembywnd(m_pl_wnd.Handle());

  if (recd && recp)
  {
    RECT d, p;
    GetClientRect(m_pl_wnd.Handle(), &p);
    GetClientRect(m_decks_wnd.Handle(), &d);

    int dh = d.bottom - d.top;
    int ph = p.bottom - p.top;

    RECT orec = m_resize.get_orig_rect();
    int hi = (int)(orec.bottom * 0.20f);
    int lo = (int)(orec.bottom * 0.80f);

    if (dh > ph)
    {
      recd->orig.bottom = hi;
      recd->scales[3] = 0.20f;

      recp->orig.top = hi;
      recp->scales[1] = 0.20f;
    }
    else
    {
      recd->orig.bottom = lo;
      recd->scales[3] = 0.80f;

      recp->orig.top = lo;
      recp->scales[1] = 0.80f;
    }

    m_resize.onResize();
  }
#endif
}

void RST_MainWnd::ShowKillRegionInfo(const char *info)
{
  m_cut_copy_paste_info.Set(info);
  m_cut_copy_paste_ms = 0;
  KillTimer(m_hwnd, RST_MAIN_TITLE_BAR);
  SetTimer(m_hwnd, RST_CUT_COPY_PASTE_INFO, RST_CUT_COPY_PASTE_INFO_MS, NULL);
}

RST_FileTasks *RST_MainWnd::FileTask()
{
  return &m_file_task;
}

bool RST_MainWnd::IsEngineOnline() const
{
  return g_main_loop_thread.GetSize() && g_audio_streamer;
}

void RST_MainWnd::SetEngineOnline()
{
  WDL_FastString asn;

#if defined(_WIN32)
  asn.Set(g_ini_file->read_str("audio_system_name", "WaveOut", "preferences"));
#elif defined(__linux__)
  asn.Set(g_ini_file->read_str("audio_system_name", "PulseAudio", "preferences"));
#endif

  g_audio_streamer = CreateAudioStreamer(asn.Get());

  if (!g_audio_streamer)
  {
    g_ini_file->write_int("last_page", 2, "preferences");
    SendMessage(m_hwnd, WM_COMMAND, ID_PREFERENCES, 0);
    return;
  }

  //g_main_loop_thread.Resize(GetCPUCores());
  g_main_loop_thread.Resize(1);

  //WDL_FastString ss;
  //ss.SetFormatted(64, "CPU cores: %d\n", GetCPUCores());
  //OutputDebugString(ss.Get());

  g_main_loop_done = 0;

  for (int i = 0; i < g_main_loop_thread.GetSize(); i++)
  {
    unsigned int thread_id;
    g_main_loop_thread.Get()[i] = (HANDLE)_beginthreadex(
      NULL, 0, MainLoopThreadFunction, 0, 0, &thread_id);
  }

  for (int i = 0; i < g_main_loop_thread.GetSize(); i++)
  {
    SetThreadPriority(g_main_loop_thread.Get()[i], THREAD_PRIORITY_ABOVE_NORMAL);
  }

  if (!g_audio_streamer)
  {
    WDL_FastString hw_error;
    hw_error.Set("Terpsichore cannot find the audio system: ");
    hw_error.Append(asn.Get());
    //MessageBox(g_main_wnd->Handle(), hw_error.Get(), "Terpsichore error", MB_OK);
  }
  else if (g_audio_streamer->Open())
  {
    if (!g_audio_streamer->IsRunning())
    {
      g_audio_streamer->Start(&AudioOnSamples);
    }
  }
  else
  {
    WDL_FastString hw_error;
    hw_error.Set("Terpsichore cannot initialize the audio system: ");
    hw_error.Append(asn.Get());
    //MessageBox(g_main_wnd->Handle(), hw_error.Get(), "Terpsichore error", MB_OK);
  }
}

void RST_MainWnd::SetEngineOffline()
{
  if (!g_audio_streamer) return;

  if (g_audio_streamer->IsRunning())
  {
    g_audio_streamer->Stop();
  }

  g_audio_streamer->Close();

  delete g_audio_streamer;
  g_audio_streamer = NULL;

  g_main_loop_done = 1;

  for (int i = 0; i < g_main_loop_thread.GetSize(); i++)
  {
    if (g_main_loop_thread.Get()[i])
    {
      WaitForSingleObject(g_main_loop_thread.Get()[i], INFINITE);
      CloseHandle(g_main_loop_thread.Get()[i]);
      g_main_loop_thread.Get()[i] = NULL;
    }
  }

  g_main_loop_thread.Resize(0);
}

void RST_MainWnd::OnInitDialog(WPARAM wparam, LPARAM lparam)
{
  double wall0 = time_precise();

  m_file_task.SetParent(m_hwnd);

  if (IsStableRelease())
  {
    SetWindowText(m_hwnd, TERPSICHORE_NAME_MARKETING);
  }
  else
  {
    SetWindowText(m_hwnd, TERPSICHORE_FULL_VERSION " (" TERPSICHORE_ARCH ")");
  }

#if defined(_WIN32)
  SetClassLongPtr(m_hwnd, GCLP_HICON, (INT_PTR)LoadImage(g_inst,
    MAKEINTRESOURCE(IDI_ICON1), IMAGE_ICON, 128, 128, LR_SHARED));
  //SetClassLongPtr(m_hwnd, GCLP_HICON, (INT_PTR)LoadImage(g_inst,
  //  MAKEINTRESOURCE(IDI_ICON1), IMAGE_ICON, 32, 32, LR_SHARED));
  //SetClassLongPtr(m_hwnd, GCLP_HICONSM, (INT_PTR)LoadImage(g_inst,
  //  MAKEINTRESOURCE(IDI_ICON1), IMAGE_ICON, 16, 16, LR_SHARED));
#endif

  g_keymap = new RST_KeyboardMap;

  WDL_FastString theme_fn(g_module_path.Get());
  theme_fn.Append("themes" WDL_DIRCHAR_STR "mask.terpsichore_theme");
  g_theme = new RST_PackedTheme();

  if (!g_theme->LoadResource(theme_fn.Get()))
  {
    g_theme->UnloadResource();
  }
  else
  {
    g_icon_theme.LoadImages();
  }

  g_main_loop = new RST_MainLoop;

  SetEngineOnline();
  wdl_log("main: online\n");

  RECT main_rect, deck_rect;
  GetClientRect(m_hwnd, &main_rect);

  CreateDialogParam(g_inst, MAKEINTRESOURCE(IDD_DECKS), m_hwnd,
    RST_DecksWnd::ST_DecksWndProc, (LPARAM)&m_decks_wnd);
  SetWindowPos(m_decks_wnd.Handle(), NULL, main_rect.left,
    main_rect.top, (main_rect.right - main_rect.left), 125,
    //(int)((main_rect.bottom - main_rect.top) * 0.20f),
    SWP_NOACTIVATE | SWP_NOZORDER);

  CreateDialogParam(g_inst, MAKEINTRESOURCE(IDD_PLAYLIST), m_hwnd,
    RST_PlayListWnd::ST_PlayListWndProc, (LPARAM)&m_pl_wnd);
  GetClientRect(m_decks_wnd.Handle(), &deck_rect);
  SetWindowPos(m_pl_wnd.Handle(), NULL, main_rect.left,
    /* deck_rect.bottom */ 125, main_rect.right - main_rect.left,
    (main_rect.bottom - deck_rect.bottom),
    SWP_NOACTIVATE | SWP_NOZORDER);

  m_resize.init(m_hwnd);
  m_resize.init_itemhwnd(m_decks_wnd.Handle(), 0.0f, 0.0f, 1.0f, 0.0f);
  m_resize.init_itemhwnd(m_pl_wnd.Handle(), 0.0f, 0.0f, 1.0f, 1.0f);
  //m_resize.init_item(IDC_CUSTOM_MAINDRAW, 0.0f, 0.0f, 1.0f, 1.0f);

  m_main_menu = LoadMenu(g_inst, MAKEINTRESOURCE(IDR_MAIN_MENUBAR));
  SetMenu(m_hwnd, m_main_menu);

  m_x = g_ini_file->read_int("main_wnd_x", 50, TERPSICHORE_NAME);
  m_y = g_ini_file->read_int("main_wnd_y", 50, TERPSICHORE_NAME);
  m_w = g_ini_file->read_int("main_wnd_w", 1024, TERPSICHORE_NAME);
  m_h = g_ini_file->read_int("main_wnd_h", 768, TERPSICHORE_NAME);
  SetWindowPos(m_hwnd, NULL, m_x, m_y, m_w, m_h, SWP_NOACTIVATE);

  //EnableMenuItem(GetMenu(m_hwnd), ID_ADDURL, MF_GRAYED);
  //EnableMenuItem(GetMenu(m_hwnd), ID_PLAYURL, MF_GRAYED);

  ShowWindow(m_pl_wnd.Handle(), SW_SHOW);

  int maximized = g_ini_file->read_int("main_wnd_maximized", 0, TERPSICHORE_NAME);

  if (maximized)
  {
    ShowWindow(m_hwnd, SW_SHOWMAXIMIZED);
  }
  else
  {
    ShowWindow(m_hwnd, SW_SHOWNORMAL);
  }

  double wall1 = time_precise();

  m_bgc = (0 << 16) | (0 << 8) | (0 << 0);
  m_bgc = g_ini_file->read_int("pl_bg_color", m_bgc, "preferences");
  m_txtc = (192 << 16) | (192 << 8) | (192 << 0);
  m_txtc = g_ini_file->read_int("pl_txt_color", m_txtc, "preferences");

  SetTimer(m_hwnd, RST_MAIN_MENU_BAR, RST_MAIN_MENU_BAR_MS, NULL);
  SetTimer(m_hwnd, RST_MAIN_TITLE_BAR, RST_MAIN_TITLE_BAR_MS, NULL);
  SetTimer(m_hwnd, RST_RENDER_MAIN_DRAW, RST_RENDER_MAIN_DRAW_MS, NULL);
  SetTimer(m_hwnd, RST_PROGRESS_WINDOW, RST_PROGRESS_WINDOW_MS, NULL);

  if (!IsStableRelease())
  {
    const char warn_msg[] =
    {
      "Welcome to " TERPSICHORE_NAME_MARKETING " (the \"Software\").\n\n"
      "THIS IS AN UNSTABLE RELEASE STILL IN DEVELOPEMENT PHASE\n"
      "AND IS PROVIDED ON AN \"AS IS\" BASIS AND IS BELIEVED\n"
      "TO CONTAIN DEFECTS AND A PRIMARY PURPOSE OF THIS\n"
      "RELEASE IS TO OBTAIN FEEDBACK ON SOFTWARE PERFORMANCE\n"
      "AND THE IDENTIFICATION OF DEFECTS.\n\n"
      "IT IS ADVISED TO USE CAUTION AND NOT TO RELY IN ANY WAY\n"
      "ON THE CORRECT FUNCTIONING OF THIS RELEASE.\n\n"
      TERPSICHORE_NAME_MARKETING " version: " TERPSICHORE_NAKED_VERSION " (" TERPSICHORE_ARCH ")\n\n"
      "For the latest production release visit: " TERPSICHORE_WEBSITE_URL "\n\n"
      "Startup time: %.3f\n\n"
      TERPSICHORE_COPYRIGHT
    };

    WDL_FastString tmp;

    tmp.SetFormatted(1024, warn_msg, wall1 - wall0);

    MessageBox(m_hwnd, tmp.Get(), "Release for testing purposes", MB_OK);
  }
}

void RST_MainWnd::OnTimer(WPARAM wparam, LPARAM lparam)
{
  if (wparam == RST_DISABLE_PLAYLIST_CONTROL)
  {
    if (!GetCapture() || !m_title_capture)
    {
      KillTimer(m_hwnd, RST_DISABLE_PLAYLIST_CONTROL);
      SendMessage(GetDlgItem(m_pl_wnd.Handle(), IDC_LIST1), WM_SETREDRAW, TRUE, 0);
    }
  }

  if (wparam == RST_MAIN_MENU_BAR)
  {
    WDL_HeapBuf *kr = NULL;

    g_main_loop_mutex.Enter();
    g_main_loop->GetKillRegion(&kr);
    bool is_playing = g_main_loop->IsPlaying();
    g_main_loop_mutex.Leave();

    if (kr)
    {
      EnableMenuItem(GetMenu(m_hwnd), ID_EDIT_PASTE_TRACKS, MF_ENABLED);
    }
    else
    {
      //EnableMenuItem(GetMenu(m_hwnd), ID_EDIT_PASTE_TRACKS, MF_DISABLED);
      EnableMenuItem(GetMenu(m_hwnd), ID_EDIT_PASTE_TRACKS, MF_GRAYED);
    }

    if (!is_playing && !GetActiveWindow() && IsEngineOnline())
    {
      SendMessage(m_hwnd, WM_ACTIVATEAPP, FALSE, 0);
      wdl_log("main timer: offline\n");
    }
  }

  if (wparam == RST_MAIN_TITLE_BAR)
  {
    if (!GetCapture())
    {
      if (IsStableRelease())
      {
        SetWindowText(g_main_wnd->Handle(), TERPSICHORE_NAME_MARKETING);
      }
      else
      {
        SetWindowText(g_main_wnd->Handle(), TERPSICHORE_FULL_VERSION " (" TERPSICHORE_ARCH ")");
      }
    }
  }

  if (wparam == RST_CUT_COPY_PASTE_INFO)
  {
    m_cut_copy_paste_ms += RST_CUT_COPY_PASTE_INFO_MS;

    if (m_cut_copy_paste_ms < 500)
    {
      if (IsStableRelease())
      {
        m_titstr.SetFormatted(128, "%s ", TERPSICHORE_NAME_MARKETING);
      }
      else
      {
        m_titstr.SetFormatted(128, "%s ", TERPSICHORE_FULL_VERSION " (" TERPSICHORE_ARCH ")");
      }

      m_titstr.Append(m_cut_copy_paste_info.Get());

      SetWindowText(g_main_wnd->Handle(), m_titstr.Get());
    }
    else
    {
      KillTimer(m_hwnd, RST_CUT_COPY_PASTE_INFO);
      SetTimer(m_hwnd, RST_MAIN_TITLE_BAR, RST_MAIN_TITLE_BAR_MS, NULL);
    }
  }

  if (wparam == RST_PROGRESS_WINDOW)
  {
    if (m_file_task.WantProgressWindow())
    {
      if (!m_ft_prog_wnd.Handle())
      {
        DialogBoxParam(g_inst, MAKEINTRESOURCE(IDD_FILE_TASKS_PROGRESS),
          m_hwnd, RST_FileTasksProgressWnd::ST_FileTasksProgressWndProc, (LPARAM)&m_ft_prog_wnd);
        ShowWindow(m_ft_prog_wnd.Handle(), SW_SHOW);
      }
    }
    else
    {
      if (m_ft_prog_wnd.Handle())
      {
        EndDialog(m_ft_prog_wnd.Handle(), 0);
      }
    }

    if (!g_loadwnd && g_antidote.IsTranscoding())
    {
      DialogBox(g_inst, MAKEINTRESOURCE(IDD_LOAD), m_hwnd, RST_LoadWndProc);
    }
  }

  if (wparam == RST_RENDER_MAIN_DRAW)
  {
    m_decks_wnd.UpdateInfo();
  }
}

void RST_MainWnd::OnNCLButtonDown(WPARAM wparam, LPARAM lparam)
{
  m_title_capture = true;
  SetTimer(m_hwnd, RST_DISABLE_PLAYLIST_CONTROL, RST_DISABLE_PLAYLIST_CONTROL_MS, NULL);
  SendMessage(GetDlgItem(m_pl_wnd.Handle(), IDC_LIST1), WM_SETREDRAW, FALSE, 0);
}

int RST_MainWnd::OnActivateApp(WPARAM wparam, LPARAM lparam)
{
#if 0
  g_main_loop_mutex.Enter();
  bool is_playing = g_main_loop->IsPlaying();
  g_main_loop_mutex.Leave();

  if (!is_playing && !m_pr_wnd.Handle())
  {
    if (wparam == TRUE) // activated
    {
      if (!IsEngineOnline())
      {
        SetEngineOnline();
        wdl_log("activateapp: online\n");
      }

      SetTimer(m_hwnd, RST_MAIN_TITLE_BAR,
        RST_MAIN_TITLE_BAR_MS, NULL);
      SetTimer(m_hwnd, RST_MAIN_MENU_BAR,
        RST_MAIN_MENU_BAR_MS, NULL);

      if (IsStableRelease())
      {
        SetWindowText(m_hwnd, TERPSICHORE_NAME_MARKETING);
      }
      else
      {
        SetWindowText(m_hwnd, TERPSICHORE_FULL_VERSION " (" TERPSICHORE_ARCH ")");
      }
    }
    else if (wparam == FALSE) // deactivated
    {
      if (IsEngineOnline())
      {
        SetEngineOffline();
        wdl_log("activateapp: offline\n");

        KillTimer(m_hwnd, RST_MAIN_TITLE_BAR);
        KillTimer(m_hwnd, RST_MAIN_MENU_BAR);
      }

      if (IsStableRelease())
      {
        m_titstr.SetFormatted(128, "%s ", TERPSICHORE_NAME_MARKETING);
      }
      else
      {
        m_titstr.SetFormatted(128, "%s ", TERPSICHORE_FULL_VERSION " (" TERPSICHORE_ARCH ")");
      }

      m_titstr.Append("| Offline");

      SetWindowText(m_hwnd, m_titstr.Get());
    }
  }
#endif

  return 0;
}

void RST_MainWnd::OnSysCommand(WPARAM wparam, LPARAM lparam)
{
  if (wparam == SC_CLOSE)
  {
    SendMessage(m_hwnd, WM_COMMAND, ID_EXIT, 0);
  }
}

int RST_MainWnd::OnClose(WPARAM wparam, LPARAM lparam)
{
#if defined(__linux__)
  has_requested_quit = true;
#endif
  return 0;
}

void RST_MainWnd::OnPaint(WPARAM wparam, LPARAM lparam)
{
  m_painter.PaintBegin(m_hwnd, RGB(51, 51, 51));
  m_painter.PaintEnd();
}

void RST_MainWnd::OnSize(WPARAM wparam, LPARAM lparam)
{
  if (wparam != SIZE_MINIMIZED)
  {
    m_title_capture = false;
    m_resize.onResize();
  }

  if (wparam != SIZE_MINIMIZED && wparam != SIZE_MAXIMIZED)
  {
    RECT r;
    GetWindowRect(m_hwnd, &r);

    m_x = r.left;
    m_y = r.top;
    m_w = r.right - r.left;
    m_h = r.bottom - r.top;
  }
}

void RST_MainWnd::OnMove(WPARAM wparam, LPARAM lparam)
{
  int xpos = (int)(short) LOWORD(lparam); // horizontal position
  int ypos = (int)(short) HIWORD(lparam); // vertical position

  if (xpos >= 0 && ypos >= 0 && GetCapture() == m_hwnd)
  {
    RECT r;
    GetWindowRect(m_hwnd, &r);

    m_x = r.left;
    m_y = r.top;
    m_w = r.right - r.left;
    m_h = r.bottom - r.top;
  }
}

int RST_MainWnd::OnDestroy(WPARAM wparam, LPARAM lparam)
{
  KillTimer(m_hwnd, RST_MAIN_MENU_BAR);
  KillTimer(m_hwnd, RST_MAIN_TITLE_BAR);
  KillTimer(m_hwnd, RST_RENDER_MAIN_DRAW);
  KillTimer(m_hwnd, RST_PROGRESS_WINDOW);

#if defined(_WIN32)
  WINDOWPLACEMENT wp = { sizeof(wp) };
  GetWindowPlacement(m_hwnd, &wp);

  //if (wp.showCmd != SW_MINIMIZE && wp.showCmd != SW_SHOWMINIMIZED &&
  //  wp.showCmd != SW_MAXIMIZE && wp.showCmd != SW_SHOWMAXIMIZED)
  //{
  //  RECT r;
  //  GetWindowRect(m_hwnd, &r);

  //  g_ini_file->write_int("main_wnd_x", r.left, TERPSICHORE_NAME);
  //  g_ini_file->write_int("main_wnd_y", r.top, TERPSICHORE_NAME);
  //  g_ini_file->write_int("main_wnd_w", r.right - r.left, TERPSICHORE_NAME);
  //  g_ini_file->write_int("main_wnd_h", r.bottom - r.top, TERPSICHORE_NAME);
  //}
  g_ini_file->write_int("main_wnd_x", m_x, TERPSICHORE_NAME);
  g_ini_file->write_int("main_wnd_y", m_y, TERPSICHORE_NAME);
  g_ini_file->write_int("main_wnd_w", m_w, TERPSICHORE_NAME);
  g_ini_file->write_int("main_wnd_h", m_h, TERPSICHORE_NAME);

  if (wp.showCmd == SW_SHOWMAXIMIZED)
  {
    g_ini_file->write_int("main_wnd_maximized", 1, TERPSICHORE_NAME);
  }
  else
  {
    g_ini_file->write_int("main_wnd_maximized", 0, TERPSICHORE_NAME);
  }
#else
  RECT r;
  GetWindowRect(m_hwnd, &r);
  if (r.left >= 0 && r.top >= 0)
  {
    g_ini_file->write_int("main_wnd_x", r.left, TERPSICHORE_NAME);
    g_ini_file->write_int("main_wnd_y", r.top, TERPSICHORE_NAME);
    g_ini_file->write_int("main_wnd_w", r.right - r.left, TERPSICHORE_NAME);
    g_ini_file->write_int("main_wnd_h", r.bottom - r.top, TERPSICHORE_NAME);
  }
  //g_ini_file->write_int("main_wnd_x", m_x, TERPSICHORE_NAME);
  //g_ini_file->write_int("main_wnd_y", m_y, TERPSICHORE_NAME);
  //g_ini_file->write_int("main_wnd_w", m_w, TERPSICHORE_NAME);
  //g_ini_file->write_int("main_wnd_h", m_h, TERPSICHORE_NAME);
#endif

  g_main_loop_mutex.Enter();
  if (g_main_loop->IsPlayActiveDeckA())
  {
    g_main_loop->SetStateDeckA(false);
    g_main_loop->EjectDeckA();
  }
  if (g_main_loop->IsPlayActiveDeckB())
  {
    g_main_loop->SetStateDeckB(false);
    g_main_loop->EjectDeckB();
  }
  g_main_loop_mutex.Leave();


  if (IsEngineOnline())
  {
    SetEngineOffline();
    wdl_log("main: offline\n");
  }

  delete g_main_loop;
  delete g_keymap;
  delete g_theme;

  DestroyWindow(m_pl_wnd.Handle());
  DestroyWindow(m_decks_wnd.Handle());

  //DestroyMenu(m_main_menu);

#if defined(_WIN32)
  PostQuitMessage(0);
#elif defined(__APPLE__)
  SWELL_PostQuitMessage(m_hwnd);
#endif

  m_hwnd = NULL;

#if defined(__linux__)
  has_requested_quit = true;
#endif
  return 0;
}

void RST_MainWnd::OnCommand(WPARAM wparam, LPARAM lparam)
{
  switch (LOWORD(wparam))
  {
  case IDOK:
  case IDNO:
  case IDCANCEL:
    {

    }
    break;
  case ID_FILE_OPENFOLDER:
    m_file_task.OpenFolder();
    break;
  case ID_FILE_OPENFILES:
    m_file_task.OpenFiles();
    break;
  case ID_ADDFOLDER:
    m_file_task.AddFolder();
    break;
  case ID_ADDFILES:
    m_file_task.AddFiles();
    break;
  case ID_FILE_LOADPLAYLIST:
    m_file_task.LoadPlayList();
    break;
  case ID_FILE_SAVEPLAYLIST:
    m_file_task.SavePlayList();
    break;
  case ID_PREFERENCES:
    {
      if (!m_pr_wnd.Handle())
      {
        g_main_loop_mutex.Enter();
        bool is_playing = g_main_loop->IsPlaying();
        g_main_loop_mutex.Leave();

        if (is_playing)
        {
          int rc = MessageBox(g_main_wnd->Handle(),
            "Do you want to stop playback?", "Preferences", MB_YESNO);
          if (rc == IDYES)
          {
            g_main_loop_mutex.Enter();
            g_main_loop->SetStateDeckA(false);
            g_main_loop->SetStateDeckB(false);
            g_main_loop_mutex.Leave();
          }
          else return;
        }

        CreateDialogParam(g_inst, MAKEINTRESOURCE(IDD_PREFERENCES),
          m_hwnd, RST_PreferencesWnd::ST_PreferencesWndProc, (LPARAM)&m_pr_wnd);
        ShowWindow(m_pr_wnd.Handle(), SW_SHOW);
      }
      else
      {
        SetFocus(m_pr_wnd.Handle());
      }
    }
    break;
  case ID_HELP_LICENSE:
    {
      if (!m_lic_wnd.Handle())
      {
        CreateDialogParam(g_inst,
          MAKEINTRESOURCE(IDD_LICENSE),
          m_hwnd, RST_LicenseWnd::ST_LicenseWndProc,
          (LPARAM)&m_lic_wnd);
        ShowWindow(m_lic_wnd.Handle(), SW_SHOW);
      }
      else
      {
        SetFocus(m_lic_wnd.Handle());
      }
    }
    break;
  case ID_HELP_ABOUTTERPSICHORE:
    {
      if (!m_about_wnd.Handle())
      {
        CreateDialogParam(g_inst,
          MAKEINTRESOURCE(IDD_ABOUT),
          m_hwnd, RST_AboutWnd::ST_AboutWndProc,
          (LPARAM)&m_about_wnd);
        ShowWindow(m_about_wnd.Handle(), SW_SHOW);
      }
      else
      {
        SetFocus(m_about_wnd.Handle());
      }
    }
    break;
  case ID_HELP_CHANGELOG:
    {
      if (!m_cl_wnd.Handle())
      {
        CreateDialogParam(g_inst,
          MAKEINTRESOURCE(IDD_CHANGELOG),
          m_hwnd, RST_ChangeLogWnd::ST_ChangelogWndProc,
          (LPARAM)&m_cl_wnd);
        ShowWindow(m_cl_wnd.Handle(), SW_SHOW);
      }
      else
      {
        SetFocus(m_cl_wnd.Handle());
      }
    }
    break;
  case ID_HELP_VERSION:
    {
      const char version_string[] =
      {
        TERPSICHORE_NAME_MARKETING " version: " TERPSICHORE_NAKED_VERSION " (" TERPSICHORE_ARCH ")\n\n"
        "Build timestamp: " TERPSICHORE_TIMESTAMP "\n"
        "Git commit: " TERPSICHORE_GIT_SHA
      };

      MessageBox(m_hwnd, version_string, "Version details", MB_OK);
    }
    break;
  //case ID_HELP_TWITTER:
  //  {
  //    ShellExecute(m_hwnd, "", "https://twitter.com/grafmin", "", "", SW_SHOWNORMAL);
  //  }
  //  break;
  case ID_HELP_TERPSICHOREHOME:
    {
      ShellExecute(m_hwnd, "", TERPSICHORE_WEBSITE_URL, "", "", SW_SHOWNORMAL);
    }
    break;
  case ID_HELP_DOCUMENTATION:
    {
      WDL_FastString doc;

      doc.Set(g_module_path.Get());
      doc.Append("documentation.txt");

      ShellExecute(m_hwnd, "", "notepad.exe", doc.Get(), "", SW_SHOWNORMAL);
    }
    break;
  case ID_EXIT:
    DestroyWindow(m_hwnd);
    break;
  case ID_EDIT_FINDINPL:
    {
      if (!m_find_wnd.Handle())
      {
        CreateDialogParam(g_inst, MAKEINTRESOURCE(IDD_FIND),
          m_hwnd, RST_FindWnd::ST_FindWndProc, (LPARAM)&m_find_wnd);
        ShowWindow(m_find_wnd.Handle(), SW_SHOW);
      }
      else
      {
        SetFocus(m_find_wnd.Handle());
      }
    }
    break;
  case ID_EDIT_CUT_TRACKS:
    m_pl_wnd.CutTracks();
    break;
  case ID_EDIT_COPY_TRACKS:
    m_pl_wnd.CopyTracks();
    break;
  case ID_EDIT_PASTE_TRACKS:
    m_pl_wnd.PasteTracks();
    break;
  case ID_EDIT_SORT:
    m_pl_wnd.SortMenu();
    break;
  case ID_EDIT_REVERSEPLAYLIST:
    m_pl_wnd.Reverse();
    break;
  case ID_EDIT_RANDOMIZEPLAYLIST:
    m_pl_wnd.Randomize();
    break;
  case ID_EDIT_SELECTALL:
    m_pl_wnd.SelectAll();
    break;
  case ID_EDIT_SELECTNONE:
    m_pl_wnd.SelectNone();
    break;
  case ID_EDIT_INVERTSELECTION:
    m_pl_wnd.InvertSelection();
    break;
  case ID_EDIT_REMOVESELECTED:
    m_pl_wnd.RemoveSelected();
    break;
  case ID_EDIT_CROPSELECTED:
    m_pl_wnd.CropSelected();
    break;
  case ID_EDIT_CLEARPLAYLIST:
    m_pl_wnd.RemoveAll();
    break;
  case ID_EDIT_REMOVEMISSINGFILES:
    m_pl_wnd.RemoveMissingFiles();
    break;
  case ID_EDIT_MOVESELECTEDUP:
    m_pl_wnd.MoveUp();
    break;
  case ID_EDIT_MOVESELECTEDDOWN:
    m_pl_wnd.MoveDown();
    break;
  case ID_EDIT_REMOVEDUPLICATEENTRIES:
    m_pl_wnd.RemoveDuplicateEntries();
    break;
  }
}

WDL_DLGRET RST_MainWnd::ST_MainWndProc(HWND hwnd, UINT msg, WPARAM wparam, LPARAM lparam)
{
  RST_MainWnd *self = (RST_MainWnd *)GetWindowLongPtr(hwnd, GWLP_USERDATA);

  if (!self && msg == WM_INITDIALOG)
  {
    SetWindowLongPtr(hwnd, GWLP_USERDATA, lparam);
    self = (RST_MainWnd *)lparam;
    self->m_hwnd = hwnd;
  }

  if (WDL_likely(self))
  {
    return self->MainWndProc(msg, wparam, lparam);
  }
  else
  {
    return 0;
  }
}

WDL_DLGRET RST_MainWnd::MainWndProc(UINT msg, WPARAM wparam, LPARAM lparam)
{
  switch (msg)
  {
    case WM_INITDIALOG: OnInitDialog(wparam, lparam); break;
    case WM_TIMER: OnTimer(wparam, lparam); break;
    case WM_NCLBUTTONDOWN: OnNCLButtonDown(wparam, lparam); break;
    case WM_ACTIVATEAPP: OnActivateApp(wparam, lparam); break;
    case WM_SYSCOMMAND: OnSysCommand(wparam, lparam); break;
    case WM_CLOSE: OnClose(wparam, lparam); break;
    case WM_PAINT: OnPaint(wparam, lparam); break;
    case WM_SIZE: OnSize(wparam, lparam); break;
    case WM_MOVE: OnMove(wparam, lparam); break;
    case WM_DESTROY: OnDestroy(wparam, lparam); break;
    case WM_COMMAND: OnCommand(wparam, lparam); break;
    case WM_ERASEBKGND: return 1; break;
  }

  return 0;
}

void RST_MainWnd::SetColor(int bg, int fg)
{
  m_bgc = bg;
  m_txtc = fg;
}

int RST_MainWnd::ProcessMessage(MSG *msg)
{
  if (msg->hwnd == m_hwnd || IsChild(m_hwnd, msg->hwnd))
  {
    if (msg->wParam == VK_CONTROL &&
        (msg->message == WM_KEYDOWN ||
         msg->message == WM_KEYUP))
    {
      if (GetCapture() == m_hwnd)
      {
        //InvalidateRect(m_hwnd, NULL, FALSE);
      }
    }

    //if (msg->message == WM_MOUSEWHEEL)
    //{
    //  SendMessage(msg->hwnd, WM_MOUSEWHEEL, msg->wParam, msg->lParam);

    //  return 1;
    //}

    if (msg->message == WM_KEYDOWN || msg->message == WM_SYSKEYDOWN /* || msg->message == WM_CHAR */)
    {
      // todo: keyboard table
      //if (msg->wParam == 'A' || (msg->wParam == VK_INSERT && msg->message == WM_KEYDOWN))
      //{
      //  return 1;
      //}

      if (msg->lParam & FVIRTKEY)
      {
        if (msg->wParam == g_keymap->GetKey("PL_Delete"))
        {
          if (g_keymap->MatchModifier("PL_Delete"))
          {
            if (!GetCapture())
            {
              m_pl_wnd.RemoveSelected();
            }

            return 1;
          }
        }
        if (msg->wParam == g_keymap->GetKey("PL_Crop"))
        {
          if (g_keymap->MatchModifier("PL_Crop"))
          {
            if (!GetCapture())
            {
              m_pl_wnd.CropSelected();
            }

            return 1;
          }
        }
        if (msg->wParam == g_keymap->GetKey("PL_Clear"))
        {
          if (g_keymap->MatchModifier("PL_Clear"))
          {
            if (!GetCapture())
            {
              m_pl_wnd.RemoveAll();
            }

            return 1;
          }
        }
        if (msg->wParam == g_keymap->GetKey("PL_RemoveMissingFiles"))
        {
          if (g_keymap->MatchModifier("PL_RemoveMissingFiles"))
          {
            if (!GetCapture())
            {
              m_pl_wnd.RemoveMissingFiles();
            }

            return 1;
          }
        }
        if (msg->wParam == g_keymap->GetKey("PL_RemoveDuplicateEntries"))
        {
          if (g_keymap->MatchModifier("PL_RemoveDuplicateEntries"))
          {
            if (!GetCapture())
            {
              m_pl_wnd.RemoveDuplicateEntries();
            }

            return 1;
          }
        }
        if (msg->wParam == g_keymap->GetKey("MoveUp"))
        {
          if (g_keymap->MatchModifier("MoveUp"))
          {
            if (!GetCapture())
            {
              m_pl_wnd.MoveUp();
            }

            return 1;
          }
        }
        if (msg->wParam == g_keymap->GetKey("MoveDown"))
        {
          if (g_keymap->MatchModifier("MoveDown"))
          {
            if (!GetCapture())
            {
              m_pl_wnd.MoveDown();
            }

            return 1;
          }
        }
        if (msg->wParam == g_keymap->GetKey("AboutTerpsichore"))
        {
          if (g_keymap->MatchModifier("AboutTerpsichore"))
          {
            if (!GetCapture())
            {
              SendMessage(m_hwnd, WM_COMMAND, ID_HELP_ABOUTTERPSICHORE, 0);
            }

            return 1;
          }
        }
        if (msg->wParam == g_keymap->GetKey("Changelog"))
        {
          if (g_keymap->MatchModifier("Changelog"))
          {
            if (!GetCapture())
            {
              SendMessage(m_hwnd, WM_COMMAND, ID_HELP_CHANGELOG, 0);
            }

            return 1;
          }
        }
        if (msg->wParam == g_keymap->GetKey("Version"))
        {
          if (g_keymap->MatchModifier("Version"))
          {
            if (!GetCapture())
            {
              SendMessage(m_hwnd, WM_COMMAND, ID_HELP_VERSION, 0);
            }

            return 1;
          }
        }
        if (msg->wParam == g_keymap->GetKey("DecreaseShiftDeckA"))
        {
          if (g_keymap->MatchModifier("DecreaseShiftDeckA"))
          {
            if (!GetCapture())
            {
              g_main_loop_mutex.Enter();
              g_main_loop->DecreaseShiftDeckA();
              g_main_loop_mutex.Leave();
            }

            return 1;
          }
        }
        if (msg->wParam == g_keymap->GetKey("IncreaseShiftDeckA"))
        {
          if (g_keymap->MatchModifier("IncreaseShiftDeckA"))
          {
            if (!GetCapture())
            {
              g_main_loop_mutex.Enter();
              g_main_loop->IncreaseShiftDeckA();
              g_main_loop_mutex.Leave();
            }

            return 1;
          }
        }
        if (msg->wParam == g_keymap->GetKey("DecreaseShiftSmallDeckA"))
        {
          if (g_keymap->MatchModifier("DecreaseShiftSmallDeckA"))
          {
            if (!GetCapture())
            {
              g_main_loop_mutex.Enter();
              g_main_loop->DecreaseShiftSmallDeckA();
              g_main_loop_mutex.Leave();
            }

            return 1;
          }
        }
        if (msg->wParam == g_keymap->GetKey("IncreaseShiftSmallDeckA"))
        {
          if (g_keymap->MatchModifier("IncreaseShiftSmallDeckA"))
          {
            if (!GetCapture())
            {
              g_main_loop_mutex.Enter();
              g_main_loop->IncreaseShiftSmallDeckA();
              g_main_loop_mutex.Leave();
            }

            return 1;
          }
        }
        if (msg->wParam == g_keymap->GetKey("DecreaseTempoDeckA"))
        {
          if (g_keymap->MatchModifier("DecreaseTempoDeckA"))
          {
            if (!GetCapture())
            {
              g_main_loop_mutex.Enter();
              g_main_loop->DecreaseTempoDeckA();
              g_main_loop_mutex.Leave();
            }

            return 1;
          }
        }
        if (msg->wParam == g_keymap->GetKey("IncreaseTempoDeckA"))
        {
          if (g_keymap->MatchModifier("IncreaseTempoDeckA"))
          {
            if (!GetCapture())
            {
              g_main_loop_mutex.Enter();
              g_main_loop->IncreaseTempoDeckA();
              g_main_loop_mutex.Leave();
            }

            return 1;
          }
        }
        if (msg->wParam == g_keymap->GetKey("DecreaseTempoSmallDeckA"))
        {
          if (g_keymap->MatchModifier("DecreaseTempoSmallDeckA"))
          {
            if (!GetCapture())
            {
              g_main_loop_mutex.Enter();
              g_main_loop->DecreaseTempoSmallDeckA();
              g_main_loop_mutex.Leave();
            }

            return 1;
          }
        }
        if (msg->wParam == g_keymap->GetKey("IncreaseTempoSmallDeckA"))
        {
          if (g_keymap->MatchModifier("IncreaseTempoSmallDeckA"))
          {
            if (!GetCapture())
            {
              g_main_loop_mutex.Enter();
              g_main_loop->IncreaseTempoSmallDeckA();
              g_main_loop_mutex.Leave();
            }

            return 1;
          }
        }
        if (msg->wParam == g_keymap->GetKey("DecreaseShiftDeckB"))
        {
          if (g_keymap->MatchModifier("DecreaseShiftDeckB"))
          {
            if (!GetCapture())
            {
              g_main_loop_mutex.Enter();
              g_main_loop->DecreaseShiftDeckB();
              g_main_loop_mutex.Leave();
            }

            return 1;
          }
        }
        if (msg->wParam == g_keymap->GetKey("IncreaseShiftDeckB"))
        {
          if (g_keymap->MatchModifier("IncreaseShiftDeckB"))
          {
            if (!GetCapture())
            {
              g_main_loop_mutex.Enter();
              g_main_loop->IncreaseShiftDeckB();
              g_main_loop_mutex.Leave();
            }

            return 1;
          }
        }
        if (msg->wParam == g_keymap->GetKey("DecreaseShiftSmallDeckB"))
        {
          if (g_keymap->MatchModifier("DecreaseShiftSmallDeckB"))
          {
            if (!GetCapture())
            {
              g_main_loop_mutex.Enter();
              g_main_loop->DecreaseShiftSmallDeckB();
              g_main_loop_mutex.Leave();
            }

            return 1;
          }
        }
        if (msg->wParam == g_keymap->GetKey("IncreaseShiftSmallDeckB"))
        {
          if (g_keymap->MatchModifier("IncreaseShiftSmallDeckB"))
          {
            if (!GetCapture())
            {
              g_main_loop_mutex.Enter();
              g_main_loop->IncreaseShiftSmallDeckB();
              g_main_loop_mutex.Leave();
            }

            return 1;
          }
        }
        if (msg->wParam == g_keymap->GetKey("DecreaseTempoDeckB"))
        {
          if (g_keymap->MatchModifier("DecreaseTempoDeckB"))
          {
            if (!GetCapture())
            {
              g_main_loop_mutex.Enter();
              g_main_loop->DecreaseTempoDeckB();
              g_main_loop_mutex.Leave();
            }

            return 1;
          }
        }
        if (msg->wParam == g_keymap->GetKey("IncreaseTempoDeckB"))
        {
          if (g_keymap->MatchModifier("IncreaseTempoDeckB"))
          {
            if (!GetCapture())
            {
              g_main_loop_mutex.Enter();
              g_main_loop->IncreaseTempoDeckB();
              g_main_loop_mutex.Leave();
            }

            return 1;
          }
        }
        if (msg->wParam == g_keymap->GetKey("DecreaseTempoSmallDeckB"))
        {
          if (g_keymap->MatchModifier("DecreaseTempoSmallDeckB"))
          {
            if (!GetCapture())
            {
              g_main_loop_mutex.Enter();
              g_main_loop->DecreaseTempoSmallDeckB();
              g_main_loop_mutex.Leave();
            }

            return 1;
          }
        }
        if (msg->wParam == g_keymap->GetKey("IncreaseTempoSmallDeckB"))
        {
          if (g_keymap->MatchModifier("IncreaseTempoSmallDeckB"))
          {
            if (!GetCapture())
            {
              g_main_loop_mutex.Enter();
              g_main_loop->IncreaseTempoSmallDeckB();
              g_main_loop_mutex.Leave();
            }

            return 1;
          }
        }
        if (msg->wParam == g_keymap->GetKey("ResetShiftTempoDeckA"))
        {
          if (g_keymap->MatchModifier("ResetShiftTempoDeckA"))
          {
            if (!GetCapture())
            {
              g_main_loop_mutex.Enter();
              g_main_loop->ResetShiftDeckA();
              g_main_loop->ResetTempoDeckA();
              g_main_loop_mutex.Leave();
            }

            return 1;
          }
        }
        if (msg->wParam == g_keymap->GetKey("ResetShiftTempoDeckB"))
        {
          if (g_keymap->MatchModifier("ResetShiftTempoDeckB"))
          {
            if (!GetCapture())
            {
              g_main_loop_mutex.Enter();
              g_main_loop->ResetShiftDeckB();
              g_main_loop->ResetTempoDeckB();
              g_main_loop_mutex.Leave();
            }

            return 1;
          }
        }
        if (msg->wParam == g_keymap->GetKey("LoadDeckA"))
        {
          if (g_keymap->MatchModifier("LoadDeckA"))
          {
            if (!GetCapture())
            {
              g_playlist_wnd->LoadDeckA();
            }

            return 1;
          }
        }
        if (msg->wParam == g_keymap->GetKey("LoadDeckB"))
        {
          if (g_keymap->MatchModifier("LoadDeckB"))
          {
            if (!GetCapture())
            {
              g_playlist_wnd->LoadDeckB();
            }

            return 1;
          }
        }
        if (msg->wParam == g_keymap->GetKey("EjectDeckA")) // && msg->message == WM_KEYDOWN)
        {
          if (g_keymap->MatchModifier("EjectDeckA"))
          {
            if (!GetCapture())
            {
              g_main_loop_mutex.Enter();
              g_main_loop->EjectDeckA();
              g_main_loop_mutex.Leave();
            }

            return 1;
          }
        }
        if (msg->wParam == g_keymap->GetKey("EjectDeckB")) // && msg->message == WM_KEYDOWN)
        {
          if (g_keymap->MatchModifier("EjectDeckB"))
          {
            if (!GetCapture())
            {
              g_main_loop_mutex.Enter();
              g_main_loop->EjectDeckB();
              g_main_loop_mutex.Leave();
            }

            return 1;
          }
        }
      } // FVIRTKEY


      if (msg->wParam == g_keymap->GetKey("EatEscape"))
      {
        if (g_keymap->MatchModifier("EatEscape"))
        {
          if (!GetCapture())
          {
            //Do nothing
          }

          if (g_playlist_wnd && g_playlist_wnd->Handle() == GetCapture())
          {
            ReleaseCapture();
          }

          return 1;
        }
      }
      if (msg->wParam == g_keymap->GetKey("ShowPlayList"))
      {
        if (g_keymap->MatchModifier("ShowPlayList"))
        {
          if (!GetCapture())
          {
            ShowPlayList();
          }

          return 1;
        }
      }
      if (msg->wParam == g_keymap->GetKey("ExitTerpsichore"))
      {
        if (g_keymap->MatchModifier("ExitTerpsichore"))
        {
          if (!GetCapture())
          {
            SendMessage(m_hwnd, WM_DESTROY, 0, 0);
          }

          return 1;
        }
      }
      if (msg->wParam == g_keymap->GetKey("PL_SortMenu"))
      {
        if (g_keymap->MatchModifier("PL_SortMenu"))
        {
          if (!GetCapture())
          {
            m_pl_wnd.SortMenu();
          }

          return 1;
        }
      }
      if (msg->wParam == g_keymap->GetKey("PL_Reverse"))
      {
        if (g_keymap->MatchModifier("PL_Reverse"))
        {
          if (!GetCapture())
          {
            m_pl_wnd.Reverse();
          }

          return 1;
        }
      }
      if (msg->wParam == g_keymap->GetKey("PL_Randomize"))
      {
        if (g_keymap->MatchModifier("PL_Randomize"))
        {
          if (!GetCapture())
          {
            m_pl_wnd.Randomize();
          }

          return 1;
        }
      }
      if (msg->wParam == g_keymap->GetKey("FindInPL"))
      {
        if (g_keymap->MatchModifier("FindInPL"))
        {
          if (!GetCapture())
          {
            SendMessage(m_hwnd, WM_COMMAND, ID_EDIT_FINDINPL, 0);
          }

          return 1;
        }
      }
      if (msg->wParam == g_keymap->GetKey("CutTracks"))
      {
        if (g_keymap->MatchModifier("CutTracks"))
        {
          if (!GetCapture())
          {
            SendMessage(m_hwnd, WM_COMMAND, ID_EDIT_CUT_TRACKS, 0);
          }

          return 1;
        }
      }
      if (msg->wParam == g_keymap->GetKey("CopyTracks"))
      {
        if (g_keymap->MatchModifier("CopyTracks"))
        {
          if (!GetCapture())
          {
            SendMessage(m_hwnd, WM_COMMAND, ID_EDIT_COPY_TRACKS, 0);
          }

          return 1;
        }
      }
      if (msg->wParam == g_keymap->GetKey("PasteTracks"))
      {
        if (g_keymap->MatchModifier("PasteTracks"))
        {
          if (!GetCapture())
          {
            SendMessage(m_hwnd, WM_COMMAND, ID_EDIT_PASTE_TRACKS, 0);
          }

          return 1;
        }
      }
      if (msg->wParam == g_keymap->GetKey("PL_SelectAll"))
      {
        if (g_keymap->MatchModifier("PL_SelectAll"))
        {
          if (!GetCapture())
          {
            m_pl_wnd.SelectAll();
          }

          return 1;
        }
      }
      if (msg->wParam == g_keymap->GetKey("PL_SelectNone"))
      {
        if (g_keymap->MatchModifier("PL_SelectNone"))
        {
          if (!GetCapture())
          {
            m_pl_wnd.SelectNone();
          }

          return 1;
        }
      }
      if (msg->wParam == g_keymap->GetKey("PL_InvertSelection"))
      {
        if (g_keymap->MatchModifier("PL_InvertSelection"))
        {
          if (!GetCapture())
          {
            m_pl_wnd.InvertSelection();
          }

          return 1;
        }
      }
      if (msg->wParam == g_keymap->GetKey("PreviewArtwork"))
      {
        if (g_keymap->MatchModifier("PreviewArtwork"))
        {
          if (!GetCapture())
          {
            SendMessage(m_pl_wnd.Handle(), WM_COMMAND, ID_PL_PREVIEWARTWORK, 0);
          }

          return 1;
        }
      }
      if (msg->wParam == g_keymap->GetKey("OpenFolder"))
      {
        if (g_keymap->MatchModifier("OpenFolder"))
        {
          if (!GetCapture())
          {
            SendMessage(m_hwnd, WM_COMMAND, ID_FILE_OPENFOLDER, 0);
          }

          return 1;
        }
      }
      if (msg->wParam == g_keymap->GetKey("OpenFiles"))
      {
        if (g_keymap->MatchModifier("OpenFiles"))
        {
          if (!GetCapture())
          {
            SendMessage(m_hwnd, WM_COMMAND, ID_FILE_OPENFILES, 0);
          }

          return 1;
        }
      }
      if (msg->wParam == g_keymap->GetKey("AddFolder"))
      {
        if (g_keymap->MatchModifier("AddFolder"))
        {
          if (!GetCapture())
          {
            SendMessage(m_hwnd, WM_COMMAND, ID_ADDFOLDER, 0);
          }

          return 1;
        }
      }
      if (msg->wParam == g_keymap->GetKey("AddFiles"))
      {
        if (g_keymap->MatchModifier("AddFiles"))
        {
          if (!GetCapture())
          {
            SendMessage(m_hwnd, WM_COMMAND, ID_ADDFILES, 0);
          }

          return 1;
        }
      }
      if (msg->wParam == g_keymap->GetKey("LoadPlayList"))
      {
        if (g_keymap->MatchModifier("LoadPlayList"))
        {
          if (!GetCapture())
          {
            SendMessage(m_hwnd, WM_COMMAND, ID_FILE_LOADPLAYLIST, 0);
          }

          return 1;
        }
      }
      if (msg->wParam == g_keymap->GetKey("SavePlayList"))
      {
        if (g_keymap->MatchModifier("SavePlayList"))
        {
          if (!GetCapture())
          {
            SendMessage(m_hwnd, WM_COMMAND, ID_FILE_SAVEPLAYLIST, 0);
          }

          return 1;
        }
      }
      if (msg->wParam == g_keymap->GetKey("Preferences"))
      {
        if (g_keymap->MatchModifier("Preferences"))
        {
          if (!GetCapture())
          {
            SendMessage(m_hwnd, WM_COMMAND, ID_PREFERENCES, 0);
          }

          return 1;
        }
      }
      if (msg->wParam == g_keymap->GetKey("Properties"))
      {
        if (g_keymap->MatchModifier("Properties"))
        {
          if (!GetCapture())
          {
            SendMessage(g_playlist_wnd->Handle(), WM_COMMAND, ID_PL_PROPERTIES, 0);
          }

          return 1;
        }
      }
      if (msg->wParam == g_keymap->GetKey("PlayDeckA"))
      {
        if (g_keymap->MatchModifier("PlayDeckA"))
        {
          if (!GetCapture())
          {
            g_main_loop_mutex.Enter();
            g_main_loop->PlayDeckA();
            g_main_loop_mutex.Leave();
          }

          return 1;
        }
      }
      if (msg->wParam == g_keymap->GetKey("PlayDeckB"))
      {
        if (g_keymap->MatchModifier("PlayDeckB"))
        {
          if (!GetCapture())
          {
            g_main_loop_mutex.Enter();
            g_main_loop->PlayDeckB();
            g_main_loop_mutex.Leave();
          }

          return 1;
        }
      }
      if (msg->wParam == g_keymap->GetKey("ToggleReverseDeckA"))
      {
        if (g_keymap->MatchModifier("ToggleReverseDeckA"))
        {
          if (!GetCapture())
          {
            g_main_loop_mutex.Enter();
            g_main_loop->ToggleReverseDeckA();
            g_main_loop_mutex.Leave();
          }

          return 1;
        }
      }
      if (msg->wParam == g_keymap->GetKey("ToggleReverseDeckB"))
      {
        if (g_keymap->MatchModifier("ToggleReverseDeckB"))
        {
          if (!GetCapture())
          {
            g_main_loop_mutex.Enter();
            g_main_loop->ToggleReverseDeckB();
            g_main_loop_mutex.Leave();
          }

          return 1;
        }
      }
      if (msg->wParam == g_keymap->GetKey("RewindDeckA"))
      {
        if (g_keymap->MatchModifier("RewindDeckA"))
        {
          if (!GetCapture())
          {
            g_main_loop_mutex.Enter();
            if (g_main_loop->IsActiveDeckA())
            {
              g_main_loop->FastForwardDeckA(false);
              g_main_loop->RewindDeckA(true);
            }
            else
            {
              g_main_loop->SeekBackwardDeckA();
            }
            g_main_loop_mutex.Leave();
          }

          return 1;
        }
      }
      if (msg->wParam == g_keymap->GetKey("FastForwardDeckA"))
      {
        if (g_keymap->MatchModifier("FastForwardDeckA"))
        {
          if (!GetCapture())
          {
            g_main_loop_mutex.Enter();
            if (g_main_loop->IsActiveDeckA())
            {
              g_main_loop->RewindDeckA(false);
              g_main_loop->FastForwardDeckA(true);
            }
            else
            {
              g_main_loop->SeekForwardDeckA();
            }
            g_main_loop_mutex.Leave();
          }

          return 1;
        }
      }
      if (msg->wParam == g_keymap->GetKey("RewindDeckB"))
      {
        if (g_keymap->MatchModifier("RewindDeckB"))
        {
          if (!GetCapture())
          {
            g_main_loop_mutex.Enter();
            if (g_main_loop->IsActiveDeckB())
            {
              g_main_loop->FastForwardDeckB(false);
              g_main_loop->RewindDeckB(true);
            }
            else
            {
              g_main_loop->SeekBackwardDeckB();
            }
            g_main_loop_mutex.Leave();
          }

          return 1;
        }
      }
      if (msg->wParam == g_keymap->GetKey("FastForwardDeckB"))
      {
        if (g_keymap->MatchModifier("FastForwardDeckB"))
        {
          if (!GetCapture())
          {
            g_main_loop_mutex.Enter();
            if (g_main_loop->IsActiveDeckB())
            {
              g_main_loop->RewindDeckB(false);
              g_main_loop->FastForwardDeckB(true);
            }
            else
            {
              g_main_loop->SeekForwardDeckB();
            }
            g_main_loop_mutex.Leave();
          }

          return 1;
        }
      }
      if (msg->wParam == g_keymap->GetKey("EnableCueDeckA"))
      {
        if (g_keymap->MatchModifier("EnableCueDeckA"))
        {
          if (!GetCapture())
          {
            g_main_loop_mutex.Enter();
            g_main_loop->EnableCueDeckA();
            g_main_loop_mutex.Leave();
          }

          return 1;
        }
      }
      if (msg->wParam == g_keymap->GetKey("EnableCueDeckB"))
      {
        if (g_keymap->MatchModifier("EnableCueDeckB"))
        {
          if (!GetCapture())
          {
            g_main_loop_mutex.Enter();
            g_main_loop->EnableCueDeckB();
            g_main_loop_mutex.Leave();
          }

          return 1;
        }
      }
      if (msg->wParam == g_keymap->GetKey("SetCueDeckA"))
      {
        if (g_keymap->MatchModifier("SetCueDeckA"))
        {
          if (!GetCapture())
          {
            g_main_loop_mutex.Enter();
            g_main_loop->SetCueDeckA();
            g_main_loop_mutex.Leave();
          }

          return 1;
        }
      }
      if (msg->wParam == g_keymap->GetKey("SetCueDeckB"))
      {
        if (g_keymap->MatchModifier("SetCueDeckB"))
        {
          if (!GetCapture())
          {
            g_main_loop_mutex.Enter();
            g_main_loop->SetCueDeckB();
            g_main_loop_mutex.Leave();
          }

          return 1;
        }
      }
      if (msg->wParam == g_keymap->GetKey("SetHotCue1DeckA"))
      {
        if (g_keymap->MatchModifier("SetHotCue1DeckA"))
        {
          if (!GetCapture())
          {
            g_main_loop_mutex.Enter();
            g_main_loop->EnableHotCueDeckA(1);
            g_main_loop_mutex.Leave();
          }

          return 1;
        }
      }
      if (msg->wParam == g_keymap->GetKey("SetHotCue2DeckA"))
      {
        if (g_keymap->MatchModifier("SetHotCue2DeckA"))
        {
          if (!GetCapture())
          {
            g_main_loop_mutex.Enter();
            g_main_loop->EnableHotCueDeckA(2);
            g_main_loop_mutex.Leave();
          }

          return 1;
        }
      }
      if (msg->wParam == g_keymap->GetKey("SetHotCue3DeckA"))
      {
        if (g_keymap->MatchModifier("SetHotCue3DeckA"))
        {
          if (!GetCapture())
          {
            g_main_loop_mutex.Enter();
            g_main_loop->EnableHotCueDeckA(3);
            g_main_loop_mutex.Leave();
          }

          return 1;
        }
      }
      if (msg->wParam == g_keymap->GetKey("SetHotCue4DeckA"))
      {
        if (g_keymap->MatchModifier("SetHotCue4DeckA"))
        {
          if (!GetCapture())
          {
            g_main_loop_mutex.Enter();
            g_main_loop->EnableHotCueDeckA(4);
            g_main_loop_mutex.Leave();
          }

          return 1;
        }
      }
      if (msg->wParam == g_keymap->GetKey("ClearHotCue1DeckA"))
      {
        if (g_keymap->MatchModifier("ClearHotCue1DeckA"))
        {
          if (!GetCapture())
          {
            g_main_loop_mutex.Enter();
            g_main_loop->ClearHotCueDeckA(1);
            g_main_loop_mutex.Leave();
          }

          return 1;
        }
      }
      if (msg->wParam == g_keymap->GetKey("ClearHotCue2DeckA"))
      {
        if (g_keymap->MatchModifier("ClearHotCue2DeckA"))
        {
          if (!GetCapture())
          {
            g_main_loop_mutex.Enter();
            g_main_loop->ClearHotCueDeckA(2);
            g_main_loop_mutex.Leave();
          }

          return 1;
        }
      }
      if (msg->wParam == g_keymap->GetKey("ClearHotCue3DeckA"))
      {
        if (g_keymap->MatchModifier("ClearHotCue3DeckA"))
        {
          if (!GetCapture())
          {
            g_main_loop_mutex.Enter();
            g_main_loop->ClearHotCueDeckA(3);
            g_main_loop_mutex.Leave();
          }

          return 1;
        }
      }
      if (msg->wParam == g_keymap->GetKey("ClearHotCue4DeckA"))
      {
        if (g_keymap->MatchModifier("ClearHotCue4DeckA"))
        {
          if (!GetCapture())
          {
            g_main_loop_mutex.Enter();
            g_main_loop->ClearHotCueDeckA(4);
            g_main_loop_mutex.Leave();
          }

          return 1;
        }
      }
      if (msg->wParam == g_keymap->GetKey("SetHotCue1DeckB"))
      {
        if (g_keymap->MatchModifier("SetHotCue1DeckB"))
        {
          if (!GetCapture())
          {
            g_main_loop_mutex.Enter();
            g_main_loop->EnableHotCueDeckB(1);
            g_main_loop_mutex.Leave();
          }

          return 1;
        }
      }
      if (msg->wParam == g_keymap->GetKey("SetHotCue2DeckB"))
      {
        if (g_keymap->MatchModifier("SetHotCue2DeckB"))
        {
          if (!GetCapture())
          {
            g_main_loop_mutex.Enter();
            g_main_loop->EnableHotCueDeckB(2);
            g_main_loop_mutex.Leave();
          }

          return 1;
        }
      }
      if (msg->wParam == g_keymap->GetKey("SetHotCue3DeckB"))
      {
        if (g_keymap->MatchModifier("SetHotCue3DeckB"))
        {
          if (!GetCapture())
          {
            g_main_loop_mutex.Enter();
            g_main_loop->EnableHotCueDeckB(3);
            g_main_loop_mutex.Leave();
          }

          return 1;
        }
      }
      if (msg->wParam == g_keymap->GetKey("SetHotCue4DeckB"))
      {
        if (g_keymap->MatchModifier("SetHotCue4DeckB"))
        {
          if (!GetCapture())
          {
            g_main_loop_mutex.Enter();
            g_main_loop->EnableHotCueDeckB(4);
            g_main_loop_mutex.Leave();
          }

          return 1;
        }
      }
      if (msg->wParam == g_keymap->GetKey("ClearHotCue1DeckB"))
      {
        if (g_keymap->MatchModifier("ClearHotCue1DeckB"))
        {
          if (!GetCapture())
          {
            g_main_loop_mutex.Enter();
            g_main_loop->ClearHotCueDeckB(1);
            g_main_loop_mutex.Leave();
          }

          return 1;
        }
      }
      if (msg->wParam == g_keymap->GetKey("ClearHotCue2DeckB"))
      {
        if (g_keymap->MatchModifier("ClearHotCue2DeckB"))
        {
          if (!GetCapture())
          {
            g_main_loop_mutex.Enter();
            g_main_loop->ClearHotCueDeckB(2);
            g_main_loop_mutex.Leave();
          }

          return 1;
        }
      }
      if (msg->wParam == g_keymap->GetKey("ClearHotCue3DeckB"))
      {
        if (g_keymap->MatchModifier("ClearHotCue3DeckB"))
        {
          if (!GetCapture())
          {
            g_main_loop_mutex.Enter();
            g_main_loop->ClearHotCueDeckB(3);
            g_main_loop_mutex.Leave();
          }

          return 1;
        }
      }
      if (msg->wParam == g_keymap->GetKey("ClearHotCue4DeckB"))
      {
        if (g_keymap->MatchModifier("ClearHotCue4DeckB"))
        {
          if (!GetCapture())
          {
            g_main_loop_mutex.Enter();
            g_main_loop->ClearHotCueDeckB(4);
            g_main_loop_mutex.Leave();
          }

          return 1;
        }
      }
      if (msg->wParam == g_keymap->GetKey("FaderDeckA"))
      {
        if (g_keymap->MatchModifier("FaderDeckA"))
        {
          if (!GetCapture())
          {
            g_main_loop_mutex.Enter();
            double pera = g_main_loop->GetFaderDeckA();
            double perb = g_main_loop->GetFaderDeckB();
            pera += 0.10;
            if (WDL_DefinitelyGreaterThan(pera, 1.0)) perb -= 0.10;
            g_main_loop->SetFaderDeckA(pera);
            g_main_loop->SetFaderDeckB(perb);
            g_main_loop_mutex.Leave();
          }

          return 1;
        }
      }
      if (msg->wParam == g_keymap->GetKey("FaderDeckB"))
      {
        if (g_keymap->MatchModifier("FaderDeckB"))
        {
          if (!GetCapture())
          {
            g_main_loop_mutex.Enter();
            double pera = g_main_loop->GetFaderDeckA();
            double perb = g_main_loop->GetFaderDeckB();
            perb += 0.10;
            if (WDL_DefinitelyGreaterThan(perb, 1.0)) pera -= 0.10;
            g_main_loop->SetFaderDeckA(pera);
            g_main_loop->SetFaderDeckB(perb);
            g_main_loop_mutex.Leave();
          }

          return 1;
        }
      }
      if (msg->wParam == g_keymap->GetKey("FaderSmallDeckA"))
      {
        if (g_keymap->MatchModifier("FaderSmallDeckA"))
        {
          if (!GetCapture())
          {
            g_main_loop_mutex.Enter();
            double pera = g_main_loop->GetFaderDeckA();
            double perb = g_main_loop->GetFaderDeckB();
            pera += 0.05; perb -= 0.05;
            g_main_loop->SetFaderDeckA(pera);
            g_main_loop->SetFaderDeckB(perb);
            g_main_loop_mutex.Leave();
          }

          return 1;
        }
      }
      if (msg->wParam == g_keymap->GetKey("FaderSmallDeckB"))
      {
        if (g_keymap->MatchModifier("FaderSmallDeckB"))
        {
          if (!GetCapture())
          {
            g_main_loop_mutex.Enter();
            double pera = g_main_loop->GetFaderDeckA();
            double perb = g_main_loop->GetFaderDeckB();
            pera -= 0.05; perb += 0.05;
            g_main_loop->SetFaderDeckA(pera);
            g_main_loop->SetFaderDeckB(perb);
            g_main_loop_mutex.Leave();
          }

          return 1;
        }
      }
      if (msg->wParam == g_keymap->GetKey("ResetFader"))
      {
        if (g_keymap->MatchModifier("ResetFader"))
        {
          if (!GetCapture())
          {
            g_main_loop_mutex.Enter();
            g_main_loop->SetFaderDeckA(1.0);
            g_main_loop->SetFaderDeckB(1.0);
            g_main_loop_mutex.Leave();
          }

          return 1;
        }
      }
    }


    if (msg->message == WM_KEYUP || msg->message == WM_SYSKEYUP /* || msg->message == WM_CHAR */)
    {
      if (msg->wParam == g_keymap->GetKey("PlayDeckA"))
      {
        if (g_keymap->MatchModifier("PlayDeckA"))
        {
          if (!GetCapture())
          {
            g_main_loop_mutex.Enter();
            g_main_loop->DisablePlayLockDeckA();
            g_main_loop_mutex.Leave();
          }

          return 1;
        }
      }
      if (msg->wParam == g_keymap->GetKey("PlayDeckB"))
      {
        if (g_keymap->MatchModifier("PlayDeckB"))
        {
          if (!GetCapture())
          {
            g_main_loop_mutex.Enter();
            g_main_loop->DisablePlayLockDeckB();
            g_main_loop_mutex.Leave();
          }

          return 1;
        }
      }
      if (msg->wParam == g_keymap->GetKey("RewindDeckA"))
      {
        if (g_keymap->MatchModifier("RewindDeckA"))
        {
          if (!GetCapture())
          {
            g_main_loop_mutex.Enter();
            g_main_loop->RewindDeckA(false);
            g_main_loop_mutex.Leave();
          }

          return 1;
        }
      }
      if (msg->wParam == g_keymap->GetKey("FastForwardDeckA"))
      {
        if (g_keymap->MatchModifier("FastForwardDeckA"))
        {
          if (!GetCapture())
          {
            g_main_loop_mutex.Enter();
            g_main_loop->FastForwardDeckA(false);
            g_main_loop_mutex.Leave();
          }

          return 1;
        }
      }
      if (msg->wParam == g_keymap->GetKey("RewindDeckB"))
      {
        if (g_keymap->MatchModifier("RewindDeckB"))
        {
          if (!GetCapture())
          {
            g_main_loop_mutex.Enter();
            g_main_loop->RewindDeckB(false);
            g_main_loop_mutex.Leave();
          }

          return 1;
        }
      }
      if (msg->wParam == g_keymap->GetKey("FastForwardDeckB"))
      {
        if (g_keymap->MatchModifier("FastForwardDeckB"))
        {
          if (!GetCapture())
          {
            g_main_loop_mutex.Enter();
            g_main_loop->FastForwardDeckB(false);
            g_main_loop_mutex.Leave();
          }

          return 1;
        }
      }
      if (msg->wParam == g_keymap->GetKey("EnableCueDeckA"))
      {
        if (g_keymap->MatchModifier("EnableCueDeckA"))
        {
          if (!GetCapture())
          {
            g_main_loop_mutex.Enter();
            g_main_loop->DisableCueDeckA();
            g_main_loop_mutex.Leave();
          }

          return 1;
        }
      }
      if (msg->wParam == g_keymap->GetKey("EnableCueDeckB"))
      {
        if (g_keymap->MatchModifier("EnableCueDeckB"))
        {
          if (!GetCapture())
          {
            g_main_loop_mutex.Enter();
            g_main_loop->DisableCueDeckB();
            g_main_loop_mutex.Leave();
          }

          return 1;
        }
      }
      if (msg->wParam == g_keymap->GetKey("SetHotCue1DeckA"))
      {
        if (g_keymap->MatchModifier("SetHotCue1DeckA"))
        {
          if (!GetCapture())
          {
            g_main_loop_mutex.Enter();
            g_main_loop->DisableHotCueDeckA(1);
            g_main_loop_mutex.Leave();
          }

          return 1;
        }
      }
      if (msg->wParam == g_keymap->GetKey("SetHotCue2DeckA"))
      {
        if (g_keymap->MatchModifier("SetHotCue2DeckA"))
        {
          if (!GetCapture())
          {
            g_main_loop_mutex.Enter();
            g_main_loop->DisableHotCueDeckA(2);
            g_main_loop_mutex.Leave();
          }

          return 1;
        }
      }
      if (msg->wParam == g_keymap->GetKey("SetHotCue3DeckA"))
      {
        if (g_keymap->MatchModifier("SetHotCue3DeckA"))
        {
          if (!GetCapture())
          {
            g_main_loop_mutex.Enter();
            g_main_loop->DisableHotCueDeckA(3);
            g_main_loop_mutex.Leave();
          }

          return 1;
        }
      }
      if (msg->wParam == g_keymap->GetKey("SetHotCue4DeckA"))
      {
        if (g_keymap->MatchModifier("SetHotCue4DeckA"))
        {
          if (!GetCapture())
          {
            g_main_loop_mutex.Enter();
            g_main_loop->DisableHotCueDeckA(4);
            g_main_loop_mutex.Leave();
          }

          return 1;
        }
      }
      if (msg->wParam == g_keymap->GetKey("SetHotCue1DeckB"))
      {
        if (g_keymap->MatchModifier("SetHotCue1DeckB"))
        {
          if (!GetCapture())
          {
            g_main_loop_mutex.Enter();
            g_main_loop->DisableHotCueDeckB(1);
            g_main_loop_mutex.Leave();
          }

          return 1;
        }
      }
      if (msg->wParam == g_keymap->GetKey("SetHotCue2DeckB"))
      {
        if (g_keymap->MatchModifier("SetHotCue2DeckB"))
        {
          if (!GetCapture())
          {
            g_main_loop_mutex.Enter();
            g_main_loop->DisableHotCueDeckB(2);
            g_main_loop_mutex.Leave();
          }

          return 1;
        }
      }
      if (msg->wParam == g_keymap->GetKey("SetHotCue3DeckB"))
      {
        if (g_keymap->MatchModifier("SetHotCue3DeckB"))
        {
          if (!GetCapture())
          {
            g_main_loop_mutex.Enter();
            g_main_loop->DisableHotCueDeckB(3);
            g_main_loop_mutex.Leave();
          }

          return 1;
        }
      }
      if (msg->wParam == g_keymap->GetKey("SetHotCue4DeckB"))
      {
        if (g_keymap->MatchModifier("SetHotCue4DeckB"))
        {
          if (!GetCapture())
          {
            g_main_loop_mutex.Enter();
            g_main_loop->DisableHotCueDeckB(4);
            g_main_loop_mutex.Leave();
          }

          return 1;
        }
      }
    }
  }

  return 0;
}

void RST_MainWnd::RunMessageLoop()
{
#if defined(_WIN32)

  for (;;) // Justin Frankel in the loop
  {
    MSG msg = { 0, };
    int vvv = GetMessage(&msg, NULL, 0, 0);

    if (!vvv)
    {
      break;
    }

    if (vvv < 0)
    {
      Sleep(10);
      continue;
    }

    if (!msg.hwnd)
    {
      DispatchMessage(&msg);
      continue;
    }

    vvv = ProcessMessage(&msg);

    if (vvv > 0)
    {
      continue;
    }

    if (IsDialogMessage(m_hwnd, &msg))
    {
      continue;
    }

    if (IsDialogMessage(m_pl_wnd.Handle(), &msg))
    {
      continue;
    }

    if (IsDialogMessage(m_pl_wnd.PropertiesHandle(), &msg))
    {
      continue;
    }

    if (IsDialogMessage(m_pl_wnd.PreviewHandle(), &msg))
    {
      continue;
    }

    if (IsDialogMessage(m_pr_wnd.Handle(), &msg))
    {
      continue;
    }

    if (IsDialogMessage(m_about_wnd.Handle(), &msg))
    {
      continue;
    }

    if (IsDialogMessage(m_ft_prog_wnd.Handle(), &msg))
    {
      continue;
    }

    if (IsDialogMessage(m_cl_wnd.Handle(), &msg))
    {
      continue;
    }

    if (IsDialogMessage(m_lic_wnd.Handle(), &msg))
    {
      continue;
    }

    if (IsDialogMessage(m_find_wnd.Handle(), &msg))
    {
      continue;
    }

    if (IsDialogMessage(m_exactvol_wnd.Handle(), &msg))
    {
      continue;
    }

    HWND parent = NULL;
    HWND temp = msg.hwnd;

    do
    {
      if (GetClassLong(temp, GCW_ATOM) == (INT)32770)
      {
        parent = temp;

        if (!(GetWindowLong(temp, GWL_STYLE) &WS_CHILD))
        {
          break;  // not a child, exit
        }
      }
    }
    while (temp = GetParent(temp));

    if (parent && IsDialogMessage(parent, &msg))
    {
      continue;
    }

    TranslateMessage(&msg);
    DispatchMessage(&msg);
  }

#else

  for (;;)
  {
    MSG msg = { 0, };
    int x = ProcessMessage(&msg);

    if (x == 0)
    {
      break;
    }
  }

#endif
}

#include <WDL/eel2/ns-eel.h>
#include <WDL/lice/lice.h>
#include <WDL/wingui/virtwnd-controls.h>
#include <WDL/wingui/scrollbar/coolscroll.h>

// an app should implement these
int WDL_STYLE_WantGlobalButtonBorders()
{
  return 0;
}

bool WDL_STYLE_WantGlobalButtonBackground(int *col)
{
  return false;
}

int WDL_STYLE_GetSysColor(int p)
{
  return GetSysColor(p);
}

void WDL_STYLE_ScaleImageCoords(int *x, int *y)
{}

bool WDL_Style_WantTextShadows(int *col)
{
  return false;
}

// this is the default, you can override per painter if you want.
// return values 0.0-1.0 for each, return false if no gradient desired
bool WDL_STYLE_GetBackgroundGradient(double *gradstart, double *gradslope)
{
  return false;
}

// for slider
LICE_IBitmap *WDL_STYLE_GetSliderBitmap2(bool vert)
{
  return NULL;
}

bool WDL_STYLE_AllowSliderMouseWheel()
{
  return false;
}

int WDL_STYLE_GetSliderDynamicCenterPos()
{
  return 0;
}

// TO BE IMPLEMENTED BY APP:
// implemented by calling app, can return a LICE_IBitmap **img for "scrollbar"
void *GetIconThemePointer(const char *name)
{
  if (!strcmp(name, "scrollbar"))
  {}

  return NULL;
}

// can be a passthrough to GetSysColor()
int CoolSB_GetSysColor(HWND m_hwnd, int val)
{
  return GetSysColor(val);
}

#include <WDL/eel2/ns-eel.h>

void NSEEL_HOSTSTUB_EnterMutex()
{}

void NSEEL_HOSTSTUB_LeaveMutex()
{}
