// Copyright (c) 2021 Giorgos Vougioukas
//
// The license can be found in the LICENSE file.

#include "terpsichore/peaks.h"
#include "terpsichore/terpsichore_plugin.h"
#include "terpsichore/plugin.h"

#include <sys/types.h>
#include <sys/stat.h>

#ifdef _WIN32
#define rst_stat_t _stat64
#define rst_stat(a,b) _stat64(a,b)
#else
#define rst_stat_t stat
#define rst_stat(a,b) stat(a,b)
#endif

#include "WDL/fpcmp.h"
#include "WDL/fileread.h"
#include "WDL/filewrite.h"
#include "WDL/pcmfmtcvt.h"
#include "WDL/win32_utf8.h"

RST_Peaks::RST_Peaks()
  : m_thread(NULL)
  , m_kill_thread(false)
  , m_fr(NULL)
  , m_peakspersec(400)
  , m_peaksready(false)
{}

RST_Peaks::~RST_Peaks()
{
  m_kill_thread = true;

  if (m_thread)
  {
    WaitForSingleObject(m_thread, INFINITE);
    CloseHandle(m_thread);
    m_thread = NULL;
  }

  if (m_fr) delete m_fr;
}

bool RST_Peaks::Open(const char *filename)
{
  m_fn.Set(filename);

  if (!m_thread)
  {
    unsigned int thread_id;
    m_thread = (HANDLE)_beginthreadex(NULL, 0, ThreadFunction, (void *)this, 0, &thread_id);
  }

  return true;
}

int RST_Peaks::GetPeaks(short *buffer, int length)
{
  if (!m_fr)
  {
    WDL_FastString pfn(m_fn.Get());
    pfn.Append(".terpsichore_peaks");
    m_fr = new WDL_NEW WDL_FileRead(pfn.Get());
  }

  if (m_fr->IsOpen())
  {
    return m_fr->Read((void *)buffer, length * sizeof(short));
  }

  return 0;
}

void RST_Peaks::Seek(double time)
{
  if (m_fr && m_fr->IsOpen())
  {
    int mipmapdata = 4 + 1 + 4 + 8 + 8 + 4 + 8;
    WDL_INT64 pos = (WDL_INT64)(time * m_peakspersec);
    m_fr->SetPosition(pos + mipmapdata);
  }
}

WDL_INT64 RST_Peaks::GetNumPeaks()
{
  if (m_fr && m_fr->IsOpen())
  {
    WDL_INT64 np;
    m_fr->SetPosition(4 + 1 + 4 + 8 + 8 + 4);
    m_fr->Read(&np, sizeof(np));
    return np;
  }

  return 0;
}

unsigned int WINAPI RST_Peaks::ThreadFunction(void *arg)
{
#if defined(_WIN32)
  CoInitialize(NULL);
#endif

  RST_Peaks *self = (RST_Peaks *)arg;

  if (WDL_NORMALLY(self))
  {
    //self->m_kill_thread = false;

    //while (!self->m_kill_thread)
    //{
    //  self->m_mutex.Enter();
    //  while (!self->WritePeaks());
    //  self->m_mutex.Leave();
    //  Sleep(50);
    //}

    self->WritePeaks();
  }

#if defined(_WIN32)
  CoUninitialize();
#endif

  return 0;
}

bool RST_Peaks::ArePeaksReady() const
{
  return m_peaksready;
}

int RST_Peaks::WritePeaks()
{
  RST_IFileInput *fin = CreateAudioInput(m_fn.Get());
  if (!fin) return 1;

  struct rst_stat_t stbuf;
  rst_stat(m_fn.Get(), &stbuf);

  WDL_FastString pfn(m_fn.Get());
  pfn.Append(".terpsichore_peaks");
  WDL_FileRead fr(pfn.Get());

  if (fr.IsOpen())
  {
    WDL_INT64 t;
    fr.SetPosition(4 + 1 + 4); fr.Read(&t, sizeof(t));
    if (t == stbuf.st_mtime) { delete fin; m_peaksready = true; return 1; }
  }

  WDL_FileWrite *fw = new WDL_NEW WDL_FileWrite(pfn.Get());
  if (!fw) return 1; if (!fw->IsOpen()) { delete fw; return 1; }

  // 4 bytes: TPAA (type)
  // 1 byte int: channels
  // 4 bytes int: samplerate
  // 8 bytes int: last modified time of source
  // 8 bytes int: file size of source
  // mipmap header
  // 4 bytes int: division factor
  // 8 bytes int: number of peaks pairs
  // mipmap data N
  // 2 bytes int: -32768 to 32767 pairs

  char hdr[4] = { 'T', 'P', 'A', 'A' };
  char ch = (char)fin->GetChannels();
  int sr = (int)fin->GetSampleRate();
  WDL_INT64 tm = (WDL_INT64)stbuf.st_mtime;
  WDL_INT64 fs = (WDL_INT64)stbuf.st_size;
  int df = (int)(fin->GetSampleRate() / m_peakspersec);
  WDL_INT64 np = 0;
  WDL_TypedBuf<short> md;
  md.Resize(ch);

  fw->Write(hdr, sizeof(hdr));
  fw->Write(&ch, sizeof(ch));
  fw->Write(&sr, sizeof(sr));
  fw->Write(&tm, sizeof(tm));
  fw->Write(&fs, sizeof(fs));
  fw->Write(&df, sizeof(df));

  WDL_TypedBuf<SAM> buf;
  buf.Resize(4096);
  int nsam = 0, diff = 0;

  while ((nsam = fin->GetSamples(buf.Get(), buf.GetSize())) > 0)
  {
    if (nsam < ch) continue;
    int i = 0 + diff;
    for (; i < nsam; i += df * ch)
    {
#if (TERPSICHORE_SAMPLE_PRECISION == 8)
      doublesToPcm(buf.Get() + i, 1, ch, md.Get(), 16, 1);
#else
      floatsToPcm(buf.Get() + i, 1, ch, md.Get(), 16, 1);
#endif
      fw->Write(md.Get(), md.GetSize() * sizeof(short));
      np += 1;
    }
    if (i > nsam) diff = i - nsam; else diff = 0;
    if (m_kill_thread) { delete fin; delete fw; DeleteFile(pfn.Get()); return 1; }
#if defined(_WIN32) && defined(_DEBUG)
    WDL_FastString rep; static int s = 0; s += nsam / ch;
    rep.SetFormatted(256, "Total samples: %d, total peaks: %d\n", s, np);
    OutputDebugString(rep.Get());
#endif
  }

  fw->SetPosition(4 + 1 + 4 + 8 + 8 + 4);
  fw->Write(&np, sizeof(np));

  m_kill_thread = true;
  delete fin; delete fw;
  m_peaksready = true;
  return 1;
}
