// Copyright (c) 2022 Giorgos Vougioukas
//
// The license can be found in the LICENSE file.

#include "terpsichore/pr_ebur128_wnd.h"
#include "terpsichore/main_wnd.h"

#include "WDL/win32_utf8.h"
#include "WDL/wdlstring.h"
#include "WDL/heapbuf.h"

RST_EBUR128Wnd::RST_EBUR128Wnd()
  : m_hwnd(NULL)
{}

RST_EBUR128Wnd::~RST_EBUR128Wnd()
{}

HWND RST_EBUR128Wnd::Handle() const
{
  return m_hwnd;
}

WDL_DLGRET RST_EBUR128Wnd::ST_EBUR128WndProc(HWND hwnd, UINT msg, WPARAM wparam, LPARAM lparam)
{
  RST_EBUR128Wnd *self = (RST_EBUR128Wnd *)GetWindowLongPtr(hwnd, GWLP_USERDATA);

  if (!self && msg == WM_INITDIALOG)
  {
    SetWindowLongPtr(hwnd, GWLP_USERDATA, lparam);
    self = (RST_EBUR128Wnd *)lparam;
    self->m_hwnd = hwnd;
  }

  if (WDL_likely(self))
  {
    return self->EBUR128WndProc(msg, wparam, lparam);
  }
  else
  {
    return 0;
  }
}

void RST_EBUR128Wnd::OnInitDialog(WPARAM wparam, LPARAM lparam)
{
  // EBU R128 standard for loudness normalization.
  int ebur128normalization = g_ini_file->read_int("ebur128_normalization", 0, "preferences");
  int ebur128downwardonly = g_ini_file->read_int("ebur128_downward_only", 1, "preferences");
  int ebur128reference = g_ini_file->read_int("ebur128_reference", 0, "preferences");
  int ebur128mode = g_ini_file->read_int("ebur128_mode", 2, "preferences");

  if (ebur128normalization)
  {
    SendMessage(GetDlgItem(m_hwnd, IDC_CHECK1), BM_SETCHECK, BST_CHECKED, 0);
    EnableWindow(GetDlgItem(m_hwnd, IDC_CHECK2), TRUE);
    EnableWindow(GetDlgItem(m_hwnd, IDC_STATIC1), TRUE);
    EnableWindow(GetDlgItem(m_hwnd, IDC_STATIC2), TRUE);
    EnableWindow(GetDlgItem(m_hwnd, IDC_COMBO1), TRUE);
    EnableWindow(GetDlgItem(m_hwnd, IDC_COMBO2), TRUE);
  }
  else
  {
    SendMessage(GetDlgItem(m_hwnd, IDC_CHECK1), BM_SETCHECK, BST_UNCHECKED, 0);
    EnableWindow(GetDlgItem(m_hwnd, IDC_CHECK2), FALSE);
    EnableWindow(GetDlgItem(m_hwnd, IDC_STATIC1), FALSE);
    EnableWindow(GetDlgItem(m_hwnd, IDC_STATIC2), FALSE);
    EnableWindow(GetDlgItem(m_hwnd, IDC_COMBO1), FALSE);
    EnableWindow(GetDlgItem(m_hwnd, IDC_COMBO2), FALSE);
  }

  if (ebur128downwardonly)
  {
    SendMessage(GetDlgItem(m_hwnd, IDC_CHECK2), BM_SETCHECK, BST_CHECKED, 0);
  }
  else
  {
    SendMessage(GetDlgItem(m_hwnd, IDC_CHECK2), BM_SETCHECK, BST_UNCHECKED, 0);
  }

  // Loudness reference.
  //   0: -23 LUFS (standard)
  //   1: -18 LUFS (RG2 - backwards compatible with RG1)
  //   2: -16 LUFS
  //   3: -14 LUFS
  //   4: -11 LUFS
  HWND loudnessreference = GetDlgItem(m_hwnd, IDC_COMBO1);
  SendMessage(loudnessreference, CB_ADDSTRING, 0, (LPARAM)"-23 LUFS (standard)");
  SendMessage(loudnessreference, CB_SETITEMDATA, 0, 0);
  SendMessage(loudnessreference, CB_ADDSTRING, 0, (LPARAM)"-18 LUFS (RG2 - backwards compatible with RG1)");
  SendMessage(loudnessreference, CB_SETITEMDATA, 1, 1);
  SendMessage(loudnessreference, CB_ADDSTRING, 0, (LPARAM)"-16 LUFS");
  SendMessage(loudnessreference, CB_SETITEMDATA, 2, 2);
  SendMessage(loudnessreference, CB_ADDSTRING, 0, (LPARAM)"-14 LUFS");
  SendMessage(loudnessreference, CB_SETITEMDATA, 3, 3);
  SendMessage(loudnessreference, CB_ADDSTRING, 0, (LPARAM)"-11 LUFS");
  SendMessage(loudnessreference, CB_SETITEMDATA, 4, 4);

  switch (ebur128reference)
  {
    case 0: SendMessage(loudnessreference, CB_SETCURSEL, 0, 0); break;
    case 1: SendMessage(loudnessreference, CB_SETCURSEL, 1, 0); break;
    case 2: SendMessage(loudnessreference, CB_SETCURSEL, 2, 0); break;
    case 3: SendMessage(loudnessreference, CB_SETCURSEL, 3, 0); break;
    case 4: SendMessage(loudnessreference, CB_SETCURSEL, 4, 0); break;
    case 5: SendMessage(loudnessreference, CB_SETCURSEL, 5, 0); break;
    default: SendMessage(loudnessreference, CB_SETCURSEL, 0, 0); break;
  }

  // Loudness mode.
  //   0: M (momentary)
  //   1: S (shortterm)
  //   2: I (integrated)
  //   3: LRA (range)
  //   4: sample peak
  //   5: true peak
  HWND loudnessmode = GetDlgItem(m_hwnd, IDC_COMBO2);
  SendMessage(loudnessmode, CB_ADDSTRING, 0, (LPARAM)"M (momentary)");
  SendMessage(loudnessmode, CB_SETITEMDATA, 0, 0);
  SendMessage(loudnessmode, CB_ADDSTRING, 0, (LPARAM)"S (shortterm)");
  SendMessage(loudnessmode, CB_SETITEMDATA, 1, 1);
  SendMessage(loudnessmode, CB_ADDSTRING, 0, (LPARAM)"I (integrated)");
  SendMessage(loudnessmode, CB_SETITEMDATA, 2, 2);
  SendMessage(loudnessmode, CB_ADDSTRING, 0, (LPARAM)"LRA (range)");
  SendMessage(loudnessmode, CB_SETITEMDATA, 3, 3);
  SendMessage(loudnessmode, CB_ADDSTRING, 0, (LPARAM)"Sample peak");
  SendMessage(loudnessmode, CB_SETITEMDATA, 4, 4);
  SendMessage(loudnessmode, CB_ADDSTRING, 0, (LPARAM)"True peak");
  SendMessage(loudnessmode, CB_SETITEMDATA, 5, 5);

  switch (ebur128mode)
  {
    case 0: SendMessage(loudnessmode, CB_SETCURSEL, 0, 0); break;
    case 1: SendMessage(loudnessmode, CB_SETCURSEL, 1, 0); break;
    case 2: SendMessage(loudnessmode, CB_SETCURSEL, 2, 0); break;
    case 3: SendMessage(loudnessmode, CB_SETCURSEL, 3, 0); break;
    case 4: SendMessage(loudnessmode, CB_SETCURSEL, 4, 0); break;
    case 5: SendMessage(loudnessmode, CB_SETCURSEL, 5, 0); break;
    default: SendMessage(loudnessmode, CB_SETCURSEL, 2, 0); break;
  }
}

void RST_EBUR128Wnd::OnDestroy(WPARAM wparam, LPARAM lparam)
{
  m_hwnd = NULL;
}

void RST_EBUR128Wnd::OnCommand(WPARAM wparam, LPARAM lparam)
{
  switch(LOWORD(wparam))
  {
  case IDC_COMBO1:
    {
      if (HIWORD(wparam) == CBN_SELCHANGE)
      {
        HWND loudnessreference = GetDlgItem(m_hwnd, IDC_COMBO1);
        int sel = (int)SendMessage(loudnessreference, CB_GETCURSEL, 0, 0);
        int ebur128reference = (int)SendMessage(loudnessreference, CB_GETITEMDATA, sel, 0);

        g_preference_settings->Set("ebur128_reference", ebur128reference);
      }
    }
    break;
  case IDC_COMBO2:
    {
      if (HIWORD(wparam) == CBN_SELCHANGE)
      {
        HWND loudnessmode = GetDlgItem(m_hwnd, IDC_COMBO2);
        int sel = (int)SendMessage(loudnessmode, CB_GETCURSEL, 0, 0);
        int ebur128mode = (int)SendMessage(loudnessmode, CB_GETITEMDATA, sel, 0);

        g_preference_settings->Set("ebur128_mode", ebur128mode);
      }
    }
    break;
  case IDC_CHECK1:
    {
      if (HIWORD(wparam) == BN_CLICKED)
      {
        int chk = (int)SendMessage(GetDlgItem(m_hwnd, IDC_CHECK1), BM_GETCHECK, 0, 0);
        if (chk == BST_CHECKED)
        {
          g_preference_settings->Set("ebur128_normalization", 1);
          EnableWindow(GetDlgItem(m_hwnd, IDC_CHECK2), TRUE);
          EnableWindow(GetDlgItem(m_hwnd, IDC_STATIC1), TRUE);
          EnableWindow(GetDlgItem(m_hwnd, IDC_STATIC2), TRUE);
          EnableWindow(GetDlgItem(m_hwnd, IDC_COMBO1), TRUE);
          EnableWindow(GetDlgItem(m_hwnd, IDC_COMBO2), TRUE);
        }
        else if (chk == BST_UNCHECKED)
        {
          g_preference_settings->Set("ebur128_normalization", 0);
          EnableWindow(GetDlgItem(m_hwnd, IDC_CHECK2), FALSE);
          EnableWindow(GetDlgItem(m_hwnd, IDC_STATIC1), FALSE);
          EnableWindow(GetDlgItem(m_hwnd, IDC_STATIC2), FALSE);
          EnableWindow(GetDlgItem(m_hwnd, IDC_COMBO1), FALSE);
          EnableWindow(GetDlgItem(m_hwnd, IDC_COMBO2), FALSE);
        }
      }
    }
    break;
  case IDC_CHECK2:
    {
      if (HIWORD(wparam) == BN_CLICKED)
      {
        int chk = (int)SendMessage(GetDlgItem(m_hwnd, IDC_CHECK2), BM_GETCHECK, 0, 0);
        if (chk == BST_CHECKED)
        {
          g_preference_settings->Set("ebur128_downward_only", 1);
        }
        else if (chk == BST_UNCHECKED)
        {
          g_preference_settings->Set("ebur128_downward_only", 0);
        }
      }
    }
    break;
  }
}

WDL_DLGRET RST_EBUR128Wnd::EBUR128WndProc(UINT msg, WPARAM wparam, LPARAM lparam)
{
  switch(msg)
  {
    case WM_INITDIALOG: OnInitDialog(wparam, lparam); break;
    case WM_DESTROY: OnDestroy(wparam, lparam); break;
    case WM_COMMAND: OnCommand(wparam, lparam); break;
  }

  return 0;
}
