// Copyright (c) 2021 Giorgos Vougioukas
//
// The license can be found in the LICENSE file.

#include "terpsichore/track.h"
#include "terpsichore/plugin.h"
#include "terpsichore/db2slider.h"
#include "terpsichore/antidote_input.h"
#include "terpsichore/decks_wnd.h"

#include <math.h>

#include "WDL/fpcmp.h"
#include "WDL/sha.h"
#include "WDL/db2val.h"

RST_Track::RST_Track()
  : m_fi(NULL)
  , m_is_open(false)
  , m_eof(false)
  , m_bq_max_duration(200) // ms
  , m_duration(0.0)
  , m_run_time(0.0)
  , m_ps(NULL)
  , m_shift(1.0)
  , m_tempo(1.0)
  , m_last_shift(1.0)
  , m_last_tempo(1.0)
  , m_min_shift(0.25)
  , m_max_shift(4.0)
  , m_min_tempo(0.25)
  , m_max_tempo(4.0)
  , m_shift_step(0.1)
  , m_tempo_step(0.1)
  , m_shift_altstep(0.01)
  , m_tempo_altstep(0.01)
  , m_seek_step(0.100)
  , m_volume(0.0)
  , m_fader(0.0)
  , m_ebur128normalization(0)
  , m_ebur128downwardonly(0)
  , m_ebur128reference(0.0)
  , m_ebur128mode(0)
  , m_ebur128momentary(0.0)
  , m_ebur128shortterm(0.0)
  , m_ebur128momentarywindow(0)
  , m_ebur128shorttermwindow(0)
  , m_ebur128integrated(0.0)
  , m_ebur128range(0.0)
  , m_ebur128samplepeak(0.0)
  , m_ebur128truepeak(0.0)
  , m_ebur128gain(0.0)
  , m_rwd(false)
  , m_fwd(false)
  , m_prevrev(false)
  , m_prevshift(0.0)
  , m_prevtempo(0.0)
  , m_wf(NULL)
  , m_peakrate(0)
  , m_pkfr(NULL)
{
  int psstep = g_ini_file->read_int("pitch_shift_step", 100, "preferences");
  int tsstep = g_ini_file->read_int("time_stretch_step", 100, "preferences");
  int psaltstep = g_ini_file->read_int("pitch_shift_altstep", 10, "preferences");
  int tsaltstep = g_ini_file->read_int("time_stretch_altstep", 10, "preferences");
  int seekstep = g_ini_file->read_int("seek_step", 100, "preferences");

  m_shift_step = wdl_clamp((double)psstep / 1000, 0.01, 4.0);
  m_tempo_step = wdl_clamp((double)tsstep / 1000, 0.01, 4.0);
  m_shift_altstep = wdl_clamp((double)psaltstep / 1000, 0.01, 4.0);
  m_tempo_altstep = wdl_clamp((double)tsaltstep / 1000, 0.01, 4.0);
  m_seek_step = wdl_clamp((double)seekstep / 1000, 0.01, 10.0);

  m_wf = new LICE_SysBitmap;
}

RST_Track::~RST_Track()
{
  if (m_pkfr) delete m_pkfr;
  if (m_fi) delete m_fi;
  if (m_ps) delete m_ps;
  if (m_wf) delete m_wf;
}

bool RST_Track::Open(const char *fn, int peakrate,
  WDL_StringKeyedArray<double> *ebur128_info)
{
  m_peakrate = peakrate;

  m_ebur128normalization = (int)ebur128_info->Get("ebur128normalization", 0.0);
  m_ebur128downwardonly = (int)ebur128_info->Get("ebur128downwardonly", 0.0);
  m_ebur128reference = ebur128_info->Get("ebur128reference", 0.0);
  m_ebur128mode = (int)ebur128_info->Get("ebur128mode", 0.0);
  m_ebur128momentary = ebur128_info->Get("ebur128momentary", 0.0);
  m_ebur128shortterm = ebur128_info->Get("ebur128shortterm", 0.0);
  m_ebur128momentarywindow = (int)ebur128_info->Get("ebur128momentarywindow", 0.0);
  m_ebur128shorttermwindow = (int)ebur128_info->Get("ebur128shorttermwindow", 0.0);
  m_ebur128integrated = ebur128_info->Get("ebur128integrated", 0.0);
  m_ebur128range = ebur128_info->Get("ebur128range", 0.0);
  m_ebur128samplepeak = ebur128_info->Get("ebur128samplepeak", 0.0);
  m_ebur128truepeak = ebur128_info->Get("ebur128truepeak", 0.0);
  m_ebur128gain = ebur128_info->Get("ebur128gain", 0.0);

  WDL_FastString tcfn(g_settings_path.Get());
  tcfn.Append("cache" WDL_DIRCHAR_STR);

  WDL_SHA1 sha;
  char hash[WDL_SHA1SIZE];
  sha.reset();
  sha.add(fn, (int)strlen(fn));
  sha.result(hash);

  tcfn.AppendFormatted(64, "%02x", (unsigned char)(hash[0] & 0xff));
  tcfn.Append(WDL_DIRCHAR_STR);

  for (int i = 1; i < WDL_SHA1SIZE; i++)
  {
    tcfn.AppendFormatted(64, "%02x", (unsigned char)(hash[i] & 0xff));
  }

  WDL_FastString pkfn(tcfn.Get());
  pkfn.Append(".pk");
  if (m_pkfr) { delete m_pkfr; m_pkfr = NULL; }
  m_pkfr = new WDL_FileRead(pkfn.Get());

  if (m_fi) { delete m_fi; m_fi = NULL; }
  RST_AntidoteInput *ap = new RST_AntidoteInput;
  if (ap->Open(tcfn.Get())) m_fi = ap;
  if (!m_fi || !m_pkfr->IsOpen()) m_is_open = false;
  else m_is_open = true;

  if (m_ps) { delete m_ps; m_ps = NULL; }
  m_ps = CreatePitchShift("SoundTouch");
  if (!m_ps) m_is_open = false;
  else m_is_open = true;

  m_bq.Empty();
  if (m_fi) m_duration = m_fi->GetLength();
  m_eof = false;

  RST_IFileTagger *ft = CreateFileTagger(fn);

  if (ft)
  {
    m_title.Set(ft->GetTitle());
    m_artist.Set(ft->GetArtist());
    delete ft;
  }

  return m_is_open;
}

bool RST_Track::IsOpen()
{
  return m_is_open;
}

bool RST_Track::IsDrained() const
{
  return m_eof && m_bq.GetSize() == 0;
}

bool RST_Track::WantMore() const
{
  if (!m_eof && m_fi)
  {
    // duration_ms = filesize in bytes /
    // (samplerate * num of channels * (bits per sample / eight))
    int duration = (int)(((double)m_bq.GetSize() / (m_fi->GetChannels() *
      GetHardwareSampleRate() * sizeof(SAM))) * 1000);

    if (duration < m_bq_max_duration) return true;
  }

  return false;
}

bool RST_Track::DoBuffering(int buffer_size)
{
  if (m_eof) return false;

  if (m_fi)
  {
    m_buffer.Resize(buffer_size);

    double start_time;
    int nsam;

    if (WDL_ApproximatelyEqual(m_shift, m_last_shift) &&
      WDL_ApproximatelyEqual(m_tempo, m_last_tempo))
    {
      start_time = m_fi->GetPosition();
      nsam = m_fi->GetSamples(m_buffer.Get(), m_buffer.GetSize());
    }
    else
    {
      m_last_shift = m_shift;
      m_last_tempo = m_tempo;

#if 1
      // Introduces lag equal the size
      // of m_bq_max_duration.

      start_time = m_fi->GetPosition();
#else
      // This option tries to eliminate
      // the lag by flushing the buffer
      // but Seek is not sample accurate
      // and as a result the transition
      // is not as smooth.

      start_time = m_bq.GetNextRunTime();

      if (DefinitelyGreaterThan(start_time, 0.0))
      {
        Seek(start_time);
      }
      else
      {
        start_time = m_fi->GetPosition();
      }
#endif

      nsam = m_fi->GetSamples(m_buffer.Get(), m_buffer.GetSize());
    }

    m_buffer.Resize(nsam, false);

    if (m_ebur128normalization)
    {
      //if ((m_ebur128mode & EBUR128_MODE_TRUE_PEAK) == EBUR128_MODE_TRUE_PEAK)
      //{
      //  for (int i = 0; i < m_buffer.GetSize(); i++)
      //  {
      //    m_buffer.Get()[i] *= DB2VAL(m_ebur128ref) / DB2VAL(m_ebur128truepeak);
      //  }
      //}
      //else if ((m_ebur128mode & EBUR128_MODE_SAMPLE_PEAK) == EBUR128_MODE_SAMPLE_PEAK)
      //{
      //  for (int i = 0; i < m_buffer.GetSize(); i++)
      //  {
      //    m_buffer.Get()[i] *= DB2VAL(m_ebur128ref) / DB2VAL(m_ebur128samplepeak);
      //  }
      //}
      //else if ((m_ebur128mode & EBUR128_MODE_LRA) == EBUR128_MODE_LRA)
      //{
      //  for (int i = 0; i < m_buffer.GetSize(); i++)
      //  {
      //    m_buffer.Get()[i] *= DB2VAL(m_ebur128ref) / DB2VAL(m_ebur128range);
      //  }
      //}
      //else if ((m_ebur128mode & EBUR128_MODE_I) == EBUR128_MODE_I)
      //{
      //  for (int i = 0; i < m_buffer.GetSize(); i++)
      //  {
      //    m_buffer.Get()[i] *= DB2VAL(m_ebur128gain);
      //  }
      //}

      if (!m_ebur128downwardonly)
      {
        for (int i = 0; i < m_buffer.GetSize(); i++)
        {
          m_buffer.Get()[i] *= DB2VAL(m_ebur128gain);
        }
      }
      else if (m_ebur128downwardonly && WDL_DefinitelyLessThan(m_ebur128gain, 0.0))
      {
        for (int i = 0; i < m_buffer.GetSize(); i++)
        {
          m_buffer.Get()[i] *= DB2VAL(m_ebur128gain);
        }
      }
    }

#if 0
    if (1) // m_rev
    {
      int nch = 2;

      for (int ch = 0; ch < nch; ch++)
      {
        for (int lo = ch, hi = m_buffer.GetSize() - nch + ch; lo < hi; lo += nch, hi -= nch)
        {
          SAM tmp = m_buffer.Get()[lo];
          m_buffer.Get()[lo] = m_buffer.Get()[hi];
          m_buffer.Get()[hi] = tmp;
        }
      }

      int bytes = m_buffer.GetSize() * sizeof(SAM);
      double fileinf = nch * GetHardwareSampleRate() * sizeof(SAM);

      double bms = bytes / fileinf / (1.0 / m_tempo); //bytes / fileinf / (1.0 / m_tempo);

      double rt = start_time - bms;
      m_fi->Seek(rt);

      //WDL_FastString ss;
      //ss.SetFormatted(111,
      //  "st: %f rt: %f bms: %f nsam: %d\n"
      //  "  real rt: %f\n"
      //  , start_time, rt, bms, nsam, m_fi->GetPosition());
      //OutputDebugString(ss.Get());
    }
#endif

    if (nsam > 0)
    {
      m_ps->set_srate(GetHardwareSampleRate());
      m_ps->set_nch(2);
      m_ps->set_shift(m_shift);
      m_ps->set_tempo(m_tempo);

      int spl = m_buffer.GetSize() / 2;

      SAM *sam = m_ps->GetBuffer(spl);
      memcpy(sam, m_buffer.Get(), m_buffer.GetSize() * sizeof(SAM));
      m_ps->BufferDone(spl);

      int want = (int)((nsam / 2) * (1.0 / m_tempo));
      if (want)
      {
        m_buffer.Resize(want * 2);
        int rd = m_ps->GetSamples(want, m_buffer.Get());
        if (rd) m_buffer.Resize(rd * 2);
      }
      m_bq.AddBlock(&m_buffer, start_time); return true;
    }
    else { m_eof = true; }
  }

  return false;
}

double RST_Track::GetDuration() const
{
  return m_duration;
}

double RST_Track::GetRunTime() const
{
  return m_run_time;
}

void RST_Track::UpdateRunTime(double time)
{
  m_run_time = time;
}

void RST_Track::SeekBackward()
{
  double st = GetRunTime();
  double dr = GetDuration();
  st -= m_seek_step;
  Seek(wdl_clamp(st, 0.0, dr));
}

void RST_Track::SeekForward()
{
  double st = GetRunTime();
  double dr = GetDuration();
  st += m_seek_step;
  Seek(wdl_clamp(st, 0.0, dr));
}

void RST_Track::Seek(double time)
{
  if (m_fi)
  {
    double st = time;
    double du = GetDuration();

    if (WDL_DefinitelyLessThan(st, 0.0)) { st = 0.0; m_eof = false; }
    else if (WDL_DefinitelyGreaterThan(st, du)) { st = du; m_eof = true; }
    else { m_eof = false; }

    m_bq.Flush();
    m_fi->Seek(st);
    m_run_time = st;
  }
}

const char *RST_Track::GetTitle() const
{
  return m_title.Get();
}

const char *RST_Track::GetArtist() const
{
  return m_artist.Get();
}

void RST_Track::Rewind(bool state)
{
  if (m_rwd == state) return; else m_rwd = state;
  if (m_rwd)
  {
    m_prevrev = m_fi->IsReverse();
    m_prevshift = m_shift;
    m_prevtempo = m_tempo;
    m_fi->SetReverse(true);
    m_shift = m_max_shift;
    m_tempo = m_max_tempo;
  }
  else
  {
    m_shift = m_prevshift;
    m_tempo = m_prevtempo;
    m_fi->SetReverse(m_prevrev);
  }
}

void RST_Track::FastForward(bool state)
{
  if (m_fwd == state) return; else m_fwd = state;
  if (m_fwd)
  {
    m_prevrev = m_fi->IsReverse();
    m_prevshift = m_shift;
    m_prevtempo = m_tempo;
    m_fi->SetReverse(false);
    m_shift = m_max_shift;
    m_tempo = m_max_tempo;
  }
  else
  {
    m_shift = m_prevshift;
    m_tempo = m_prevtempo;
    m_fi->SetReverse(m_prevrev);
  }
}

void RST_Track::DecreaseShift()
{
  m_shift = wdl_clamp(m_shift - m_shift_step, m_min_shift, m_max_shift);
}

void RST_Track::IncreaseShift()
{
  m_shift = wdl_clamp(m_shift + m_shift_step, m_min_shift, m_max_shift);
}

void RST_Track::DecreaseShiftSmall()
{
  m_shift = wdl_clamp(m_shift - m_shift_altstep, m_min_shift, m_max_shift);
}

void RST_Track::IncreaseShiftSmall()
{
  m_shift = wdl_clamp(m_shift + m_shift_altstep, m_min_shift, m_max_shift);
}

void RST_Track::DecreaseTempo()
{
  m_tempo = wdl_clamp(m_tempo - m_tempo_step, m_min_tempo, m_max_tempo);
}

void RST_Track::IncreaseTempo()
{
  m_tempo = wdl_clamp(m_tempo + m_tempo_step, m_min_tempo, m_max_tempo);
}

void RST_Track::DecreaseTempoSmall()
{
  m_tempo = wdl_clamp(m_tempo - m_tempo_altstep, m_min_tempo, m_max_tempo);
}

void RST_Track::IncreaseTempoSmall()
{
  m_tempo = wdl_clamp(m_tempo + m_tempo_altstep, m_min_tempo, m_max_tempo);
}

void RST_Track::ResetShift()
{
  m_shift = 1.0;
}

void RST_Track::ResetTempo()
{
  m_tempo = 1.0;
}

double RST_Track::GetVolume() const
{
  return m_volume;
}

void RST_Track::SetVolume(double decibel)
{
  m_volume = wdl_clamp(decibel, -150.0, 0.0);
}

double RST_Track::GetFader() const
{
  int sl = DB2SLIDER(m_fader);
  double per = (double)sl / 1000;
  per = wdl_clamp(per, 0.0, 1.0);
  return per;
}

double RST_Track::GetFaderDecibel() const
{
  return m_fader;
}

void RST_Track::SetFader(double percentage)
{
  double per = wdl_clamp(percentage, 0.0, 1.0);
  int sl = (int)(per * 1000);
  m_fader = SLIDER2DB(sl);
}

bool RST_Track::IsReverse() const
{
  return m_fi->IsReverse();
}

void RST_Track::ToggleReverse()
{
  bool st = !m_fi->IsReverse();
  m_fi->SetReverse(st);
}

LICE_IBitmap *RST_Track::GetWaveForm(HWND ctrl)
{
  if (IsOpen())
  {
    RECT r;
    GetWindowRect(ctrl, &r);
    const int spc = 2;
    int w = r.right - r.left;
    int h = r.bottom - r.top;
    int hh = (h / 2) - spc / 2;
    m_wf->resize(w, h);
    int nch = 2;

    WDL_INT64 pos = (WDL_INT64)(GetRunTime() * 100.0) *
      (WDL_INT64)(m_peakrate / 100) * (WDL_INT64)nch * (WDL_INT64)sizeof(float);
    pos = pos - ((w / 2) * nch * sizeof(float));
    int diff = 0;
    if (pos < 0)
    {
      diff = (int)wdl_abs(pos) / nch / (int)sizeof(float);
      pos = 0;
    }
    m_pkfr->SetPosition(pos);

    WDL_TypedBuf<float> buf;
    buf.Resize(w * nch);

    int read = m_pkfr->Read(buf.Get(), buf.GetSize() * sizeof(float));
    buf.Resize(read / sizeof(float), false);

    LICE_Clear(m_wf, LICE_RGBA(51, 51, 51, 255));

    int lineindex = 0 + diff;
    for (WDL_INT64 i = 0; i < buf.GetSize(); i += nch)
    {
      int lineheight = wdl_clamp((int)(buf.Get()[i] * hh), 0, hh);
      int y1 = (hh - lineheight) / 2;
      int y2 = y1 + lineheight;

      LICE_Line(m_wf, lineindex, y1, lineindex, y2,
        LICE_RGBA(55, 141, 247, 255));
      lineindex++; if (lineindex > w) break;
    }

    lineindex = 0 + diff;
    for (WDL_INT64 i = 1; i < buf.GetSize(); i += nch)
    {
      int lineheight = wdl_clamp((int)(buf.Get()[i] * hh), 0, hh);
      int y1 = (hh - lineheight) / 2;
      int y2 = y1 + lineheight;

      LICE_Line(m_wf, lineindex, y1 + hh + 1 + spc, lineindex,
        y2 + hh + 1 + spc, LICE_RGBA(55, 141, 247, 255));
      lineindex++; if (lineindex > w) break;
    }

    return m_wf;
  }

  return NULL;
}
