// Copyright (c) 2022 Giorgos Vougioukas
//
// The license can be found in the LICENSE file.

#if defined(_WIN32) && defined(_DEBUG)
#include <vld.h>
#endif

//#define ENABLE_THA_OPUS_CONSOLE

#include "thalia/thalia_plugin_public.h"
#include "thalia_opus/opus_input.h"
#include "thalia_opus/opus_apic.h"
#include "thalia_opus/opus_tag.h"
#include "thalia_opus/opus_entry_point.h"

#include "WDL/wdlcstring.h"

double (*THA_GetAudioDeviceSamplerate)();
void (*THA_GetDiskReadMode)(int *rmode, int *rbufsize, int *rnbufs);
void (*THA_GetResampleMode)(bool *interp, int *filtercnt,
  bool *sinc, int *sinc_size, int *sinc_interpsize);

static HINSTANCE opus_instance;
static HWND opus_main_hwnd;

//opus_int64 tha_filelength(void *user_data)
//{
//  WDL_FileRead *fr = (WDL_FileRead *)user_data;
//  return (opus_int64)fr->GetSize();
//}

int tha_seek(void *user_data, opus_int64 offset, int whence)
{
  WDL_FileRead *fr = (WDL_FileRead *)user_data;

  switch (whence)
  {
  case SEEK_SET:
    {
      fr->SetPosition((WDL_INT64)offset);
    } break;
  case SEEK_CUR:
    {
      fr->SetPosition(fr->GetPosition() + (WDL_INT64)offset);
    } break;
  case SEEK_END:
    {
      fr->SetPosition(fr->GetSize() + (WDL_INT64)offset);
    } break;
  default: break;
  }

  return 0;
}

int tha_read(void *user_data, unsigned char *ptr, int count)
{
  WDL_FileRead *fr = (WDL_FileRead *)user_data;

  if (fr->GetPosition() + (WDL_INT64)count > fr->GetSize())
  {
    count = (int)(fr->GetSize() - fr->GetPosition());
  }

  fr->Read((unsigned char *)ptr, (int)count);
  return count;
}

//opus_int64 tha_write(const void *ptr, opus_int64 count, void *user_data)
//{
//  WDL_FileWrite *fw = (WDL_FileWrite *)user_data;
//  fw->Write(ptr, (int)count);
//  return count;
//}

opus_int64 tha_tell(void *user_data)
{
  WDL_FileRead *fr = (WDL_FileRead *)user_data;
  return (opus_int64)fr->GetPosition();
}

THA_IFileInput *CreateFromType(const char *type)
{
  if (!strcmp(type, "OPUS"))
  {
    return new THA_OpusInput;
  }

  return NULL;
}

THA_IFileInput *CreateFromFile(const char *filename)
{
  if (!wdl_filename_cmp(WDL_get_fileext(filename), ".opus"))
  {
    THA_OpusInput *p = new THA_OpusInput;

    if (p->Open(filename))
    {
      return p;
    }

    delete p;
  }

  return NULL;
}

struct THA_FileInputRegister opus_input_reg =
{
  CreateFromType,
  CreateFromFile,
};

THA_IFileTag *CreateOpusTag(const char *filename)
{
  if (!wdl_filename_cmp(WDL_get_fileext(filename), ".opus"))
  {
    THA_OpusTag *p = new THA_OpusTag;

    if (p && p->Open(filename))
    {
      return p;
    }
    else
    {
      delete p;
    }
  }

  return NULL;
}

THA_FileTagRegister opus_tag_reg =
{
  &CreateOpusTag
};

THA_IFilePic *CreateOpusAPIC(const char *filename)
{
  if (!wdl_filename_cmp(WDL_get_fileext(filename), ".opus"))
  {
    THA_OpusAPIC *p = new THA_OpusAPIC;

    if (p->Open(filename))
    {
      return p;
    }
    else
    {
      delete p;
    }
  }

  return NULL;
}

THA_FilePicRegister opus_apic_reg =
{
  &CreateOpusAPIC
};

extern "C"
{
  THA_PLUGIN_EXPORT int THA_PLUGIN_ENTRYPOINT(
    THA_PLUGIN_HINSTANCE instance, THA_PluginInfo *rec)
  {
#if defined(_WIN32) && defined(_DEBUG) && defined(ENABLE_THA_OPUS_CONSOLE)
    // The first will use the console only if
    // the application is started by the command 
    // line. The second one will always enable
    // the console.
    //if (AttachConsole(ATTACH_PARENT_PROCESS))
    if (AttachConsole(ATTACH_PARENT_PROCESS) || AllocConsole())
    {
        freopen("CONOUT$", "w", stdout);
        freopen("CONOUT$", "w", stderr);
    }
#endif

    opus_instance = instance;

    if (rec)
    {
      if (rec->caller_version != THA_PLUGIN_VERSION || !rec->GetFunc)
      {
        return 0;
      }

      opus_main_hwnd = rec->hwnd_main;

      *((void **)&THA_GetAudioDeviceSamplerate) = rec->GetFunc("THA_GetAudioDeviceSamplerate");
      *((void **)&THA_GetDiskReadMode) = rec->GetFunc("THA_GetDiskReadMode");
      *((void **)&THA_GetResampleMode) = rec->GetFunc("THA_GetResampleMode");

      if (!rec->Register || !THA_GetAudioDeviceSamplerate || !THA_GetDiskReadMode ||
        !THA_GetResampleMode)
      {
        return 0;
      }

      rec->Register("input", &opus_input_reg);
      rec->Register("tag", &opus_tag_reg);
      //rec->Register("apic", &opus_apic_reg);

      return 1;
    }
    else
    {
      return 0;
    }
  }
}
